/***************************************************************************
 *   Copyright (C) 2004 by Ingo Stierand, Stierand-LinuxIt                 *
 *   linuxit@web.de                                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "kinstaller.h"

#include <qdragobject.h>
#include <qregexp.h>
#include <kprinter.h>
#include <qpainter.h>
#include <qpaintdevicemetrics.h>
#include <qurl.h>

#include <kapp.h>
#include <kglobal.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kdeversion.h>
#include <kmenubar.h>
#include <kmessagebox.h>
#include <kstatusbar.h>
#include <kkeydialog.h>
#include <kaccel.h>
#include <kio/netaccess.h>
#include <kfiledialog.h>
#include <kdirselectdialog.h>
#include <kconfig.h>
#include <kurl.h>
#include <kurldrag.h>
#include <kurlrequesterdlg.h>

#include <kedittoolbar.h>

#include <kstdaccel.h>
#include <kaction.h>
#include <kstdaction.h>

#include "kinstallerdefs.h"


#define KINSTALLER_LOG  "\nkinstaller: "


KInstaller::KInstaller()
    : KMainWindow( 0, "KInstaller" ),
      m_view(new KInstallerView(this)),
      m_printer(0)
{
    // accept dnd
    setAcceptDrops(false);

    // tell the KMainWindow that this is indeed the main widget
    setCentralWidget(m_view);

    // then, setup our actions
    setupActions();

    // and a status bar
    statusBar()->show();

    // apply the saved mainwindow settings, if any, and ask the mainwindow
    // to automatically save settings if changed: window size, toolbar
    // position, icon size, etc.
    setAutoSaveSettings();

    m_settings = new TInstallerSettings();
    m_pkgInfos = new TPackageInfoList();
    
    vRunning = false;
    vCanceled = false;
    
    m_firstRun = true;
    
    m_setupWizard = new KSetupWizard();
    m_prefDialog = new KInstallerPreferences();
    m_pkgInfoDialog = new KPkgInfoDialog(this);

    connect(m_setupWizard, SIGNAL(signalGuessKDEBaseClicked(QString &)),
            this,   SLOT(slotGuessKDEBaseClicked(QString &)));
    connect(m_setupWizard, SIGNAL(signalSelectStorageDirClicked(QString &)),
            this,   SLOT(slotSelectStorageDirClicked(QString &)));
    connect(m_prefDialog, SIGNAL(signalGuessKDEBaseClicked(QString &)),
            this,   SLOT(slotGuessKDEBaseClicked(QString &)));
    connect(m_prefDialog, SIGNAL(signalSelectStorageDirClicked(QString &)),
            this,   SLOT(slotSelectStorageDirClicked(QString &)));
    
    // allow the view to change the statusbar and caption
/*
    connect(m_view, SIGNAL(signalChangeStatusbar(const QString&)),
            this,   SLOT(changeStatusbar(const QString&)));
    connect(m_view, SIGNAL(signalChangeCaption(const QString&)),
            this,   SLOT(changeCaption(const QString&)));
*/
    connect(m_view, SIGNAL(signalSelectSourceClicked()),
            this,   SLOT(slotSelectSourceClicked()));
    connect(m_view, SIGNAL(signalInstallClicked()),
            this,   SLOT(slotInstallClicked()));
    connect(m_view, SIGNAL(signalDeleteClicked()),
            this,   SLOT(slotDeleteClicked()));
    connect(m_view, SIGNAL(signalUninstallClicked()),
            this,   SLOT(slotUninstallClicked()));

    m_guessProc = new KInstallerProcess(this);
    connect(m_guessProc,SIGNAL(signalProcessStdout(const QString& )),
            this,       SLOT(slotGuessProcessStdout(const QString& )));
    connect(m_guessProc,SIGNAL(signalProcessStderr(const QString& )),
            this,       SLOT(slotGuessProcessStdout(const QString& )));
    
    m_instProc = new KInstallerProcess(this);
    connect(m_instProc,SIGNAL(signalProcessStdout(const QString& )),
            this,    SLOT(slotInstallProcessStdout(const QString& )));
    connect(m_instProc,SIGNAL(signalProcessStderr(const QString& )),
            this,    SLOT(slotInstallProcessStderr(const QString& )));

    m_uninstProc = new KInstallerProcess(this);
    connect(m_uninstProc,SIGNAL(signalProcessStdout(const QString& )),
            this,    SLOT(slotUninstallProcessStdout(const QString& )));
    connect(m_uninstProc,SIGNAL(signalProcessStderr(const QString& )),
            this,    SLOT(slotUninstallProcessStderr(const QString& )));

    // reading the settings
    readGlobalConfig();
    collectPackages();

    // ... and applying them to the GUI
    m_view->applySettings(m_settings);
    m_view->updatePackageInfos(m_pkgInfos);

    if (m_firstRun)
      slotSettingsWizard();
}


KInstaller::~KInstaller()
{
}


void KInstaller::load(const KURL& url)
{
  if (!url.isEmpty())
  {
    m_view->setSourceURL(url.path());
  }
}


void KInstaller::setupActions()
{
//    KStdAction::openNew(this, SLOT(fileNew()), actionCollection());
    KStdAction::open(this, SLOT(fileOpen()), actionCollection());
//    KStdAction::save(this, SLOT(fileSave()), actionCollection());
//    KStdAction::saveAs(this, SLOT(fileSaveAs()), actionCollection());
//    KStdAction::print(this, SLOT(filePrint()), actionCollection());
    KStdAction::quit(kapp, SLOT(quit()), actionCollection());

//    m_toolbarAction = KStdAction::showToolbar(this, SLOT(optionsShowToolbar()), actionCollection());
//    m_statusbarAction = KStdAction::showStatusbar(this, SLOT(optionsShowStatusbar()), actionCollection());

    m_wizardAction = new KAction(i18n("Start setup wizard"), 0,
                                 this, SLOT(slotSettingsWizard()),
                                 actionCollection(), "settings_wizard");

//    KStdAction::keyBindings(this, SLOT(optionsConfigureKeys()), actionCollection());
//    KStdAction::configureToolbars(this, SLOT(optionsConfigureToolbars()), actionCollection());
    KStdAction::preferences(this, SLOT(optionsPreferences()), actionCollection());
    
    createGUI();
    toolBar()->hide();
}


void KInstaller::saveProperties(KConfig *config)
{
    // the 'config' object points to the session managed
    // config file.  anything you write here will be available
    // later when this app is restored

/*
      if (!m_view->currentURL().isEmpty()) {
#if KDE_IS_VERSION(3,1,3)
        config->writePathEntry("lastURL", m_view->currentURL());
#else
        config->writeEntry("lastURL", m_view->currentURL());
#endif
    }
  */
}


void KInstaller::readProperties(KConfig *config)
{
    // the 'config' object points to the session managed
    // config file.  this function is automatically called whenever
    // the app is being restored.  read in here whatever you wrote
    // in 'saveProperties'

//    QString url = config->readPathEntry("lastURL");

//    if (!url.isEmpty())
//        m_view->openURL(KURL(url));
}


#define CONFIG_GROUPVIEW        "KInstaller.View"
#define CONFIG_KEYWINDOWPOS     "windoPos"
#define CONFIG_KEYSOURCEURL     "sourceUrl"

/*!
 */
void KInstaller::readGlobalConfig()
{
  KConfig * config = KGlobal::config();
  
  
  // read options
  m_firstRun = true;

  // is the config file v0.1?
  if (!config->hasGroup(CONFIG_GROUPVIEW))
  {
    config->setGroup(CONFIG_GROUPGENERAL);
    m_view->setSourceURL(config->readEntry(CONFIG_KEYSOURCEURL,""));
  }
  else
  {
    config->setGroup(CONFIG_GROUPVIEW);
    if (config->hasKey(CONFIG_KEYWINDOWPOS))
    {
      move(config->readPointEntry(CONFIG_KEYWINDOWPOS));
      m_firstRun = false;
    }
    
    // source/dest options
    m_view->setSourceURL(config->readEntry(CONFIG_KEYSOURCEURL,""));
  }

  m_settings->readConfig(config);

  // we have to try to provide the user with a valid KDE base directory
  if (m_settings->KDEBase().isEmpty())
    m_settings->setKDEBase(guessKDEBase());
}


/*!
 */
void KInstaller::writeGlobalConfig()
{
  KConfig * config = KGlobal::config();
    
  
  config->setGroup(CONFIG_GROUPVIEW);
  // general settings
  config->writeEntry(CONFIG_KEYSOURCEURL,m_view->sourceURL());
  config->writeEntry(CONFIG_KEYWINDOWPOS,pos());

  // kinstaller settings
  m_settings->writeConfig(config);
  
  // We need to call 'sync' to write back all changes
  //  Maybe this is a bug in KDE? The config is only written
  //  when internal keys are changed.
  config->sync();
}


void KInstaller::dragEnterEvent(QDragEnterEvent *event)
{
    // accept uri drops only
    event->accept(KURLDrag::canDecode(event));
}


void KInstaller::dropEvent(QDropEvent *event)
{
    // this is a very simplistic implementation of a drop event.  we
    // will only accept a dropped URL.  the Qt dnd code can do *much*
    // much more, so please read the docs there
    KURL::List urls;

    // see if we can decode a URI.. if not, just ignore it
    if (KURLDrag::decode(event, urls) && !urls.isEmpty())
    {
        // okay, we have a URI.. process it
        const KURL &url = urls.first();

        // load in the file
        load(url);
    }
}


void KInstaller::fileNew()
{
    // this slot is called whenever the File->New menu is selected,
    // the New shortcut is pressed (usually CTRL+N) or the New toolbar
    // button is clicked

    // create a new window
    (new KInstaller)->show();
}


void KInstaller::fileOpen()
{
    // this slot is called whenever the File->Open menu is selected,
    // the Open shortcut is pressed (usually CTRL+O) or the Open toolbar
    // button is clicked
/*
    // this brings up the generic open dialog
    KURL url = KURLRequesterDlg::getURL(QString::null, this, i18n("Open Location") );
*/
  // standard filedialog
  QString dir;
  
  if (m_view->sourceURL().isEmpty())
    dir = QDir::current().absPath();
  else
    dir = m_view->sourceURL();
  
  KURL url = KFileDialog::getOpenURL(dir,
                                     "*.tar *.tar.gz *.tgz *.tar.bz2|"
                                     + i18n("tar packages")
                                     + " (.tar .tar.gz .tgz .tar.bz2)\n*|"
                                     + i18n("All files"),
                                     this, i18n("Select source package"));
  if (!url.isEmpty())
  {
    m_view->setSourceURL(url.path());
  }
}


void KInstaller::filePrint()
{
    // this slot is called whenever the File->Print menu is selected,
    // the Print shortcut is pressed (usually CTRL+P) or the Print toolbar
    // button is clicked
/*
    if (!m_printer) m_printer = new KPrinter;
    if (m_printer->setup(this))
    {
        // setup the printer.  with Qt, you always "print" to a
        // QPainter.. whether the output medium is a pixmap, a screen,
        // or paper
        QPainter p;
        p.begin(m_printer);

        // we let our view do the actual printing
        QPaintDeviceMetrics metrics(m_printer);
        m_view->print(&p, metrics.height(), metrics.width());

        // and send the result to the printer
        p.end();
    }
*/
}


void KInstaller::optionsShowToolbar()
{
    // this is all very cut and paste code for showing/hiding the
    // toolbar
    if (m_toolbarAction->isChecked())
        toolBar()->show();
    else
        toolBar()->hide();
}


void KInstaller::optionsShowStatusbar()
{
    // this is all very cut and paste code for showing/hiding the
    // statusbar
    if (m_statusbarAction->isChecked())
        statusBar()->show();
    else
        statusBar()->hide();
}


/*
void KInstaller::optionsConfigureKeys()
{
    KKeyDialog::configureKeys(actionCollection(), "kinstallerui.rc");
}
*/


void KInstaller::optionsConfigureToolbars()
{
    // use the standard toolbar editor
#if defined(KDE_MAKE_VERSION)
# if KDE_VERSION >= KDE_MAKE_VERSION(3,1,0)
    saveMainWindowSettings(KGlobal::config(), autoSaveGroup());
# else
    saveMainWindowSettings(KGlobal::config());
# endif
#else
    saveMainWindowSettings(KGlobal::config());
#endif
}


void KInstaller::newToolbarConfig()
{
    // this slot is called when user clicks "Ok" or "Apply" in the toolbar editor.
    // recreate our GUI, and re-apply the settings (e.g. "text under icons", etc.)
    createGUI();

#if defined(KDE_MAKE_VERSION)
# if KDE_VERSION >= KDE_MAKE_VERSION(3,1,0)
    applyMainWindowSettings(KGlobal::config(), autoSaveGroup());
# else
    applyMainWindowSettings(KGlobal::config());
# endif
#else
    applyMainWindowSettings(KGlobal::config());
#endif
}


void KInstaller::optionsPreferences()
{
  // popup some sort of preference dialog, here
  if (m_prefDialog->execute(m_settings))
  {
    writeGlobalConfig();
    m_view->applySettings(m_settings);
  }
}


/*!
 */
void KInstaller::slotSettingsWizard()
{
  if (m_setupWizard->execute(m_settings))
  {
    writeGlobalConfig();
    m_view->applySettings(m_settings);
  }
}


void KInstaller::changeStatusbar(const QString& text)
{
    // display the text on the statusbar
    statusBar()->message(text);
}


void KInstaller::changeCaption(const QString& text)
{
    // display the text on the caption
    setCaption(text);
}


/*!
 */
bool KInstaller::queryExit()
{
  writeGlobalConfig();
  
  return(!vRunning);
}


/*!
 */
void KInstaller::slotSelectSourceClicked()
{
  fileOpen();
}


/*!
 */
void KInstaller::slotSelectStorageDirClicked(QString & dir)
{
  QString tmp = dir;

  
  if (tmp.isEmpty())
    tmp = QDir::current().absPath();
  
  tmp = selectDirectory(dir,i18n("Select storage directory"));
  
  if (!tmp.isEmpty())
    dir = tmp;
}


/*!
 */
void KInstaller::slotGuessKDEBaseClicked(QString & dir)
{
  dir = guessKDEBase();

  if (dir.isEmpty())
  {
    showInfo(i18n("Sorry but KInstaller wasn't able to determine the KDE base"\
        "directory. Please enter the directory by hand."));
  }
}


/*!
 */
QString KInstaller::guessKDEBase()
{
  QStringList vBaseDirs;
  
  
  vBaseDirs.append("/opt/kde3");
  vBaseDirs.append("/usr");
  vBaseDirs.append("/usr/local");

  for (QStringList::Iterator it = vBaseDirs.begin(); it != vBaseDirs.end(); ++it)
  {
    if (testKDEBaseDir(*it))
    {
      return(*it);
    }
  }

  if (!m_guessProc->exec("kde-config","--prefix"))
    return("");
  else
    return(m_guessStr);
}


/*!
 */
bool KInstaller::testKDEBaseDir(const QString & path)
{
  QDir dir;
  QFileInfo fi;
  
  
  dir = QDir(path.stripWhiteSpace());
  fi.setFile(dir.absFilePath("bin/klipper"));
  qDebug("test for %s",fi.filePath().latin1());
  return(fi.exists() || fi.isSymLink());
}


/*!
 */
void KInstaller::collectPackages()
{
  QDir recDir = QDir::home().absFilePath(DEFAULT_PGKINFODIR);
  QStringList entries;
  QString name;
  TPackageInfo * info;


  entries = recDir.entryList("*" + QString(PKGINFO_EXT));
  for (QStringList::const_iterator it = entries.begin(); it != entries.end(); ++it)
  {
    name = *it;
    qDebug("found %s",recDir.absFilePath(name).latin1());

    info = new TPackageInfo();
    if (!info->readInfo(recDir.absFilePath(name)))
    {
    }

    m_pkgInfos->addInfo(info);
  }
}


/*!
 */
QString KInstaller::selectDirectory(const QString & startDir, const QString & caption)
{
  QString dir;
  
  // we invoking the standard KDE directory dialog to do this job
  dir = KFileDialog::getExistingDirectory(startDir,this,caption);
  return(dir);
}


/*!
 */
void KInstaller::showError(const QString & error, const QString & title)
{
  if (title != 0)
    KMessageBox::error(this,error,title);
  else
    KMessageBox::error(this,error,i18n("error"));
}


/*!
 */
void KInstaller::showInfo(const QString & msg, const QString & title)
{
  if (title != 0)
    KMessageBox::information(this,msg,i18n("info"));
  else
    KMessageBox::information(this,msg,title);
}


#define processInstallCancelRequest \
          if (vCanceled) {\
            if (installCleanup(tid,false))\
              m_view->appendInstallLog(KINSTALLER_LOG + i18n("operation successfully canceled."));\
            return(false);\
          }

/*!
 */
void KInstaller::slotInstallClicked()
{
  TInstallData * tid;
  bool ret;


  // when a previous action is canceled but the cancel isn't
  // already processed we abort here
  if (vCanceled)
  {
    return;
  }
  // when a previous action is in progress we cancel here
  else if (vRunning)
  {
    m_view->appendInstallLog(KINSTALLER_LOG + i18n("user abort"));
    // the install process is killed
    m_instProc->requestAbort();
    // set view state 
    m_view->setViewState(KInstallerView::InstallCanceled);
    vRunning = false; vCanceled = true;
    return;
  }
  // here we start a new action
  else
  {
    m_view->setViewState(KInstallerView::Install);
    vRunning = true; vCanceled = false;
    menuBar()->setEnabled(false);
  }

  tid = new TInstallData();

  ret = install(tid);

  delete tid;

  if (ret)
  {
    m_view->setSourceURL("");
    m_view->updatePackageInfos(m_pkgInfos);
  }

  vRunning = false; vCanceled = false;
  m_view->setViewState(KInstallerView::Ready);
  menuBar()->setEnabled(true);
}


/*!
 */
bool KInstaller::install(TInstallData * tid)
{
  // first the installation environment is checked.
  m_view->clearLogs();
  m_view->appendInstallLog(KINSTALLER_LOG + i18n("checking environment ..."));

  if (!installCheckEnvironment(tid))
  {
    return(false);
  }
  
  // a possible cancel request is processed here
  processInstallCancelRequest;
  
  m_view->appendInstallLog(KINSTALLER_LOG + i18n("prepare installation ..."));

  if (!installPrepare(tid))
  {
    installCleanup(tid,false);
    return(false);
  }
  
  // a possible cancel request is processed here
  processInstallCancelRequest;
  
  // untar source file to destination folder
  m_view->appendInstallLog(KINSTALLER_LOG + i18n("untar source file ..."));
  
  if (!installUntar(tid))
  {
    installCleanup(tid,false);
    return(false);
  }
  
  // a possible cancel request is processed here
  processInstallCancelRequest;
  
  // configure
  m_view->appendInstallLog(KINSTALLER_LOG + i18n("execute 'configure' ..."));

  if (!installConfigure(tid))
  {
    installCleanup(tid,false);
    return(false);
  }
  
  // a possible cancel request is processed here
  processInstallCancelRequest;
  
  // make
  m_view->appendInstallLog(KINSTALLER_LOG + i18n("execute 'make' ..."));
  
  if (!installMake(tid))
  {
    installCleanup(tid,false);
    return(false);
  }
  
  // a possible cancel request is processed here
  processInstallCancelRequest;
  
  // make install
  m_view->appendInstallLog(KINSTALLER_LOG + i18n("install ..."));
    
  if (!installInstall(tid))
  {
    installCleanup(tid,false);
    return(false);
  }
  
  // a possible cancel request is processed here
  processInstallCancelRequest;
  
  // cleanup
  m_view->appendInstallLog(KINSTALLER_LOG + i18n("cleaning up ..."));
  
  if (!installCleanup(tid,true))
  {
    return(false);
  }
    
  if (!installPost(tid))
  {
    return(false);
  }
  
  m_view->appendInstallLog(KINSTALLER_LOG + i18n("ready."));
  
  return(true);
}


/*!
 */
bool KInstaller::installCheckEnvironment(TInstallData * tid)
{
  QUrl url;
  QFileInfo fi;
  QString storagedir;
  QDir dir;


  // first we set the flag that the installation directory isn't created yet
  tid->installDirCreated = false;
  
  // checking if source file is valid
  url = m_view->sourceURL();
  if (!url.isValid())
  {
    m_view->appendInstallLog(KINSTALLER_LOG + i18n("source file isn't valid."));
    showError(
        i18n("Sorry, but the source file you selected doesn't seem to exist.")
        + "\n"
        + i18n("Please select a correct file."));
    return(false);
  }
  
  if (url.isLocalFile())
  {
    // first we have a look if the source file is ok
    fi.setFile(m_view->sourceURL());
    if (!fi.exists())
    {
      m_view->appendInstallLog(KINSTALLER_LOG + i18n("source file doesn't exist."));
      showError(
        i18n("Sorry, but the source file you selected doesn't seem to exist.")
        + "\n"
        + i18n("Please select a correct file."));
      return(false);
    }
  }

  // checking if storage directory is correct
  storagedir = m_settings->storageDir();
  dir = QDir(storagedir);
  if (storagedir.isEmpty() || !dir.exists())
  {
    m_view->appendInstallLog(KINSTALLER_LOG + i18n("storage directory doesn't exist."));
    showError(
        i18n("Sorry, but the storage directory you selected doesn't seem to exist.")
        + "\n"
        + i18n("Please choose another storage directory. ")
        + i18n("For changing the storage directory you can use the 'Settings' menu."),
        i18n("Write error"));
    return(false);
  }
  fi.setFile(dir.absPath());
  if (!fi.isWritable())
  {
    m_view->appendInstallLog(KINSTALLER_LOG + i18n("storage directory isn't writable."));
    showError(
        i18n("Sorry, but the storage directory you selected isn't writable.")
        + "\n"
        + i18n("Please check for correct write access or choose another storage directory. ")
        + i18n("For changing the storage directory you can use the 'Settings' menu."),
        i18n("Write error"));
    return(false);
  }

  // test for valid KDE base directory  
  if (!testKDEBaseDir(m_settings->KDEBase()))
  {
    m_view->appendInstallLog(KINSTALLER_LOG + i18n("KDE base directory isn't correct."));
    showError(i18n("Sorry, but the KDE base directory you selected isn't correct.\n"\
        "Please select the correct KDE base."));
    return(false);
  }
  
  return(true);
}


/*!
 */
bool KInstaller::installPrepareSplitName(const QString & name,
                                 QString * base, QString * ver, QString *rel)
{
  QRegExp baseExp("(.*)(\\.tar|\\.tar\\.gz|\\.tgz|\\.tar.bz2)");
  QRegExp verExp("[0-9][0-9\\.]*-?[a-zA-Z]*[0-9]?");
  QRegExp relExp("[0-9]+");
  int pos;
  QString tmp;
  

  // first we try to extract the extension from the name
  pos = baseExp.search(name);
  if (pos > -1)
  {
    tmp = baseExp.cap(1);
  }
  else
  {
    *base = name;
    *ver = "";
    *rel = "";

    return(false);
  }

  // By counting from the right we try to find the version and
  // release part of the file
  if (tmp.section('-',-2,-2).isEmpty())
  {
    *base = tmp;
    *ver = "";
    *rel = "";
  }
  else if (tmp.section('-',-3,-3).isEmpty())
  {
    *base = tmp.section('-',-2,-2);
    *ver = tmp.section('-',-1,-1);
    *rel = "";
  }
  else
  {
    *base = tmp.section('-',0,-3);
    *ver = tmp.section('-',-2,-2);
    *rel = tmp.section('-',-1,-1);
  }

  // We take a look if the release is a number
  if (!relExp.exactMatch(*rel))
  {
    qDebug("%s doesn't match release",rel->latin1());
    // If this is not the case we test if version is valid.
    // If not the base is appended with the version string
    if (!verExp.exactMatch(*ver))
    {
      if (!ver->isEmpty())
        *base = *base + '-' + *ver;
      
      *ver = *rel;
    }
    else
    {
      if (!rel->isEmpty())
        *ver = *ver + '-' + *rel;
    }
    
    *rel = "";
  }
  
  if (!verExp.exactMatch(*ver))
  {
    qDebug("%s doesn't match version",ver->latin1());
    if (!ver->isEmpty())
      *base = *base + '-' + *ver;
    
    if (relExp.exactMatch(*rel))
      *ver = *rel;
    else
      *ver = "";
    
    *rel = "";
  }
  
  return(true);
}


/*!
 */
bool KInstaller::installPrepare(TInstallData * tid)
{
  QString pkgName, pkgVersion, pkgRelease;
  QUrl srcurl;
  QDir dir;
  QFileInfo sf,pf;
  QString tmp;

  
  srcurl = m_view->sourceURL();
  sf.setFile(srcurl.path());
  dir = QDir(m_settings->storageDir());
  pf.setFile(dir.absFilePath(sf.fileName()));

  // here we set the storage directory and package file
  tid->packageDir = dir;
  tid->packageFile = pf.absFilePath();
  
  // the package name and version is calculated
  installPrepareSplitName(sf.fileName(),&pkgName,&pkgVersion,&pkgRelease);
  
  if (!m_pkgInfoDialog->execute(&pkgName,&pkgVersion,&pkgRelease))
  {
    m_view->appendInstallLog(KINSTALLER_LOG + i18n("user abort"));
    return(false);
  }

  tid->packageName = pkgName;
  tid->packageVersion = pkgVersion;
  if (pkgRelease.isEmpty())
    tid->packageRelease = "0";
  else
    tid->packageRelease = pkgRelease;
  
  tid->packageFileBase = pkgName + "-" + pkgVersion + "-" + pkgRelease;
    
  // If the package should be stored after install we test
  // if the source package already exists within the package directory
  if (m_settings->keepPackage())
  {
    if (pf.exists())
    {
      if (KMessageBox::questionYesNo(this,
          i18n("There exists already a tar package")
              + "'" + pf.fileName() + "'"
              + i18n("at the storage directory.")
              + "\n"
              + i18n("Do you want to install it anyway")
              + " "
              + i18n("(the old one will be overwritten)?"),
          i18n("Package already exists")) != KMessageBox::Yes)
      {
        return(false);
      }
    }

    tid->packageKeep = true;
  }
  else
  {
    tid->packageKeep = false;
  }
  
  // setting up the source file etc. and download it if needed
  if (srcurl.isLocalFile())
  {
    tid->sourceFile = srcurl;
    // we don't delete this file on failure
    tid->packageDeleteOnFail = false;
  }
  else
  {
    // the local source file is calculated which is temporary
    dir = QDir("/tmp");
    tid->sourceFile = dir.absFilePath(PACKAGENAMEBASE + sf.fileName());
    // we delete this file on failure
    tid->packageDeleteOnFail = true;
    // downloading ...
  }

  // at last we create an temporary directory for the install process
  tid->dateTime = QDateTime::currentDateTime();
  tid->dateLabel = tid->dateTime.toString("yyyyMMddhhmmss");
  tid->installDirName = tid->packageFileBase + QString("-%1").arg(tid->dateLabel);
  dir = dir.absFilePath(tid->installDirName);
  if (!dir.mkdir(dir.absPath()))
  {
    m_view->appendInstallLog(KINSTALLER_LOG + i18n("cannot create directory"));
    showError(
        i18n("Sorry, but the compilation directory")
        + "'" + dir.absPath() + "'"
        + i18n("couldn't be created.")
        + "\n"
        + i18n("Please check for correct write access to the storage directory."));
        return(false);
  }

  tid->installDir = dir;
  // we set the flag that the installation directory is created yet
  tid->installDirCreated = true;

  if (m_settings->useConfigureOptions())
    tid->configureOpts = m_settings->configureOptions();
  else
    tid->configureOpts = "";

  tid->destDir = QDir(m_settings->KDEBase());

  tid->makeDirKeep = m_settings->keepInstallfiles();
  
  return(true);
}


/*!
 */
bool KInstaller::installUntar(TInstallData * tid)
{
  QString filename;
  QString flags;
  QStringList entries;
  int numEntries = 0;
  

  filename = tid->sourceFile.path();
  
  // here we test for proper compress flag
  if (filename.endsWith("tar"))
    flags = "-x";
  else if (filename.endsWith("tar.gz"))
    flags = "-xz";
  else if (filename.endsWith("tgz"))
    flags = "-xz";
  else if (filename.endsWith("tar.bz2"))
    flags = "-xj";

  // yet the file is untar'ed
  if (!m_instProc->exec("tar",flags,"-C",tid->installDir.absPath(),"-f",filename))
  {
    m_view->appendInstallLog(KINSTALLER_LOG + i18n("error while executing tar"));
    showError(
        i18n("Sorry, but unarchiving the source file has failed.")
        + "\n"
        + i18n("Please consult the log entries for further informations."),
        i18n("install error"));
    return(false);
  }

  // now we are searching for the make directory
  entries = tid->installDir.entryList("*");
  for (QStringList::const_iterator it = entries.begin(); it != entries.end(); ++it)
  {
    if (*it != QString(".") && *it != QString(".."))
    {
//      qDebug(i18n("found sub directory " + *it).ascii());
      ++numEntries;
      if (numEntries == 1)
      {
        tid->makeDirName = *it;
        tid->makeDir = tid->installDir.absFilePath(tid->makeDirName);
      }
      else
      {
        m_view->appendInstallLog(KINSTALLER_LOG + i18n("tar file contains more than one directory"));
        showError(i18n("The tar package contains no or more than one root directory.\n"\
            "Sorry, but KInstaller cannot handle such packages."),
            i18n("No or multiple root directories"));
        return(false);
      }        
    }
  }
  
  return(true);
}


/*!
 */
bool KInstaller::installConfigure(TInstallData * tid)
{
  QString arg1 = tid->makeDir.absPath();
  QString arg2 = tid->destDir.absPath();
  QString arg3 = tid->configureOpts;
  QDir dir;
  QFileInfo fi;
  
  
  fi.setFile(tid->makeDir.absFilePath("configure"));
  if (!fi.exists())
  {
    m_view->appendInstallLog(KINSTALLER_LOG + i18n("configure not found"));
    showError(
        i18n("Sorry, but the file 'configure' doesn't exist.")
        + "\n"
        + i18n("Is this really a tar package of a program?"),
    i18n("install error"));
    return(false);
  }
  
  if (!fi.isExecutable())
  {
    m_view->appendInstallLog(KINSTALLER_LOG + i18n("configure not accessible"));
    showError(
        i18n("Sorry, but the file 'configure' doesn't seem to be executable.")
        + "\n"
        + i18n("This normally means that the package you want to install")
        + " "
        + i18n("is not a program package you can install with KInstaller."),
    i18n("install error"));
    return(false);
  }
  
  if (!m_instProc->exec(
       QString("cd %1; ./configure --prefix=%2 %3").arg(arg1).arg(arg2).arg(arg3)))
  {
    m_view->appendInstallLog(KINSTALLER_LOG + i18n("error while executing configure"));
    showError(
        i18n("Sorry, but executing the command 'configure' has failed.")
        + "\n"
        + i18n("Please consult the log entries for further informations."),
        i18n("install error"));
    return(false);
  }

  return(true);
}


/*!
 */
bool KInstaller::installMake(TInstallData * tid)
{
  QString arg1 = tid->makeDir.absPath();

  
  if (!m_instProc->exec(QString("cd %1; make").arg(arg1)))
  {
    m_view->appendInstallLog(KINSTALLER_LOG + i18n("error while executing make"));
    showError(
        i18n("Sorry, but executing the command 'make' has failed.")
        + "\n"
        + i18n("Please consult the log entries for further informations."),
        i18n("install error"));

    return(false);
  }

  return(true);
}


/*!
 */
bool KInstaller::installInstall(TInstallData * tid)
{
  QString arg1 = tid->makeDir.absPath();

  
  if (!m_instProc->exec("kdesu","-t",QString("cd %1; make install").arg(arg1)))
  {
    m_view->appendInstallLog(KINSTALLER_LOG + i18n("error while installing"));
    showError(
        i18n("Sorry, but executing the command 'make install' has failed.")
        + "\n"
        + i18n("Please consult the log entries for further informations."),
        i18n("install error"));
    return(false);
  }

  return(true);
}


/*!
 */
bool KInstaller::installCleanup(TInstallData * tid, bool success)
{
  QString arg1,arg2;

  
  if (success && tid->makeDirKeep)
  {
    // we do nothing;)
/*    // we move the make directory into the storage directory if needed
    arg1 = tid->makeDir.absPath();
    arg2 = tid->packageDir.absPath() + "/" + tid->makeDirName + "-" + tid->dateLabel;
    
    if (!m_instProc->exec("mv","-f",arg1,arg2))
    {
      m_view->appendLog(i18n("error while moving make directory"));
      return(false);
    }*/
  }

  if (!success && tid->installDirCreated)
  {
    // we delete the package source directory
    if (!m_instProc->exec("rm","-rf",tid->installDir.absPath()))
    {
      m_view->appendInstallLog(KINSTALLER_LOG + i18n("error while cleaning up"));
      showInfo(i18n("During installation the directory")
          + "\n" + tid->installDir.absPath() + "\n"
          + i18n("has been created. But now KInstall wasn't able to remove it.")
          + "\n"
          + i18n("Please remove it by hand.")
          + i18n("Your software should be installed correct anyway."));
  
      // @todo a failed install cleanup processing is not allowed to treat the whole process failed
//      return(false);
    }
  }
    
  // we move the make package into the storage directory if needed
  if (success && tid->packageKeep)
  {
    arg1 = tid->sourceFile.path();
    arg2 = tid->packageFile.path();

    if (!m_instProc->exec("mv","-f",arg1,arg2))
    {
      m_view->appendInstallLog(KINSTALLER_LOG + i18n("error while moving tar package to storage directory"));
      showInfo(i18n("KInstall wasn't able to move the tar package\n"\
          "to the storage directory.")
          + "\n\n"
          + i18n("This only means that you will not find it there for later use. ")
          + i18n("Your software should be installed correct anyway."));
      // @todo a failed install cleanup processing is not allowed to treat the whole process failed
//      return(false);
    }
  }
  
  return(true);
}


/*!
 */
bool KInstaller::installPost(TInstallData * tid)
{
  QDir recDir = QDir::home().absFilePath(DEFAULT_PGKINFODIR);
  QString recFileName = recDir.absFilePath(tid->installDirName + PKGINFO_EXT);
  TPackageInfo * info = new TPackageInfo();

  if (!recDir.exists())
    recDir.mkdir(recDir.absPath());

  info->dateTime = tid->dateTime;
  info->pkgFile = tid->packageFile.path();
  info->pkgName = tid->packageName;
  info->pkgVersion = tid->packageVersion;
  info->pkgRelease = tid->packageRelease;
  info->installType = "make";
  info->installDir = tid->installDir.absPath();
  info->makeDir = tid->makeDir.absPath();
  
  if (!info->writeInfo(recFileName))
  {
    m_view->appendInstallLog(KINSTALLER_LOG + i18n("error while creating package info"));
    showError(
        i18n("KInstall wasn't able to create a package info file.")
        + "\n\n"
        + i18n("Sorry, this means that you will not find the package "\
          "at the uninstall page, and so you are not able to uninstall "\
          "it later using KInstaller.")
        + i18n("Your software should be installed correct anyway."));
    // @todo a failed install post processing is not allowed to treat the whole process failed
//    return(false);
  }

  m_pkgInfos->addInfo(info);
  
  return(true);
}



#define processUninstallCancelRequest \
        if (vCanceled) {\
          m_uninstProc->requestAbort();\
          if (uninstallCleanup(info,false))\
            m_view->appendUninstallLog(KINSTALLER_LOG + i18n("operation successfully canceled."));\
          return(true);\
}


/*!
 */
void KInstaller::slotDeleteClicked()
{
  if (KMessageBox::questionYesNo(this,
      i18n("A deleted package cannot be uninstalled by KInstaller anymore. ")
      + i18n("Please keep in mind that you probably want to uninstall "\
             "the program later.")
      + "\n"
      + i18n("Are you really sure you want to delete the package?")
                                ) == KMessageBox::Yes)
    slotUninstall(true);
}


/*!
 */
void KInstaller::slotUninstallClicked()
{
  slotUninstall(false);
}


/*!
 */
void KInstaller::slotUninstall(bool deleteOnly)
{
  TPackageInfo * info;
  int ret;
  

  // when a previous action is canceled but the cancel isn't
  // already processed we abort here
  if (vCanceled)
  {
    return;
  }
  // when a previous action is in progress we cancel here
  else if (vRunning)
  {
    // the uninstall process is killed
    m_uninstProc->requestAbort();
    
    m_view->appendUninstallLog(KINSTALLER_LOG + i18n("user abort"));
    m_view->setViewState(KInstallerView::UninstallCanceled);
    vRunning = false; vCanceled = true;
    return;
  }
  // here we start a new action
  else
  {
//    if (deleteOnly)
    m_view->setViewState(KInstallerView::Uninstall);
    menuBar()->setEnabled(false);
    vRunning = true; vCanceled = false;
  }

  info = m_view->selectedPackageInfo();
  if (info == 0)
  {
    m_view->appendUninstallLog(KINSTALLER_LOG + i18n("no package selected"));
    showError(i18n("There is no package selected for uninstallation.\n"\
        "Please first select a package you want to uninstall."),
    i18n("no package selected"));
    return;
  }
  
  m_view->clearLogs();

  if (!uninstall(info,deleteOnly))
  {
    // This is removed because it may confuse the user.
/*    ret = KMessageBox::questionYesNo(this,
              i18n("The uninstallation has failed. This probably means that you "\
                  "cannot use KInstaller anymore to uninstall it. Do you want to delete it from "\
                  "the package list?"),
              i18n("uninstall failed"));
    if (ret == KMessageBox::Yes)
      uninstallPost(info);*/
  }

  // update the package list view and reactivate the GUI
  m_view->updatePackageInfos(m_pkgInfos);

  vRunning = false; vCanceled = false;
  m_view->setViewState(KInstallerView::Ready);
  menuBar()->setEnabled(true);
}


/*!
 */
bool KInstaller::uninstall(TPackageInfo * info, bool deleteOnly)
{
  m_view->appendUninstallLog(KINSTALLER_LOG + i18n("checking environment ..."));

  if (!uninstallCheckEnvironment(info))
  {
//    uninstallCleanup(info,false);
    return(false);
  }
  
  // a possible cancel request is processed here
  processUninstallCancelRequest;

  if (!deleteOnly)
  {
    // make uninstall
    m_view->appendUninstallLog(KINSTALLER_LOG + i18n("uninstall ..."));
      
    if (!uninstallUninstall(info))
    {
//      uninstallCleanup(info,false);
      return(false);
    }
    
    // a possible cancel request is processed here
    processUninstallCancelRequest;
  }
  
  // cleanup
  m_view->appendUninstallLog(KINSTALLER_LOG + i18n("cleaning up ..."));
  
  if (!uninstallCleanup(info,true))
  {
    return(false);
  }

  // post uninstall
  if (!uninstallPost(info))
  {
    return(false);
  }
  
  m_view->appendUninstallLog(KINSTALLER_LOG + i18n("ready."));

  return(true);
}


/*!
 */
bool KInstaller::uninstallCheckEnvironment(TPackageInfo * info)
{
  QDir dir;

  dir = info->installDir;
  if (!dir.exists())
  {
    m_view->appendUninstallLog(KINSTALLER_LOG + i18n("make directory not found"));
    showError(i18n("The compilation files should be found within the directory\n")
        + "'" + info->installDir + "'\n"\
        + i18n("But this directory doesn't seem to exist anymore.\n")
        + i18n("Sorry, this means that you cannot uninstall the package using KInstaller. ")
        + i18n("May be you want to untar the package again and call 'configure; make uninstall' by hand."),
        i18n("directory not found"));
    return(false);
  }
  
  dir = info->makeDir;
  if (!dir.exists())
  {
    m_view->appendUninstallLog(KINSTALLER_LOG + i18n("make directory not found"));
    showError(i18n("The compilation files should be found within the directory\n")
        + "'" + info->makeDir + "'\n"\
        + i18n("But this directory doesn't seem to exist anymore.\n")
        + i18n("Sorry, this means that you cannot uninstall the package using KInstaller. ")
        + i18n("May be you want to untar the package again and call 'configure; make uninstall' by hand."),
        i18n("directory not found"));
    return(false);
  }

  // @todo if the install dir is not found provide a way to select another one
  
  return(true);
}


/*!
 */
bool KInstaller::uninstallUninstall(TPackageInfo * info)
{
  QString arg1 = info->makeDir;

  
  if (!m_uninstProc->exec("kdesu","-t",QString("cd %1; make uninstall").arg(arg1)))
  {
    m_view->appendUninstallLog(KINSTALLER_LOG + i18n("error while uninstalling"));
    showError(
        i18n("Sorry, but executing the command 'make uninstall' has failed.")
        + "\n"
        + i18n("Please consult the log entries for further informations."));

    return(false);
  }

  return(true);
}


/*!
 */
bool KInstaller::uninstallCleanup(TPackageInfo * info, bool success)
{
  QString arg1 = info->installDir;

  
  if (success)
  {
    // we delete the package source directory
    if (!m_uninstProc->exec("rm","-rf",arg1))
    {
      m_view->appendUninstallLog(KINSTALLER_LOG + i18n("error while cleaning up"));
      showInfo(i18n("During uninstallation the directory")
          + "\n" + arg1 + "\n"
          + i18n("should be removed. But this failed for some reason.")
          + "\n"
          + i18n("Please remove it by hand."));

      // @todo a failed cleanup is not allowed to treat the whole process failed
//      return(false);
    }
  }
    
  return(true);
}


/*!
 */
bool KInstaller::uninstallPost(TPackageInfo * info)
{
  QString arg1 = info->filename();

  
  // we delete the package source directory
  if (!m_uninstProc->exec("rm","-f",arg1))
  {
    m_view->appendUninstallLog(KINSTALLER_LOG + i18n("error while cleaning up"));
    showInfo(i18n("During uninstallation the package info file")
        + "\n" + arg1 + "\n"
        + i18n("couldn't be removed.")
        + "\n"
        + i18n("Please remove it by hand."));

    // @todo a failed uninstall post processing is not allowed to treat the whole process failed
//    return(false);
  }
    
  m_pkgInfos->deleteInfo(info);
  
  return(true);
}


/*!
 */
void KInstaller::slotGuessProcessStdout(const QString & str)
{
  m_guessStr = str.stripWhiteSpace();
  if (!m_guessStr.startsWith("/"))
    m_guessStr = "";
}


/*!
 */
void KInstaller::slotInstallProcessStdout(const QString & str)
{
  m_view->appendInstallLog(str);
}


/*!
 */
void KInstaller::slotInstallProcessStderr(const QString & str)
{
  m_view->appendInstallLog(str);
}
/*!
 */
void KInstaller::slotUninstallProcessStdout(const QString & str)
{
  m_view->appendUninstallLog(str);
}


/*!
 */
void KInstaller::slotUninstallProcessStderr(const QString & str)
{
  m_view->appendUninstallLog(str);
}


#include "kinstaller.moc"
