/***************************************************************************
 *
 * knetworkmanager-nminfo.cpp - A NetworkManager frontend for KDE 
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Timo Hoenig        <thoenig@suse.de>, <thoenig@nouse.net>
 *         Valentine Sinitsyn <e_val@inbox.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include <kdebug.h>
#include <klocale.h>

#include "knetworkmanager.h"
#include "knetworkmanager-nminfo_dbus.h"
#include "knetworkmanager-dialogfab.h"
#include "knetworkmanager-encryption.h"
#include "knetworkmanager-storage.h"
#include "knetworkmanager-synchronizer.h"
#include "knetworkmanager-vpn.h"

#include "knetworkmanager-nminfo.h"

PassphraseRequest::PassphraseRequest(KNetworkManager* ctx, QString obj_path, QString net_path, QString essid, bool new_key, DBusMessage* msg)
	: QObject(ctx)
{
	_ctx = ctx;
	_obj_path = obj_path;
	_net_path = net_path;
	_essid = essid;
	_new_key = new_key;
	_msg = msg;
	_canceled = false;
	_dlg = NULL;
}

PassphraseRequest::~PassphraseRequest()
{

}

void 
PassphraseRequest::slotKeyEntered(Network* /*net*/)
{
	DeviceStore *store = _ctx->getDeviceStore ();
	Device *dev = store->getDevice (_obj_path);
	Network* net = dev->getNetwork(_net_path);

	_ctx->getNetworkManagerInfo()->sendPassphrase(net, _msg);
}

void
PassphraseRequest::slotCancelRequest()
{
	_canceled = true;
	_ctx->getNetworkManagerInfo()->sendPassphraseError(_msg);
}

void
PassphraseRequest::acquireKeyFromDialog()
{
	QString msg = QString::null;
	if (_new_key)
		msg = i18n("The connection could not be established. Please verify your settings and try again.");
	_dlg = new AcquirePasswordDialog (_ctx->getTray (), "PassphraseDialog", true, 0, _ctx, _obj_path, _net_path, _essid, msg);
	connect(_dlg, SIGNAL(sendPassphrase(Network*)), this, SLOT(slotKeyEntered(Network*)));
	connect(_dlg, SIGNAL(cancelClicked()), this, SLOT(slotCancelRequest()));
	_dlg->show ();
}

void
PassphraseRequest::slotKeyRestored(bool result, bool canceled)
{
	DeviceStore *store = _ctx->getDeviceStore ();
	Device *dev = store->getDevice (_obj_path);
	Network* net = dev->getNetwork(_net_path);
	bool handled = false;

	if (_canceled)
		return;

	// if the user decided to use an other device throw the retrieved key away
	if (canceled)
	{
		// fake a canceled activation stage and send a reply to NM
		dev->setActivationStage(NM_ACT_STAGE_CANCELLED);
		this->slotCancelRequest();
		return;
	}

	// if the key was successful restored we can proceed
	if (result)
	{
		Encryption *enc = net->getEncryption ();
		// check if encryption is now valid
		if (enc->isValid(_essid))
		{
			// ok, lets send the new key
			handled = true;
			_ctx->getNetworkManagerInfo()->sendPassphrase(net, _msg);
		}
	}

	// not handled -> ask the user
	if (!handled)
	{
		acquireKeyFromDialog();
	}
}

void
PassphraseRequest::request()
{
	bool handled = false;
	DeviceStore *store = _ctx->getDeviceStore ();
	Device *dev = store->getDevice (_obj_path);
	Network* net = dev->getNetwork(_net_path);

	/* Steps for obtaining the key:
	 *  First, look in the Encryption object - maybe it was already loaded
	 *  Second, look in the wallet - maybe it is here
	 *  Third, ask the user
	 */
	if (!_new_key) {
		Encryption *enc = net->getEncryption ();
		if (enc->isValid (_essid) ) {
			// the encryption is valid -> send it immediatly
			handled = true;
			_ctx->getNetworkManagerInfo()->sendPassphrase(net, _msg);
		}
		else if (enc->hasStoredKey()) {
			// the key is stored in the wallet -> try to restore and wait for restoration
			connect(enc, SIGNAL(keyRestored(bool, bool)), this, SLOT(slotKeyRestored(bool, bool)));
			enc->restoreKeyAsync();
			handled = true;
		}
	}

	if (!handled) {
		// the encryption is not valid and no key exists in kwallet -> ask the user
		acquireKeyFromDialog();
	}
}

void
NetworkManagerInfo::userInteraction (void)
{
	NetworkManagerInfoDBus::userInteraction ();
}

void
NetworkManagerInfo::sendPassphrase (Network* net, DBusMessage* msg)
{
	NetworkManagerInfoDBus::sendKeyForNetwork (net, msg);
}

void
NetworkManagerInfo::sendPassphraseError (DBusMessage* msg)
{
	NetworkManagerInfoDBus::sendGetKeyError (msg);
}

void
NetworkManagerInfo::acquirePassphrase (QString obj_path, QString net_path, QString essid, bool new_key, DBusMessage* msg)
{
	kdDebug () << k_funcinfo << " fork ahead: user or storage" << endl;

	DeviceStore *store = _ctx->getDeviceStore ();
	Device *dev = store->getDevice (obj_path);
	Synchronizer sync(dev);
	sync.setSources(Synchronizer::Storage | Synchronizer::New);
	sync.synchronize(essid, net_path);

	/*
	  drop the current pending passphrase request without sending an answer to NM 
	  because NM cant handle multiple passphrase replies for now.
	*/
	if (_currentRequest)
		delete _currentRequest;	

	_currentRequest = new PassphraseRequest(_ctx, obj_path, net_path, essid, new_key, msg);
	_currentRequest->request();
}

QStringList
NetworkManagerInfo::getNetworks ()
{
 	return KNetworkManagerStorage::getInstance ()->networks ();
}

Network*
NetworkManagerInfo::getNetworkProperties (const QString & essid)
{
 	return KNetworkManagerStorage::getInstance ()->networkProperties (essid);
}

void 
NetworkManagerInfo::emitNetworkUpdated (Network* net, bool automatic)
{
	emit networkUpdated (net, automatic);
}

VPNConnection*
NetworkManagerInfo::getVPNConnection (const QString & name)
{
	VPN*           vpn           = _ctx->getVPN ();
	VPNConnection* vpnConnection = NULL;

	if (vpn && vpn->isAvailable ()) {
		VPNList* vpnList = vpn->getVPNList ();

		for (VPNList::iterator i = vpnList->begin (); i != vpnList->end (); ++i) {
			if ((*i)->getName () == name)
				vpnConnection = *i;
		}
	}

	return vpnConnection;
}

QStringList
NetworkManagerInfo::getVPNConnectionNames ()
{
	VPN*        vpn = _ctx->getVPN ();
	QStringList vpnConnectionNames;

	if (vpn && vpn->isAvailable ()) {
		VPNList* vpnList = vpn->getVPNList ();
		
		for (VPNList::iterator i = vpnList->begin (); i != vpnList->end (); ++i) {
			vpnConnectionNames.append ((*i)->getName ());
		}
	}

	return vpnConnectionNames;
}

void
NetworkManagerInfo::push (KNetworkManager* ctx)
{
	NetworkManagerInfoDBus::push (ctx);
	_ctx = ctx;
}

NetworkManagerInfo::NetworkManagerInfo ()
{
	_currentRequest = NULL;
}

NetworkManagerInfo::~NetworkManagerInfo ()
{

}

#include "knetworkmanager-nminfo.moc"
