/*
 * This file is part of the KDE project
 *
 *  Copyright (c) 2007 Cyrille Berger <cberger@cberger.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; version 2 of the License.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "Dodge.h"

#include <stdlib.h>
#include <vector>

#include <klocale.h>
#include <kiconloader.h>
#include <kinstance.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <ktempfile.h>
#include <kdebug.h>
#include <kgenericfactory.h>

#include <kis_multi_double_filter_widget.h>
#include <kis_iterators_pixel.h>
#include <kis_progress_display_interface.h>
#include <kis_filter_registry.h>
#include <kis_global.h>
#include <kis_transaction.h>
#include <kis_types.h>
#include <kis_selection.h>

#include <kis_convolution_painter.h>

#include <qimage.h>
#include <qpixmap.h>
#include <qbitmap.h>
#include <qpainter.h>

#include "DodgeBurnConfigurationWidget.h"

KisDodgeFilter::KisDodgeFilter() 
    : KisFilter(id(), "Dodge", i18n("&Dodge..."))
{
}

std::list<KisFilterConfiguration*> KisDodgeFilter::listOfExamplesConfiguration(KisPaintDeviceSP )
{
    std::list<KisFilterConfiguration*> list;
    list.insert(list.begin(), configuration());
    return list;
}

KisFilterConfiguration* KisDodgeFilter::configuration()
{
    KisFilterConfiguration* config = new KisFilterConfiguration(id().id(),1);
    config->setProperty("type", 0);
    config->setProperty("exposure", 0.5);
    return config;
};

KisFilterConfigWidget * KisDodgeFilter::createConfigurationWidget(QWidget* parent, KisPaintDeviceSP dev)
{
    return new DodgeBurnConfigurationWidget(parent, "");
}

KisFilterConfiguration* KisDodgeFilter::configuration(QWidget* nwidget)
{
    DodgeBurnConfigurationWidget* widget = (DodgeBurnConfigurationWidget*) nwidget;
    if( widget == 0 )
    {
        return configuration();
    } else {
        return widget->configuration(id().id(),1);
    }
}

typedef void (*funcDodge)(const Q_UINT8* , Q_UINT8* , uint, double );

template<typename _TYPE, int norm>
void DodgeShadow(const Q_UINT8* s, Q_UINT8* d, uint nbpixels, double exposure)
{
    const _TYPE* sT = (_TYPE*)(s);
    _TYPE* dT = (_TYPE*)(d);
    for(uint i = 0; i < nbpixels; i ++)
    {
        dT[i] = (_TYPE)((exposure + sT[i] / (double)norm) * norm - exposure * sT[i]) ;
    }
}

template<typename _TYPE, int norm>
void DodgeMidtones(const Q_UINT8* s, Q_UINT8* d, uint nbpixels, double exposure)
{
    const _TYPE* sT = (_TYPE*)(s);
    _TYPE* dT = (_TYPE*)(d);
    for(uint i = 0; i < nbpixels; i ++)
    {
        dT[i] = (_TYPE)(pow( sT[i] / (double)norm, exposure) * norm) ;
    }
}

template<typename _TYPE, int max>
void DodgeHighlights(const Q_UINT8* s, Q_UINT8* d, uint nbpixels, double exposure)
{
    const _TYPE* sT = (_TYPE*)(s);
    _TYPE* dT = (_TYPE*)(d);
    for(uint i = 0; i < nbpixels; i ++)
    {
        double v = exposure * sT[i];
        if(v > max) v = max;
        dT[i] = (_TYPE)( v ) ;
    }
}

void KisDodgeFilter::process(KisPaintDeviceSP src, KisPaintDeviceSP dst, 
                                   KisFilterConfiguration* config, const QRect& rect ) 
{
    Q_ASSERT(src != 0);
    Q_ASSERT(dst != 0);
    
    double exposure;
    int type;
    if(config)
    {
        exposure = config->getDouble("exposure", 0.5);
        type = config->getInt("type", SHADOWS);
    } else {
        type = SHADOWS;
        exposure = 0.5;
    }
    
    KisRectIteratorPixel dstIt = dst->createRectIterator(rect.x(), rect.y(), rect.width(), rect.height(), true );
    KisRectIteratorPixel srcIt = src->createRectIterator(rect.x(), rect.y(), rect.width(), rect.height(), false);

    int pixelsProcessed = 0;
    setProgressTotalSteps(rect.width() * rect.height());

    KisColorSpace * cs = src->colorSpace();

    funcDodge F;
    KisChannelInfo::enumChannelValueType cT = cs->channels()[0]->channelValueType();

    switch(type)
    {
        case SHADOWS:
        {
            exposure /= 3.0;
            if( cT == KisChannelInfo::UINT8 || cT == KisChannelInfo::INT8 )
            {
                F = & DodgeShadow<Q_UINT8, 255>;
            } else if( cT == KisChannelInfo::UINT16 || cT == KisChannelInfo::INT16 )
            {
                F = & DodgeShadow<Q_UINT16, 65555>;
            } else if( cT == KisChannelInfo::FLOAT32 )
            {
                F = & DodgeShadow<float, 1>;
            } else {
                return;
            }
        }
            break;
        case MIDTONES:
        {
            exposure = 1.0 / ( 1.0 + exposure);
            if( cT == KisChannelInfo::UINT8 || cT == KisChannelInfo::INT8 )
            {
                F = & DodgeMidtones<Q_UINT8, 255>;
            } else if( cT == KisChannelInfo::UINT16 || cT == KisChannelInfo::INT16 )
            {
                F = & DodgeMidtones<Q_UINT16, 65555>;
            } else if( cT == KisChannelInfo::FLOAT32 )
            {
                F = & DodgeMidtones<float, 1>;
            } else {
                return;
            }
        }
            break;
        case HIGHLIGHTS:
        {
            exposure = 1.0 + exposure / 3.0;
            if( cT == KisChannelInfo::UINT8 || cT == KisChannelInfo::INT8 )
            {
                F = & DodgeHighlights<Q_UINT8,255>;
            } else if( cT == KisChannelInfo::UINT16 || cT == KisChannelInfo::INT16 )
            {
                F = & DodgeHighlights<Q_UINT16,65555>;
            } else if( cT == KisChannelInfo::FLOAT32 )
            {
                F = & DodgeHighlights<float,1000000000>;
            } else {
                return;
            }
        }
            break;
    }
    Q_INT32 nC = cs->nColorChannels();

    while( ! srcIt.isDone() )
    {
        if(srcIt.isSelected())
        {
            F( srcIt.oldRawData(), dstIt.rawData(), nC, exposure);
        }
        setProgress(++pixelsProcessed);
        ++srcIt;
        ++dstIt;
    }

    setProgressDone(); // Must be called even if you don't really support progression
}
