/***************************************************************************
 *   Copyright (C) 2004-2008 by Giovanni Venturi                           *
 *   giovanni@ksniffer.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#include <netinet/in.h>

#include <qstring.h>
#include <qstringlist.h>

#include <klocale.h>

#include "tcp-protocol.h"
#include "portnumbernamelist.h"
#include "../packet.h"
#include "../packetstructures.h"

TcpProtocol::TcpProtocol()
{
  m_packet = NULL;
  m_tcp = (struct TcpHdr *) NULL;
  m_portNumberNameList = NULL;
}


TcpProtocol::TcpProtocol(Packet *packet, PortNumberNameList *ppnl)
{
  m_packet = packet;
  m_tcp = (struct TcpHdr *) m_packet;
  m_portNumberNameList = ppnl;
}


void TcpProtocol::setPacket(Packet *packet)
{
  m_packet = packet;
  m_tcp = (struct TcpHdr *) m_packet;
}


void TcpProtocol::setPortNumberList(PortNumberNameList *pnnl)
{
  m_portNumberNameList = pnnl;
}


void TcpProtocol::giveNameToProtocol()
{
  m_giveNameToProtocol = true;
}


void TcpProtocol::DontGiveNameToProtocol()
{
  m_giveNameToProtocol = false;
}


uint16_t TcpProtocol::sourcePort() const
{
  return ntohs(m_tcp->th_sport);
}


uint16_t TcpProtocol::destinationPort() const
{
  return ntohs(m_tcp->th_dport);
}


QString TcpProtocol::flags() const
{
  char *flagVector = "FSRPAU", p[6];
  int bit = 0;
  QString result;

  while(bit < 6) {
    if (m_tcp->th_flags & (1 << bit))
    {
      p[bit] = flagVector[bit];
      switch (p[bit])
      {
        case 'F':
          result += "FIN, ";
          break;
        case 'S':
          result += "SYN, ";
          break;
        case 'R':
          result += "RST, ";
          break;
        case 'P':
          result += "PSH, ";
          break;
        case 'A':
          result += "ACK, ";
          break;
        case 'U':
          result += "URG, ";
          break;
      }
    }
    bit++;
  }
  result = result.left(result.length() - 2 );

  return result;
}


QString TcpProtocol::protocolName() const
{
  return "TCP";
}


QString TcpProtocol::strInfo() const
{
  return QString("%1 > %2 [%3] Seq=%4 Win=%5 Len=%6")
      .arg(sourcePort()).arg(destinationPort()).arg(flags())
      .arg(sequenceNumber()).arg(win()).arg(dataLength());
}


uint16_t TcpProtocol::win() const
{
  return ntohs(m_tcp->th_win);
}


uint16_t TcpProtocol::sequenceNumber() const
{
  return ntohl(m_tcp->th_seq);
}


uint16_t TcpProtocol::dataLength() const
{
  return m_tcp->th_off << 2;
}


QStringList TcpProtocol::headerLines() const
{
  QStringList list;
  QString portNumberName;

  portNumberName = m_portNumberNameList->detectPortNumberName( sourcePort() );
  if ( portNumberName.isEmpty() )
    list << i18n("TCP protocol field", "Source port: %1").arg( sourcePort() );
  else
    list << i18n("TCP protocol field", "Source port: %1 (%2)").arg( sourcePort() ).arg( portNumberName );

  portNumberName = m_portNumberNameList->detectPortNumberName( destinationPort() );
  if ( portNumberName.isEmpty() )
    list << i18n("TCP protocol field", "Destination port: %1").arg( destinationPort() );
  else
    list << i18n("TCP protocol field", "Destination port: %1 (%2)").arg( destinationPort() ).arg( portNumberName );

  list << i18n("TCP protocol field", "Window size: %1").arg( win() );
  list << i18n("TCP protocol field", "Sequence number: %1").arg( sequenceNumber() );
  list << i18n("TCP protocol field", "Header length: %1 bytes").arg( dataLength() );
  list << i18n("TCP protocol field", "Flags: %1").arg( flags() );

  return list;
}
