/***************************************************************************
 *
 *  $Id: kzencommandthread.cpp,v 1.26 2005/07/03 17:01:48 muszilla Exp $
 *
 *  Copyright (C) 2005 by Andreas Mussgiller
 *  muszilla@users.sourceforge.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the
 *  Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#include <iostream>

#include <kurl.h>
#include <klocale.h>
#include <kdebug.h>
#include <qfile.h>

#include <fileref.h>
#include <mpegfile.h>
#include <tag.h>

#include "kzenexplorer.h"
#include "kzenexplorerview.h"
#include "kzentrack.h"
#include "kzenartist.h"
#include "kzenalbum.h"
#include "kzentageditor.h"
#include "kzenconfigdialog.h"
#include "kzenmediafiles.h"

#include "kzencommandthread.h"

static njb_t * fNJB = NULL;

u_int64_t KZenCommandThread::fBytesSent = 0;
u_int64_t KZenCommandThread::fBytesTotal = 0;

static KZenCommandThread * fgThread = NULL;

KZenCommandThread::KZenCommandThread(KZenExplorer * explorer)
  :QThread()
{
  fRun = false;
  fLock = false;
  fExplorer = explorer;
  fgThread = this;
  fCurrentPercentage = -1; 
  fCancelRequest = false;

  fPlaying = false;
}

KZenCommandThread::~KZenCommandThread()
{
  
}

void KZenCommandThread::setPercentage(int percentage)
{
  if (fCurrentPercentage!=percentage) {
    QApplication::postEvent(fExplorer, new KZenProgressEvent(percentage));
    fCurrentPercentage = percentage;
    if (fCurrentPercentage==100) {
      fCurrentPercentage = -1;
    }
  }
}

int KZenCommandThread::progress(u_int64_t sent, u_int64_t total, const char*,unsigned,void *)
{
  fBytesSent = sent;
  fBytesTotal = total;

  int percentage = (int)(100.0*(double)fBytesSent/(double)fBytesTotal);
  fgThread->setPercentage(percentage);
  
  return 0;
}

void KZenCommandThread::setNJB(njb_t * njb)
{
  fNJB = njb;
}

void KZenCommandThread::run()
{
  KZenCommand * cmd;
  
  int count = 0;

  fRun = true;
  while (fRun) {
    if (!fLock && !fCommandQueue.isEmpty()) {
      fExplorer->setBusy(true);
      while (!fLock && (cmd = fCommandQueue.dequeue())) {
	processCommand(cmd);
	delete cmd;
      }
      fExplorer->setBusy(false);
    }
    msleep(100);

    if (fPlaying) {
      count++;
      if (count==2) {
	u_int16_t sec;
	int change = 0;
	
	NJB_Elapsed_Time(fNJB, &sec, &change);

	if (change) {
	  kdDebug() << "CommandThread --> playback: change" << endl;

	  kdDebug() << "CommandThread --> playback: stop" << endl;
	  fPlaying = false;
	  NJB_Stop_Play(fNJB);
	  
	  if (!fTrackQueue.empty()) {
	    int trackid = fTrackQueue.front();
	    fTrackQueue.pop();

	    kdDebug() << "CommandThread --> playback: new track = " << trackid << endl;
	    
	    if (NJB_Play_Track(fNJB, trackid)==-1) {
	      NJB_Error_Dump(fNJB, stderr);
	      fPlaying = false;
	      while (!fTrackQueue.empty()) fTrackQueue.pop();
	    } else {
	      fPlaying = true;
	    }
	  }
	}

	count = 0;
      }
    }
  }
}

void KZenCommandThread::processCommand(KZenCommand * cmd)
{
  if (cmd->getCommandType()==KZenCommand::CmdUpdateUsage) {
    kdDebug() << "CommandThread --> update usage " << cmd->getName() << endl;

    u_int64_t total;
    u_int64_t free;
    
    long long Total;
    long long Free;
    
    if (NJB_Get_Disk_Usage(fNJB, &total, &free) == -1) {
      NJB_Error_Dump(fNJB,stderr);
      Total = 20;
      Free = 0;
    } else {
      Total = total/(1024*1024);
      Free = free/(1024*1024);
    }
    
    fExplorer->getView()->setUsage(Total-Free, Total);
  }
  
  if (cmd->getCommandType()==KZenCommand::CmdAddTrack) {
    if (!fCancelRequest) {
      u_int32_t trackID;
      
      njb_songid_t * songid = NJB_Songid_New();
      njb_songid_frame_t * frame;
      
      char temp[255];
      
      sprintf(temp,"%s",cmd->getTrack()->getCodecName());
      frame = NJB_Songid_Frame_New_Codec(temp);
      NJB_Songid_Addframe(songid, frame);
      
      frame = NJB_Songid_Frame_New_Filesize(cmd->getTrack()->getFileSize());
      NJB_Songid_Addframe(songid, frame);
      
      sprintf(temp,"%s",cmd->getTrack()->getTitle().latin1());
      frame = NJB_Songid_Frame_New_Title(temp);
      NJB_Songid_Addframe(songid, frame);
      
      sprintf(temp,"%s",cmd->getTrack()->getAlbum().latin1());
      frame = NJB_Songid_Frame_New_Album(temp);
      NJB_Songid_Addframe(songid, frame);
      
      sprintf(temp,"%s",cmd->getTrack()->getArtist().latin1());
      frame = NJB_Songid_Frame_New_Artist(temp);
      NJB_Songid_Addframe(songid, frame);
      
      sprintf(temp,"%s",cmd->getTrack()->getGenre().latin1());
      frame = NJB_Songid_Frame_New_Genre(temp);
      NJB_Songid_Addframe(songid, frame);
      
      frame = NJB_Songid_Frame_New_Year(cmd->getTrack()->getYear());
      NJB_Songid_Addframe(songid, frame);
      
      frame = NJB_Songid_Frame_New_Tracknum(cmd->getTrack()->getTrackNumber());
      NJB_Songid_Addframe(songid, frame);
      
      frame = NJB_Songid_Frame_New_Length(cmd->getTrack()->getLength());
      NJB_Songid_Addframe(songid, frame);
      
      sprintf(temp,"%s",cmd->getTrack()->getFileName().latin1());
      frame = NJB_Songid_Frame_New_Filename(temp);
      NJB_Songid_Addframe(songid, frame);
      
      if (NJB_Send_Track(fNJB, QFile::encodeName(cmd->getURL().path()).data(), songid, 
			 KZenCommandThread::progress, NULL, &trackID) == -1 ) {
	NJB_Error_Dump(fNJB,stderr);
      } else {
	cmd->getTrack()->setTrackID(trackID);
	fExplorer->getTracks()->append(cmd->getTrack());
	
	KZenArtist * temp;
	QString artist = cmd->getTrack()->getArtist();
	temp = 0;
	for (uint a=0;a<fExplorer->getArtists()->count();a++) {
	  temp = fExplorer->getArtists()->at(a);
	  if (artist==temp->getName()) break;
	  temp = 0;
	}
	if (!temp) {
	  temp = new KZenArtist(artist);
	  fExplorer->getArtists()->append(temp);
	}
	temp->addTrack(cmd->getTrack());
	cmd->getTrack()->setZenArtist(temp);
	
	KZenGenre * gtemp;
	QString genre = cmd->getTrack()->getGenre();
	gtemp = 0;
	for (uint a=0;a<fExplorer->getGenres()->count();a++) {
	  gtemp = fExplorer->getGenres()->at(a);
	  if (genre==gtemp->getName()) break;
	  gtemp = 0;
	}
	if (!gtemp) {
	  gtemp = new KZenGenre(genre);
	  fExplorer->getGenres()->append(gtemp);
	}
	gtemp->addTrack(cmd->getTrack());
	cmd->getTrack()->setZenGenre(gtemp);
	
	if (cmd->getPlayList()) {
	  cmd->getPlayList()->addTrack(cmd->getTrack());
	}
	
	KZenSmartPlayList * spl;
	for (uint i=0;i<fExplorer->getSmartPlayLists()->count();i++) {
	  spl = fExplorer->getSmartPlayLists()->at(i);
	  spl->addTrack(cmd->getTrack());
	}
	
	QApplication::postEvent(fExplorer, new KZenItemEvent(cmd->getTrack()));
      }
      NJB_Songid_Destroy(songid);
    } else {
      QApplication::postEvent(fExplorer, new KZenProgressEvent(100));
    }
  }
  
  if (cmd->getCommandType()==KZenCommand::CmdSaveTrack) {
    if (!fCancelRequest) {
      bool valid;
      KURL url = KZenMediaFiles::saveFilename(cmd->getTrack(),cmd->getURL(),valid);
      kdDebug() << "SaveName=" << url.url().utf8() << endl;
      kdDebug() << "valid=" << (int)valid << endl;
      if (valid) {
	if (NJB_Get_Track(fNJB,
			  cmd->getTrack()->getTrackID(),
			  cmd->getTrack()->getFileSize(),
			  QFile::encodeName(url.path()).data(),
			  KZenCommandThread::progress, NULL) == -1) {
	  NJB_Error_Dump(fNJB, stderr);
	}

	TagLib::FileRef f = TagLib::FileRef(new TagLib::MPEG::File(QFile::encodeName(url.path()).data()));
	if (!f.isNull() && f.tag()) {
	  TagLib::Tag * tag = f.tag();
	  
	  tag->setArtist(cmd->getTrack()->getArtist().latin1());
	  tag->setAlbum(cmd->getTrack()->getAlbum().latin1());
	  tag->setTitle(cmd->getTrack()->getTitle().latin1());
	  tag->setTrack(cmd->getTrack()->getTrackNumber());
	  tag->setGenre(cmd->getTrack()->getGenre().latin1());
	  tag->setYear(cmd->getTrack()->getYear());
	  tag->setComment("");

	  f.save();
	}

      } else {
	QApplication::postEvent(fExplorer, new KZenProgressEvent(100));
      }
    } else {
      QApplication::postEvent(fExplorer, new KZenProgressEvent(100));
    }
  }
  
  if (cmd->getCommandType()==KZenCommand::CmdNewPlayList) {
    if (cmd->getPlayList()==0) {
      kdDebug() << "CommandThread --> new playList " << cmd->getName() << endl;
 
      njb_playlist_t * tPlayList = NJB_Playlist_New();
      NJB_Playlist_Set_Name(tPlayList, cmd->getName());
      
      if (NJB_Update_Playlist(fNJB, tPlayList) == -1) {
	NJB_Error_Dump(fNJB,stderr);
      }
      
      KZenPlayList * pl = new KZenPlayList(tPlayList, fExplorer->getTracks());
      fExplorer->addPlayList(pl);
      
      fExplorer->getView()->addPlayList(pl);

      fExplorer->emitSelectNaviItem(pl->getItem());
    } else {
      kdDebug() << "CommandThread --> new playList from selection " << cmd->getPlayList()->getName() << endl;

      njb_playlist_t * tPlayList = NJB_Playlist_New();
      NJB_Playlist_Set_Name(tPlayList, cmd->getPlayList()->getName());
   
      if (NJB_Update_Playlist(fNJB, tPlayList) == -1) {
	NJB_Error_Dump(fNJB,stderr);
      }
      
      cmd->getPlayList()->setID(tPlayList->plid);

      NJB_Playlist_Destroy(tPlayList);
    }
  }

  if (cmd->getCommandType()==KZenCommand::CmdNewSmartPlayList) {
    njb_playlist_t * tPlayList = NJB_Playlist_New();
    NJB_Playlist_Set_Name(tPlayList, cmd->getPlayList()->getName());
    
    if (NJB_Update_Playlist(fNJB, tPlayList) == -1) {
      NJB_Error_Dump(fNJB,stderr);
    }

    KZenSmartPlayList * spl = (KZenSmartPlayList*)cmd->getPlayList();
    spl->init(tPlayList, fExplorer->getTracks());

    fExplorer->addSmartPlayList(spl);
    
    fExplorer->getView()->addPlayList(spl);

    fExplorer->emitSelectNaviItem(spl->getItem());
  }
  
  if (cmd->getCommandType()==KZenCommand::CmdUpdateTrack) {
    if ((fExplorer->getEditor()->isTitleChecked() && 
	 cmd->getTrack()->getTitle()!=fExplorer->getEditor()->getTitle()) ||
	(fExplorer->getEditor()->isAlbumChecked() && 
	 cmd->getTrack()->getAlbum()!=fExplorer->getEditor()->getAlbum()) ||
	(fExplorer->getEditor()->isArtistChecked() && 
	 cmd->getTrack()->getArtist()!=fExplorer->getEditor()->getArtist()) ||
	(fExplorer->getEditor()->isGenreChecked() && 
	 cmd->getTrack()->getGenre()!=fExplorer->getEditor()->getGenre()) ||
	(fExplorer->getEditor()->isTrackNumberChecked() && 
	 cmd->getTrack()->getTrackNumber()!=fExplorer->getEditor()->getTrackNumber()) ||
	(fExplorer->getEditor()->isYearChecked() && 
	 cmd->getTrack()->getYear()!=fExplorer->getEditor()->getYear())) {
      
      njb_songid_t * songid = NJB_Songid_New();
      njb_songid_frame_t * frame;
      
      char temp[255];
      
      frame = NJB_Songid_Frame_New_Filename(cmd->getTrack()->getFileName().latin1());
      NJB_Songid_Addframe(songid, frame);
      
      frame = NJB_Songid_Frame_New_Filesize(cmd->getTrack()->getFileSize());
      NJB_Songid_Addframe(songid, frame);
      
      frame = NJB_Songid_Frame_New_Length(cmd->getTrack()->getLength());
      NJB_Songid_Addframe(songid, frame);
      
      if (fExplorer->getEditor()->isTitleChecked()) {
	sprintf(temp,"%s",fExplorer->getEditor()->getTitle());
      } else {
	sprintf(temp,"%s",cmd->getTrack()->getTitle().latin1());
      }
      frame = NJB_Songid_Frame_New_Title(temp);
      NJB_Songid_Addframe(songid, frame);
      
      if (fExplorer->getEditor()->isAlbumChecked()) {
	sprintf(temp,"%s",fExplorer->getEditor()->getAlbum());
      } else {
	sprintf(temp,"%s",cmd->getTrack()->getAlbum().latin1());
      }
      frame = NJB_Songid_Frame_New_Album(temp);
      NJB_Songid_Addframe(songid, frame);
      
      if (fExplorer->getEditor()->isArtistChecked()) {
	sprintf(temp,"%s",fExplorer->getEditor()->getArtist());
      } else {
	sprintf(temp,"%s",cmd->getTrack()->getArtist().latin1());
      }
      frame = NJB_Songid_Frame_New_Artist(temp);
      NJB_Songid_Addframe(songid, frame);
      
      if (fExplorer->getEditor()->isGenreChecked()) {
	sprintf(temp,"%s",fExplorer->getEditor()->getGenre());
      } else {
	sprintf(temp,"%s",cmd->getTrack()->getGenre().latin1());
      }
      frame = NJB_Songid_Frame_New_Genre(temp);
      NJB_Songid_Addframe(songid, frame);
      
      if (fExplorer->getEditor()->isYearChecked()) {
	frame = NJB_Songid_Frame_New_Year(fExplorer->getEditor()->getYear());
      } else {
	frame = NJB_Songid_Frame_New_Year(cmd->getTrack()->getYear());
      }
      NJB_Songid_Addframe(songid, frame);
      
      if (fExplorer->getEditor()->isTrackNumberChecked()) {
	frame = NJB_Songid_Frame_New_Tracknum(fExplorer->getEditor()->getTrackNumber());
      } else {
      frame = NJB_Songid_Frame_New_Tracknum(cmd->getTrack()->getTrackNumber());
      }
      NJB_Songid_Addframe(songid, frame);
      
      if (NJB_Replace_Track_Tag(fNJB, cmd->getTrack()->getTrackID(), songid) == -1 ) {
	NJB_Error_Dump(fNJB,stderr);
      }else {
	if (fExplorer->getEditor()->isTitleChecked()) 
	  cmd->getTrack()->setTitle(fExplorer->getEditor()->getTitle());
	if (fExplorer->getEditor()->isAlbumChecked()) 
	  cmd->getTrack()->setAlbum(fExplorer->getEditor()->getAlbum());
	if (fExplorer->getEditor()->isArtistChecked()) 
	  cmd->getTrack()->setArtist(fExplorer->getEditor()->getArtist());
	if (fExplorer->getEditor()->isGenreChecked()) 
	  cmd->getTrack()->setGenre(fExplorer->getEditor()->getGenre());
	if (fExplorer->getEditor()->isTrackNumberChecked()) 
	  cmd->getTrack()->setTrackNumber(fExplorer->getEditor()->getTrackNumber());
	if (fExplorer->getEditor()->isYearChecked()) 
	  cmd->getTrack()->setYear(fExplorer->getEditor()->getYear());
	cmd->getTrack()->refresh();
      }
      NJB_Songid_Destroy(songid);
    }
    
    if (fExplorer->getEditor()->isRatingChecked()) 
      cmd->getTrack()->setRating(fExplorer->getEditor()->getRating());
    if (fExplorer->getEditor()->isCategoryChecked()) 
      cmd->getTrack()->setCategories(fExplorer->getEditor()->getCategories());
    
    cmd->getTrack()->refresh();
    fExplorer->getView()->trackUpdated(cmd->getTrack());
    fExplorer->refreshSmartPlayLists(cmd->getTrack());
  }

  if (cmd->getCommandType()==KZenCommand::CmdUpdateTrackInline) {
    
    njb_songid_t * songid = NJB_Songid_New();
    njb_songid_frame_t * frame;
    
    frame = NJB_Songid_Frame_New_Filename(cmd->getTrack()->getFileName().latin1());
    NJB_Songid_Addframe(songid, frame);
    
    frame = NJB_Songid_Frame_New_Filesize(cmd->getTrack()->getFileSize());
    NJB_Songid_Addframe(songid, frame);
    
    frame = NJB_Songid_Frame_New_Length(cmd->getTrack()->getLength());
    NJB_Songid_Addframe(songid, frame);
    
    frame = NJB_Songid_Frame_New_Title(cmd->getTrack()->getTitle().latin1());
    NJB_Songid_Addframe(songid, frame);
      
    frame = NJB_Songid_Frame_New_Album(cmd->getTrack()->getAlbum().latin1());
    NJB_Songid_Addframe(songid, frame);
    
    frame = NJB_Songid_Frame_New_Artist(cmd->getTrack()->getArtist().latin1());
    NJB_Songid_Addframe(songid, frame);
    
    frame = NJB_Songid_Frame_New_Genre(cmd->getTrack()->getGenre().latin1());
    NJB_Songid_Addframe(songid, frame);
    
    frame = NJB_Songid_Frame_New_Year(cmd->getTrack()->getYear());
    NJB_Songid_Addframe(songid, frame);
    
    frame = NJB_Songid_Frame_New_Tracknum(cmd->getTrack()->getTrackNumber());
    NJB_Songid_Addframe(songid, frame);
    
    if (NJB_Replace_Track_Tag(fNJB, cmd->getTrack()->getTrackID(), songid) == -1 ) {
      NJB_Error_Dump(fNJB,stderr);
    }
    NJB_Songid_Destroy(songid);

    cmd->getTrack()->refresh();
    fExplorer->getView()->trackUpdated(cmd->getTrack());
    fExplorer->refreshSmartPlayLists(cmd->getTrack());
  }
  
  if (cmd->getCommandType()==KZenCommand::CmdUpdatePlayList) {
    
    njb_playlist_t * tPlayList = NJB_Playlist_New();
    NJB_Playlist_Set_Name(tPlayList, cmd->getPlayList()->getName());
    tPlayList->plid = cmd->getPlayList()->getID();

    njb_playlist_track_t * track;
    KZenTrack * Ztrack;
    for (uint i=0;i<cmd->getPlayList()->getTrackList()->count();i++) {
      Ztrack = cmd->getPlayList()->getTrackList()->at(i);
      track = NJB_Playlist_Track_New(Ztrack->getTrackID());
      NJB_Playlist_Addtrack(tPlayList, track, NJB_PL_END);
    }
    tPlayList->_state = NJB_PL_CHTRACKS;
    
    if (NJB_Update_Playlist(fNJB, tPlayList) == -1) {
      NJB_Error_Dump(fNJB,stderr);
    }
    NJB_Playlist_Destroy(tPlayList);
    
    cmd->getPlayList()->setModified(false); 
    fExplorer->setUpdateEnabled(false);
  }

  if (cmd->getCommandType()==KZenCommand::CmdRenamePlayList) {
    njb_playlist_t * tPlayList = NJB_Playlist_New();
    NJB_Playlist_Set_Name(tPlayList, cmd->getName());
    tPlayList->plid = cmd->getPlayList()->getID();
    tPlayList->_state = NJB_PL_CHNAME;
    
    if (NJB_Update_Playlist(fNJB, tPlayList) == -1) {
      NJB_Error_Dump(fNJB,stderr);
    } else {
      cmd->getPlayList()->setName(cmd->getName());
      fExplorer->getView()->refresh();
    }
  }

  if (cmd->getCommandType()==KZenCommand::CmdDestroyTrack) {

    if (NJB_Delete_Track(fNJB,cmd->getTrack()->getTrackID()) == -1) {
      NJB_Error_Dump(fNJB,stderr);
    } else {
      KZenArtist * artist;
      KZenAlbum * album;
      KZenGenre * genre;
      QString name;
      for (uint a=0;a<fExplorer->getArtists()->count();a++) {
	artist = fExplorer->getArtists()->at(a);
	name = artist->getName();
	if (name==cmd->getTrack()->getArtist()) {
	  artist->getTrackList()->remove(cmd->getTrack());
	  
	  for (uint b=0;b<artist->getAlbumList()->count();b++) {
	    album = artist->getAlbumList()->at(b);
	    name = album->getName();
	    if (name==cmd->getTrack()->getAlbum()) {
	      album->getTrackList()->remove(cmd->getTrack());
	    }
	    
	    if (album->getTrackList()->count()==0) {
	      fExplorer->emitRemoveAlbum(artist->getName(),album->getName());
	      artist->getAlbumList()->remove(album);
	    }
	  }

	  if (artist->getTrackList()->count()==0) {
	    fExplorer->emitRemoveArtist(artist->getName());
	    fExplorer->getArtists()->remove(artist);
	  }
	}
      }
      for (uint a=0;a<fExplorer->getGenres()->count();a++) {
	genre = fExplorer->getGenres()->at(a);
	name = genre->getName();
	if (name==cmd->getTrack()->getGenre()) {
	  genre->removeTrack(cmd->getTrack());

	  if (genre->getTrackList()->count()==0) {
	    fExplorer->emitRemoveGenre(name);
	    fExplorer->getGenres()->remove(genre);
	  }
	}
      }

      KZenPlayList * pl;
      for (uint si=0;si<fExplorer->getPlayListList()->count();si++) {
	pl = fExplorer->getPlayListList()->at(si);
	if (pl->getTrackList()->remove(cmd->getTrack())) pl->setModified(true);
      }

      KZenSmartPlayList * spl;
      for (uint si=0;si<fExplorer->getSmartPlayLists()->count();si++) {
	spl = fExplorer->getSmartPlayLists()->at(si);
	if (spl->getTrackList()->remove(cmd->getTrack())) spl->setModified(true);
      }
      
      fExplorer->emitRemoveTrackItem(cmd->getTrack()->getItem());
      
      fExplorer->getTracks()->remove(cmd->getTrack());
    }
  }

  if (cmd->getCommandType()==KZenCommand::CmdDeletePlayList) {

    if (NJB_Delete_Playlist(fNJB, cmd->getPlayList()->getID()) == -1) {
      NJB_Error_Dump(fNJB,stderr);
    }

    fExplorer->getPlayListList()->remove(cmd->getPlayList());
    fExplorer->getSmartPlayLists()->remove((KZenSmartPlayList*)cmd->getPlayList());

    fExplorer->emitRemoveNaviItem(cmd->getPlayList()->getItem());
  }

  if (cmd->getCommandType()==KZenCommand::CmdGetOwnerString) {
    
    char * ownerstring = NJB_Get_Owner_String(fNJB);
    if (!ownerstring) {
      NJB_Error_Dump(fNJB,stderr);
    }
    
    fExplorer->configDialog()->setOwnerString(ownerstring);
    
    free(ownerstring);
  }

  if (cmd->getCommandType()==KZenCommand::CmdSetOwnerString) {
    
    if (NJB_Set_Owner_String(fNJB, cmd->getName())==-1) {
      NJB_Error_Dump(fNJB,stderr);
    }
    
    char * ownerstring = NJB_Get_Owner_String(fNJB);
    if (!ownerstring) {
      NJB_Error_Dump(fNJB,stderr);
    }
    
    fExplorer->configDialog()->setOwnerString(ownerstring);
    
    free(ownerstring);
  }

  if (cmd->getCommandType()==KZenCommand::CmdGetClock) {

    njb_time_t * time;
    time = NJB_Get_Time(fNJB);

    if (time) {
      QDateTime newTime(QDate(time->year, time->month, time->day),
			QTime(time->hours, time->minutes, time->seconds));
      fExplorer->configDialog()->setClock(newTime);
      NJB_Destroy_Time(time);
    }
    
    if (NJB_Error_Pending(fNJB)) {
      NJB_Error_Dump(fNJB, stderr);
    }
  }

  if (cmd->getCommandType()==KZenCommand::CmdSetClock) {

    njb_time_t * time;
    time = NJB_Get_Time(fNJB);
    
    time->year = cmd->getDateTime().date().year();
    time->month = cmd->getDateTime().date().month();
    time->day = cmd->getDateTime().date().day();
    time->weekday = cmd->getDateTime().date().dayOfWeek();
    if (time->weekday==7) time->weekday = 0; 

    time->hours = cmd->getDateTime().time().hour();
    time->minutes = cmd->getDateTime().time().minute();
    time->seconds = cmd->getDateTime().time().second();

    if (NJB_Set_Time(fNJB, time)==-1) {
      NJB_Error_Dump(fNJB,stderr);
    }
    NJB_Destroy_Time(time);
    
    time = NJB_Get_Time(fNJB);
    if (time) {
      QDateTime newTime(QDate(time->year, time->month, time->day),
			QTime(time->hours, time->minutes, time->seconds));
      fExplorer->configDialog()->setClock(newTime);
      NJB_Destroy_Time(time);
    }
    
    if (NJB_Error_Pending(fNJB)) {
      NJB_Error_Dump(fNJB, stderr);
    }
  }

  if (cmd->getCommandType()==KZenCommand::CmdPlayTrack) {

    NJB_Stop_Play(fNJB);

    if (NJB_Play_Track(fNJB, cmd->getTrack()->getTrackID())==-1) {
      NJB_Error_Dump(fNJB, stderr);
      fPlaying = false;
      while (!fTrackQueue.empty()) fTrackQueue.pop();
    } else {
      fPlaying = true;
    }
  }

  if (cmd->getCommandType()==KZenCommand::CmdQueueTrack) {

    fTrackQueue.push(cmd->getTrack()->getTrackID());
  }

  if (cmd->getCommandType()==KZenCommand::CmdStopPlay) {

    NJB_Stop_Play(fNJB);
    fPlaying = false;
    while (!fTrackQueue.empty()) fTrackQueue.pop();
  }

  if (cmd->getCommandType()==KZenCommand::CmdUpdatePlayProgress) {
    
    u_int16_t sec;
    int change = 0;
    
    NJB_Elapsed_Time(fNJB, &sec, &change);
    
    if (change) {
      if (fTrackQueue.empty()) {
	fPlaying = false;
	NJB_Stop_Play(fNJB);
      } else {
	int trackid = fTrackQueue.front();
	
	if (NJB_Play_Track(fNJB, trackid)==-1) {
	  NJB_Error_Dump(fNJB, stderr);
	  fPlaying = false;
	  while (!fTrackQueue.empty()) fTrackQueue.pop();
	} else {
	  fPlaying = true;
	}
	
	fTrackQueue.pop();
      }
    }
  }
  
}

void KZenCommandThread::updateUsage()
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdUpdateUsage));
}

void KZenCommandThread::addTrack(KZenTrack * track, const KURL & url, KZenPlayList * pl)
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdAddTrack,track,url,pl));
}

void KZenCommandThread::saveTrack(KZenTrack * track, const KURL & url)
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdSaveTrack,track,url));
}

void KZenCommandThread::newPlayList(const QString & name)
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdNewPlayList,name));
}

void KZenCommandThread::newPlayList(KZenPlayList * pl)
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdNewPlayList,pl));
}

void KZenCommandThread::newSmartPlayList(KZenPlayList * pl)
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdNewSmartPlayList,pl));
}

void KZenCommandThread::updateTrack(KZenTrack * track)
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdUpdateTrack,track));
}

void KZenCommandThread::updateTrackInline(KZenTrack * track)
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdUpdateTrackInline,track));
}

void KZenCommandThread::updatePlayList(KZenPlayList * pl)
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdUpdatePlayList,pl));
}

void KZenCommandThread::renamePlayList(KZenPlayList * pl, const QString & name)
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdRenamePlayList,pl,name));
}

void KZenCommandThread::destroyTrack(KZenTrack * track)
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdDestroyTrack,track));
}

void KZenCommandThread::deletePlayList(KZenPlayList * pl)
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdDeletePlayList,pl));
}

void KZenCommandThread::getOwnerString()
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdGetOwnerString));
}

void KZenCommandThread::setOwnerString(const QString & name)
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdSetOwnerString,name));
}

void KZenCommandThread::getClock()
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdGetClock));
}

void KZenCommandThread::setClock(const QDateTime & clock)
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdSetClock,clock));
}

void KZenCommandThread::playTrack(KZenTrack * track)
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdPlayTrack,track));
}

void KZenCommandThread::queueTrack(KZenTrack * track)
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdQueueTrack,track));
}

void KZenCommandThread::stopPlay()
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdStopPlay));
}

void KZenCommandThread::updatePlayProgress()
{
  fCommandQueue.enqueue(new KZenCommand(KZenCommand::CmdUpdatePlayProgress));
}

KZenCommand::KZenCommand(CommandType cmd)
{
  fCmdType = cmd;
  fTrack = 0;
  fPlayList = 0;
}

KZenCommand::KZenCommand(CommandType cmd, const QString & name)
{
  fCmdType = cmd;
  fTrack = 0;
  fPlayList = 0;
  fName = name;
}

KZenCommand::KZenCommand(CommandType cmd, const QDateTime & datetime)
{
  fCmdType = cmd;
  fTrack = 0;
  fPlayList = 0;
  fDateTime = datetime;
}

KZenCommand::KZenCommand(CommandType cmd, KZenTrack * track)
{
  fCmdType = cmd;
  fTrack = track;
  fPlayList = 0;
}

KZenCommand::KZenCommand(CommandType cmd, KZenTrack * track, KZenPlayList * pl)
{
  fCmdType = cmd;
  fTrack = track;
  fPlayList = pl;
}
 
KZenCommand::KZenCommand(CommandType cmd, KZenTrack * track, const KURL & url, 
			 KZenPlayList * pl)
{
  fCmdType = cmd;
  fTrack = track;
  fPlayList = pl;
  fURL = url;
}
 
KZenCommand::KZenCommand(CommandType cmd, KZenTrack * track, const KURL & url)
{
  fCmdType = cmd;
  fTrack = track;
  fPlayList = 0;
  fURL = url;
}

KZenCommand::KZenCommand(CommandType cmd, KZenPlayList * pl)
{
  fCmdType = cmd;
  fTrack = 0;
  fPlayList = pl;
}

KZenCommand::KZenCommand(CommandType cmd, KZenPlayList * pl, const QString & name)
{
  fCmdType = cmd;
  fTrack = 0;
  fPlayList = pl;
  fName = name;
}

KZenCommand::~KZenCommand()
{

}
