#!/usr/bin/python
# -*- coding: utf-8 -*-
### BEGIN LICENSE
# Copyright (C) 2009 Jono Bacon <jono@ubuntu.com>
# Copyright (C) 2010 Michael Budde <mbudde@gmail.com>
#
#This program is free software: you can redistribute it and/or modify it
#under the terms of the GNU General Public License version 3, as published
#by the Free Software Foundation.
#
#This program is distributed in the hope that it will be useful, but
#WITHOUT ANY WARRANTY; without even the implied warranties of
#MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
#PURPOSE.  See the GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License along
#with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import re
import vobject
import urllib

import lernid.DateTime as dt

class Session(object):

    PAST, NOW, FUTURE = range(3)

    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            if k in ('title', 'description', 'instructors', 'local_start', 'local_end', 'slides', 'event'):
                setattr(self, '_'+k, v)

    @property
    def title(self):
        return self._title

    @property
    def description(self):
        return self._description

    @property
    def instructors(self):
        return self._instructors

    @property
    def start_local_date(self):
        # Translators: Local date representation
        return self._local_start.strftime(_('%d %B %Y'))

    @property
    def start_local_time(self):
        # Translators: Local time representation
        return self._local_start.strftime(_('%H:%M'))

    @property
    def end_local_time(self):
        # Translators: Local time representation
        return self._local_end.strftime(_('%H:%M'))

    @property
    def start_datetime(self):
        return self._local_start

    @property
    def end_datetime(self):
        return self._local_end

    @property
    def slides(self):
        return self._slides

    @property
    def evenet(self):
        return self._event

    @property
    def state(self):
        now, start, end = dt.now_local(), self.start_datetime, self.end_datetime
        if now < start:
            return self.FUTURE
        if start < now < end:
            return self.NOW
        if now > end:
            return self.PAST


def parse_ical(event):
    """Parse iCal schedule for event and generate a list of Session objects"""

    ical = urllib.urlopen(event.icalurl)
    cal = vobject.readOne(ical)

    sessions = []

    eventstart_local = dt.parse_time(event.eventstart, '%Y-%m-%d %H:%M:%S')
    eventend_local = dt.parse_time(event.eventend, '%Y-%m-%d %H:%M:%S')

    for session in cal.vevent_list:
        local_start = dt.as_local(session.dtstart.value)
        local_end = dt.as_local(session.dtend.value)

        if eventstart_local < local_start < eventend_local:
            session_data = parse_ical_description(session.description.value)
            sessions.append(Session(
                title = session.summary.value,
                local_start = local_start,
                local_end = local_end,
                **session_data))

    # reverse the list to get the events in chronological order
    sessions.sort(lambda x,y: cmp(x.start_datetime, y.start_datetime))
    return sessions

def parse_ical_description(desc):
    session = {}
    for line in desc.splitlines():
        items = [s.strip() for s in line.split(None, 1)]
        if len(items) != 2:
            continue
        key, value = items
        key = key[:-1].lower()
        if key in ('description', 'slides', 'event'):
            session[str(key)] = value
        elif key == 'instructor' or key == 'instructors':
            instructors = [s.strip() for s in value.split(',')]
            session['instructors'] = instructors
    return session
