/*
 * Copyright (C) 2005-2014 Junjiro R. Okajima
 *
 * This program, aufs is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * sub-routines for dentry cache
 */

#ifndef __AUFS_DCSUB_H__
#define __AUFS_DCSUB_H__

#ifdef __KERNEL__

#include <linux/dcache.h>
#include <linux/fs.h>

struct dentry;

struct au_dpage {
	int ndentry;
	struct dentry **dentries;
};

struct au_dcsub_pages {
	int ndpage;
	struct au_dpage *dpages;
};

/* ---------------------------------------------------------------------- */

/* dcsub.c */
int au_dpages_init(struct au_dcsub_pages *dpages, gfp_t gfp);
void au_dpages_free(struct au_dcsub_pages *dpages);
typedef int (*au_dpages_test)(struct dentry *dentry, void *arg);
int au_dcsub_pages(struct au_dcsub_pages *dpages, struct dentry *root,
		   au_dpages_test test, void *arg);
int au_dcsub_pages_rev(struct au_dcsub_pages *dpages, struct dentry *dentry,
		       int do_include, au_dpages_test test, void *arg);
int au_dcsub_pages_rev_aufs(struct au_dcsub_pages *dpages,
			    struct dentry *dentry, int do_include);
int au_test_subdir(struct dentry *d1, struct dentry *d2);

/* ---------------------------------------------------------------------- */

/*
 * todo: in linux-3.13, several similar (but faster) helpers are added to
 * include/linux/dcache.h. Try them (in the future).
 */

static inline int au_d_hashed_positive(struct dentry *d)
{
	int err;
	struct inode *inode = d->d_inode;

	err = 0;
	if (unlikely(d_unhashed(d) || !inode || !inode->i_nlink))
		err = -ENOENT;
	return err;
}

static inline int au_d_linkable(struct dentry *d)
{
	int err;
	struct inode *inode = d->d_inode;

	err = au_d_hashed_positive(d);
	if (err
	    && inode
	    && (inode->i_state & I_LINKABLE))
		err = 0;
	return err;
}

static inline int au_d_alive(struct dentry *d)
{
	int err;
	struct inode *inode;

	err = 0;
	if (!IS_ROOT(d))
		err = au_d_hashed_positive(d);
	else {
		inode = d->d_inode;
		if (unlikely(d_unlinked(d) || !inode || !inode->i_nlink))
			err = -ENOENT;
	}
	return err;
}

static inline int au_alive_dir(struct dentry *d)
{
	int err;

	err = au_d_alive(d);
	if (unlikely(err || IS_DEADDIR(d->d_inode)))
		err = -ENOENT;
	return err;
}

static inline int au_qstreq(struct qstr *a, struct qstr *b)
{
	return a->len == b->len
		&& !memcmp(a->name, b->name, a->len);
}

static inline int au_dcount(struct dentry *d)
{
	return (int)d_count(d);
}

#endif /* __KERNEL__ */
#endif /* __AUFS_DCSUB_H__ */
