#ifndef __BNX2X_INIT_VALUES_H__
#define __BNX2X_INIT_VALUES_H__

/* This array contains the list of operations needed to initialize the chip.
 *
 * For each block in the chip there are three init stages:
 * common - HW used by both ports,
 * port1 and port2 - initialization for a specific Ethernet port.
 * When a port is opened or closed, the management CPU tells the driver
 * whether to init/disable common HW in addition to the port HW.
 * This way the first port going up will first initializes the common HW,
 * and the last port going down also resets the common HW
 *
 * For each init stage/block there is a list of actions needed in a format:
 * {operation, register, data}
 * where:
 * OP_WR - write a value to the chip.
 * OP_RD - read a register (usually a clear on read register).
 * OP_SW - string write, write a section of consecutive addresses to the chip.
 * OP_SI - copy a string using indirect writes.
 * OP_ZR - clear a range of memory.
 * OP_ZP - unzip and copy using DMAE.
 * OP_WB - string copy using DMAE.
 *
 * The #defines mark the stages.
 *
 */

static const struct raw_op init_ops[] = {
#define PRS_COMMON_START        0
	{OP_WR, PRS_REG_INC_VALUE, 0xf},
	{OP_WR, PRS_REG_EVENT_ID_1, 0x45},
	{OP_WR, PRS_REG_EVENT_ID_2, 0x84},
	{OP_WR, PRS_REG_EVENT_ID_3, 0x6},
	{OP_WR, PRS_REG_NO_MATCH_EVENT_ID, 0x4},
	{OP_WR, PRS_REG_CM_HDR_TYPE_0, 0x0},
	{OP_WR, PRS_REG_CM_HDR_TYPE_1, 0x12170000},
	{OP_WR, PRS_REG_CM_HDR_TYPE_2, 0x22170000},
	{OP_WR, PRS_REG_CM_HDR_TYPE_3, 0x32170000},
	{OP_ZR, PRS_REG_CM_HDR_TYPE_4, 0x5},
	{OP_WR, PRS_REG_CM_HDR_LOOPBACK_TYPE_1, 0x12150000},
	{OP_WR, PRS_REG_CM_HDR_LOOPBACK_TYPE_2, 0x22150000},
	{OP_WR, PRS_REG_CM_HDR_LOOPBACK_TYPE_3, 0x32150000},
	{OP_ZR, PRS_REG_CM_HDR_LOOPBACK_TYPE_4, 0x4},
	{OP_WR, PRS_REG_CM_NO_MATCH_HDR, 0x2100000},
	{OP_WR, PRS_REG_CM_HDR_FLUSH_NO_LOAD_TYPE_0, 0x100000},
	{OP_WR, PRS_REG_CM_HDR_FLUSH_NO_LOAD_TYPE_1, 0x10100000},
	{OP_WR, PRS_REG_CM_HDR_FLUSH_NO_LOAD_TYPE_2, 0x20100000},
	{OP_WR, PRS_REG_CM_HDR_FLUSH_NO_LOAD_TYPE_3, 0x30100000},
	{OP_ZR, PRS_REG_CM_HDR_FLUSH_NO_LOAD_TYPE_4, 0x4},
	{OP_WR, PRS_REG_CM_HDR_FLUSH_LOAD_TYPE_0, 0x100000},
	{OP_WR, PRS_REG_CM_HDR_FLUSH_LOAD_TYPE_1, 0x12140000},
	{OP_WR, PRS_REG_CM_HDR_FLUSH_LOAD_TYPE_2, 0x22140000},
	{OP_WR, PRS_REG_CM_HDR_FLUSH_LOAD_TYPE_3, 0x32140000},
	{OP_ZR, PRS_REG_CM_HDR_FLUSH_LOAD_TYPE_4, 0x4},
	{OP_RD, PRS_REG_NUM_OF_PACKETS, 0x0},
	{OP_RD, PRS_REG_NUM_OF_CFC_FLUSH_MESSAGES, 0x0},
	{OP_RD, PRS_REG_NUM_OF_TRANSPARENT_FLUSH_MESSAGES, 0x0},
	{OP_RD, PRS_REG_NUM_OF_DEAD_CYCLES, 0x0},
	{OP_WR_E1H, PRS_REG_FCOE_TYPE, 0x8906},
	{OP_WR, PRS_REG_FLUSH_REGIONS_TYPE_0, 0xff},
	{OP_WR, PRS_REG_FLUSH_REGIONS_TYPE_1, 0xff},
	{OP_WR, PRS_REG_FLUSH_REGIONS_TYPE_2, 0xff},
	{OP_WR, PRS_REG_FLUSH_REGIONS_TYPE_3, 0xff},
	{OP_WR, PRS_REG_FLUSH_REGIONS_TYPE_4, 0xff},
	{OP_WR, PRS_REG_FLUSH_REGIONS_TYPE_5, 0xff},
	{OP_WR, PRS_REG_FLUSH_REGIONS_TYPE_6, 0xff},
	{OP_WR, PRS_REG_FLUSH_REGIONS_TYPE_7, 0xff},
	{OP_WR, PRS_REG_PURE_REGIONS, 0x3e},
	{OP_WR, PRS_REG_PACKET_REGIONS_TYPE_0, 0x0},
	{OP_WR, PRS_REG_PACKET_REGIONS_TYPE_1, 0x3f},
	{OP_WR, PRS_REG_PACKET_REGIONS_TYPE_2, 0x3f},
	{OP_WR, PRS_REG_PACKET_REGIONS_TYPE_3, 0x3f},
	{OP_WR, PRS_REG_PACKET_REGIONS_TYPE_4, 0x0},
	{OP_WR, PRS_REG_PACKET_REGIONS_TYPE_5, 0x3f},
	{OP_WR, PRS_REG_PACKET_REGIONS_TYPE_6, 0x3f},
	{OP_WR, PRS_REG_PACKET_REGIONS_TYPE_7, 0x3f},
#define PRS_COMMON_END          47
#define SRCH_COMMON_START       47
	{OP_WR_E1H, SRC_REG_E1HMF_ENABLE, 0x1},
#define SRCH_COMMON_END         48
#define TSDM_COMMON_START       48
	{OP_WR_E1, TSDM_REG_CFC_RSP_START_ADDR, 0x411},
	{OP_WR_E1H, TSDM_REG_CFC_RSP_START_ADDR, 0x211},
	{OP_WR_E1, TSDM_REG_CMP_COUNTER_START_ADDR, 0x400},
	{OP_WR_E1H, TSDM_REG_CMP_COUNTER_START_ADDR, 0x200},
	{OP_WR_E1, TSDM_REG_Q_COUNTER_START_ADDR, 0x404},
	{OP_WR_E1H, TSDM_REG_Q_COUNTER_START_ADDR, 0x204},
	{OP_WR_E1, TSDM_REG_PCK_END_MSG_START_ADDR, 0x419},
	{OP_WR_E1H, TSDM_REG_PCK_END_MSG_START_ADDR, 0x219},
	{OP_WR, TSDM_REG_CMP_COUNTER_MAX0, 0xffff},
	{OP_WR, TSDM_REG_CMP_COUNTER_MAX1, 0xffff},
	{OP_WR, TSDM_REG_CMP_COUNTER_MAX2, 0xffff},
	{OP_WR, TSDM_REG_CMP_COUNTER_MAX3, 0xffff},
	{OP_ZR, TSDM_REG_AGG_INT_EVENT_0, 0x2},
	{OP_WR, TSDM_REG_AGG_INT_EVENT_2, 0x34},
	{OP_WR, TSDM_REG_AGG_INT_EVENT_3, 0x35},
	{OP_ZR, TSDM_REG_AGG_INT_EVENT_4, 0x7c},
	{OP_WR, TSDM_REG_ENABLE_IN1, 0x7ffffff},
	{OP_WR, TSDM_REG_ENABLE_IN2, 0x3f},
	{OP_WR, TSDM_REG_ENABLE_OUT1, 0x7ffffff},
	{OP_WR, TSDM_REG_ENABLE_OUT2, 0xf},
	{OP_RD, TSDM_REG_NUM_OF_Q0_CMD, 0x0},
	{OP_RD, TSDM_REG_NUM_OF_Q1_CMD, 0x0},
	{OP_RD, TSDM_REG_NUM_OF_Q3_CMD, 0x0},
	{OP_RD, TSDM_REG_NUM_OF_Q4_CMD, 0x0},
	{OP_RD, TSDM_REG_NUM_OF_Q5_CMD, 0x0},
	{OP_RD, TSDM_REG_NUM_OF_Q6_CMD, 0x0},
	{OP_RD, TSDM_REG_NUM_OF_Q7_CMD, 0x0},
	{OP_RD, TSDM_REG_NUM_OF_Q8_CMD, 0x0},
	{OP_RD, TSDM_REG_NUM_OF_Q9_CMD, 0x0},
	{OP_RD, TSDM_REG_NUM_OF_Q10_CMD, 0x0},
	{OP_RD, TSDM_REG_NUM_OF_Q11_CMD, 0x0},
	{OP_RD, TSDM_REG_NUM_OF_PKT_END_MSG, 0x0},
	{OP_RD, TSDM_REG_NUM_OF_PXP_ASYNC_REQ, 0x0},
	{OP_RD, TSDM_REG_NUM_OF_ACK_AFTER_PLACE, 0x0},
	{OP_WR_E1, TSDM_REG_INIT_CREDIT_PXP_CTRL, 0x1},
	{OP_WR_ASIC, TSDM_REG_TIMER_TICK, 0x3e8},
	{OP_WR_EMUL, TSDM_REG_TIMER_TICK, 0x1},
	{OP_WR_FPGA, TSDM_REG_TIMER_TICK, 0xa},
#define TSDM_COMMON_END         86
#define TCM_COMMON_START        86
	{OP_WR, TCM_REG_XX_MAX_LL_SZ, 0x20},
	{OP_WR, TCM_REG_XX_OVFL_EVNT_ID, 0x32},
	{OP_WR, TCM_REG_TQM_TCM_HDR_P, 0x2150020},
	{OP_WR, TCM_REG_TQM_TCM_HDR_S, 0x2150020},
	{OP_WR, TCM_REG_TM_TCM_HDR, 0x30},
	{OP_WR, TCM_REG_ERR_TCM_HDR, 0x8100000},
	{OP_WR, TCM_REG_ERR_EVNT_ID, 0x33},
	{OP_WR, TCM_REG_EXPR_EVNT_ID, 0x30},
	{OP_WR, TCM_REG_STOP_EVNT_ID, 0x31},
	{OP_WR, TCM_REG_PRS_WEIGHT, 0x4},
	{OP_WR, TCM_REG_PBF_WEIGHT, 0x5},
	{OP_WR, TCM_REG_CP_WEIGHT, 0x0},
	{OP_WR, TCM_REG_TSDM_WEIGHT, 0x4},
	{OP_WR, TCM_REG_TCM_TQM_USE_Q, 0x1},
	{OP_WR, TCM_REG_GR_ARB_TYPE, 0x1},
	{OP_WR, TCM_REG_GR_LD0_PR, 0x1},
	{OP_WR, TCM_REG_GR_LD1_PR, 0x2},
	{OP_WR, TCM_REG_CFC_INIT_CRD, 0x1},
	{OP_WR, TCM_REG_FIC0_INIT_CRD, 0x40},
	{OP_WR, TCM_REG_FIC1_INIT_CRD, 0x40},
	{OP_WR, TCM_REG_TQM_INIT_CRD, 0x20},
	{OP_WR, TCM_REG_XX_INIT_CRD, 0x13},
	{OP_WR, TCM_REG_XX_MSG_NUM, 0x20},
	{OP_ZR, TCM_REG_XX_TABLE, 0xa},
	{OP_SW, TCM_REG_XX_DESCR_TABLE, 0x200000},
	{OP_WR, TCM_REG_N_SM_CTX_LD_0, 0x7},
	{OP_WR, TCM_REG_N_SM_CTX_LD_1, 0x7},
	{OP_WR, TCM_REG_N_SM_CTX_LD_2, 0x8},
	{OP_WR, TCM_REG_N_SM_CTX_LD_3, 0x8},
	{OP_ZR, TCM_REG_N_SM_CTX_LD_4, 0x4},
	{OP_WR, TCM_REG_TCM_REG0_SZ, 0x6},
	{OP_WR_E1, TCM_REG_PHYS_QNUM0_0, 0xd},
	{OP_WR_E1, TCM_REG_PHYS_QNUM0_1, 0x2d},
	{OP_WR_E1, TCM_REG_PHYS_QNUM1_0, 0x7},
	{OP_WR_E1, TCM_REG_PHYS_QNUM1_1, 0x27},
	{OP_WR_E1, TCM_REG_PHYS_QNUM2_0, 0x7},
	{OP_WR_E1, TCM_REG_PHYS_QNUM2_1, 0x27},
	{OP_WR_E1, TCM_REG_PHYS_QNUM3_0, 0x7},
	{OP_WR_E1, TCM_REG_PHYS_QNUM3_1, 0x27},
	{OP_WR, TCM_REG_TCM_STORM0_IFEN, 0x1},
	{OP_WR, TCM_REG_TCM_STORM1_IFEN, 0x1},
	{OP_WR, TCM_REG_TCM_TQM_IFEN, 0x1},
	{OP_WR, TCM_REG_STORM_TCM_IFEN, 0x1},
	{OP_WR, TCM_REG_TQM_TCM_IFEN, 0x1},
	{OP_WR, TCM_REG_TSDM_IFEN, 0x1},
	{OP_WR, TCM_REG_TM_TCM_IFEN, 0x1},
	{OP_WR, TCM_REG_PRS_IFEN, 0x1},
	{OP_WR, TCM_REG_PBF_IFEN, 0x1},
	{OP_WR, TCM_REG_USEM_IFEN, 0x1},
	{OP_WR, TCM_REG_CSEM_IFEN, 0x1},
	{OP_WR, TCM_REG_CDU_AG_WR_IFEN, 0x1},
	{OP_WR, TCM_REG_CDU_AG_RD_IFEN, 0x1},
	{OP_WR, TCM_REG_CDU_SM_WR_IFEN, 0x1},
	{OP_WR, TCM_REG_CDU_SM_RD_IFEN, 0x1},
	{OP_WR, TCM_REG_TCM_CFC_IFEN, 0x1},
#define TCM_COMMON_END          141
#define TCM_FUNC0_START         141
	{OP_WR_E1H, TCM_REG_PHYS_QNUM0_0, 0xd},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM1_0, 0x7},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM2_0, 0x7},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM3_0, 0x7},
#define TCM_FUNC0_END           145
#define TCM_FUNC1_START         145
	{OP_WR_E1H, TCM_REG_PHYS_QNUM0_1, 0x2d},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM1_1, 0x27},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM2_1, 0x27},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM3_1, 0x27},
#define TCM_FUNC1_END           149
#define TCM_FUNC2_START         149
	{OP_WR_E1H, TCM_REG_PHYS_QNUM0_0, 0x1d},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM1_0, 0x17},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM2_0, 0x17},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM3_0, 0x17},
#define TCM_FUNC2_END           153
#define TCM_FUNC3_START         153
	{OP_WR_E1H, TCM_REG_PHYS_QNUM0_1, 0x3d},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM1_1, 0x37},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM2_1, 0x37},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM3_1, 0x37},
#define TCM_FUNC3_END           157
#define TCM_FUNC4_START         157
	{OP_WR_E1H, TCM_REG_PHYS_QNUM0_0, 0x4d},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM1_0, 0x47},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM2_0, 0x47},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM3_0, 0x47},
#define TCM_FUNC4_END           161
#define TCM_FUNC5_START         161
	{OP_WR_E1H, TCM_REG_PHYS_QNUM0_1, 0x6d},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM1_1, 0x67},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM2_1, 0x67},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM3_1, 0x67},
#define TCM_FUNC5_END           165
#define TCM_FUNC6_START         165
	{OP_WR_E1H, TCM_REG_PHYS_QNUM0_0, 0x5d},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM1_0, 0x57},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM2_0, 0x57},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM3_0, 0x57},
#define TCM_FUNC6_END           169
#define TCM_FUNC7_START         169
	{OP_WR_E1H, TCM_REG_PHYS_QNUM0_1, 0x7d},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM1_1, 0x77},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM2_1, 0x77},
	{OP_WR_E1H, TCM_REG_PHYS_QNUM3_1, 0x77},
#define TCM_FUNC7_END           173
#define BRB1_COMMON_START       173
	{OP_SW, BRB1_REG_LL_RAM, 0x2000020},
	{OP_WR, BRB1_REG_SOFT_RESET, 0x1},
	{OP_RD, BRB1_REG_NUM_OF_FULL_CYCLES_4, 0x0},
	{OP_SW, BRB1_REG_FREE_LIST_PRS_CRDT, 0x30220},
	{OP_WR, BRB1_REG_SOFT_RESET, 0x0},
#define BRB1_COMMON_END         178
#define BRB1_PORT0_START        178
	{OP_WR_E1, BRB1_REG_PAUSE_LOW_THRESHOLD_0, 0xb8},
	{OP_WR_E1, BRB1_REG_PAUSE_HIGH_THRESHOLD_0, 0x114},
	{OP_RD, BRB1_REG_NUM_OF_PAUSE_CYCLES_0, 0x0},
	{OP_RD, BRB1_REG_NUM_OF_FULL_CYCLES_0, 0x0},
#define BRB1_PORT0_END          182
#define BRB1_PORT1_START        182
	{OP_WR_E1, BRB1_REG_PAUSE_LOW_THRESHOLD_1, 0xb8},
	{OP_WR_E1, BRB1_REG_PAUSE_HIGH_THRESHOLD_1, 0x114},
	{OP_RD, BRB1_REG_NUM_OF_PAUSE_CYCLES_1, 0x0},
	{OP_RD, BRB1_REG_NUM_OF_FULL_CYCLES_1, 0x0},
#define BRB1_PORT1_END          186
#define TSEM_COMMON_START       186
	{OP_RD, TSEM_REG_MSG_NUM_FIC0, 0x0},
	{OP_RD, TSEM_REG_MSG_NUM_FIC1, 0x0},
	{OP_RD, TSEM_REG_MSG_NUM_FOC0, 0x0},
	{OP_RD, TSEM_REG_MSG_NUM_FOC1, 0x0},
	{OP_RD, TSEM_REG_MSG_NUM_FOC2, 0x0},
	{OP_RD, TSEM_REG_MSG_NUM_FOC3, 0x0},
	{OP_WR, TSEM_REG_ARB_ELEMENT0, 0x1},
	{OP_WR, TSEM_REG_ARB_ELEMENT1, 0x2},
	{OP_WR, TSEM_REG_ARB_ELEMENT2, 0x3},
	{OP_WR, TSEM_REG_ARB_ELEMENT3, 0x0},
	{OP_WR, TSEM_REG_ARB_ELEMENT4, 0x4},
	{OP_WR, TSEM_REG_ARB_CYCLE_SIZE, 0x1},
	{OP_WR, TSEM_REG_TS_0_AS, 0x0},
	{OP_WR, TSEM_REG_TS_1_AS, 0x1},
	{OP_WR, TSEM_REG_TS_2_AS, 0x4},
	{OP_WR, TSEM_REG_TS_3_AS, 0x0},
	{OP_WR, TSEM_REG_TS_4_AS, 0x1},
	{OP_WR, TSEM_REG_TS_5_AS, 0x3},
	{OP_WR, TSEM_REG_TS_6_AS, 0x0},
	{OP_WR, TSEM_REG_TS_7_AS, 0x1},
	{OP_WR, TSEM_REG_TS_8_AS, 0x4},
	{OP_WR, TSEM_REG_TS_9_AS, 0x0},
	{OP_WR, TSEM_REG_TS_10_AS, 0x1},
	{OP_WR, TSEM_REG_TS_11_AS, 0x3},
	{OP_WR, TSEM_REG_TS_12_AS, 0x0},
	{OP_WR, TSEM_REG_TS_13_AS, 0x1},
	{OP_WR, TSEM_REG_TS_14_AS, 0x4},
	{OP_WR, TSEM_REG_TS_15_AS, 0x0},
	{OP_WR, TSEM_REG_TS_16_AS, 0x4},
	{OP_WR, TSEM_REG_TS_17_AS, 0x3},
	{OP_ZR, TSEM_REG_TS_18_AS, 0x2},
	{OP_WR, TSEM_REG_ENABLE_IN, 0x3fff},
	{OP_WR, TSEM_REG_ENABLE_OUT, 0x3ff},
	{OP_WR, TSEM_REG_FIC0_DISABLE, 0x0},
	{OP_WR, TSEM_REG_FIC1_DISABLE, 0x0},
	{OP_WR, TSEM_REG_PAS_DISABLE, 0x0},
	{OP_WR, TSEM_REG_THREADS_LIST, 0xff},
	{OP_ZR, TSEM_REG_PASSIVE_BUFFER, 0x400},
	{OP_WR, TSEM_REG_FAST_MEMORY + 0x18bc0, 0x1},
	{OP_WR, TSEM_REG_FAST_MEMORY + 0x18000, 0x34},
	{OP_WR, TSEM_REG_FAST_MEMORY + 0x18040, 0x18},
	{OP_WR, TSEM_REG_FAST_MEMORY + 0x18080, 0xc},
	{OP_WR, TSEM_REG_FAST_MEMORY + 0x180c0, 0x20},
	{OP_WR_ASIC, TSEM_REG_FAST_MEMORY + 0x18300, 0x7a120},
	{OP_WR_EMUL, TSEM_REG_FAST_MEMORY + 0x18300, 0x138},
	{OP_WR_FPGA, TSEM_REG_FAST_MEMORY + 0x18300, 0x1388},
	{OP_WR, TSEM_REG_FAST_MEMORY + 0x183c0, 0x1f4},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x2000, 0xb2},
	{OP_WR_E1H, TSEM_REG_FAST_MEMORY + 0x11480, 0x1},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x23c8, 0xc1},
	{OP_WR_EMUL_E1H, TSEM_REG_FAST_MEMORY + 0x11480, 0x0},
	{OP_SW_E1, TSEM_REG_FAST_MEMORY + 0x23c8 + 0x304, 0x10223},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x1000, 0x2b3},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x1020, 0xc8},
	{OP_SW_E1H, TSEM_REG_FAST_MEMORY + 0x1000 + 0xacc, 0x10223},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x1000, 0x2},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0xa020, 0xc8},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x1c18, 0x4},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0xa000, 0x2},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x800, 0x2},
	{OP_WR_E1H, TSEM_REG_FAST_MEMORY + 0x1ad0, 0x0},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x808, 0x2},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x3b28, 0x6},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x810, 0x4},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5000, 0x2},
	{OP_SW_E1, TSEM_REG_FAST_MEMORY + 0x1fb0, 0x40224},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5008, 0x4},
	{OP_SW_E1, TSEM_REG_FAST_MEMORY + 0x4cb0, 0x80228},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5018, 0x4},
	{OP_ZP_E1, TSEM_REG_INT_TABLE, 0x940000},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5028, 0x4},
	{OP_WR_64_E1, TSEM_REG_INT_TABLE + 0x360, 0x140230},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5038, 0x4},
	{OP_ZP_E1, TSEM_REG_PRAM, 0x30b10000},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5048, 0x4},
	{OP_ZP_E1, TSEM_REG_PRAM + 0x8000, 0x33c50c2d},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5058, 0x4},
	{OP_ZP_E1, TSEM_REG_PRAM + 0x10000, 0xbc6191f},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5068, 0x4},
	{OP_WR_64_E1, TSEM_REG_PRAM + 0x117f0, 0x5d020232},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5078, 0x2},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x4000, 0x2},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x4008, 0x2},
	{OP_SW_E1H, TSEM_REG_FAST_MEMORY + 0x6140, 0x200224},
	{OP_ZP_E1H, TSEM_REG_INT_TABLE, 0x960000},
	{OP_WR_64_E1H, TSEM_REG_INT_TABLE + 0x360, 0x140244},
	{OP_ZP_E1H, TSEM_REG_PRAM, 0x30cc0000},
	{OP_ZP_E1H, TSEM_REG_PRAM + 0x8000, 0x33df0c33},
	{OP_ZP_E1H, TSEM_REG_PRAM + 0x10000, 0xdce192b},
	{OP_WR_64_E1H, TSEM_REG_PRAM + 0x11c70, 0x5c720246},
#define TSEM_COMMON_END         276
#define TSEM_PORT0_START        276
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x22c8, 0x20},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x2000, 0x16c},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x4000, 0xfc},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0xb000, 0x28},
	{OP_WR_E1, TSEM_REG_FAST_MEMORY + 0x4b60, 0x0},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0xb140, 0xc},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x1400, 0xa},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x32c0, 0x12},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x1450, 0x6},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x3350, 0xfa},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x1500, 0xe},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x8108, 0x2},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x1570, 0x12},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x9c0, 0xbe},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x800, 0x2},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x820, 0xe},
	{OP_SW_E1, TSEM_REG_FAST_MEMORY + 0x1fb0, 0x20234},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x2908, 0x2},
#define TSEM_PORT0_END          294
#define TSEM_PORT1_START        294
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x2348, 0x20},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x25b0, 0x16c},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x43f0, 0xfc},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0xb0a0, 0x28},
	{OP_WR_E1, TSEM_REG_FAST_MEMORY + 0x4b64, 0x0},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0xb170, 0xc},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x1428, 0xa},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x3308, 0x12},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x1468, 0x6},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x3738, 0xfa},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x1538, 0xe},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x8110, 0x2},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x15b8, 0x12},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0xcb8, 0xbe},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x808, 0x2},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x858, 0xe},
	{OP_SW_E1, TSEM_REG_FAST_MEMORY + 0x1fb8, 0x20236},
	{OP_ZR_E1, TSEM_REG_FAST_MEMORY + 0x2910, 0x2},
#define TSEM_PORT1_END          312
#define TSEM_FUNC0_START        312
	{OP_WR_E1H, TSEM_REG_FAST_MEMORY + 0x2b60, 0x0},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x3000, 0xe},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x31c0, 0x8},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5000, 0x2},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5080, 0x12},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x4000, 0x2},
#define TSEM_FUNC0_END          318
#define TSEM_FUNC1_START        318
	{OP_WR_E1H, TSEM_REG_FAST_MEMORY + 0x2b64, 0x0},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x3038, 0xe},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x31e0, 0x8},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5010, 0x2},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x50c8, 0x12},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x4008, 0x2},
#define TSEM_FUNC1_END          324
#define TSEM_FUNC2_START        324
	{OP_WR_E1H, TSEM_REG_FAST_MEMORY + 0x2b68, 0x0},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x3070, 0xe},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x3200, 0x8},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5020, 0x2},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5110, 0x12},
	{OP_SW_E1H, TSEM_REG_FAST_MEMORY + 0x4010, 0x20248},
#define TSEM_FUNC2_END          330
#define TSEM_FUNC3_START        330
	{OP_WR_E1H, TSEM_REG_FAST_MEMORY + 0x2b6c, 0x0},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x30a8, 0xe},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x3220, 0x8},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5030, 0x2},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5158, 0x12},
	{OP_SW_E1H, TSEM_REG_FAST_MEMORY + 0x4018, 0x2024a},
#define TSEM_FUNC3_END          336
#define TSEM_FUNC4_START        336
	{OP_WR_E1H, TSEM_REG_FAST_MEMORY + 0x2b70, 0x0},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x30e0, 0xe},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x3240, 0x8},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5040, 0x2},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x51a0, 0x12},
	{OP_SW_E1H, TSEM_REG_FAST_MEMORY + 0x4020, 0x2024c},
#define TSEM_FUNC4_END          342
#define TSEM_FUNC5_START        342
	{OP_WR_E1H, TSEM_REG_FAST_MEMORY + 0x2b74, 0x0},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x3118, 0xe},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x3260, 0x8},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5050, 0x2},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x51e8, 0x12},
	{OP_SW_E1H, TSEM_REG_FAST_MEMORY + 0x4028, 0x2024e},
#define TSEM_FUNC5_END          348
#define TSEM_FUNC6_START        348
	{OP_WR_E1H, TSEM_REG_FAST_MEMORY + 0x2b78, 0x0},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x3150, 0xe},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x3280, 0x8},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5060, 0x2},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5230, 0x12},
	{OP_SW_E1H, TSEM_REG_FAST_MEMORY + 0x4030, 0x20250},
#define TSEM_FUNC6_END          354
#define TSEM_FUNC7_START        354
	{OP_WR_E1H, TSEM_REG_FAST_MEMORY + 0x2b7c, 0x0},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x3188, 0xe},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x32a0, 0x8},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5070, 0x2},
	{OP_ZR_E1H, TSEM_REG_FAST_MEMORY + 0x5278, 0x12},
	{OP_SW_E1H, TSEM_REG_FAST_MEMORY + 0x4038, 0x20252},
#define TSEM_FUNC7_END          360
#define MISC_COMMON_START       360
	{OP_WR_E1, MISC_REG_GRC_TIMEOUT_EN, 0x1},
	{OP_WR, MISC_REG_PLL_STORM_CTRL_1, 0x71d2911},
	{OP_WR, MISC_REG_PLL_STORM_CTRL_2, 0x0},
	{OP_WR, MISC_REG_PLL_STORM_CTRL_3, 0x9c0424},
	{OP_WR, MISC_REG_PLL_STORM_CTRL_4, 0x0},
	{OP_WR, MISC_REG_LCPLL_CTRL_1, 0x209},
	{OP_WR_E1, MISC_REG_SPIO, 0xff000000},
#define MISC_COMMON_END         367
#define MISC_FUNC0_START        367
	{OP_WR_E1H, MISC_REG_NIG_WOL_P0, 0x0},
#define MISC_FUNC0_END          368
#define MISC_FUNC1_START        368
	{OP_WR_E1H, MISC_REG_NIG_WOL_P1, 0x0},
#define MISC_FUNC1_END          369
#define MISC_FUNC2_START        369
	{OP_WR_E1H, MISC_REG_NIG_WOL_P0, 0x0},
#define MISC_FUNC2_END          370
#define MISC_FUNC3_START        370
	{OP_WR_E1H, MISC_REG_NIG_WOL_P1, 0x0},
#define MISC_FUNC3_END          371
#define MISC_FUNC4_START        371
	{OP_WR_E1H, MISC_REG_NIG_WOL_P0, 0x0},
#define MISC_FUNC4_END          372
#define MISC_FUNC5_START        372
	{OP_WR_E1H, MISC_REG_NIG_WOL_P1, 0x0},
#define MISC_FUNC5_END          373
#define MISC_FUNC6_START        373
	{OP_WR_E1H, MISC_REG_NIG_WOL_P0, 0x0},
#define MISC_FUNC6_END          374
#define MISC_FUNC7_START        374
	{OP_WR_E1H, MISC_REG_NIG_WOL_P1, 0x0},
#define MISC_FUNC7_END          375
#define NIG_COMMON_START        375
	{OP_WR, NIG_REG_PBF_LB_IN_EN, 0x1},
	{OP_WR, NIG_REG_PRS_REQ_IN_EN, 0x1},
	{OP_WR, NIG_REG_EGRESS_DEBUG_IN_EN, 0x1},
	{OP_WR, NIG_REG_BRB_LB_OUT_EN, 0x1},
	{OP_WR, NIG_REG_PRS_EOP_OUT_EN, 0x1},
#define NIG_COMMON_END          380
#define NIG_PORT0_START         380
	{OP_WR, NIG_REG_LLH0_CM_HEADER, 0x300000},
	{OP_WR, NIG_REG_LLH0_EVENT_ID, 0x28},
	{OP_WR, NIG_REG_LLH0_ERROR_MASK, 0x0},
	{OP_WR, NIG_REG_LLH0_XCM_MASK, 0x4},
	{OP_WR, NIG_REG_LLH0_BRB1_NOT_MCP, 0x1},
	{OP_WR, NIG_REG_STATUS_INTERRUPT_PORT0, 0x0},
	{OP_WR_E1H, NIG_REG_LLH0_CLS_TYPE, 0x1},
	{OP_WR, NIG_REG_LLH0_XCM_INIT_CREDIT, 0x30},
	{OP_WR, NIG_REG_BRB0_PAUSE_IN_EN, 0x1},
	{OP_WR, NIG_REG_EGRESS_PBF0_IN_EN, 0x1},
	{OP_WR, NIG_REG_BRB0_OUT_EN, 0x1},
	{OP_WR, NIG_REG_XCM0_OUT_EN, 0x1},
#define NIG_PORT0_END           392
#define NIG_PORT1_START         392
	{OP_WR, NIG_REG_LLH1_CM_HEADER, 0x300000},
	{OP_WR, NIG_REG_LLH1_EVENT_ID, 0x28},
	{OP_WR, NIG_REG_LLH1_ERROR_MASK, 0x0},
	{OP_WR, NIG_REG_LLH1_XCM_MASK, 0x4},
	{OP_WR, NIG_REG_LLH1_BRB1_NOT_MCP, 0x1},
	{OP_WR, NIG_REG_STATUS_INTERRUPT_PORT1, 0x0},
	{OP_WR_E1H, NIG_REG_LLH1_CLS_TYPE, 0x1},
	{OP_WR, NIG_REG_LLH1_XCM_INIT_CREDIT, 0x30},
	{OP_WR, NIG_REG_BRB1_PAUSE_IN_EN, 0x1},
	{OP_WR, NIG_REG_EGRESS_PBF1_IN_EN, 0x1},
	{OP_WR, NIG_REG_BRB1_OUT_EN, 0x1},
	{OP_WR, NIG_REG_XCM1_OUT_EN, 0x1},
#define NIG_PORT1_END           404
#define UPB_COMMON_START        404
	{OP_WR, GRCBASE_UPB + PB_REG_CONTROL, 0x20},
#define UPB_COMMON_END          405
#define CSDM_COMMON_START       405
	{OP_WR_E1, CSDM_REG_CFC_RSP_START_ADDR, 0xa11},
	{OP_WR_E1H, CSDM_REG_CFC_RSP_START_ADDR, 0x211},
	{OP_WR_E1, CSDM_REG_CMP_COUNTER_START_ADDR, 0xa00},
	{OP_WR_E1H, CSDM_REG_CMP_COUNTER_START_ADDR, 0x200},
	{OP_WR_E1, CSDM_REG_Q_COUNTER_START_ADDR, 0xa04},
	{OP_WR_E1H, CSDM_REG_Q_COUNTER_START_ADDR, 0x204},
	{OP_WR, CSDM_REG_CMP_COUNTER_MAX0, 0xffff},
	{OP_WR, CSDM_REG_CMP_COUNTER_MAX1, 0xffff},
	{OP_WR, CSDM_REG_CMP_COUNTER_MAX2, 0xffff},
	{OP_WR, CSDM_REG_CMP_COUNTER_MAX3, 0xffff},
	{OP_WR, CSDM_REG_AGG_INT_EVENT_0, 0xc6},
	{OP_WR, CSDM_REG_AGG_INT_EVENT_1, 0x0},
	{OP_WR, CSDM_REG_AGG_INT_EVENT_2, 0x34},
	{OP_WR, CSDM_REG_AGG_INT_EVENT_3, 0x35},
	{OP_ZR, CSDM_REG_AGG_INT_EVENT_4, 0x1c},
	{OP_WR, CSDM_REG_AGG_INT_T_0, 0x1},
	{OP_ZR, CSDM_REG_AGG_INT_T_1, 0x5f},
	{OP_WR, CSDM_REG_ENABLE_IN1, 0x7ffffff},
	{OP_WR, CSDM_REG_ENABLE_IN2, 0x3f},
	{OP_WR, CSDM_REG_ENABLE_OUT1, 0x7ffffff},
	{OP_WR, CSDM_REG_ENABLE_OUT2, 0xf},
	{OP_RD, CSDM_REG_NUM_OF_Q0_CMD, 0x0},
	{OP_RD, CSDM_REG_NUM_OF_Q1_CMD, 0x0},
	{OP_RD, CSDM_REG_NUM_OF_Q3_CMD, 0x0},
	{OP_RD, CSDM_REG_NUM_OF_Q4_CMD, 0x0},
	{OP_RD, CSDM_REG_NUM_OF_Q5_CMD, 0x0},
	{OP_RD, CSDM_REG_NUM_OF_Q6_CMD, 0x0},
	{OP_RD, CSDM_REG_NUM_OF_Q7_CMD, 0x0},
	{OP_RD, CSDM_REG_NUM_OF_Q8_CMD, 0x0},
	{OP_RD, CSDM_REG_NUM_OF_Q9_CMD, 0x0},
	{OP_RD, CSDM_REG_NUM_OF_Q10_CMD, 0x0},
	{OP_RD, CSDM_REG_NUM_OF_Q11_CMD, 0x0},
	{OP_RD, CSDM_REG_NUM_OF_PKT_END_MSG, 0x0},
	{OP_RD, CSDM_REG_NUM_OF_PXP_ASYNC_REQ, 0x0},
	{OP_RD, CSDM_REG_NUM_OF_ACK_AFTER_PLACE, 0x0},
	{OP_WR_E1, CSDM_REG_INIT_CREDIT_PXP_CTRL, 0x1},
	{OP_WR_ASIC, CSDM_REG_TIMER_TICK, 0x3e8},
	{OP_WR_EMUL, CSDM_REG_TIMER_TICK, 0x1},
	{OP_WR_FPGA, CSDM_REG_TIMER_TICK, 0xa},
#define CSDM_COMMON_END         444
#define USDM_COMMON_START       444
	{OP_WR_E1, USDM_REG_CFC_RSP_START_ADDR, 0xa11},
	{OP_WR_E1H, USDM_REG_CFC_RSP_START_ADDR, 0x411},
	{OP_WR_E1, USDM_REG_CMP_COUNTER_START_ADDR, 0xa00},
	{OP_WR_E1H, USDM_REG_CMP_COUNTER_START_ADDR, 0x400},
	{OP_WR_E1, USDM_REG_Q_COUNTER_START_ADDR, 0xa04},
	{OP_WR_E1H, USDM_REG_Q_COUNTER_START_ADDR, 0x404},
	{OP_WR_E1, USDM_REG_PCK_END_MSG_START_ADDR, 0xa21},
	{OP_WR_E1H, USDM_REG_PCK_END_MSG_START_ADDR, 0x421},
	{OP_WR, USDM_REG_CMP_COUNTER_MAX0, 0xffff},
	{OP_WR, USDM_REG_CMP_COUNTER_MAX1, 0xffff},
	{OP_WR, USDM_REG_CMP_COUNTER_MAX2, 0xffff},
	{OP_WR, USDM_REG_CMP_COUNTER_MAX3, 0xffff},
	{OP_WR, USDM_REG_AGG_INT_EVENT_0, 0x46},
	{OP_WR, USDM_REG_AGG_INT_EVENT_1, 0x5},
	{OP_WR, USDM_REG_AGG_INT_EVENT_2, 0x34},
	{OP_WR, USDM_REG_AGG_INT_EVENT_3, 0x35},
	{OP_ZR, USDM_REG_AGG_INT_EVENT_4, 0x5c},
	{OP_WR, USDM_REG_AGG_INT_MODE_0, 0x1},
	{OP_ZR, USDM_REG_AGG_INT_MODE_1, 0x1f},
	{OP_WR, USDM_REG_ENABLE_IN1, 0x7ffffff},
	{OP_WR, USDM_REG_ENABLE_IN2, 0x3f},
	{OP_WR, USDM_REG_ENABLE_OUT1, 0x7ffffff},
	{OP_WR, USDM_REG_ENABLE_OUT2, 0xf},
	{OP_RD, USDM_REG_NUM_OF_Q0_CMD, 0x0},
	{OP_RD, USDM_REG_NUM_OF_Q1_CMD, 0x0},
	{OP_RD, USDM_REG_NUM_OF_Q2_CMD, 0x0},
	{OP_RD, USDM_REG_NUM_OF_Q3_CMD, 0x0},
	{OP_RD, USDM_REG_NUM_OF_Q4_CMD, 0x0},
	{OP_RD, USDM_REG_NUM_OF_Q5_CMD, 0x0},
	{OP_RD, USDM_REG_NUM_OF_Q6_CMD, 0x0},
	{OP_RD, USDM_REG_NUM_OF_Q7_CMD, 0x0},
	{OP_RD, USDM_REG_NUM_OF_Q8_CMD, 0x0},
	{OP_RD, USDM_REG_NUM_OF_Q9_CMD, 0x0},
	{OP_RD, USDM_REG_NUM_OF_Q10_CMD, 0x0},
	{OP_RD, USDM_REG_NUM_OF_Q11_CMD, 0x0},
	{OP_RD, USDM_REG_NUM_OF_PKT_END_MSG, 0x0},
	{OP_RD, USDM_REG_NUM_OF_PXP_ASYNC_REQ, 0x0},
	{OP_RD, USDM_REG_NUM_OF_ACK_AFTER_PLACE, 0x0},
	{OP_WR_E1, USDM_REG_INIT_CREDIT_PXP_CTRL, 0x1},
	{OP_WR_ASIC, USDM_REG_TIMER_TICK, 0x3e8},
	{OP_WR_EMUL, USDM_REG_TIMER_TICK, 0x1},
	{OP_WR_FPGA, USDM_REG_TIMER_TICK, 0xa},
#define USDM_COMMON_END         486
#define CCM_COMMON_START        486
	{OP_WR, CCM_REG_XX_OVFL_EVNT_ID, 0x32},
	{OP_WR, CCM_REG_CQM_CCM_HDR_P, 0x2150020},
	{OP_WR, CCM_REG_CQM_CCM_HDR_S, 0x2150020},
	{OP_WR, CCM_REG_ERR_CCM_HDR, 0x8100000},
	{OP_WR, CCM_REG_ERR_EVNT_ID, 0x33},
	{OP_WR, CCM_REG_TSEM_WEIGHT, 0x0},
	{OP_WR, CCM_REG_XSEM_WEIGHT, 0x4},
	{OP_WR, CCM_REG_USEM_WEIGHT, 0x4},
	{OP_ZR, CCM_REG_PBF_WEIGHT, 0x2},
	{OP_WR, CCM_REG_CQM_P_WEIGHT, 0x2},
	{OP_WR, CCM_REG_CCM_CQM_USE_Q, 0x1},
	{OP_WR, CCM_REG_CNT_AUX1_Q, 0x2},
	{OP_WR, CCM_REG_CNT_AUX2_Q, 0x2},
	{OP_WR, CCM_REG_INV_DONE_Q, 0x1},
	{OP_WR, CCM_REG_GR_ARB_TYPE, 0x1},
	{OP_WR, CCM_REG_GR_LD0_PR, 0x1},
	{OP_WR, CCM_REG_GR_LD1_PR, 0x2},
	{OP_WR, CCM_REG_CFC_INIT_CRD, 0x1},
	{OP_WR, CCM_REG_CQM_INIT_CRD, 0x20},
	{OP_WR, CCM_REG_FIC0_INIT_CRD, 0x40},
	{OP_WR, CCM_REG_FIC1_INIT_CRD, 0x40},
	{OP_WR, CCM_REG_XX_INIT_CRD, 0x3},
	{OP_WR, CCM_REG_XX_MSG_NUM, 0x18},
	{OP_ZR, CCM_REG_XX_TABLE, 0x12},
	{OP_SW_E1, CCM_REG_XX_DESCR_TABLE, 0x240238},
	{OP_SW_E1H, CCM_REG_XX_DESCR_TABLE, 0x240254},
	{OP_WR, CCM_REG_N_SM_CTX_LD_0, 0x1},
	{OP_WR, CCM_REG_N_SM_CTX_LD_1, 0x2},
	{OP_WR, CCM_REG_N_SM_CTX_LD_2, 0x8},
	{OP_WR, CCM_REG_N_SM_CTX_LD_3, 0x8},
	{OP_ZR, CCM_REG_N_SM_CTX_LD_4, 0x4},
	{OP_WR, CCM_REG_CCM_REG0_SZ, 0x4},
	{OP_WR_E1, CCM_REG_QOS_PHYS_QNUM0_0, 0x9},
	{OP_WR_E1, CCM_REG_QOS_PHYS_QNUM0_1, 0x29},
	{OP_WR_E1, CCM_REG_QOS_PHYS_QNUM1_0, 0xa},
	{OP_WR_E1, CCM_REG_QOS_PHYS_QNUM1_1, 0x2a},
	{OP_WR_E1, CCM_REG_QOS_PHYS_QNUM2_0, 0x7},
	{OP_WR_E1, CCM_REG_QOS_PHYS_QNUM2_1, 0x27},
	{OP_WR_E1, CCM_REG_QOS_PHYS_QNUM3_0, 0x7},
	{OP_WR_E1, CCM_REG_QOS_PHYS_QNUM3_1, 0x27},
	{OP_WR_E1, CCM_REG_PHYS_QNUM1_0, 0xc},
	{OP_WR_E1, CCM_REG_PHYS_QNUM1_1, 0x2c},
	{OP_WR_E1, CCM_REG_PHYS_QNUM2_0, 0xc},
	{OP_WR_E1, CCM_REG_PHYS_QNUM2_1, 0x2c},
	{OP_WR_E1, CCM_REG_PHYS_QNUM3_0, 0xc},
	{OP_WR_E1, CCM_REG_PHYS_QNUM3_1, 0x2c},
	{OP_WR, CCM_REG_CCM_STORM0_IFEN, 0x1},
	{OP_WR, CCM_REG_CCM_STORM1_IFEN, 0x1},
	{OP_WR, CCM_REG_CCM_CQM_IFEN, 0x1},
	{OP_WR, CCM_REG_STORM_CCM_IFEN, 0x1},
	{OP_WR, CCM_REG_CQM_CCM_IFEN, 0x1},
	{OP_WR, CCM_REG_CSDM_IFEN, 0x1},
	{OP_WR, CCM_REG_TSEM_IFEN, 0x1},
	{OP_WR, CCM_REG_XSEM_IFEN, 0x1},
	{OP_WR, CCM_REG_USEM_IFEN, 0x1},
	{OP_WR, CCM_REG_PBF_IFEN, 0x1},
	{OP_WR, CCM_REG_CDU_AG_WR_IFEN, 0x1},
	{OP_WR, CCM_REG_CDU_AG_RD_IFEN, 0x1},
	{OP_WR, CCM_REG_CDU_SM_WR_IFEN, 0x1},
	{OP_WR, CCM_REG_CDU_SM_RD_IFEN, 0x1},
	{OP_WR, CCM_REG_CCM_CFC_IFEN, 0x1},
#define CCM_COMMON_END          547
#define CCM_FUNC0_START         547
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM0_0, 0x9},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM1_0, 0xa},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM2_0, 0x7},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM3_0, 0x7},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM1_0, 0xc},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM2_0, 0xb},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM3_0, 0x7},
#define CCM_FUNC0_END           554
#define CCM_FUNC1_START         554
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM0_1, 0x29},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM1_1, 0x2a},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM2_1, 0x27},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM3_1, 0x27},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM1_1, 0x2c},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM2_1, 0x2b},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM3_1, 0x27},
#define CCM_FUNC1_END           561
#define CCM_FUNC2_START         561
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM0_0, 0x19},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM1_0, 0x1a},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM2_0, 0x17},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM3_0, 0x17},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM1_0, 0x1c},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM2_0, 0x1b},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM3_0, 0x17},
#define CCM_FUNC2_END           568
#define CCM_FUNC3_START         568
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM0_1, 0x39},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM1_1, 0x3a},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM2_1, 0x37},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM3_1, 0x37},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM1_1, 0x3c},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM2_1, 0x3b},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM3_1, 0x37},
#define CCM_FUNC3_END           575
#define CCM_FUNC4_START         575
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM0_0, 0x49},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM1_0, 0x4a},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM2_0, 0x47},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM3_0, 0x47},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM1_0, 0x4c},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM2_0, 0x4b},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM3_0, 0x47},
#define CCM_FUNC4_END           582
#define CCM_FUNC5_START         582
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM0_1, 0x69},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM1_1, 0x6a},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM2_1, 0x67},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM3_1, 0x67},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM1_1, 0x6c},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM2_1, 0x6b},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM3_1, 0x67},
#define CCM_FUNC5_END           589
#define CCM_FUNC6_START         589
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM0_0, 0x59},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM1_0, 0x5a},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM2_0, 0x57},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM3_0, 0x57},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM1_0, 0x5c},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM2_0, 0x5b},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM3_0, 0x57},
#define CCM_FUNC6_END           596
#define CCM_FUNC7_START         596
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM0_1, 0x79},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM1_1, 0x7a},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM2_1, 0x77},
	{OP_WR_E1H, CCM_REG_QOS_PHYS_QNUM3_1, 0x77},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM1_1, 0x7c},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM2_1, 0x7b},
	{OP_WR_E1H, CCM_REG_PHYS_QNUM3_1, 0x77},
#define CCM_FUNC7_END           603
#define UCM_COMMON_START        603
	{OP_WR, UCM_REG_XX_OVFL_EVNT_ID, 0x32},
	{OP_WR, UCM_REG_UQM_UCM_HDR_P, 0x2150020},
	{OP_WR, UCM_REG_UQM_UCM_HDR_S, 0x2150020},
	{OP_WR, UCM_REG_TM_UCM_HDR, 0x30},
	{OP_WR, UCM_REG_ERR_UCM_HDR, 0x8100000},
	{OP_WR, UCM_REG_ERR_EVNT_ID, 0x33},
	{OP_WR, UCM_REG_EXPR_EVNT_ID, 0x30},
	{OP_WR, UCM_REG_STOP_EVNT_ID, 0x31},
	{OP_WR, UCM_REG_TSEM_WEIGHT, 0x3},
	{OP_WR, UCM_REG_CSEM_WEIGHT, 0x0},
	{OP_WR, UCM_REG_CP_WEIGHT, 0x0},
	{OP_WR, UCM_REG_UQM_P_WEIGHT, 0x6},
	{OP_WR, UCM_REG_UCM_UQM_USE_Q, 0x1},
	{OP_WR, UCM_REG_INV_CFLG_Q, 0x1},
	{OP_WR, UCM_REG_GR_ARB_TYPE, 0x1},
	{OP_WR, UCM_REG_GR_LD0_PR, 0x1},
	{OP_WR, UCM_REG_GR_LD1_PR, 0x2},
	{OP_WR, UCM_REG_CFC_INIT_CRD, 0x1},
	{OP_WR, UCM_REG_FIC0_INIT_CRD, 0x40},
	{OP_WR, UCM_REG_FIC1_INIT_CRD, 0x40},
	{OP_WR, UCM_REG_TM_INIT_CRD, 0x4},
	{OP_WR, UCM_REG_UQM_INIT_CRD, 0x20},
	{OP_WR, UCM_REG_XX_INIT_CRD, 0xe},
	{OP_WR, UCM_REG_XX_MSG_NUM, 0x1b},
	{OP_ZR, UCM_REG_XX_TABLE, 0x12},
	{OP_SW_E1, UCM_REG_XX_DESCR_TABLE, 0x1b025c},
	{OP_SW_E1H, UCM_REG_XX_DESCR_TABLE, 0x1b0278},
	{OP_WR, UCM_REG_N_SM_CTX_LD_0, 0x10},
	{OP_WR, UCM_REG_N_SM_CTX_LD_1, 0x7},
	{OP_WR, UCM_REG_N_SM_CTX_LD_2, 0xf},
	{OP_WR, UCM_REG_N_SM_CTX_LD_3, 0x10},
	{OP_ZR_E1, UCM_REG_N_SM_CTX_LD_4, 0x4},
	{OP_WR_E1H, UCM_REG_N_SM_CTX_LD_4, 0xd},
	{OP_ZR_E1H, UCM_REG_N_SM_CTX_LD_5, 0x3},
	{OP_WR, UCM_REG_UCM_REG0_SZ, 0x3},
	{OP_WR_E1, UCM_REG_PHYS_QNUM0_0, 0xf},
	{OP_WR_E1, UCM_REG_PHYS_QNUM0_1, 0x2f},
	{OP_WR_E1, UCM_REG_PHYS_QNUM1_0, 0xe},
	{OP_WR_E1, UCM_REG_PHYS_QNUM1_1, 0x2e},
	{OP_WR, UCM_REG_UCM_STORM0_IFEN, 0x1},
	{OP_WR, UCM_REG_UCM_STORM1_IFEN, 0x1},
	{OP_WR, UCM_REG_UCM_UQM_IFEN, 0x1},
	{OP_WR, UCM_REG_STORM_UCM_IFEN, 0x1},
	{OP_WR, UCM_REG_UQM_UCM_IFEN, 0x1},
	{OP_WR, UCM_REG_USDM_IFEN, 0x1},
	{OP_WR, UCM_REG_TM_UCM_IFEN, 0x1},
	{OP_WR, UCM_REG_UCM_TM_IFEN, 0x1},
	{OP_WR, UCM_REG_TSEM_IFEN, 0x1},
	{OP_WR, UCM_REG_CSEM_IFEN, 0x1},
	{OP_WR, UCM_REG_XSEM_IFEN, 0x1},
	{OP_WR, UCM_REG_DORQ_IFEN, 0x1},
	{OP_WR, UCM_REG_CDU_AG_WR_IFEN, 0x1},
	{OP_WR, UCM_REG_CDU_AG_RD_IFEN, 0x1},
	{OP_WR, UCM_REG_CDU_SM_WR_IFEN, 0x1},
	{OP_WR, UCM_REG_CDU_SM_RD_IFEN, 0x1},
	{OP_WR, UCM_REG_UCM_CFC_IFEN, 0x1},
#define UCM_COMMON_END          659
#define UCM_FUNC0_START         659
	{OP_WR_E1H, UCM_REG_PHYS_QNUM0_0, 0xf},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM1_0, 0xe},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM2_0, 0x0},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM3_0, 0x0},
#define UCM_FUNC0_END           663
#define UCM_FUNC1_START         663
	{OP_WR_E1H, UCM_REG_PHYS_QNUM0_1, 0x2f},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM1_1, 0x2e},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM2_1, 0x0},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM3_1, 0x0},
#define UCM_FUNC1_END           667
#define UCM_FUNC2_START         667
	{OP_WR_E1H, UCM_REG_PHYS_QNUM0_0, 0x1f},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM1_0, 0x1e},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM2_0, 0x0},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM3_0, 0x0},
#define UCM_FUNC2_END           671
#define UCM_FUNC3_START         671
	{OP_WR_E1H, UCM_REG_PHYS_QNUM0_1, 0x3f},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM1_1, 0x3e},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM2_1, 0x0},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM3_1, 0x0},
#define UCM_FUNC3_END           675
#define UCM_FUNC4_START         675
	{OP_WR_E1H, UCM_REG_PHYS_QNUM0_0, 0x4f},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM1_0, 0x4e},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM2_0, 0x0},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM3_0, 0x0},
#define UCM_FUNC4_END           679
#define UCM_FUNC5_START         679
	{OP_WR_E1H, UCM_REG_PHYS_QNUM0_1, 0x6f},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM1_1, 0x6e},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM2_1, 0x0},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM3_1, 0x0},
#define UCM_FUNC5_END           683
#define UCM_FUNC6_START         683
	{OP_WR_E1H, UCM_REG_PHYS_QNUM0_0, 0x5f},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM1_0, 0x5e},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM2_0, 0x0},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM3_0, 0x0},
#define UCM_FUNC6_END           687
#define UCM_FUNC7_START         687
	{OP_WR_E1H, UCM_REG_PHYS_QNUM0_1, 0x7f},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM1_1, 0x7e},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM2_1, 0x0},
	{OP_WR_E1H, UCM_REG_PHYS_QNUM3_1, 0x0},
#define UCM_FUNC7_END           691
#define USEM_COMMON_START       691
	{OP_RD, USEM_REG_MSG_NUM_FIC0, 0x0},
	{OP_RD, USEM_REG_MSG_NUM_FIC1, 0x0},
	{OP_RD, USEM_REG_MSG_NUM_FOC0, 0x0},
	{OP_RD, USEM_REG_MSG_NUM_FOC1, 0x0},
	{OP_RD, USEM_REG_MSG_NUM_FOC2, 0x0},
	{OP_RD, USEM_REG_MSG_NUM_FOC3, 0x0},
	{OP_WR, USEM_REG_ARB_ELEMENT0, 0x1},
	{OP_WR, USEM_REG_ARB_ELEMENT1, 0x2},
	{OP_WR, USEM_REG_ARB_ELEMENT2, 0x3},
	{OP_WR, USEM_REG_ARB_ELEMENT3, 0x0},
	{OP_WR, USEM_REG_ARB_ELEMENT4, 0x4},
	{OP_WR, USEM_REG_ARB_CYCLE_SIZE, 0x1},
	{OP_WR, USEM_REG_TS_0_AS, 0x0},
	{OP_WR, USEM_REG_TS_1_AS, 0x1},
	{OP_WR, USEM_REG_TS_2_AS, 0x4},
	{OP_WR, USEM_REG_TS_3_AS, 0x0},
	{OP_WR, USEM_REG_TS_4_AS, 0x1},
	{OP_WR, USEM_REG_TS_5_AS, 0x3},
	{OP_WR, USEM_REG_TS_6_AS, 0x0},
	{OP_WR, USEM_REG_TS_7_AS, 0x1},
	{OP_WR, USEM_REG_TS_8_AS, 0x4},
	{OP_WR, USEM_REG_TS_9_AS, 0x0},
	{OP_WR, USEM_REG_TS_10_AS, 0x1},
	{OP_WR, USEM_REG_TS_11_AS, 0x3},
	{OP_WR, USEM_REG_TS_12_AS, 0x0},
	{OP_WR, USEM_REG_TS_13_AS, 0x1},
	{OP_WR, USEM_REG_TS_14_AS, 0x4},
	{OP_WR, USEM_REG_TS_15_AS, 0x0},
	{OP_WR, USEM_REG_TS_16_AS, 0x4},
	{OP_WR, USEM_REG_TS_17_AS, 0x3},
	{OP_ZR, USEM_REG_TS_18_AS, 0x2},
	{OP_WR, USEM_REG_ENABLE_IN, 0x3fff},
	{OP_WR, USEM_REG_ENABLE_OUT, 0x3ff},
	{OP_WR, USEM_REG_FIC0_DISABLE, 0x0},
	{OP_WR, USEM_REG_FIC1_DISABLE, 0x0},
	{OP_WR, USEM_REG_PAS_DISABLE, 0x0},
	{OP_WR, USEM_REG_THREADS_LIST, 0xffff},
	{OP_ZR, USEM_REG_PASSIVE_BUFFER, 0x800},
	{OP_WR, USEM_REG_FAST_MEMORY + 0x18bc0, 0x1},
	{OP_WR, USEM_REG_FAST_MEMORY + 0x18000, 0x1a},
	{OP_WR, USEM_REG_FAST_MEMORY + 0x18040, 0x4e},
	{OP_WR, USEM_REG_FAST_MEMORY + 0x18080, 0x10},
	{OP_WR, USEM_REG_FAST_MEMORY + 0x180c0, 0x20},
	{OP_WR_ASIC, USEM_REG_FAST_MEMORY + 0x18300, 0x7a120},
	{OP_WR_EMUL, USEM_REG_FAST_MEMORY + 0x18300, 0x138},
	{OP_WR_FPGA, USEM_REG_FAST_MEMORY + 0x18300, 0x1388},
	{OP_WR, USEM_REG_FAST_MEMORY + 0x183c0, 0x1f4},
	{OP_WR_ASIC, USEM_REG_FAST_MEMORY + 0x18380, 0x1dcd6500},
	{OP_WR_EMUL, USEM_REG_FAST_MEMORY + 0x18380, 0x4c4b4},
	{OP_WR_FPGA, USEM_REG_FAST_MEMORY + 0x18380, 0x4c4b40},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x5000, 0x102},
	{OP_WR_EMUL_E1H, USEM_REG_FAST_MEMORY + 0x11480, 0x0},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x1020, 0xc8},
	{OP_WR_E1H, USEM_REG_FAST_MEMORY + 0x11480, 0x1},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x1000, 0x2},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x2000, 0x102},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x57e8, 0x4},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x8020, 0xc8},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x57d0, 0x5},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x8000, 0x2},
	{OP_SW_E1, USEM_REG_FAST_MEMORY + 0x57d0 + 0x14, 0x10277},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x3760, 0x4},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x1e20, 0x42},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x3738, 0x9},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x4b68, 0x2},
	{OP_SW_E1H, USEM_REG_FAST_MEMORY + 0x3738 + 0x24, 0x10293},
	{OP_SW_E1, USEM_REG_FAST_MEMORY + 0x4b68 + 0x8, 0x20278},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x3180, 0x42},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x4b10, 0x2},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5000, 0x400},
	{OP_SW_E1, USEM_REG_FAST_MEMORY + 0x2830, 0x2027a},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x4000, 0x2},
	{OP_SW_E1H, USEM_REG_FAST_MEMORY + 0x4000 + 0x8, 0x20294},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x6b68, 0x2},
	{OP_SW_E1H, USEM_REG_FAST_MEMORY + 0x6b68 + 0x8, 0x20296},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x6b10, 0x2},
	{OP_SW_E1H, USEM_REG_FAST_MEMORY + 0x74c0, 0x20298},
	{OP_WR, USEM_REG_FAST_MEMORY + 0x10800, 0x1000000},
	{OP_SW_E1, USEM_REG_FAST_MEMORY + 0x10c00, 0x10027c},
	{OP_SW_E1H, USEM_REG_FAST_MEMORY + 0x10c00, 0x10029a},
	{OP_WR, USEM_REG_FAST_MEMORY + 0x10800, 0x0},
	{OP_SW_E1, USEM_REG_FAST_MEMORY + 0x10c40, 0x10028c},
	{OP_SW_E1H, USEM_REG_FAST_MEMORY + 0x10c40, 0x1002aa},
	{OP_ZP_E1, USEM_REG_INT_TABLE, 0xc20000},
	{OP_ZP_E1H, USEM_REG_INT_TABLE, 0xc40000},
	{OP_WR_64_E1, USEM_REG_INT_TABLE + 0x368, 0x13029c},
	{OP_WR_64_E1H, USEM_REG_INT_TABLE + 0x368, 0x1302ba},
	{OP_ZP_E1, USEM_REG_PRAM, 0x311c0000},
	{OP_ZP_E1H, USEM_REG_PRAM, 0x31070000},
	{OP_ZP_E1, USEM_REG_PRAM + 0x8000, 0x33450c47},
	{OP_ZP_E1H, USEM_REG_PRAM + 0x8000, 0x330e0c42},
	{OP_ZP_E1, USEM_REG_PRAM + 0x10000, 0x38561919},
	{OP_ZP_E1H, USEM_REG_PRAM + 0x10000, 0x389b1906},
	{OP_WR_64_E1, USEM_REG_PRAM + 0x17fe0, 0x5004029e},
	{OP_ZP_E1H, USEM_REG_PRAM + 0x18000, 0x132272d},
	{OP_WR_64_E1H, USEM_REG_PRAM + 0x18250, 0x4fb602bc},
#define USEM_COMMON_END         787
#define USEM_PORT0_START        787
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x1400, 0xa0},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x9000, 0xa0},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x1900, 0xa},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x9500, 0x28},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x1950, 0x2e},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x9640, 0x34},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x1d00, 0x4},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x3080, 0x20},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x1d20, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x3288, 0x96},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x5440, 0x72},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5000, 0x20},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x4b78, 0x52},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5100, 0x20},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x4e08, 0xc},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5200, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5300, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5400, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5500, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5600, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5700, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5800, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5900, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5a00, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5b00, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5c00, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5d00, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5e00, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5f00, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x6b78, 0x52},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x6e08, 0xc},
#define USEM_PORT0_END          818
#define USEM_PORT1_START        818
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x1680, 0xa0},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x9280, 0xa0},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x1928, 0xa},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x95a0, 0x28},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x1a08, 0x2e},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x9710, 0x34},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x1d10, 0x4},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x3100, 0x20},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x1da0, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x34e0, 0x96},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x5608, 0x72},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5080, 0x20},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x4cc0, 0x52},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5180, 0x20},
	{OP_ZR_E1, USEM_REG_FAST_MEMORY + 0x4e38, 0xc},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5280, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5380, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5480, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5580, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5680, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5780, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5880, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5980, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5a80, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5b80, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5c80, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5d80, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5e80, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x5f80, 0x20},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x6cc0, 0x52},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x6e38, 0xc},
#define USEM_PORT1_END          849
#define USEM_FUNC0_START        849
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x3000, 0x4},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x4010, 0x2},
#define USEM_FUNC0_END          851
#define USEM_FUNC1_START        851
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x3010, 0x4},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x4020, 0x2},
#define USEM_FUNC1_END          853
#define USEM_FUNC2_START        853
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x3020, 0x4},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x4030, 0x2},
#define USEM_FUNC2_END          855
#define USEM_FUNC3_START        855
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x3030, 0x4},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x4040, 0x2},
#define USEM_FUNC3_END          857
#define USEM_FUNC4_START        857
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x3040, 0x4},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x4050, 0x2},
#define USEM_FUNC4_END          859
#define USEM_FUNC5_START        859
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x3050, 0x4},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x4060, 0x2},
#define USEM_FUNC5_END          861
#define USEM_FUNC6_START        861
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x3060, 0x4},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x4070, 0x2},
#define USEM_FUNC6_END          863
#define USEM_FUNC7_START        863
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x3070, 0x4},
	{OP_ZR_E1H, USEM_REG_FAST_MEMORY + 0x4080, 0x2},
#define USEM_FUNC7_END          865
#define CSEM_COMMON_START       865
	{OP_RD, CSEM_REG_MSG_NUM_FIC0, 0x0},
	{OP_RD, CSEM_REG_MSG_NUM_FIC1, 0x0},
	{OP_RD, CSEM_REG_MSG_NUM_FOC0, 0x0},
	{OP_RD, CSEM_REG_MSG_NUM_FOC1, 0x0},
	{OP_RD, CSEM_REG_MSG_NUM_FOC2, 0x0},
	{OP_RD, CSEM_REG_MSG_NUM_FOC3, 0x0},
	{OP_WR, CSEM_REG_ARB_ELEMENT0, 0x1},
	{OP_WR, CSEM_REG_ARB_ELEMENT1, 0x2},
	{OP_WR, CSEM_REG_ARB_ELEMENT2, 0x3},
	{OP_WR, CSEM_REG_ARB_ELEMENT3, 0x0},
	{OP_WR, CSEM_REG_ARB_ELEMENT4, 0x4},
	{OP_WR, CSEM_REG_ARB_CYCLE_SIZE, 0x1},
	{OP_WR, CSEM_REG_TS_0_AS, 0x0},
	{OP_WR, CSEM_REG_TS_1_AS, 0x1},
	{OP_WR, CSEM_REG_TS_2_AS, 0x4},
	{OP_WR, CSEM_REG_TS_3_AS, 0x0},
	{OP_WR, CSEM_REG_TS_4_AS, 0x1},
	{OP_WR, CSEM_REG_TS_5_AS, 0x3},
	{OP_WR, CSEM_REG_TS_6_AS, 0x0},
	{OP_WR, CSEM_REG_TS_7_AS, 0x1},
	{OP_WR, CSEM_REG_TS_8_AS, 0x4},
	{OP_WR, CSEM_REG_TS_9_AS, 0x0},
	{OP_WR, CSEM_REG_TS_10_AS, 0x1},
	{OP_WR, CSEM_REG_TS_11_AS, 0x3},
	{OP_WR, CSEM_REG_TS_12_AS, 0x0},
	{OP_WR, CSEM_REG_TS_13_AS, 0x1},
	{OP_WR, CSEM_REG_TS_14_AS, 0x4},
	{OP_WR, CSEM_REG_TS_15_AS, 0x0},
	{OP_WR, CSEM_REG_TS_16_AS, 0x4},
	{OP_WR, CSEM_REG_TS_17_AS, 0x3},
	{OP_ZR, CSEM_REG_TS_18_AS, 0x2},
	{OP_WR, CSEM_REG_ENABLE_IN, 0x3fff},
	{OP_WR, CSEM_REG_ENABLE_OUT, 0x3ff},
	{OP_WR, CSEM_REG_FIC0_DISABLE, 0x0},
	{OP_WR, CSEM_REG_FIC1_DISABLE, 0x0},
	{OP_WR, CSEM_REG_PAS_DISABLE, 0x0},
	{OP_WR, CSEM_REG_THREADS_LIST, 0xffff},
	{OP_ZR, CSEM_REG_PASSIVE_BUFFER, 0x800},
	{OP_WR, CSEM_REG_FAST_MEMORY + 0x18bc0, 0x1},
	{OP_WR, CSEM_REG_FAST_MEMORY + 0x18000, 0x10},
	{OP_WR, CSEM_REG_FAST_MEMORY + 0x18040, 0x12},
	{OP_WR, CSEM_REG_FAST_MEMORY + 0x18080, 0x30},
	{OP_WR, CSEM_REG_FAST_MEMORY + 0x180c0, 0xe},
	{OP_WR, CSEM_REG_FAST_MEMORY + 0x183c0, 0x1f4},
	{OP_ZR_E1, CSEM_REG_FAST_MEMORY + 0x5000, 0x42},
	{OP_WR_E1H, CSEM_REG_FAST_MEMORY + 0x11480, 0x1},
	{OP_ZR_E1, CSEM_REG_FAST_MEMORY + 0x1020, 0xc8},
	{OP_WR_EMUL_E1H, CSEM_REG_FAST_MEMORY + 0x11480, 0x0},
	{OP_ZR_E1, CSEM_REG_FAST_MEMORY + 0x1000, 0x2},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x1000, 0x42},
	{OP_ZR_E1, CSEM_REG_FAST_MEMORY + 0x2000, 0xc0},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x7020, 0xc8},
	{OP_ZR_E1, CSEM_REG_FAST_MEMORY + 0x3070, 0x80},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x7000, 0x2},
	{OP_ZR_E1, CSEM_REG_FAST_MEMORY + 0x4280, 0x4},
	{OP_WR_E1H, CSEM_REG_FAST_MEMORY + 0x11e8, 0x0},
	{OP_ZR_E1, CSEM_REG_FAST_MEMORY + 0x25c0, 0x240},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x3000, 0xc0},
	{OP_SW_E1, CSEM_REG_FAST_MEMORY + 0x2ec8, 0x802a0},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x4070, 0x80},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x5280, 0x4},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x6280, 0x240},
	{OP_SW_E1H, CSEM_REG_FAST_MEMORY + 0x6b88, 0x2002be},
	{OP_WR, CSEM_REG_FAST_MEMORY + 0x10800, 0x13fffff},
	{OP_SW_E1, CSEM_REG_FAST_MEMORY + 0x10c00, 0x1002a8},
	{OP_SW_E1H, CSEM_REG_FAST_MEMORY + 0x10c00, 0x1002de},
	{OP_WR, CSEM_REG_FAST_MEMORY + 0x10800, 0x0},
	{OP_SW_E1, CSEM_REG_FAST_MEMORY + 0x10c40, 0x1002b8},
	{OP_SW_E1H, CSEM_REG_FAST_MEMORY + 0x10c40, 0x1002ee},
	{OP_ZP_E1, CSEM_REG_INT_TABLE, 0x6e0000},
	{OP_ZP_E1H, CSEM_REG_INT_TABLE, 0x6f0000},
	{OP_WR_64_E1, CSEM_REG_INT_TABLE + 0x380, 0x1002c8},
	{OP_WR_64_E1H, CSEM_REG_INT_TABLE + 0x380, 0x1002fe},
	{OP_ZP_E1, CSEM_REG_PRAM, 0x32580000},
	{OP_ZP_E1H, CSEM_REG_PRAM, 0x31fa0000},
	{OP_ZP_E1, CSEM_REG_PRAM + 0x8000, 0x18270c96},
	{OP_ZP_E1H, CSEM_REG_PRAM + 0x8000, 0x19040c7f},
	{OP_WR_64_E1, CSEM_REG_PRAM + 0xb210, 0x682402ca},
	{OP_WR_64_E1H, CSEM_REG_PRAM + 0xb430, 0x67e00300},
#define CSEM_COMMON_END         944
#define CSEM_PORT0_START        944
	{OP_ZR_E1, CSEM_REG_FAST_MEMORY + 0x1400, 0xa0},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x8000, 0xa0},
	{OP_ZR_E1, CSEM_REG_FAST_MEMORY + 0x1900, 0x10},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x8500, 0x40},
	{OP_ZR_E1, CSEM_REG_FAST_MEMORY + 0x1980, 0x30},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x8700, 0x3c},
	{OP_WR_E1, CSEM_REG_FAST_MEMORY + 0x5118, 0x0},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x4040, 0x6},
	{OP_ZR_E1, CSEM_REG_FAST_MEMORY + 0x2300, 0xe},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x6040, 0x30},
	{OP_ZR_E1, CSEM_REG_FAST_MEMORY + 0x3040, 0x6},
	{OP_ZR_E1, CSEM_REG_FAST_MEMORY + 0x2410, 0x30},
#define CSEM_PORT0_END          956
#define CSEM_PORT1_START        956
	{OP_ZR_E1, CSEM_REG_FAST_MEMORY + 0x1680, 0xa0},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x8280, 0xa0},
	{OP_ZR_E1, CSEM_REG_FAST_MEMORY + 0x1940, 0x10},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x8600, 0x40},
	{OP_ZR_E1, CSEM_REG_FAST_MEMORY + 0x1a40, 0x30},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x87f0, 0x3c},
	{OP_WR_E1, CSEM_REG_FAST_MEMORY + 0x511c, 0x0},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x4058, 0x6},
	{OP_ZR_E1, CSEM_REG_FAST_MEMORY + 0x2338, 0xe},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x6100, 0x30},
	{OP_ZR_E1, CSEM_REG_FAST_MEMORY + 0x3058, 0x6},
	{OP_ZR_E1, CSEM_REG_FAST_MEMORY + 0x24d0, 0x30},
#define CSEM_PORT1_END          968
#define CSEM_FUNC0_START        968
	{OP_WR_E1H, CSEM_REG_FAST_MEMORY + 0x1148, 0x0},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x3300, 0x2},
#define CSEM_FUNC0_END          970
#define CSEM_FUNC1_START        970
	{OP_WR_E1H, CSEM_REG_FAST_MEMORY + 0x114c, 0x0},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x3308, 0x2},
#define CSEM_FUNC1_END          972
#define CSEM_FUNC2_START        972
	{OP_WR_E1H, CSEM_REG_FAST_MEMORY + 0x1150, 0x0},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x3310, 0x2},
#define CSEM_FUNC2_END          974
#define CSEM_FUNC3_START        974
	{OP_WR_E1H, CSEM_REG_FAST_MEMORY + 0x1154, 0x0},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x3318, 0x2},
#define CSEM_FUNC3_END          976
#define CSEM_FUNC4_START        976
	{OP_WR_E1H, CSEM_REG_FAST_MEMORY + 0x1158, 0x0},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x3320, 0x2},
#define CSEM_FUNC4_END          978
#define CSEM_FUNC5_START        978
	{OP_WR_E1H, CSEM_REG_FAST_MEMORY + 0x115c, 0x0},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x3328, 0x2},
#define CSEM_FUNC5_END          980
#define CSEM_FUNC6_START        980
	{OP_WR_E1H, CSEM_REG_FAST_MEMORY + 0x1160, 0x0},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x3330, 0x2},
#define CSEM_FUNC6_END          982
#define CSEM_FUNC7_START        982
	{OP_WR_E1H, CSEM_REG_FAST_MEMORY + 0x1164, 0x0},
	{OP_ZR_E1H, CSEM_REG_FAST_MEMORY + 0x3338, 0x2},
#define CSEM_FUNC7_END          984
#define XPB_COMMON_START        984
	{OP_WR, GRCBASE_XPB + PB_REG_CONTROL, 0x20},
#define XPB_COMMON_END          985
#define DQ_COMMON_START         985
	{OP_WR, DORQ_REG_MODE_ACT, 0x2},
	{OP_WR, DORQ_REG_NORM_CID_OFST, 0x3},
	{OP_WR, DORQ_REG_OUTST_REQ, 0x4},
	{OP_WR, DORQ_REG_DPM_CID_ADDR, 0x8},
	{OP_WR, DORQ_REG_RSP_INIT_CRD, 0x2},
	{OP_WR, DORQ_REG_NORM_CMHEAD_TX, 0x90},
	{OP_WR, DORQ_REG_CMHEAD_RX, 0x90},
	{OP_WR, DORQ_REG_SHRT_CMHEAD, 0x800090},
	{OP_WR, DORQ_REG_ERR_CMHEAD, 0x8140000},
	{OP_WR, DORQ_REG_AGG_CMD0, 0x8a},
	{OP_WR, DORQ_REG_AGG_CMD1, 0x80},
	{OP_WR, DORQ_REG_AGG_CMD2, 0x90},
	{OP_WR, DORQ_REG_AGG_CMD3, 0x80},
	{OP_WR, DORQ_REG_SHRT_ACT_CNT, 0x6},
	{OP_WR, DORQ_REG_DQ_FIFO_FULL_TH, 0x7d0},
	{OP_WR, DORQ_REG_DQ_FIFO_AFULL_TH, 0x76c},
	{OP_WR, DORQ_REG_REGN, 0x7c1004},
	{OP_WR, DORQ_REG_IF_EN, 0xf},
#define DQ_COMMON_END           1003
#define TIMERS_COMMON_START     1003
	{OP_ZR, TM_REG_CLIN_PRIOR0_CLIENT, 0x2},
	{OP_WR, TM_REG_LIN_SETCLR_FIFO_ALFULL_THR, 0x1c},
	{OP_WR, TM_REG_CFC_AC_CRDCNT_VAL, 0x1},
	{OP_WR, TM_REG_CFC_CLD_CRDCNT_VAL, 0x1},
	{OP_WR, TM_REG_CLOUT_CRDCNT0_VAL, 0x1},
	{OP_WR, TM_REG_CLOUT_CRDCNT1_VAL, 0x1},
	{OP_WR, TM_REG_CLOUT_CRDCNT2_VAL, 0x1},
	{OP_WR, TM_REG_EXP_CRDCNT_VAL, 0x1},
	{OP_WR_E1, TM_REG_PCIARB_CRDCNT_VAL, 0x1},
	{OP_WR_E1H, TM_REG_PCIARB_CRDCNT_VAL, 0x2},
	{OP_WR_ASIC, TM_REG_TIMER_TICK_SIZE, 0x3d090},
	{OP_WR_EMUL, TM_REG_TIMER_TICK_SIZE, 0x9c},
	{OP_WR_FPGA, TM_REG_TIMER_TICK_SIZE, 0x9c4},
	{OP_WR, TM_REG_CL0_CONT_REGION, 0x8},
	{OP_WR, TM_REG_CL1_CONT_REGION, 0xc},
	{OP_WR, TM_REG_CL2_CONT_REGION, 0x10},
	{OP_WR, TM_REG_TM_CONTEXT_REGION, 0x20},
	{OP_WR, TM_REG_EN_TIMERS, 0x1},
	{OP_WR, TM_REG_EN_REAL_TIME_CNT, 0x1},
	{OP_WR, TM_REG_EN_CL0_INPUT, 0x1},
	{OP_WR, TM_REG_EN_CL1_INPUT, 0x1},
	{OP_WR, TM_REG_EN_CL2_INPUT, 0x1},
#define TIMERS_COMMON_END       1025
#define TIMERS_PORT0_START      1025
	{OP_ZR, TM_REG_LIN0_PHY_ADDR, 0x2},
#define TIMERS_PORT0_END        1026
#define TIMERS_PORT1_START      1026
	{OP_ZR, TM_REG_LIN1_PHY_ADDR, 0x2},
#define TIMERS_PORT1_END        1027
#define XSDM_COMMON_START       1027
	{OP_WR_E1, XSDM_REG_CFC_RSP_START_ADDR, 0x614},
	{OP_WR_E1H, XSDM_REG_CFC_RSP_START_ADDR, 0x424},
	{OP_WR_E1, XSDM_REG_CMP_COUNTER_START_ADDR, 0x600},
	{OP_WR_E1H, XSDM_REG_CMP_COUNTER_START_ADDR, 0x410},
	{OP_WR_E1, XSDM_REG_Q_COUNTER_START_ADDR, 0x604},
	{OP_WR_E1H, XSDM_REG_Q_COUNTER_START_ADDR, 0x414},
	{OP_WR, XSDM_REG_CMP_COUNTER_MAX0, 0xffff},
	{OP_WR, XSDM_REG_CMP_COUNTER_MAX1, 0xffff},
	{OP_WR, XSDM_REG_CMP_COUNTER_MAX2, 0xffff},
	{OP_WR, XSDM_REG_CMP_COUNTER_MAX3, 0xffff},
	{OP_WR, XSDM_REG_AGG_INT_EVENT_0, 0x20},
	{OP_WR, XSDM_REG_AGG_INT_EVENT_1, 0x20},
	{OP_WR, XSDM_REG_AGG_INT_EVENT_2, 0x34},
	{OP_WR, XSDM_REG_AGG_INT_EVENT_3, 0x35},
	{OP_WR, XSDM_REG_AGG_INT_EVENT_4, 0x23},
	{OP_WR, XSDM_REG_AGG_INT_EVENT_5, 0x24},
	{OP_WR, XSDM_REG_AGG_INT_EVENT_6, 0x25},
	{OP_WR, XSDM_REG_AGG_INT_EVENT_7, 0x26},
	{OP_WR, XSDM_REG_AGG_INT_EVENT_8, 0x27},
	{OP_WR, XSDM_REG_AGG_INT_EVENT_9, 0x29},
	{OP_WR, XSDM_REG_AGG_INT_EVENT_10, 0x2a},
	{OP_WR, XSDM_REG_AGG_INT_EVENT_11, 0x2b},
	{OP_ZR, XSDM_REG_AGG_INT_EVENT_12, 0x54},
	{OP_WR, XSDM_REG_AGG_INT_MODE_0, 0x1},
	{OP_ZR, XSDM_REG_AGG_INT_MODE_1, 0x1f},
	{OP_WR, XSDM_REG_ENABLE_IN1, 0x7ffffff},
	{OP_WR, XSDM_REG_ENABLE_IN2, 0x3f},
	{OP_WR, XSDM_REG_ENABLE_OUT1, 0x7ffffff},
	{OP_WR, XSDM_REG_ENABLE_OUT2, 0xf},
	{OP_RD, XSDM_REG_NUM_OF_Q0_CMD, 0x0},
	{OP_RD, XSDM_REG_NUM_OF_Q1_CMD, 0x0},
	{OP_RD, XSDM_REG_NUM_OF_Q3_CMD, 0x0},
	{OP_RD, XSDM_REG_NUM_OF_Q4_CMD, 0x0},
	{OP_RD, XSDM_REG_NUM_OF_Q5_CMD, 0x0},
	{OP_RD, XSDM_REG_NUM_OF_Q6_CMD, 0x0},
	{OP_RD, XSDM_REG_NUM_OF_Q7_CMD, 0x0},
	{OP_RD, XSDM_REG_NUM_OF_Q8_CMD, 0x0},
	{OP_RD, XSDM_REG_NUM_OF_Q9_CMD, 0x0},
	{OP_RD, XSDM_REG_NUM_OF_Q10_CMD, 0x0},
	{OP_RD, XSDM_REG_NUM_OF_Q11_CMD, 0x0},
	{OP_RD, XSDM_REG_NUM_OF_PKT_END_MSG, 0x0},
	{OP_RD, XSDM_REG_NUM_OF_PXP_ASYNC_REQ, 0x0},
	{OP_RD, XSDM_REG_NUM_OF_ACK_AFTER_PLACE, 0x0},
	{OP_WR_E1, XSDM_REG_INIT_CREDIT_PXP_CTRL, 0x1},
	{OP_WR_ASIC, XSDM_REG_TIMER_TICK, 0x3e8},
	{OP_WR_EMUL, XSDM_REG_TIMER_TICK, 0x1},
	{OP_WR_FPGA, XSDM_REG_TIMER_TICK, 0xa},
#define XSDM_COMMON_END         1074
#define QM_COMMON_START         1074
	{OP_WR, QM_REG_ACTCTRINITVAL_0, 0x6},
	{OP_WR, QM_REG_ACTCTRINITVAL_1, 0x5},
	{OP_WR, QM_REG_ACTCTRINITVAL_2, 0xa},
	{OP_WR, QM_REG_ACTCTRINITVAL_3, 0x5},
	{OP_WR, QM_REG_PCIREQAT, 0x2},
	{OP_WR, QM_REG_CMINITCRD_0, 0x4},
	{OP_WR, QM_REG_CMINITCRD_1, 0x4},
	{OP_WR, QM_REG_CMINITCRD_2, 0x4},
	{OP_WR, QM_REG_CMINITCRD_3, 0x4},
	{OP_WR, QM_REG_CMINITCRD_4, 0x4},
	{OP_WR, QM_REG_CMINITCRD_5, 0x4},
	{OP_WR, QM_REG_CMINITCRD_6, 0x4},
	{OP_WR, QM_REG_CMINITCRD_7, 0x4},
	{OP_WR, QM_REG_OUTLDREQ, 0x4},
	{OP_WR, QM_REG_CTXREG_0, 0x7c},
	{OP_WR, QM_REG_CTXREG_1, 0x3d},
	{OP_WR, QM_REG_CTXREG_2, 0x3f},
	{OP_WR, QM_REG_CTXREG_3, 0x9c},
	{OP_WR, QM_REG_ENSEC, 0x7},
	{OP_ZR, QM_REG_QVOQIDX_0, 0x5},
	{OP_WR, QM_REG_WRRWEIGHTS_0, 0x1010101},
	{OP_WR, QM_REG_QVOQIDX_5, 0x0},
	{OP_WR, QM_REG_QVOQIDX_6, 0x4},
	{OP_WR, QM_REG_QVOQIDX_7, 0x4},
	{OP_WR, QM_REG_QVOQIDX_8, 0x2},
	{OP_WR, QM_REG_WRRWEIGHTS_1, 0x8012004},
	{OP_WR, QM_REG_QVOQIDX_9, 0x5},
	{OP_WR, QM_REG_QVOQIDX_10, 0x5},
	{OP_WR, QM_REG_QVOQIDX_11, 0x5},
	{OP_WR, QM_REG_QVOQIDX_12, 0x5},
	{OP_WR, QM_REG_WRRWEIGHTS_2, 0x20081001},
	{OP_WR, QM_REG_QVOQIDX_13, 0x8},
	{OP_WR, QM_REG_QVOQIDX_14, 0x6},
	{OP_WR, QM_REG_QVOQIDX_15, 0x7},
	{OP_WR, QM_REG_QVOQIDX_16, 0x0},
	{OP_WR, QM_REG_WRRWEIGHTS_3, 0x1010120},
	{OP_ZR, QM_REG_QVOQIDX_17, 0x4},
	{OP_WR, QM_REG_WRRWEIGHTS_4, 0x1010101},
	{OP_ZR_E1, QM_REG_QVOQIDX_21, 0x4},
	{OP_WR_E1H, QM_REG_QVOQIDX_21, 0x0},
	{OP_WR_E1, QM_REG_WRRWEIGHTS_5, 0x1010101},
	{OP_WR_E1H, QM_REG_QVOQIDX_22, 0x4},
	{OP_ZR_E1, QM_REG_QVOQIDX_25, 0x4},
	{OP_WR_E1H, QM_REG_QVOQIDX_23, 0x4},
	{OP_WR_E1, QM_REG_WRRWEIGHTS_6, 0x1010101},
	{OP_WR_E1H, QM_REG_QVOQIDX_24, 0x2},
	{OP_ZR_E1, QM_REG_QVOQIDX_29, 0x3},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_5, 0x8012004},
	{OP_WR_E1H, QM_REG_QVOQIDX_25, 0x5},
	{OP_WR_E1H, QM_REG_QVOQIDX_26, 0x5},
	{OP_WR_E1H, QM_REG_QVOQIDX_27, 0x5},
	{OP_WR_E1H, QM_REG_QVOQIDX_28, 0x5},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_6, 0x20081001},
	{OP_WR_E1H, QM_REG_QVOQIDX_29, 0x8},
	{OP_WR_E1H, QM_REG_QVOQIDX_30, 0x6},
	{OP_WR_E1H, QM_REG_QVOQIDX_31, 0x7},
	{OP_WR, QM_REG_QVOQIDX_32, 0x1},
	{OP_WR_E1, QM_REG_WRRWEIGHTS_7, 0x1010101},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_7, 0x1010120},
	{OP_WR, QM_REG_QVOQIDX_33, 0x1},
	{OP_WR, QM_REG_QVOQIDX_34, 0x1},
	{OP_WR, QM_REG_QVOQIDX_35, 0x1},
	{OP_WR, QM_REG_QVOQIDX_36, 0x1},
	{OP_WR, QM_REG_WRRWEIGHTS_8, 0x1010101},
	{OP_WR, QM_REG_QVOQIDX_37, 0x1},
	{OP_WR, QM_REG_QVOQIDX_38, 0x4},
	{OP_WR, QM_REG_QVOQIDX_39, 0x4},
	{OP_WR, QM_REG_QVOQIDX_40, 0x2},
	{OP_WR, QM_REG_WRRWEIGHTS_9, 0x8012004},
	{OP_WR, QM_REG_QVOQIDX_41, 0x5},
	{OP_WR, QM_REG_QVOQIDX_42, 0x5},
	{OP_WR, QM_REG_QVOQIDX_43, 0x5},
	{OP_WR, QM_REG_QVOQIDX_44, 0x5},
	{OP_WR, QM_REG_WRRWEIGHTS_10, 0x20081001},
	{OP_WR, QM_REG_QVOQIDX_45, 0x8},
	{OP_WR, QM_REG_QVOQIDX_46, 0x6},
	{OP_WR, QM_REG_QVOQIDX_47, 0x7},
	{OP_WR, QM_REG_QVOQIDX_48, 0x1},
	{OP_WR, QM_REG_WRRWEIGHTS_11, 0x1010120},
	{OP_WR, QM_REG_QVOQIDX_49, 0x1},
	{OP_WR, QM_REG_QVOQIDX_50, 0x1},
	{OP_WR, QM_REG_QVOQIDX_51, 0x1},
	{OP_WR, QM_REG_QVOQIDX_52, 0x1},
	{OP_WR, QM_REG_WRRWEIGHTS_12, 0x1010101},
	{OP_WR, QM_REG_QVOQIDX_53, 0x1},
	{OP_WR_E1, QM_REG_QVOQIDX_54, 0x1},
	{OP_WR_E1H, QM_REG_QVOQIDX_54, 0x4},
	{OP_WR_E1, QM_REG_QVOQIDX_55, 0x1},
	{OP_WR_E1H, QM_REG_QVOQIDX_55, 0x4},
	{OP_WR_E1, QM_REG_QVOQIDX_56, 0x1},
	{OP_WR_E1H, QM_REG_QVOQIDX_56, 0x2},
	{OP_WR_E1, QM_REG_WRRWEIGHTS_13, 0x1010101},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_13, 0x8012004},
	{OP_WR_E1, QM_REG_QVOQIDX_57, 0x1},
	{OP_WR_E1H, QM_REG_QVOQIDX_57, 0x5},
	{OP_WR_E1, QM_REG_QVOQIDX_58, 0x1},
	{OP_WR_E1H, QM_REG_QVOQIDX_58, 0x5},
	{OP_WR_E1, QM_REG_QVOQIDX_59, 0x1},
	{OP_WR_E1H, QM_REG_QVOQIDX_59, 0x5},
	{OP_WR_E1, QM_REG_QVOQIDX_60, 0x1},
	{OP_WR_E1H, QM_REG_QVOQIDX_60, 0x5},
	{OP_WR_E1, QM_REG_WRRWEIGHTS_14, 0x1010101},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_14, 0x20081001},
	{OP_WR_E1, QM_REG_QVOQIDX_61, 0x1},
	{OP_WR_E1H, QM_REG_QVOQIDX_61, 0x8},
	{OP_WR_E1, QM_REG_QVOQIDX_62, 0x1},
	{OP_WR_E1H, QM_REG_QVOQIDX_62, 0x6},
	{OP_WR_E1, QM_REG_QVOQIDX_63, 0x1},
	{OP_WR_E1H, QM_REG_QVOQIDX_63, 0x7},
	{OP_WR_E1, QM_REG_WRRWEIGHTS_15, 0x1010101},
	{OP_WR_E1H, QM_REG_QVOQIDX_64, 0x0},
	{OP_WR_E1, QM_REG_VOQQMASK_0_LSB, 0xffff003f},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_15, 0x1010120},
	{OP_ZR_E1, QM_REG_VOQQMASK_0_MSB, 0x2},
	{OP_ZR_E1H, QM_REG_QVOQIDX_65, 0x4},
	{OP_WR_E1, QM_REG_VOQQMASK_1_MSB, 0xffff003f},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_16, 0x1010101},
	{OP_WR_E1, QM_REG_VOQQMASK_2_LSB, 0x100},
	{OP_WR_E1H, QM_REG_QVOQIDX_69, 0x0},
	{OP_WR_E1, QM_REG_VOQQMASK_2_MSB, 0x100},
	{OP_WR_E1H, QM_REG_QVOQIDX_70, 0x4},
	{OP_WR_E1H, QM_REG_QVOQIDX_71, 0x4},
	{OP_WR_E1H, QM_REG_QVOQIDX_72, 0x2},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_17, 0x8012004},
	{OP_WR_E1H, QM_REG_QVOQIDX_73, 0x5},
	{OP_WR_E1H, QM_REG_QVOQIDX_74, 0x5},
	{OP_WR_E1H, QM_REG_QVOQIDX_75, 0x5},
	{OP_WR_E1H, QM_REG_QVOQIDX_76, 0x5},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_18, 0x20081001},
	{OP_WR_E1H, QM_REG_QVOQIDX_77, 0x8},
	{OP_WR_E1H, QM_REG_QVOQIDX_78, 0x6},
	{OP_WR_E1H, QM_REG_QVOQIDX_79, 0x7},
	{OP_WR_E1H, QM_REG_QVOQIDX_80, 0x0},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_19, 0x1010120},
	{OP_ZR_E1H, QM_REG_QVOQIDX_81, 0x4},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_20, 0x1010101},
	{OP_WR_E1H, QM_REG_QVOQIDX_85, 0x0},
	{OP_WR_E1H, QM_REG_QVOQIDX_86, 0x4},
	{OP_WR_E1H, QM_REG_QVOQIDX_87, 0x4},
	{OP_WR_E1H, QM_REG_QVOQIDX_88, 0x2},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_21, 0x8012004},
	{OP_WR_E1H, QM_REG_QVOQIDX_89, 0x5},
	{OP_WR_E1H, QM_REG_QVOQIDX_90, 0x5},
	{OP_WR_E1H, QM_REG_QVOQIDX_91, 0x5},
	{OP_WR_E1H, QM_REG_QVOQIDX_92, 0x5},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_22, 0x20081001},
	{OP_WR_E1H, QM_REG_QVOQIDX_93, 0x8},
	{OP_WR_E1H, QM_REG_QVOQIDX_94, 0x6},
	{OP_WR_E1H, QM_REG_QVOQIDX_95, 0x7},
	{OP_WR_E1H, QM_REG_QVOQIDX_96, 0x1},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_23, 0x1010120},
	{OP_WR_E1H, QM_REG_QVOQIDX_97, 0x1},
	{OP_WR_E1H, QM_REG_QVOQIDX_98, 0x1},
	{OP_WR_E1H, QM_REG_QVOQIDX_99, 0x1},
	{OP_WR_E1H, QM_REG_QVOQIDX_100, 0x1},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_24, 0x1010101},
	{OP_WR_E1H, QM_REG_QVOQIDX_101, 0x1},
	{OP_WR_E1H, QM_REG_QVOQIDX_102, 0x4},
	{OP_WR_E1H, QM_REG_QVOQIDX_103, 0x4},
	{OP_WR_E1H, QM_REG_QVOQIDX_104, 0x2},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_25, 0x8012004},
	{OP_WR_E1H, QM_REG_QVOQIDX_105, 0x5},
	{OP_WR_E1H, QM_REG_QVOQIDX_106, 0x5},
	{OP_WR_E1H, QM_REG_QVOQIDX_107, 0x5},
	{OP_WR_E1H, QM_REG_QVOQIDX_108, 0x5},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_26, 0x20081001},
	{OP_WR_E1H, QM_REG_QVOQIDX_109, 0x8},
	{OP_WR_E1H, QM_REG_QVOQIDX_110, 0x6},
	{OP_WR_E1H, QM_REG_QVOQIDX_111, 0x7},
	{OP_WR_E1H, QM_REG_QVOQIDX_112, 0x1},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_27, 0x1010120},
	{OP_WR_E1H, QM_REG_QVOQIDX_113, 0x1},
	{OP_WR_E1H, QM_REG_QVOQIDX_114, 0x1},
	{OP_WR_E1H, QM_REG_QVOQIDX_115, 0x1},
	{OP_WR_E1H, QM_REG_QVOQIDX_116, 0x1},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_28, 0x1010101},
	{OP_WR_E1H, QM_REG_QVOQIDX_117, 0x1},
	{OP_WR_E1H, QM_REG_QVOQIDX_118, 0x4},
	{OP_WR_E1H, QM_REG_QVOQIDX_119, 0x4},
	{OP_WR_E1H, QM_REG_QVOQIDX_120, 0x2},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_29, 0x8012004},
	{OP_WR_E1H, QM_REG_QVOQIDX_121, 0x5},
	{OP_WR_E1H, QM_REG_QVOQIDX_122, 0x5},
	{OP_WR_E1H, QM_REG_QVOQIDX_123, 0x5},
	{OP_WR_E1H, QM_REG_QVOQIDX_124, 0x5},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_30, 0x20081001},
	{OP_WR_E1H, QM_REG_QVOQIDX_125, 0x8},
	{OP_WR_E1H, QM_REG_QVOQIDX_126, 0x6},
	{OP_WR_E1H, QM_REG_QVOQIDX_127, 0x7},
	{OP_WR_E1H, QM_REG_WRRWEIGHTS_31, 0x1010120},
	{OP_WR_E1H, QM_REG_VOQQMASK_0_LSB, 0x3f003f},
	{OP_WR_E1H, QM_REG_VOQQMASK_0_MSB, 0x0},
	{OP_WR_E1H, QM_REG_VOQQMASK_0_LSB_EXT_A, 0x3f003f},
	{OP_WR_E1H, QM_REG_VOQQMASK_0_MSB_EXT_A, 0x0},
	{OP_WR_E1H, QM_REG_VOQQMASK_1_LSB, 0x0},
	{OP_WR_E1H, QM_REG_VOQQMASK_1_MSB, 0x3f003f},
	{OP_WR_E1H, QM_REG_VOQQMASK_1_LSB_EXT_A, 0x0},
	{OP_WR_E1H, QM_REG_VOQQMASK_1_MSB_EXT_A, 0x3f003f},
	{OP_WR_E1H, QM_REG_VOQQMASK_2_LSB, 0x1000100},
	{OP_WR_E1H, QM_REG_VOQQMASK_2_MSB, 0x1000100},
	{OP_WR_E1H, QM_REG_VOQQMASK_2_LSB_EXT_A, 0x1000100},
	{OP_WR_E1H, QM_REG_VOQQMASK_2_MSB_EXT_A, 0x1000100},
	{OP_ZR, QM_REG_VOQQMASK_3_LSB, 0x2},
	{OP_WR_E1, QM_REG_VOQQMASK_4_LSB, 0xc0},
	{OP_WR_E1H, QM_REG_VOQQMASK_3_LSB_EXT_A, 0x0},
	{OP_WR_E1, QM_REG_VOQQMASK_4_MSB, 0xc0},
	{OP_WR_E1H, QM_REG_VOQQMASK_3_MSB_EXT_A, 0x0},
	{OP_WR_E1, QM_REG_VOQQMASK_5_LSB, 0x1e00},
	{OP_WR_E1H, QM_REG_VOQQMASK_4_LSB, 0xc000c0},
	{OP_WR_E1, QM_REG_VOQQMASK_5_MSB, 0x1e00},
	{OP_WR_E1H, QM_REG_VOQQMASK_4_MSB, 0xc000c0},
	{OP_WR_E1, QM_REG_VOQQMASK_6_LSB, 0x4000},
	{OP_WR_E1H, QM_REG_VOQQMASK_4_LSB_EXT_A, 0xc000c0},
	{OP_WR_E1, QM_REG_VOQQMASK_6_MSB, 0x4000},
	{OP_WR_E1H, QM_REG_VOQQMASK_4_MSB_EXT_A, 0xc000c0},
	{OP_WR_E1, QM_REG_VOQQMASK_7_LSB, 0x8000},
	{OP_WR_E1H, QM_REG_VOQQMASK_5_LSB, 0x1e001e00},
	{OP_WR_E1, QM_REG_VOQQMASK_7_MSB, 0x8000},
	{OP_WR_E1H, QM_REG_VOQQMASK_5_MSB, 0x1e001e00},
	{OP_WR_E1, QM_REG_VOQQMASK_8_LSB, 0x2000},
	{OP_WR_E1H, QM_REG_VOQQMASK_5_LSB_EXT_A, 0x1e001e00},
	{OP_WR_E1, QM_REG_VOQQMASK_8_MSB, 0x2000},
	{OP_WR_E1H, QM_REG_VOQQMASK_5_MSB_EXT_A, 0x1e001e00},
	{OP_ZR_E1, QM_REG_VOQQMASK_9_LSB, 0x7},
	{OP_WR_E1H, QM_REG_VOQQMASK_6_LSB, 0x40004000},
	{OP_WR_E1H, QM_REG_VOQQMASK_6_MSB, 0x40004000},
	{OP_WR_E1H, QM_REG_VOQQMASK_6_LSB_EXT_A, 0x40004000},
	{OP_WR_E1H, QM_REG_VOQQMASK_6_MSB_EXT_A, 0x40004000},
	{OP_WR_E1H, QM_REG_VOQQMASK_7_LSB, 0x80008000},
	{OP_WR_E1H, QM_REG_VOQQMASK_7_MSB, 0x80008000},
	{OP_WR_E1H, QM_REG_VOQQMASK_7_LSB_EXT_A, 0x80008000},
	{OP_WR_E1H, QM_REG_VOQQMASK_7_MSB_EXT_A, 0x80008000},
	{OP_WR_E1H, QM_REG_VOQQMASK_8_LSB, 0x20002000},
	{OP_WR_E1H, QM_REG_VOQQMASK_8_MSB, 0x20002000},
	{OP_WR_E1H, QM_REG_VOQQMASK_8_LSB_EXT_A, 0x20002000},
	{OP_WR_E1H, QM_REG_VOQQMASK_8_MSB_EXT_A, 0x20002000},
	{OP_ZR_E1H, QM_REG_VOQQMASK_9_LSB, 0x2},
	{OP_WR_E1H, QM_REG_VOQQMASK_9_LSB_EXT_A, 0x0},
	{OP_WR_E1H, QM_REG_VOQQMASK_9_MSB_EXT_A, 0x0},
	{OP_WR_E1H, QM_REG_VOQQMASK_10_LSB, 0x0},
	{OP_WR_E1H, QM_REG_VOQQMASK_10_MSB, 0x0},
	{OP_WR_E1H, QM_REG_VOQQMASK_10_LSB_EXT_A, 0x0},
	{OP_WR_E1H, QM_REG_VOQQMASK_10_MSB_EXT_A, 0x0},
	{OP_WR_E1H, QM_REG_VOQQMASK_11_LSB, 0x0},
	{OP_WR_E1H, QM_REG_VOQQMASK_11_MSB, 0x0},
	{OP_WR_E1H, QM_REG_VOQQMASK_11_LSB_EXT_A, 0x0},
	{OP_WR_E1H, QM_REG_VOQQMASK_11_MSB_EXT_A, 0x0},
	{OP_WR_E1H, QM_REG_VOQPORT_0, 0x0},
	{OP_WR, QM_REG_VOQPORT_1, 0x1},
	{OP_ZR, QM_REG_VOQPORT_2, 0xa},
	{OP_WR, QM_REG_CMINTVOQMASK_0, 0xc08},
	{OP_WR, QM_REG_CMINTVOQMASK_1, 0x40},
	{OP_WR, QM_REG_CMINTVOQMASK_2, 0x100},
	{OP_WR, QM_REG_CMINTVOQMASK_3, 0x20},
	{OP_WR, QM_REG_CMINTVOQMASK_4, 0x17},
	{OP_WR, QM_REG_CMINTVOQMASK_5, 0x80},
	{OP_WR, QM_REG_CMINTVOQMASK_6, 0x200},
	{OP_WR, QM_REG_CMINTVOQMASK_7, 0x0},
	{OP_WR_E1, QM_REG_HWAEMPTYMASK_LSB, 0xffff01ff},
	{OP_WR_E1H, QM_REG_HWAEMPTYMASK_LSB, 0x1ff01ff},
	{OP_WR_E1, QM_REG_HWAEMPTYMASK_MSB, 0xffff01ff},
	{OP_WR_E1H, QM_REG_HWAEMPTYMASK_MSB, 0x1ff01ff},
	{OP_WR_E1H, QM_REG_HWAEMPTYMASK_LSB_EXT_A, 0x1ff01ff},
	{OP_WR_E1H, QM_REG_HWAEMPTYMASK_MSB_EXT_A, 0x1ff01ff},
	{OP_WR, QM_REG_ENBYPVOQMASK, 0x13},
	{OP_WR, QM_REG_VOQCREDITAFULLTHR, 0x13f},
	{OP_WR, QM_REG_VOQINITCREDIT_0, 0x140},
	{OP_WR, QM_REG_VOQINITCREDIT_1, 0x140},
	{OP_ZR, QM_REG_VOQINITCREDIT_2, 0x2},
	{OP_WR, QM_REG_VOQINITCREDIT_4, 0xc0},
	{OP_ZR, QM_REG_VOQINITCREDIT_5, 0x7},
	{OP_WR, QM_REG_TASKCRDCOST_0, 0x48},
	{OP_WR, QM_REG_TASKCRDCOST_1, 0x48},
	{OP_ZR, QM_REG_TASKCRDCOST_2, 0x2},
	{OP_WR, QM_REG_TASKCRDCOST_4, 0x48},
	{OP_ZR, QM_REG_TASKCRDCOST_5, 0x7},
	{OP_WR, QM_REG_BYTECRDINITVAL, 0x8000},
	{OP_WR, QM_REG_BYTECRDCOST, 0x25e4},
	{OP_WR, QM_REG_BYTECREDITAFULLTHR, 0x7fff},
	{OP_WR_E1, QM_REG_ENBYTECRD_LSB, 0x7},
	{OP_WR_E1H, QM_REG_ENBYTECRD_LSB, 0x70007},
	{OP_WR_E1, QM_REG_ENBYTECRD_MSB, 0x7},
	{OP_WR_E1H, QM_REG_ENBYTECRD_MSB, 0x70007},
	{OP_WR_E1H, QM_REG_ENBYTECRD_LSB_EXT_A, 0x70007},
	{OP_WR_E1H, QM_REG_ENBYTECRD_MSB_EXT_A, 0x70007},
	{OP_WR, QM_REG_BYTECRDPORT_LSB, 0x0},
	{OP_WR, QM_REG_BYTECRDPORT_MSB, 0xffffffff},
	{OP_WR_E1, QM_REG_FUNCNUMSEL_LSB, 0x0},
	{OP_WR_E1H, QM_REG_BYTECRDPORT_LSB_EXT_A, 0x0},
	{OP_WR_E1, QM_REG_FUNCNUMSEL_MSB, 0xffffffff},
	{OP_WR_E1H, QM_REG_BYTECRDPORT_MSB_EXT_A, 0xffffffff},
	{OP_WR_E1H, QM_REG_PQ2PCIFUNC_0, 0x0},
	{OP_WR_E1H, QM_REG_PQ2PCIFUNC_1, 0x2},
	{OP_WR_E1H, QM_REG_PQ2PCIFUNC_2, 0x1},
	{OP_WR_E1H, QM_REG_PQ2PCIFUNC_3, 0x3},
	{OP_WR_E1H, QM_REG_PQ2PCIFUNC_4, 0x4},
	{OP_WR_E1H, QM_REG_PQ2PCIFUNC_5, 0x6},
	{OP_WR_E1H, QM_REG_PQ2PCIFUNC_6, 0x5},
	{OP_WR_E1H, QM_REG_PQ2PCIFUNC_7, 0x7},
	{OP_WR, QM_REG_CMINTEN, 0xff},
#define QM_COMMON_END           1374
#define PBF_COMMON_START        1374
	{OP_WR, PBF_REG_INIT, 0x1},
	{OP_WR, PBF_REG_INIT_P4, 0x1},
	{OP_WR, PBF_REG_MAC_LB_ENABLE, 0x1},
	{OP_WR, PBF_REG_IF_ENABLE_REG, 0x7fff},
	{OP_WR, PBF_REG_INIT_P4, 0x0},
	{OP_WR, PBF_REG_INIT, 0x0},
	{OP_WR, PBF_REG_DISABLE_NEW_TASK_PROC_P4, 0x0},
#define PBF_COMMON_END          1381
#define PBF_PORT0_START         1381
	{OP_WR, PBF_REG_INIT_P0, 0x1},
	{OP_WR, PBF_REG_MAC_IF0_ENABLE, 0x1},
	{OP_WR, PBF_REG_INIT_P0, 0x0},
	{OP_WR, PBF_REG_DISABLE_NEW_TASK_PROC_P0, 0x0},
#define PBF_PORT0_END           1385
#define PBF_PORT1_START         1385
	{OP_WR, PBF_REG_INIT_P1, 0x1},
	{OP_WR, PBF_REG_MAC_IF1_ENABLE, 0x1},
	{OP_WR, PBF_REG_INIT_P1, 0x0},
	{OP_WR, PBF_REG_DISABLE_NEW_TASK_PROC_P1, 0x0},
#define PBF_PORT1_END           1389
#define XCM_COMMON_START        1389
	{OP_WR, XCM_REG_XX_OVFL_EVNT_ID, 0x32},
	{OP_WR, XCM_REG_XQM_XCM_HDR_P, 0x3150020},
	{OP_WR, XCM_REG_XQM_XCM_HDR_S, 0x3150020},
	{OP_WR, XCM_REG_TM_XCM_HDR, 0x1000030},
	{OP_WR, XCM_REG_ERR_XCM_HDR, 0x8100000},
	{OP_WR, XCM_REG_ERR_EVNT_ID, 0x33},
	{OP_WR, XCM_REG_EXPR_EVNT_ID, 0x30},
	{OP_WR, XCM_REG_STOP_EVNT_ID, 0x31},
	{OP_WR, XCM_REG_STORM_WEIGHT, 0x2},
	{OP_WR, XCM_REG_TSEM_WEIGHT, 0x5},
	{OP_WR, XCM_REG_CSEM_WEIGHT, 0x2},
	{OP_WR, XCM_REG_USEM_WEIGHT, 0x2},
	{OP_WR, XCM_REG_PBF_WEIGHT, 0x7},
	{OP_WR, XCM_REG_NIG1_WEIGHT, 0x1},
	{OP_WR, XCM_REG_CP_WEIGHT, 0x0},
	{OP_WR, XCM_REG_XSDM_WEIGHT, 0x5},
	{OP_WR, XCM_REG_XQM_P_WEIGHT, 0x3},
	{OP_WR, XCM_REG_XCM_XQM_USE_Q, 0x1},
	{OP_WR, XCM_REG_XQM_BYP_ACT_UPD, 0x6},
	{OP_WR, XCM_REG_UNA_GT_NXT_Q, 0x0},
	{OP_WR, XCM_REG_AUX1_Q, 0x2},
	{OP_WR, XCM_REG_AUX_CNT_FLG_Q_19, 0x1},
	{OP_WR, XCM_REG_GR_ARB_TYPE, 0x1},
	{OP_WR, XCM_REG_GR_LD0_PR, 0x1},
	{OP_WR, XCM_REG_GR_LD1_PR, 0x2},
	{OP_WR, XCM_REG_CFC_INIT_CRD, 0x1},
	{OP_WR, XCM_REG_FIC0_INIT_CRD, 0x40},
	{OP_WR, XCM_REG_FIC1_INIT_CRD, 0x40},
	{OP_WR, XCM_REG_TM_INIT_CRD, 0x4},
	{OP_WR, XCM_REG_XQM_INIT_CRD, 0x20},
	{OP_WR, XCM_REG_XX_INIT_CRD, 0x2},
	{OP_WR_E1, XCM_REG_XX_MSG_NUM, 0x1f},
	{OP_WR_E1H, XCM_REG_XX_MSG_NUM, 0x20},
	{OP_ZR, XCM_REG_XX_TABLE, 0x12},
	{OP_SW_E1, XCM_REG_XX_DESCR_TABLE, 0x1f02cc},
	{OP_SW_E1H, XCM_REG_XX_DESCR_TABLE, 0x1f0302},
	{OP_WR, XCM_REG_N_SM_CTX_LD_0, 0xf},
	{OP_WR, XCM_REG_N_SM_CTX_LD_1, 0x7},
	{OP_WR, XCM_REG_N_SM_CTX_LD_2, 0xb},
	{OP_WR, XCM_REG_N_SM_CTX_LD_3, 0xe},
	{OP_ZR_E1, XCM_REG_N_SM_CTX_LD_4, 0x4},
	{OP_WR_E1H, XCM_REG_N_SM_CTX_LD_4, 0xc},
	{OP_ZR_E1H, XCM_REG_N_SM_CTX_LD_5, 0x3},
	{OP_WR, XCM_REG_XCM_REG0_SZ, 0x4},
	{OP_WR, XCM_REG_XCM_STORM0_IFEN, 0x1},
	{OP_WR, XCM_REG_XCM_STORM1_IFEN, 0x1},
	{OP_WR, XCM_REG_XCM_XQM_IFEN, 0x1},
	{OP_WR, XCM_REG_STORM_XCM_IFEN, 0x1},
	{OP_WR, XCM_REG_XQM_XCM_IFEN, 0x1},
	{OP_WR, XCM_REG_XSDM_IFEN, 0x1},
	{OP_WR, XCM_REG_TM_XCM_IFEN, 0x1},
	{OP_WR, XCM_REG_XCM_TM_IFEN, 0x1},
	{OP_WR, XCM_REG_TSEM_IFEN, 0x1},
	{OP_WR, XCM_REG_CSEM_IFEN, 0x1},
	{OP_WR, XCM_REG_USEM_IFEN, 0x1},
	{OP_WR, XCM_REG_DORQ_IFEN, 0x1},
	{OP_WR, XCM_REG_PBF_IFEN, 0x1},
	{OP_WR, XCM_REG_NIG0_IFEN, 0x1},
	{OP_WR, XCM_REG_NIG1_IFEN, 0x1},
	{OP_WR, XCM_REG_CDU_AG_WR_IFEN, 0x1},
	{OP_WR, XCM_REG_CDU_AG_RD_IFEN, 0x1},
	{OP_WR, XCM_REG_CDU_SM_WR_IFEN, 0x1},
	{OP_WR, XCM_REG_CDU_SM_RD_IFEN, 0x1},
	{OP_WR, XCM_REG_XCM_CFC_IFEN, 0x1},
#define XCM_COMMON_END          1453
#define XCM_PORT0_START         1453
	{OP_WR_E1, XCM_REG_GLB_DEL_ACK_TMR_VAL_0, 0xc8},
	{OP_WR_E1, XCM_REG_GLB_DEL_ACK_MAX_CNT_0, 0x2},
	{OP_WR_E1, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD00, 0x0},
	{OP_WR_E1, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD10, 0x0},
	{OP_WR_E1, XCM_REG_WU_DA_CNT_CMD00, 0x2},
	{OP_WR_E1, XCM_REG_WU_DA_CNT_CMD10, 0x2},
	{OP_WR_E1, XCM_REG_WU_DA_CNT_UPD_VAL00, 0xff},
	{OP_WR_E1, XCM_REG_WU_DA_CNT_UPD_VAL10, 0xff},
#define XCM_PORT0_END           1461
#define XCM_PORT1_START         1461
	{OP_WR_E1, XCM_REG_GLB_DEL_ACK_TMR_VAL_1, 0xc8},
	{OP_WR_E1, XCM_REG_GLB_DEL_ACK_MAX_CNT_1, 0x2},
	{OP_WR_E1, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD01, 0x0},
	{OP_WR_E1, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD11, 0x0},
	{OP_WR_E1, XCM_REG_WU_DA_CNT_CMD01, 0x2},
	{OP_WR_E1, XCM_REG_WU_DA_CNT_CMD11, 0x2},
	{OP_WR_E1, XCM_REG_WU_DA_CNT_UPD_VAL01, 0xff},
	{OP_WR_E1, XCM_REG_WU_DA_CNT_UPD_VAL11, 0xff},
#define XCM_PORT1_END           1469
#define XCM_FUNC0_START         1469
	{OP_WR_E1H, XCM_REG_GLB_DEL_ACK_TMR_VAL_0, 0xc8},
	{OP_WR_E1H, XCM_REG_GLB_DEL_ACK_MAX_CNT_0, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD00, 0x0},
	{OP_WR_E1H, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD10, 0x0},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_CMD00, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_CMD10, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_UPD_VAL00, 0xff},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_UPD_VAL10, 0xff},
	{OP_WR_E1H, XCM_REG_PHYS_QNUM3_0, 0x0},
#define XCM_FUNC0_END           1478
#define XCM_FUNC1_START         1478
	{OP_WR_E1H, XCM_REG_GLB_DEL_ACK_TMR_VAL_1, 0xc8},
	{OP_WR_E1H, XCM_REG_GLB_DEL_ACK_MAX_CNT_1, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD01, 0x0},
	{OP_WR_E1H, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD11, 0x0},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_CMD01, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_CMD11, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_UPD_VAL01, 0xff},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_UPD_VAL11, 0xff},
	{OP_WR_E1H, XCM_REG_PHYS_QNUM3_1, 0x0},
#define XCM_FUNC1_END           1487
#define XCM_FUNC2_START         1487
	{OP_WR_E1H, XCM_REG_GLB_DEL_ACK_TMR_VAL_0, 0xc8},
	{OP_WR_E1H, XCM_REG_GLB_DEL_ACK_MAX_CNT_0, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD00, 0x0},
	{OP_WR_E1H, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD10, 0x0},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_CMD00, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_CMD10, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_UPD_VAL00, 0xff},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_UPD_VAL10, 0xff},
	{OP_WR_E1H, XCM_REG_PHYS_QNUM3_0, 0x0},
#define XCM_FUNC2_END           1496
#define XCM_FUNC3_START         1496
	{OP_WR_E1H, XCM_REG_GLB_DEL_ACK_TMR_VAL_1, 0xc8},
	{OP_WR_E1H, XCM_REG_GLB_DEL_ACK_MAX_CNT_1, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD01, 0x0},
	{OP_WR_E1H, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD11, 0x0},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_CMD01, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_CMD11, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_UPD_VAL01, 0xff},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_UPD_VAL11, 0xff},
	{OP_WR_E1H, XCM_REG_PHYS_QNUM3_1, 0x0},
#define XCM_FUNC3_END           1505
#define XCM_FUNC4_START         1505
	{OP_WR_E1H, XCM_REG_GLB_DEL_ACK_TMR_VAL_0, 0xc8},
	{OP_WR_E1H, XCM_REG_GLB_DEL_ACK_MAX_CNT_0, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD00, 0x0},
	{OP_WR_E1H, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD10, 0x0},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_CMD00, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_CMD10, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_UPD_VAL00, 0xff},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_UPD_VAL10, 0xff},
	{OP_WR_E1H, XCM_REG_PHYS_QNUM3_0, 0x0},
#define XCM_FUNC4_END           1514
#define XCM_FUNC5_START         1514
	{OP_WR_E1H, XCM_REG_GLB_DEL_ACK_TMR_VAL_1, 0xc8},
	{OP_WR_E1H, XCM_REG_GLB_DEL_ACK_MAX_CNT_1, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD01, 0x0},
	{OP_WR_E1H, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD11, 0x0},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_CMD01, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_CMD11, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_UPD_VAL01, 0xff},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_UPD_VAL11, 0xff},
	{OP_WR_E1H, XCM_REG_PHYS_QNUM3_1, 0x0},
#define XCM_FUNC5_END           1523
#define XCM_FUNC6_START         1523
	{OP_WR_E1H, XCM_REG_GLB_DEL_ACK_TMR_VAL_0, 0xc8},
	{OP_WR_E1H, XCM_REG_GLB_DEL_ACK_MAX_CNT_0, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD00, 0x0},
	{OP_WR_E1H, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD10, 0x0},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_CMD00, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_CMD10, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_UPD_VAL00, 0xff},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_UPD_VAL10, 0xff},
	{OP_WR_E1H, XCM_REG_PHYS_QNUM3_0, 0x0},
#define XCM_FUNC6_END           1532
#define XCM_FUNC7_START         1532
	{OP_WR_E1H, XCM_REG_GLB_DEL_ACK_TMR_VAL_1, 0xc8},
	{OP_WR_E1H, XCM_REG_GLB_DEL_ACK_MAX_CNT_1, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD01, 0x0},
	{OP_WR_E1H, XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD11, 0x0},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_CMD01, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_CMD11, 0x2},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_UPD_VAL01, 0xff},
	{OP_WR_E1H, XCM_REG_WU_DA_CNT_UPD_VAL11, 0xff},
	{OP_WR_E1H, XCM_REG_PHYS_QNUM3_1, 0x0},
#define XCM_FUNC7_END           1541
#define XSEM_COMMON_START       1541
	{OP_RD, XSEM_REG_MSG_NUM_FIC0, 0x0},
	{OP_RD, XSEM_REG_MSG_NUM_FIC1, 0x0},
	{OP_RD, XSEM_REG_MSG_NUM_FOC0, 0x0},
	{OP_RD, XSEM_REG_MSG_NUM_FOC1, 0x0},
	{OP_RD, XSEM_REG_MSG_NUM_FOC2, 0x0},
	{OP_RD, XSEM_REG_MSG_NUM_FOC3, 0x0},
	{OP_WR, XSEM_REG_ARB_ELEMENT0, 0x1},
	{OP_WR, XSEM_REG_ARB_ELEMENT1, 0x2},
	{OP_WR, XSEM_REG_ARB_ELEMENT2, 0x3},
	{OP_WR, XSEM_REG_ARB_ELEMENT3, 0x0},
	{OP_WR, XSEM_REG_ARB_ELEMENT4, 0x4},
	{OP_WR, XSEM_REG_ARB_CYCLE_SIZE, 0x1},
	{OP_WR, XSEM_REG_TS_0_AS, 0x0},
	{OP_WR, XSEM_REG_TS_1_AS, 0x1},
	{OP_WR, XSEM_REG_TS_2_AS, 0x4},
	{OP_WR, XSEM_REG_TS_3_AS, 0x0},
	{OP_WR, XSEM_REG_TS_4_AS, 0x1},
	{OP_WR, XSEM_REG_TS_5_AS, 0x3},
	{OP_WR, XSEM_REG_TS_6_AS, 0x0},
	{OP_WR, XSEM_REG_TS_7_AS, 0x1},
	{OP_WR, XSEM_REG_TS_8_AS, 0x4},
	{OP_WR, XSEM_REG_TS_9_AS, 0x0},
	{OP_WR, XSEM_REG_TS_10_AS, 0x1},
	{OP_WR, XSEM_REG_TS_11_AS, 0x3},
	{OP_WR, XSEM_REG_TS_12_AS, 0x0},
	{OP_WR, XSEM_REG_TS_13_AS, 0x1},
	{OP_WR, XSEM_REG_TS_14_AS, 0x4},
	{OP_WR, XSEM_REG_TS_15_AS, 0x0},
	{OP_WR, XSEM_REG_TS_16_AS, 0x4},
	{OP_WR, XSEM_REG_TS_17_AS, 0x3},
	{OP_ZR, XSEM_REG_TS_18_AS, 0x2},
	{OP_WR, XSEM_REG_ENABLE_IN, 0x3fff},
	{OP_WR, XSEM_REG_ENABLE_OUT, 0x3ff},
	{OP_WR, XSEM_REG_FIC0_DISABLE, 0x0},
	{OP_WR, XSEM_REG_FIC1_DISABLE, 0x0},
	{OP_WR, XSEM_REG_PAS_DISABLE, 0x0},
	{OP_WR, XSEM_REG_THREADS_LIST, 0xffff},
	{OP_ZR, XSEM_REG_PASSIVE_BUFFER, 0x800},
	{OP_WR, XSEM_REG_FAST_MEMORY + 0x18bc0, 0x1},
	{OP_WR, XSEM_REG_FAST_MEMORY + 0x18000, 0x0},
	{OP_WR, XSEM_REG_FAST_MEMORY + 0x18040, 0x18},
	{OP_WR, XSEM_REG_FAST_MEMORY + 0x18080, 0xc},
	{OP_WR, XSEM_REG_FAST_MEMORY + 0x180c0, 0x66},
	{OP_WR_ASIC, XSEM_REG_FAST_MEMORY + 0x18300, 0x7a120},
	{OP_WR_EMUL, XSEM_REG_FAST_MEMORY + 0x18300, 0x138},
	{OP_WR_FPGA, XSEM_REG_FAST_MEMORY + 0x18300, 0x1388},
	{OP_WR, XSEM_REG_FAST_MEMORY + 0x183c0, 0x1f4},
	{OP_WR_ASIC, XSEM_REG_FAST_MEMORY + 0x18340, 0x1f4},
	{OP_WR_EMUL, XSEM_REG_FAST_MEMORY + 0x18340, 0x0},
	{OP_WR_FPGA, XSEM_REG_FAST_MEMORY + 0x18340, 0x5},
	{OP_WR_EMUL, XSEM_REG_FAST_MEMORY + 0x18380, 0x4c4b4},
	{OP_WR_ASIC, XSEM_REG_FAST_MEMORY + 0x18380, 0x1dcd6500},
	{OP_WR_EMUL_E1H, XSEM_REG_FAST_MEMORY + 0x11480, 0x0},
	{OP_WR_FPGA, XSEM_REG_FAST_MEMORY + 0x18380, 0x4c4b40},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x3d00, 0x4},
	{OP_WR_E1H, XSEM_REG_FAST_MEMORY + 0x11480, 0x1},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x3000, 0x48},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x28a8, 0x4},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x1020, 0xc8},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x2080, 0x48},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x1000, 0x2},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x9020, 0xc8},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x3128, 0x8e},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x9000, 0x2},
	{OP_WR_E1, XSEM_REG_FAST_MEMORY + 0x3368, 0x0},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x21a8, 0x86},
	{OP_SW_E1, XSEM_REG_FAST_MEMORY + 0x3370, 0x202eb},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x2000, 0x20},
	{OP_SW_E1, XSEM_REG_FAST_MEMORY + 0x3b90, 0x402ed},
	{OP_WR_E1H, XSEM_REG_FAST_MEMORY + 0x23c8, 0x0},
	{OP_WR_E1, XSEM_REG_FAST_MEMORY + 0x1518, 0x1},
	{OP_SW_E1H, XSEM_REG_FAST_MEMORY + 0x23d0, 0x20321},
	{OP_WR_E1, XSEM_REG_FAST_MEMORY + 0x1830, 0x0},
	{OP_SW_E1H, XSEM_REG_FAST_MEMORY + 0x2498, 0x40323},
	{OP_WR_E1, XSEM_REG_FAST_MEMORY + 0x1838, 0x0},
	{OP_WR_E1H, XSEM_REG_FAST_MEMORY + 0x2ac8, 0x0},
	{OP_SW_E1, XSEM_REG_FAST_MEMORY + 0x1820, 0x202f1},
	{OP_WR_E1H, XSEM_REG_FAST_MEMORY + 0x2ab8, 0x0},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x4ac0, 0x2},
	{OP_WR_E1H, XSEM_REG_FAST_MEMORY + 0x3010, 0x1},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x4b00, 0x4},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x4040, 0x10},
	{OP_SW_E1, XSEM_REG_FAST_MEMORY + 0x1f50, 0x202f3},
	{OP_SW_E1H, XSEM_REG_FAST_MEMORY + 0x4000, 0x100327},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x6ac0, 0x2},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x6b00, 0x4},
	{OP_SW_E1H, XSEM_REG_FAST_MEMORY + 0x83b0, 0x20337},
	{OP_WR, XSEM_REG_FAST_MEMORY + 0x10800, 0x0},
	{OP_SW_E1, XSEM_REG_FAST_MEMORY + 0x10c00, 0x1002f5},
	{OP_SW_E1H, XSEM_REG_FAST_MEMORY + 0x10c00, 0x100339},
	{OP_WR, XSEM_REG_FAST_MEMORY + 0x10800, 0x1000000},
	{OP_SW_E1, XSEM_REG_FAST_MEMORY + 0x10c40, 0x80305},
	{OP_SW_E1H, XSEM_REG_FAST_MEMORY + 0x10c40, 0x80349},
	{OP_WR, XSEM_REG_FAST_MEMORY + 0x10800, 0x2000000},
	{OP_SW_E1, XSEM_REG_FAST_MEMORY + 0x10c60, 0x8030d},
	{OP_SW_E1H, XSEM_REG_FAST_MEMORY + 0x10c60, 0x80351},
	{OP_ZP_E1, XSEM_REG_INT_TABLE, 0xa90000},
	{OP_ZP_E1H, XSEM_REG_INT_TABLE, 0xac0000},
	{OP_WR_64_E1, XSEM_REG_INT_TABLE + 0x368, 0x130315},
	{OP_WR_64_E1H, XSEM_REG_INT_TABLE + 0x368, 0x130359},
	{OP_ZP_E1, XSEM_REG_PRAM, 0x344e0000},
	{OP_ZP_E1H, XSEM_REG_PRAM, 0x34620000},
	{OP_ZP_E1, XSEM_REG_PRAM + 0x8000, 0x38840d14},
	{OP_ZP_E1H, XSEM_REG_PRAM + 0x8000, 0x38240d19},
	{OP_ZP_E1, XSEM_REG_PRAM + 0x10000, 0x3e711b35},
	{OP_ZP_E1H, XSEM_REG_PRAM + 0x10000, 0x3e971b22},
	{OP_ZP_E1, XSEM_REG_PRAM + 0x18000, 0x1dd02ad2},
	{OP_ZP_E1H, XSEM_REG_PRAM + 0x18000, 0x21542ac8},
	{OP_WR_64_E1, XSEM_REG_PRAM + 0x1c0d0, 0x47e60317},
	{OP_WR_64_E1H, XSEM_REG_PRAM + 0x1c8d0, 0x46e6035b},
#define XSEM_COMMON_END         1651
#define XSEM_PORT0_START        1651
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x3ba0, 0x10},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0xc000, 0xfc},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x3c20, 0x1c},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x24a8, 0x10},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x1400, 0xa},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x2528, 0x1c},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x1450, 0x6},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x2608, 0x1c},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x3378, 0xfc},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x26e8, 0x1c},
	{OP_WR_E1, XSEM_REG_FAST_MEMORY + 0x3b58, 0x0},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x27c8, 0x1c},
	{OP_SW_E1, XSEM_REG_FAST_MEMORY + 0x3d10, 0x100319},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0xa000, 0x28},
	{OP_WR_E1, XSEM_REG_FAST_MEMORY + 0x1500, 0x0},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0xa140, 0xc},
	{OP_WR_E1, XSEM_REG_FAST_MEMORY + 0x1508, 0x1},
	{OP_WR_E1H, XSEM_REG_FAST_MEMORY + 0x3000, 0x1},
	{OP_ZR, XSEM_REG_FAST_MEMORY + 0x5020, 0x2},
	{OP_ZR, XSEM_REG_FAST_MEMORY + 0x5030, 0x2},
	{OP_ZR, XSEM_REG_FAST_MEMORY + 0x5000, 0x2},
	{OP_ZR, XSEM_REG_FAST_MEMORY + 0x5010, 0x2},
	{OP_WR_E1, XSEM_REG_FAST_MEMORY + 0x5040, 0x0},
	{OP_WR_E1H, XSEM_REG_FAST_MEMORY + 0x5208, 0x1},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x5048, 0xe},
	{OP_SW_E1H, XSEM_REG_FAST_MEMORY + 0x6ac8, 0x2035d},
	{OP_WR_E1, XSEM_REG_FAST_MEMORY + 0x50b8, 0x1},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x6b10, 0x42},
	{OP_SW_E1, XSEM_REG_FAST_MEMORY + 0x4ac8, 0x20329},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x6d20, 0x4},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x4b10, 0x42},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x4d20, 0x4},
#define XSEM_PORT0_END          1683
#define XSEM_PORT1_START        1683
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x3be0, 0x10},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0xc3f0, 0xfc},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x3c90, 0x1c},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x24e8, 0x10},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x1428, 0xa},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x2598, 0x1c},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x1468, 0x6},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x2678, 0x1c},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x3768, 0xfc},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x2758, 0x1c},
	{OP_WR_E1, XSEM_REG_FAST_MEMORY + 0x3b5c, 0x0},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x2838, 0x1c},
	{OP_SW_E1, XSEM_REG_FAST_MEMORY + 0x3d50, 0x10032b},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0xa0a0, 0x28},
	{OP_WR_E1, XSEM_REG_FAST_MEMORY + 0x1504, 0x0},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0xa170, 0xc},
	{OP_WR_E1, XSEM_REG_FAST_MEMORY + 0x150c, 0x1},
	{OP_WR_E1H, XSEM_REG_FAST_MEMORY + 0x3004, 0x1},
	{OP_ZR, XSEM_REG_FAST_MEMORY + 0x5028, 0x2},
	{OP_ZR, XSEM_REG_FAST_MEMORY + 0x5038, 0x2},
	{OP_ZR, XSEM_REG_FAST_MEMORY + 0x5008, 0x2},
	{OP_ZR, XSEM_REG_FAST_MEMORY + 0x5018, 0x2},
	{OP_WR_E1, XSEM_REG_FAST_MEMORY + 0x5044, 0x0},
	{OP_WR_E1H, XSEM_REG_FAST_MEMORY + 0x520c, 0x1},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x5080, 0xe},
	{OP_SW_E1H, XSEM_REG_FAST_MEMORY + 0x6ad0, 0x2035f},
	{OP_WR_E1, XSEM_REG_FAST_MEMORY + 0x50bc, 0x1},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x6c18, 0x42},
	{OP_SW_E1, XSEM_REG_FAST_MEMORY + 0x4ad0, 0x2033b},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x6d30, 0x4},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x4c18, 0x42},
	{OP_ZR_E1, XSEM_REG_FAST_MEMORY + 0x4d30, 0x4},
#define XSEM_PORT1_END          1715
#define XSEM_FUNC0_START        1715
	{OP_WR_E1H, XSEM_REG_FAST_MEMORY + 0xc7e0, 0x0},
	{OP_SW_E1H, XSEM_REG_FAST_MEMORY + 0x28b8, 0x100361},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x5048, 0xe},
#define XSEM_FUNC0_END          1718
#define XSEM_FUNC1_START        1718
	{OP_WR_E1H, XSEM_REG_FAST_MEMORY + 0xc7e4, 0x0},
	{OP_SW_E1H, XSEM_REG_FAST_MEMORY + 0x28f8, 0x100371},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x5080, 0xe},
#define XSEM_FUNC1_END          1721
#define XSEM_FUNC2_START        1721
	{OP_WR_E1H, XSEM_REG_FAST_MEMORY + 0xc7e8, 0x0},
	{OP_SW_E1H, XSEM_REG_FAST_MEMORY + 0x2938, 0x100381},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x50b8, 0xe},
#define XSEM_FUNC2_END          1724
#define XSEM_FUNC3_START        1724
	{OP_WR_E1H, XSEM_REG_FAST_MEMORY + 0xc7ec, 0x0},
	{OP_SW_E1H, XSEM_REG_FAST_MEMORY + 0x2978, 0x100391},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x50f0, 0xe},
#define XSEM_FUNC3_END          1727
#define XSEM_FUNC4_START        1727
	{OP_WR_E1H, XSEM_REG_FAST_MEMORY + 0xc7f0, 0x0},
	{OP_SW_E1H, XSEM_REG_FAST_MEMORY + 0x29b8, 0x1003a1},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x5128, 0xe},
#define XSEM_FUNC4_END          1730
#define XSEM_FUNC5_START        1730
	{OP_WR_E1H, XSEM_REG_FAST_MEMORY + 0xc7f4, 0x0},
	{OP_SW_E1H, XSEM_REG_FAST_MEMORY + 0x29f8, 0x1003b1},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x5160, 0xe},
#define XSEM_FUNC5_END          1733
#define XSEM_FUNC6_START        1733
	{OP_WR_E1H, XSEM_REG_FAST_MEMORY + 0xc7f8, 0x0},
	{OP_SW_E1H, XSEM_REG_FAST_MEMORY + 0x2a38, 0x1003c1},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x5198, 0xe},
#define XSEM_FUNC6_END          1736
#define XSEM_FUNC7_START        1736
	{OP_WR_E1H, XSEM_REG_FAST_MEMORY + 0xc7fc, 0x0},
	{OP_SW_E1H, XSEM_REG_FAST_MEMORY + 0x2a78, 0x1003d1},
	{OP_ZR_E1H, XSEM_REG_FAST_MEMORY + 0x51d0, 0xe},
#define XSEM_FUNC7_END          1739
#define CDU_COMMON_START        1739
	{OP_WR, CDU_REG_CDU_CONTROL0, 0x1},
	{OP_WR_E1H, CDU_REG_MF_MODE, 0x1},
	{OP_WR, CDU_REG_CDU_CHK_MASK0, 0x3d000},
	{OP_WR, CDU_REG_CDU_CHK_MASK1, 0x3d},
	{OP_WB_E1, CDU_REG_L1TT, 0x200033d},
	{OP_WB_E1H, CDU_REG_L1TT, 0x20003e1},
	{OP_WB_E1, CDU_REG_MATT, 0x20053d},
	{OP_WB_E1H, CDU_REG_MATT, 0x2805e1},
	{OP_ZR_E1, CDU_REG_MATT + 0x80, 0x2},
	{OP_WB_E1, CDU_REG_MATT + 0x88, 0x6055d},
	{OP_ZR, CDU_REG_MATT + 0xa0, 0x18},
#define CDU_COMMON_END          1750
#define DMAE_COMMON_START       1750
	{OP_ZR, DMAE_REG_CMD_MEM, 0xe0},
	{OP_WR, DMAE_REG_CRC16C_INIT, 0x0},
	{OP_WR, DMAE_REG_CRC16T10_INIT, 0x1},
	{OP_WR_E1, DMAE_REG_PXP_REQ_INIT_CRD, 0x1},
	{OP_WR_E1H, DMAE_REG_PXP_REQ_INIT_CRD, 0x2},
	{OP_WR, DMAE_REG_PCI_IFEN, 0x1},
	{OP_WR, DMAE_REG_GRC_IFEN, 0x1},
#define DMAE_COMMON_END         1757
#define PXP_COMMON_START        1757
	{OP_WB_E1, PXP_REG_HST_INBOUND_INT + 0x400, 0x50563},
	{OP_WB_E1H, PXP_REG_HST_INBOUND_INT + 0x400, 0x50609},
	{OP_WB_E1, PXP_REG_HST_INBOUND_INT + 0x420, 0x50568},
	{OP_WB_E1H, PXP_REG_HST_INBOUND_INT, 0x5060e},
	{OP_WB_E1, PXP_REG_HST_INBOUND_INT, 0x5056d},
#define PXP_COMMON_END          1762
#define CFC_COMMON_START        1762
	{OP_ZR_E1H, CFC_REG_LINK_LIST, 0x100},
	{OP_WR, CFC_REG_CONTROL0, 0x10},
	{OP_WR, CFC_REG_DISABLE_ON_ERROR, 0x3fff},
	{OP_WR, CFC_REG_LCREQ_WEIGHTS, 0x84924a},
#define CFC_COMMON_END          1766
#define HC_COMMON_START         1766
	{OP_ZR_E1, HC_REG_USTORM_ADDR_FOR_COALESCE, 0x4},
#define HC_COMMON_END           1767
#define HC_PORT0_START          1767
	{OP_WR_E1, HC_REG_CONFIG_0, 0x1080},
	{OP_ZR_E1, HC_REG_UC_RAM_ADDR_0, 0x2},
	{OP_WR_E1, HC_REG_ATTN_NUM_P0, 0x10},
	{OP_WR_E1, HC_REG_LEADING_EDGE_0, 0xffff},
	{OP_WR_E1, HC_REG_TRAILING_EDGE_0, 0xffff},
	{OP_WR_E1, HC_REG_AGG_INT_0, 0x0},
	{OP_WR_E1, HC_REG_ATTN_IDX, 0x0},
	{OP_ZR_E1, HC_REG_ATTN_BIT, 0x2},
	{OP_WR_E1, HC_REG_VQID_0, 0x2b5},
	{OP_WR_E1, HC_REG_PCI_CONFIG_0, 0x0},
	{OP_ZR_E1, HC_REG_P0_PROD_CONS, 0x4a},
	{OP_WR_E1, HC_REG_INT_MASK, 0x1ffff},
	{OP_ZR_E1, HC_REG_PBA_COMMAND, 0x2},
	{OP_WR_E1, HC_REG_CONFIG_0, 0x1a80},
	{OP_ZR_E1, HC_REG_STATISTIC_COUNTERS, 0x24},
	{OP_ZR_E1, HC_REG_STATISTIC_COUNTERS + 0x120, 0x4a},
	{OP_ZR_E1, HC_REG_STATISTIC_COUNTERS + 0x370, 0x4a},
	{OP_ZR_E1, HC_REG_STATISTIC_COUNTERS + 0x5c0, 0x4a},
#define HC_PORT0_END            1785
#define HC_PORT1_START          1785
	{OP_WR_E1, HC_REG_CONFIG_1, 0x1080},
	{OP_ZR_E1, HC_REG_UC_RAM_ADDR_1, 0x2},
	{OP_WR_E1, HC_REG_ATTN_NUM_P1, 0x10},
	{OP_WR_E1, HC_REG_LEADING_EDGE_1, 0xffff},
	{OP_WR_E1, HC_REG_TRAILING_EDGE_1, 0xffff},
	{OP_WR_E1, HC_REG_AGG_INT_1, 0x0},
	{OP_WR_E1, HC_REG_ATTN_IDX + 0x4, 0x0},
	{OP_ZR_E1, HC_REG_ATTN_BIT + 0x8, 0x2},
	{OP_WR_E1, HC_REG_VQID_1, 0x2b5},
	{OP_WR_E1, HC_REG_PCI_CONFIG_1, 0x0},
	{OP_ZR_E1, HC_REG_P1_PROD_CONS, 0x4a},
	{OP_WR_E1, HC_REG_INT_MASK + 0x4, 0x1ffff},
	{OP_ZR_E1, HC_REG_PBA_COMMAND + 0x8, 0x2},
	{OP_WR_E1, HC_REG_CONFIG_1, 0x1a80},
	{OP_ZR_E1, HC_REG_STATISTIC_COUNTERS + 0x90, 0x24},
	{OP_ZR_E1, HC_REG_STATISTIC_COUNTERS + 0x248, 0x4a},
	{OP_ZR_E1, HC_REG_STATISTIC_COUNTERS + 0x498, 0x4a},
	{OP_ZR_E1, HC_REG_STATISTIC_COUNTERS + 0x6e8, 0x4a},
#define HC_PORT1_END            1803
#define HC_FUNC0_START          1803
	{OP_WR_E1H, HC_REG_CONFIG_0, 0x1080},
	{OP_WR_E1H, HC_REG_FUNC_NUM_P0, 0x0},
	{OP_WR_E1H, HC_REG_ATTN_NUM_P0, 0x10},
	{OP_WR_E1H, HC_REG_ATTN_IDX, 0x0},
	{OP_ZR_E1H, HC_REG_ATTN_BIT, 0x2},
	{OP_WR_E1H, HC_REG_VQID_0, 0x2b5},
	{OP_WR_E1H, HC_REG_PCI_CONFIG_0, 0x0},
	{OP_ZR_E1H, HC_REG_P0_PROD_CONS, 0x4a},
	{OP_WR_E1H, HC_REG_INT_MASK, 0x1ffff},
	{OP_ZR_E1H, HC_REG_PBA_COMMAND, 0x2},
	{OP_WR_E1H, HC_REG_CONFIG_0, 0x1a80},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS, 0x24},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x120, 0x4a},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x370, 0x4a},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x5c0, 0x4a},
#define HC_FUNC0_END            1818
#define HC_FUNC1_START          1818
	{OP_WR_E1H, HC_REG_CONFIG_1, 0x1080},
	{OP_WR_E1H, HC_REG_FUNC_NUM_P1, 0x1},
	{OP_WR_E1H, HC_REG_ATTN_NUM_P1, 0x10},
	{OP_WR_E1H, HC_REG_ATTN_IDX + 0x4, 0x0},
	{OP_ZR_E1H, HC_REG_ATTN_BIT + 0x8, 0x2},
	{OP_WR_E1H, HC_REG_VQID_1, 0x2b5},
	{OP_WR_E1H, HC_REG_PCI_CONFIG_1, 0x0},
	{OP_ZR_E1H, HC_REG_P1_PROD_CONS, 0x4a},
	{OP_WR_E1H, HC_REG_INT_MASK + 0x4, 0x1ffff},
	{OP_ZR_E1H, HC_REG_PBA_COMMAND + 0x8, 0x2},
	{OP_WR_E1H, HC_REG_CONFIG_1, 0x1a80},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x90, 0x24},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x248, 0x4a},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x498, 0x4a},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x6e8, 0x4a},
#define HC_FUNC1_END            1833
#define HC_FUNC2_START          1833
	{OP_WR_E1H, HC_REG_CONFIG_0, 0x1080},
	{OP_WR_E1H, HC_REG_FUNC_NUM_P0, 0x2},
	{OP_WR_E1H, HC_REG_ATTN_NUM_P0, 0x10},
	{OP_WR_E1H, HC_REG_ATTN_IDX, 0x0},
	{OP_ZR_E1H, HC_REG_ATTN_BIT, 0x2},
	{OP_WR_E1H, HC_REG_VQID_0, 0x2b5},
	{OP_WR_E1H, HC_REG_PCI_CONFIG_0, 0x0},
	{OP_ZR_E1H, HC_REG_P0_PROD_CONS, 0x4a},
	{OP_WR_E1H, HC_REG_INT_MASK, 0x1ffff},
	{OP_ZR_E1H, HC_REG_PBA_COMMAND, 0x2},
	{OP_WR_E1H, HC_REG_CONFIG_0, 0x1a80},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS, 0x24},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x120, 0x4a},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x370, 0x4a},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x5c0, 0x4a},
#define HC_FUNC2_END            1848
#define HC_FUNC3_START          1848
	{OP_WR_E1H, HC_REG_CONFIG_1, 0x1080},
	{OP_WR_E1H, HC_REG_FUNC_NUM_P1, 0x3},
	{OP_WR_E1H, HC_REG_ATTN_NUM_P1, 0x10},
	{OP_WR_E1H, HC_REG_ATTN_IDX + 0x4, 0x0},
	{OP_ZR_E1H, HC_REG_ATTN_BIT + 0x8, 0x2},
	{OP_WR_E1H, HC_REG_VQID_1, 0x2b5},
	{OP_WR_E1H, HC_REG_PCI_CONFIG_1, 0x0},
	{OP_ZR_E1H, HC_REG_P1_PROD_CONS, 0x4a},
	{OP_WR_E1H, HC_REG_INT_MASK + 0x4, 0x1ffff},
	{OP_ZR_E1H, HC_REG_PBA_COMMAND + 0x8, 0x2},
	{OP_WR_E1H, HC_REG_CONFIG_1, 0x1a80},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x90, 0x24},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x248, 0x4a},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x498, 0x4a},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x6e8, 0x4a},
#define HC_FUNC3_END            1863
#define HC_FUNC4_START          1863
	{OP_WR_E1H, HC_REG_CONFIG_0, 0x1080},
	{OP_WR_E1H, HC_REG_FUNC_NUM_P0, 0x4},
	{OP_WR_E1H, HC_REG_ATTN_NUM_P0, 0x10},
	{OP_WR_E1H, HC_REG_ATTN_IDX, 0x0},
	{OP_ZR_E1H, HC_REG_ATTN_BIT, 0x2},
	{OP_WR_E1H, HC_REG_VQID_0, 0x2b5},
	{OP_WR_E1H, HC_REG_PCI_CONFIG_0, 0x0},
	{OP_ZR_E1H, HC_REG_P0_PROD_CONS, 0x4a},
	{OP_WR_E1H, HC_REG_INT_MASK, 0x1ffff},
	{OP_ZR_E1H, HC_REG_PBA_COMMAND, 0x2},
	{OP_WR_E1H, HC_REG_CONFIG_0, 0x1a80},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS, 0x24},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x120, 0x4a},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x370, 0x4a},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x5c0, 0x4a},
#define HC_FUNC4_END            1878
#define HC_FUNC5_START          1878
	{OP_WR_E1H, HC_REG_CONFIG_1, 0x1080},
	{OP_WR_E1H, HC_REG_FUNC_NUM_P1, 0x5},
	{OP_WR_E1H, HC_REG_ATTN_NUM_P1, 0x10},
	{OP_WR_E1H, HC_REG_ATTN_IDX + 0x4, 0x0},
	{OP_ZR_E1H, HC_REG_ATTN_BIT + 0x8, 0x2},
	{OP_WR_E1H, HC_REG_VQID_1, 0x2b5},
	{OP_WR_E1H, HC_REG_PCI_CONFIG_1, 0x0},
	{OP_ZR_E1H, HC_REG_P1_PROD_CONS, 0x4a},
	{OP_WR_E1H, HC_REG_INT_MASK + 0x4, 0x1ffff},
	{OP_ZR_E1H, HC_REG_PBA_COMMAND + 0x8, 0x2},
	{OP_WR_E1H, HC_REG_CONFIG_1, 0x1a80},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x90, 0x24},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x248, 0x4a},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x498, 0x4a},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x6e8, 0x4a},
#define HC_FUNC5_END            1893
#define HC_FUNC6_START          1893
	{OP_WR_E1H, HC_REG_CONFIG_0, 0x1080},
	{OP_WR_E1H, HC_REG_FUNC_NUM_P0, 0x6},
	{OP_WR_E1H, HC_REG_ATTN_NUM_P0, 0x10},
	{OP_WR_E1H, HC_REG_ATTN_IDX, 0x0},
	{OP_ZR_E1H, HC_REG_ATTN_BIT, 0x2},
	{OP_WR_E1H, HC_REG_VQID_0, 0x2b5},
	{OP_WR_E1H, HC_REG_PCI_CONFIG_0, 0x0},
	{OP_ZR_E1H, HC_REG_P0_PROD_CONS, 0x4a},
	{OP_WR_E1H, HC_REG_INT_MASK, 0x1ffff},
	{OP_ZR_E1H, HC_REG_PBA_COMMAND, 0x2},
	{OP_WR_E1H, HC_REG_CONFIG_0, 0x1a80},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS, 0x24},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x120, 0x4a},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x370, 0x4a},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x5c0, 0x4a},
#define HC_FUNC6_END            1908
#define HC_FUNC7_START          1908
	{OP_WR_E1H, HC_REG_CONFIG_1, 0x1080},
	{OP_WR_E1H, HC_REG_FUNC_NUM_P1, 0x7},
	{OP_WR_E1H, HC_REG_ATTN_NUM_P1, 0x10},
	{OP_WR_E1H, HC_REG_ATTN_IDX + 0x4, 0x0},
	{OP_ZR_E1H, HC_REG_ATTN_BIT + 0x8, 0x2},
	{OP_WR_E1H, HC_REG_VQID_1, 0x2b5},
	{OP_WR_E1H, HC_REG_PCI_CONFIG_1, 0x0},
	{OP_ZR_E1H, HC_REG_P1_PROD_CONS, 0x4a},
	{OP_WR_E1H, HC_REG_INT_MASK + 0x4, 0x1ffff},
	{OP_ZR_E1H, HC_REG_PBA_COMMAND + 0x8, 0x2},
	{OP_WR_E1H, HC_REG_CONFIG_1, 0x1a80},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x90, 0x24},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x248, 0x4a},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x498, 0x4a},
	{OP_ZR_E1H, HC_REG_STATISTIC_COUNTERS + 0x6e8, 0x4a},
#define HC_FUNC7_END            1923
#define PXP2_COMMON_START       1923
	{OP_WR_E1, PXP2_REG_PGL_CONTROL0, 0xe38340},
	{OP_WR_E1H, PXP2_REG_RQ_DRAM_ALIGN, 0x1},
	{OP_WR, PXP2_REG_PGL_CONTROL1, 0x3c10},
	{OP_WR_E1H, PXP2_REG_RQ_ELT_DISABLE, 0x1},
	{OP_WR_E1H, PXP2_REG_WR_REV_MODE, 0x0},
	{OP_WR, PXP2_REG_PGL_INT_TSDM_0, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_TSDM_1, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_TSDM_2, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_TSDM_3, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_TSDM_4, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_TSDM_5, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_TSDM_6, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_TSDM_7, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_USDM_1, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_USDM_2, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_USDM_3, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_USDM_4, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_USDM_5, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_USDM_6, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_USDM_7, 0xffffffff},
	{OP_WR_E1H, PXP2_REG_PGL_INT_XSDM_1, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_XSDM_2, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_XSDM_3, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_XSDM_4, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_XSDM_5, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_XSDM_6, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_XSDM_7, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_CSDM_0, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_CSDM_1, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_CSDM_2, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_CSDM_3, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_CSDM_4, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_CSDM_5, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_CSDM_6, 0xffffffff},
	{OP_WR, PXP2_REG_PGL_INT_CSDM_7, 0xffffffff},
	{OP_WR_E1, PXP2_REG_PGL_INT_XSDM_0, 0xffff3330},
	{OP_WR_E1H, PXP2_REG_PGL_INT_XSDM_0, 0xff802000},
	{OP_WR_E1, PXP2_REG_PGL_INT_XSDM_1, 0xffff3340},
	{OP_WR_E1H, PXP2_REG_PGL_INT_USDM_0, 0xf0005000},
	{OP_WR_E1, PXP2_REG_PGL_INT_USDM_0, 0xf0003000},
	{OP_WR, PXP2_REG_RD_MAX_BLKS_VQ6, 0x8},
	{OP_WR, PXP2_REG_RD_MAX_BLKS_VQ9, 0x8},
	{OP_WR, PXP2_REG_RD_MAX_BLKS_VQ10, 0x8},
	{OP_WR, PXP2_REG_RD_MAX_BLKS_VQ11, 0x2},
	{OP_WR, PXP2_REG_RD_MAX_BLKS_VQ17, 0x4},
	{OP_WR, PXP2_REG_RD_MAX_BLKS_VQ18, 0x5},
	{OP_WR, PXP2_REG_RD_MAX_BLKS_VQ19, 0x4},
	{OP_WR, PXP2_REG_RD_MAX_BLKS_VQ22, 0x0},
	{OP_WR, PXP2_REG_RD_START_INIT, 0x1},
	{OP_WR, PXP2_REG_WR_DMAE_TH, 0x3f},
	{OP_WR, PXP2_REG_RQ_BW_RD_ADD0, 0x40},
	{OP_WR, PXP2_REG_PSWRQ_BW_ADD1, 0x1808},
	{OP_WR, PXP2_REG_PSWRQ_BW_ADD2, 0x803},
	{OP_WR, PXP2_REG_PSWRQ_BW_ADD3, 0x803},
	{OP_WR, PXP2_REG_RQ_BW_RD_ADD4, 0x40},
	{OP_WR, PXP2_REG_RQ_BW_RD_ADD5, 0x3},
	{OP_WR, PXP2_REG_PSWRQ_BW_ADD6, 0x803},
	{OP_WR, PXP2_REG_PSWRQ_BW_ADD7, 0x803},
	{OP_WR, PXP2_REG_PSWRQ_BW_ADD8, 0x803},
	{OP_WR, PXP2_REG_PSWRQ_BW_ADD9, 0x10003},
	{OP_WR, PXP2_REG_PSWRQ_BW_ADD10, 0x803},
	{OP_WR, PXP2_REG_PSWRQ_BW_ADD11, 0x803},
	{OP_WR, PXP2_REG_RQ_BW_RD_ADD12, 0x3},
	{OP_WR, PXP2_REG_RQ_BW_RD_ADD13, 0x3},
	{OP_WR, PXP2_REG_RQ_BW_RD_ADD14, 0x3},
	{OP_WR, PXP2_REG_RQ_BW_RD_ADD15, 0x3},
	{OP_WR, PXP2_REG_RQ_BW_RD_ADD16, 0x3},
	{OP_WR, PXP2_REG_RQ_BW_RD_ADD17, 0x3},
	{OP_WR, PXP2_REG_RQ_BW_RD_ADD18, 0x3},
	{OP_WR, PXP2_REG_RQ_BW_RD_ADD19, 0x3},
	{OP_WR, PXP2_REG_RQ_BW_RD_ADD20, 0x3},
	{OP_WR, PXP2_REG_RQ_BW_RD_ADD22, 0x3},
	{OP_WR, PXP2_REG_RQ_BW_RD_ADD23, 0x3},
	{OP_WR, PXP2_REG_RQ_BW_RD_ADD24, 0x3},
	{OP_WR, PXP2_REG_RQ_BW_RD_ADD25, 0x3},
	{OP_WR, PXP2_REG_RQ_BW_RD_ADD26, 0x3},
	{OP_WR, PXP2_REG_RQ_BW_RD_ADD27, 0x3},
	{OP_WR, PXP2_REG_PSWRQ_BW_ADD28, 0x2403},
	{OP_WR, PXP2_REG_RQ_BW_WR_ADD29, 0x2f},
	{OP_WR, PXP2_REG_RQ_BW_WR_ADD30, 0x9},
	{OP_WR, PXP2_REG_RQ_BW_RD_UBOUND0, 0x19},
	{OP_WR, PXP2_REG_PSWRQ_BW_UB1, 0x184},
	{OP_WR, PXP2_REG_PSWRQ_BW_UB2, 0x183},
	{OP_WR, PXP2_REG_PSWRQ_BW_UB3, 0x306},
	{OP_WR, PXP2_REG_RQ_BW_RD_UBOUND4, 0x19},
	{OP_WR, PXP2_REG_RQ_BW_RD_UBOUND5, 0x6},
	{OP_WR, PXP2_REG_PSWRQ_BW_UB6, 0x306},
	{OP_WR, PXP2_REG_PSWRQ_BW_UB7, 0x306},
	{OP_WR, PXP2_REG_PSWRQ_BW_UB8, 0x306},
	{OP_WR, PXP2_REG_PSWRQ_BW_UB9, 0xc86},
	{OP_WR, PXP2_REG_PSWRQ_BW_UB10, 0x306},
	{OP_WR, PXP2_REG_PSWRQ_BW_UB11, 0x306},
	{OP_WR, PXP2_REG_RQ_BW_RD_UBOUND12, 0x6},
	{OP_WR, PXP2_REG_RQ_BW_RD_UBOUND13, 0x6},
	{OP_WR, PXP2_REG_RQ_BW_RD_UBOUND14, 0x6},
	{OP_WR, PXP2_REG_RQ_BW_RD_UBOUND15, 0x6},
	{OP_WR, PXP2_REG_RQ_BW_RD_UBOUND16, 0x6},
	{OP_WR, PXP2_REG_RQ_BW_RD_UBOUND17, 0x6},
	{OP_WR, PXP2_REG_RQ_BW_RD_UBOUND18, 0x6},
	{OP_WR, PXP2_REG_RQ_BW_RD_UBOUND19, 0x6},
	{OP_WR, PXP2_REG_RQ_BW_RD_UBOUND20, 0x6},
	{OP_WR, PXP2_REG_RQ_BW_RD_UBOUND22, 0x6},
	{OP_WR, PXP2_REG_RQ_BW_RD_UBOUND23, 0x6},
	{OP_WR, PXP2_REG_RQ_BW_RD_UBOUND24, 0x6},
	{OP_WR, PXP2_REG_RQ_BW_RD_UBOUND25, 0x6},
	{OP_WR, PXP2_REG_RQ_BW_RD_UBOUND26, 0x6},
	{OP_WR, PXP2_REG_RQ_BW_RD_UBOUND27, 0x6},
	{OP_WR, PXP2_REG_PSWRQ_BW_UB28, 0x306},
	{OP_WR, PXP2_REG_RQ_BW_WR_UBOUND29, 0x13},
	{OP_WR, PXP2_REG_RQ_BW_WR_UBOUND30, 0x6},
	{OP_WR, PXP2_REG_PSWRQ_BW_L1, 0x1004},
	{OP_WR, PXP2_REG_PSWRQ_BW_L2, 0x1004},
	{OP_WR, PXP2_REG_PSWRQ_BW_RD, 0x106440},
	{OP_WR, PXP2_REG_PSWRQ_BW_WR, 0x106440},
	{OP_WR_E1H, PXP2_REG_RQ_ILT_MODE, 0x1},
	{OP_WR, PXP2_REG_RQ_RBC_DONE, 0x1},
	{OP_WR_E1H, PXP2_REG_PGL_CONTROL0, 0xe38340},
#define PXP2_COMMON_END         2040
#define MISC_AEU_COMMON_START   2040
	{OP_ZR, MISC_REG_AEU_GENERAL_ATTN_0, 0x16},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE1_NIG_0, 0x55540000},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE2_NIG_0, 0x55555555},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE3_NIG_0, 0x5555},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE4_NIG_0, 0xf0000000},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE1_PXP_0, 0x55540000},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE2_PXP_0, 0x55555555},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE3_PXP_0, 0x5555},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE4_PXP_0, 0xf0000000},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE1_NIG_1, 0x55540000},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE2_NIG_1, 0x55555555},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE3_NIG_1, 0x5555},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE4_NIG_1, 0xf0000000},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE1_PXP_1, 0x0},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE2_PXP_1, 0x10000},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE3_PXP_1, 0x5014},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE4_PXP_1, 0x0},
	{OP_WR_E1H, MISC_REG_AEU_CLR_LATCH_SIGNAL, 0xc00},
	{OP_WR_E1H, MISC_REG_AEU_GENERAL_MASK, 0x3},
#define MISC_AEU_COMMON_END     2059
#define MISC_AEU_PORT0_START    2059
	{OP_WR_E1, MISC_REG_AEU_ENABLE1_FUNC_0_OUT_0, 0xbf5c0000},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE1_FUNC_0_OUT_0, 0xff5c0000},
	{OP_WR_E1, MISC_REG_AEU_ENABLE2_FUNC_0_OUT_0, 0xfff51fef},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE2_FUNC_0_OUT_0, 0xfff55fff},
	{OP_WR, MISC_REG_AEU_ENABLE3_FUNC_0_OUT_0, 0xffff},
	{OP_WR_E1, MISC_REG_AEU_ENABLE4_FUNC_0_OUT_0, 0x500003e0},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE4_FUNC_0_OUT_0, 0xf00003e0},
	{OP_WR, MISC_REG_AEU_ENABLE1_FUNC_0_OUT_1, 0x0},
	{OP_WR, MISC_REG_AEU_ENABLE2_FUNC_0_OUT_1, 0xa000},
	{OP_ZR, MISC_REG_AEU_ENABLE3_FUNC_0_OUT_1, 0x5},
	{OP_WR, MISC_REG_AEU_ENABLE4_FUNC_0_OUT_2, 0xfe00000},
	{OP_ZR_E1, MISC_REG_AEU_ENABLE1_FUNC_0_OUT_3, 0x14},
	{OP_ZR_E1H, MISC_REG_AEU_ENABLE1_FUNC_0_OUT_3, 0x7},
	{OP_WR_E1, MISC_REG_AEU_ENABLE1_NIG_0, 0x55540000},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE4_FUNC_0_OUT_4, 0x400},
	{OP_WR_E1, MISC_REG_AEU_ENABLE2_NIG_0, 0x55555555},
	{OP_ZR_E1H, MISC_REG_AEU_ENABLE1_FUNC_0_OUT_5, 0x3},
	{OP_WR_E1, MISC_REG_AEU_ENABLE3_NIG_0, 0x5555},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE4_FUNC_0_OUT_5, 0x1000},
	{OP_WR_E1, MISC_REG_AEU_ENABLE4_NIG_0, 0x0},
	{OP_ZR_E1H, MISC_REG_AEU_ENABLE1_FUNC_0_OUT_6, 0x3},
	{OP_WR_E1, MISC_REG_AEU_ENABLE1_PXP_0, 0x55540000},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE4_FUNC_0_OUT_6, 0x4000},
	{OP_WR_E1, MISC_REG_AEU_ENABLE2_PXP_0, 0x55555555},
	{OP_ZR_E1H, MISC_REG_AEU_ENABLE1_FUNC_0_OUT_7, 0x3},
	{OP_WR_E1, MISC_REG_AEU_ENABLE3_PXP_0, 0x5555},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE4_FUNC_0_OUT_7, 0x10000},
	{OP_WR_E1, MISC_REG_AEU_ENABLE4_PXP_0, 0x0},
	{OP_ZR_E1H, MISC_REG_AEU_INVERTER_1_FUNC_0, 0x4},
	{OP_WR_E1, MISC_REG_AEU_INVERTER_1_FUNC_0, 0x0},
	{OP_ZR_E1, MISC_REG_AEU_INVERTER_2_FUNC_0, 0x3},
	{OP_WR_E1, MISC_REG_AEU_MASK_ATTN_FUNC_0, 0x7},
#define MISC_AEU_PORT0_END      2091
#define MISC_AEU_PORT1_START    2091
	{OP_WR_E1, MISC_REG_AEU_ENABLE1_FUNC_1_OUT_0, 0xbf5c0000},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE1_FUNC_1_OUT_0, 0xff5c0000},
	{OP_WR_E1, MISC_REG_AEU_ENABLE2_FUNC_1_OUT_0, 0xfff51fef},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE2_FUNC_1_OUT_0, 0xfff55fff},
	{OP_WR, MISC_REG_AEU_ENABLE3_FUNC_1_OUT_0, 0xffff},
	{OP_WR_E1, MISC_REG_AEU_ENABLE4_FUNC_1_OUT_0, 0x500003e0},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE4_FUNC_1_OUT_0, 0xf00003e0},
	{OP_WR, MISC_REG_AEU_ENABLE1_FUNC_1_OUT_1, 0x0},
	{OP_WR, MISC_REG_AEU_ENABLE2_FUNC_1_OUT_1, 0xa000},
	{OP_ZR, MISC_REG_AEU_ENABLE3_FUNC_1_OUT_1, 0x5},
	{OP_WR, MISC_REG_AEU_ENABLE4_FUNC_1_OUT_2, 0xfe00000},
	{OP_ZR_E1, MISC_REG_AEU_ENABLE1_FUNC_1_OUT_3, 0x14},
	{OP_ZR_E1H, MISC_REG_AEU_ENABLE1_FUNC_1_OUT_3, 0x7},
	{OP_WR_E1, MISC_REG_AEU_ENABLE1_NIG_1, 0x55540000},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE4_FUNC_1_OUT_4, 0x800},
	{OP_WR_E1, MISC_REG_AEU_ENABLE2_NIG_1, 0x55555555},
	{OP_ZR_E1H, MISC_REG_AEU_ENABLE1_FUNC_1_OUT_5, 0x3},
	{OP_WR_E1, MISC_REG_AEU_ENABLE3_NIG_1, 0x5555},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE4_FUNC_1_OUT_5, 0x2000},
	{OP_WR_E1, MISC_REG_AEU_ENABLE4_NIG_1, 0x0},
	{OP_ZR_E1H, MISC_REG_AEU_ENABLE1_FUNC_1_OUT_6, 0x3},
	{OP_WR_E1, MISC_REG_AEU_ENABLE1_PXP_1, 0x55540000},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE4_FUNC_1_OUT_6, 0x8000},
	{OP_WR_E1, MISC_REG_AEU_ENABLE2_PXP_1, 0x55555555},
	{OP_ZR_E1H, MISC_REG_AEU_ENABLE1_FUNC_1_OUT_7, 0x3},
	{OP_WR_E1, MISC_REG_AEU_ENABLE3_PXP_1, 0x5555},
	{OP_WR_E1H, MISC_REG_AEU_ENABLE4_FUNC_1_OUT_7, 0x20000},
	{OP_WR_E1, MISC_REG_AEU_ENABLE4_PXP_1, 0x0},
	{OP_ZR_E1H, MISC_REG_AEU_INVERTER_1_FUNC_1, 0x4},
	{OP_WR_E1, MISC_REG_AEU_INVERTER_1_FUNC_1, 0x0},
	{OP_ZR_E1, MISC_REG_AEU_INVERTER_2_FUNC_1, 0x3},
	{OP_WR_E1, MISC_REG_AEU_MASK_ATTN_FUNC_1, 0x7},
#define MISC_AEU_PORT1_END      2123

};

/*(DEBLOBBED)*/

#endif /*__BNX2X_INIT_VALUES_H__*/
