# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###

import gst
#import gst.interfaces
import gobject

import vfs

from gst_daap import DaapSource
from plugins.player import PlayerManager

USE_FADE = False
FADE_STEP = 0.05
TIMEOUT_FADE_IN = 100
TIMEOUT_FADE_OUT = 100

DEFAULT_QUEUE_SIZE = (3 * gst.SECOND)
DEFAULT_QUEUE_MIN_THRESHOLD = ((DEFAULT_QUEUE_SIZE * 30) / 100)
DEFAULT_QUEUE_THRESHOLD = ((DEFAULT_QUEUE_SIZE * 95) / 100)

STREAM_URI = [ "http://", "mms://", "mmsh://",  "mmsu://", "mmst://" ]

def trace(func):
    def wrapper(*arg):
        class_name = ""
        if hasattr(func,'im_class'):
            class_name = func.im_class__name__+":"
        print '***'
        print 'DEBUG:%s' % (class_name+func.func_name)
        print arg
        print '***'
        res = func(*arg)
        return res
    return wrapper


def is_stream(uri):
    scheme = vfs.get_scheme(uri)
    return scheme in STREAM_URI

class IncorrectBinPlugin(Exception):
    pass

class PlayBin(gst.Pipeline):

    __src = None
    __is_stream = False
    __plugins = []

    def __init__(self,*args,**kwargs):
        gst.Pipeline.__init__(self,*args,**kwargs)

        self.auto_clock()

        self.__decoder = gst.element_factory_make('decodebin')
        self.__convert = gst.element_factory_make('audioconvert')

        def on_new_decoded_pad(element, pad, last):
            caps = pad.get_caps()
            name = caps[0].get_name()
            apad = self.__tee.get_pad('sink')
            if 'audio' in name:
                if not apad.is_linked(): # Only link once
                    pad.link(apad)

        self.__decoder.connect('new-decoded-pad', on_new_decoded_pad)

        
        self.__volume = gst.element_factory_make('volume', 'vlm')

        try: self.__sink = gst.element_factory_make('gconfaudiosink', 'sink')
        except: self.__sink = gst.element_factory_make('autoaudiosink', 'sink')
        
        self.__queue = gst.element_factory_make("queue")
        #self.__queue.set_property("m-level--time",DEFAULT_QUEUE_SIZE)
        #self.__queue.set_property("min-threshold-time",DEFAULT_QUEUE_THRESHOLD)

        self.__resample = gst.element_factory_make("audioresample")

        self.__tee = gst.element_factory_make("tee")


        self.add(self.__queue)
        self.add(self.__tee)
        self.add(self.__decoder)
        self.add(self.__convert)
        self.add(self.__resample)
        self.add(self.__volume)
        self.add(self.__sink)

        self.__queue.link(self.__decoder)
        self.__tee.link(self.__convert)
        self.__convert.link(self.__resample)
        self.__resample.link(self.__volume)
        self.__volume.link(self.__sink)

        self.set_property("auto-flush-bus",True)
        self.__plugin_manager = PlayerManager()

    def unload_plugins(self):
        
        # Wait state are changed
        self.set_state(gst.STATE_NULL)
        self.get_state()
        
        for p in self.__plugins:
            if p["type"]=="tee":
                self.__tee.unlink(p["bin"])
                self.remove(p["bin"])
            else:
                print "W:PlayBin:queue plugin not implemented"

        self.__plugins = []

        assert self.__tee.get_property("num-src-pads") == 1

    def load_plugins(self):
        self.__plugin_manager.scan()

        # Wait state are changed
        self.set_state(gst.STATE_NULL)
        self.get_state()
        
        #clean the bin
        for p in self.__plugins:
            if p["type"]=="tee":
                self.__tee.unlink(p["bin"])
                self.remove(p["bin"])
            else:
                raise NotImplemented
        
        assert self.__tee.get_property("num-src-pads") == 1

        self.__plugins = []
        tee_plug = 0
        for Kind in self.__plugin_manager.list(False):
            try: 
                p = Kind()
                bin = p.get_bin()
                type = self.__get_bin_type(bin)
            except Exception, e:
                print "W:PlayBin:load_plugins:",Kind," : ",e
                continue
            else:
                if type == "tee":
                    self.add(bin)
                    self.__tee.link_pads("src%d",bin,"sink")
                    tee_plug += 1
                else:
                   print "W:PlayBin:queue plugin not implemented"

                self.__plugins.append({"Kind":p,"bin":bin,"type":type})

        assert self.__tee.get_property("num-src-pads") == tee_plug + 1
        return self.__pipeline_test()

    def __get_bin_type(self,bin):
        """check if bin is correct and return
        if have 1 src and 1 sink or only one src
        """
        if not isinstance(bin,gst.Bin):
            raise IncorrectBinPlugin("PluginBin must be a subclass of gst.Bin")

        n_src_pags = 0
        iter = bin.src_pads()
        while True:
            try: pad = iter.next()
            except StopIteration: break
            if not pad.is_linked() and pad.can_link(self.__tee.get_pad("src%d")):
                n_src_pags += 1

        n_sink_pags = 0
        iter = bin.sink_pads()
        while True:
            try: pad = iter.next()
            except StopIteration: break
            if not pad.is_linked():
                n_sink_pags += 1

        if n_src_pags != 1:
            raise IncorrectBinPlugin("Bin can only have 1 src pad, "+str(n_src_pags)+" found")

        if n_sink_pags > 1:
            raise IncorrectBinPlugin("Bin can have 1 sink pad maximun, "+str(n_src_pags)+" found")
        
        if n_sink_pags == 0:
            return "tee"
        else:
            return "queue"

    def get_property(self,key):
        if key == "volume":
            return self.__volume.get_property("volume")
        elif key == "uri":
            if self.__src:
                return self.__src.get_property("location")
            return None
        elif key == "source":
            raise NotImplemented
        else:
            return super(PlayBin,self).get_property(key)

    def set_property(self,key,value):
        #print "D:PlayBin(",self.get_property("name"),"):set_property(",key,",",value,")"
        if key == "volume":
           return self.__volume.set_property("volume",value)
        elif key == "uri":
            if self.__src:
                self.set_state(gst.STATE_NULL)
                self.__src.unlink(self.__queue)
                self.remove(self.__src)
                del self.__src
                self.__src = None
            scheme = vfs.get_scheme( value )
            if scheme == "daap://":
                self.__src = DaapSource("src")
            else:
                self.__src = gst.element_make_from_uri(gst.URI_SRC, value ,"src")

            if not self.__src:
                return False

            self.__is_stream = is_stream(value)

            if proto != "http://" and hasattr(self.__src,"iradio-mode"):
                self.__src.set_property('iradio-mode',True)

            self.__src.set_property("location", value)
            self.add(self.__src)
            self.__src.link(self.__queue)
            return True
        else:
            return super(PlayBin,self).set_property(key,value)

#    def query_duration(self,format):
#        return self.__src.query_duration(format)
#
#    def query_position(self,format):
#        return self.__src.query_position(format)

class PlayBinWrapper(gobject.GObject):
    __volume = 0
    """
    Show if bin have already been started and stopped
    """
    __cur_started = False

    __id_cur_fade = None
    __id_prev_fade = None
    __cur_message_id = None
    __message_func = None
    def __init__(self):
        super(PlayBinWrapper,self).__init__()
        self.__curbin = PlayBin("bin1")
        self.__prevbin = PlayBin("bin2222")
    
    def set_message_func(self,func):
        if not callable(func):
            raise Exception("func must be a function")
        self.__message_func = func

    def __on_message(self,bus,message):
        """Forward message from the current bin to the player"""
        if self.__message_func:
            self.__message_func(bus,message)
    
    def __fade_in(self,bin):
        cur_vol = bin.get_property("volume")
        print "D:PlayBinWrapper(",bin.get_property("name"),"):__fade_in(",bin,"):",cur_vol

        if bin.get_state()[1] == gst.STATE_NULL:
            bin.set_property("volume", self.__volume)
            self.__id_cur_fade = None
            return False

        if cur_vol < self.__volume - FADE_STEP :
            bin.set_property("volume", cur_vol + FADE_STEP)
            return True
        else:
            bin.set_property("volume", self.__volume)
            self.__id_cur_fade = None
            return False

    def __fade_out(self,bin,i):
        cur_vol = bin.get_property("volume")
        print "D:PlayBinWrapper(",bin.get_property("name"),"):__fade_out(",bin,"):",cur_vol," === ",i

        if bin.get_state()[1] == gst.STATE_NULL:
            self.__id_prev_fade = None
            bin.set_property("volume", 0)
            return False

        if cur_vol >  FADE_STEP :
            bin.set_property("volume", cur_vol - FADE_STEP)
            return True
        else:
            bin.set_property("volume", 0)
            bin.set_state(gst.STATE_NULL)
            self.__id_prev_fade = None
            return False

    def get_state(self,*args):
        if USE_FADE and not self.__cur_started: 
            return gst.STATE_CHANGE_SUCCESS, gst.STATE_NULL, gst.STATE_VOID_PENDING
        else:
            return self.__curbin.get_state(*args)

    def set_state(self,state):
        """ change the state of the curbin
        for gst.STATE_NULL or gst.STATE_PLAY fade can be used to setup volume
        """
        if state == gst.STATE_PAUSED or state == gst.STATE_NULL:
            # stop prevbin
            if self.__id_prev_fade:
                gobject.source_remove( self.__id_prev_fade )
                self.__id_prev_fade = None
            self.__prevbin.set_state( gst.STATE_NULL )

            # stop fade of curbin
            if self.__id_cur_fade:
                gobject.source_remove( self.__id_cur_fade )
                self.__id_cur_fade = None

        if state == gst.STATE_PAUSED:
            self.__curbin.set_property( "volume" , self.__volume )

        if self.__cur_started and state == gst.STATE_NULL : 
            self.__cur_started = False
            if USE_FADE:
                self.__id_prev_fade = gobject.timeout_add( TIMEOUT_FADE_OUT , self.__fade_out, self.__curbin,False)
                self.__id_prev_fade = gobject.idle_add( self.__fade_out, self.__curbin ,True)
                return gst.STATE_CHANGE_SUCCESS, gst.STATE_NULL, gst.STATE_VOID_PENDING

        if not self.__cur_started and state == gst.STATE_PLAYING:
            self.__cur_started = True
            if USE_FADE:
                self.__curbin.set_property( "volume" , 0 )
                self.__id_cur_fade = gobject.timeout_add( TIMEOUT_FADE_IN , self.__fade_in, self.__curbin)

        state_ret = self.__curbin.set_state(state)
        timeout = 4
        while state_ret == gst.STATE_CHANGE_ASYNC and timeout > 0:
            state_ret,state,pending_state = self.__curbin.get_state(1 * gst.SECOND)
            timeout -= 1

        return state_ret

    def get_property(self,key):
        if key == "volume":
            return self.__volume

        return self.__curbin.get_property(key)

    def set_property(self,key,value):
        if key == "uri":
            
            # Stop current fade if any
            if self.__id_cur_fade:
                gobject.source_remove( self.__id_cur_fade )
                self.__id_cur_fade = None

            # stop prevbin
            #FIXME: wait pipeline is NULL ?
            if self.__id_prev_fade:
                gobject.source_remove( self.__id_prev_fade )
                self.__id_prev_fade = None
            self.__prevbin.set_state( gst.STATE_NULL )
            
            self.__prevbin.set_property("uri",value)
            self.__prevbin.set_property("volume",self.__volume)
            
            #Disconnect the message of old pipeline
            if self.__cur_message_id:
                bus = self.__curbin.get_bus()
                bus.add_signal_watch()
                bus.disconnect(self.__cur_message_id)
            
            #Connect message of the new pipeline
            bus = self.__prevbin.get_bus()
            bus.add_signal_watch()
            self.__cur_message_id = bus.connect('message', self.__on_message)
            
            #swicth cur and prev pipeline
            tmp = self.__curbin
            self.__curbin = self.__prevbin
            self.__prevbin = tmp

            return True

        elif key == "volume":
            self.__volume = value
            if not USE_FADE:
               self.__prevbin.set_property(key, value)

        return self.__curbin.set_property(key,value)

    def query_duration(self,format):
        return self.__curbin.query_duration(format)

    def query_position(self,format):
        return self.__curbin.query_position(format)

    def send_event(self,event):
        return self.__curbin.send_event(event)

    def set_new_stream_time(self,time):
        return self.__curbin.set_new_stream_time(time)


    def reload_pipeline(self,*args):
        print "I:PlayBinWrapper:reload_pipeline"

        try:gobject.source_remove(self.__id_reload_stop)
        except: self.__id_reload_stop = None
        else: self.__id_reload_stop = None

        # Stop current fade if any
        if self.__id_cur_fade:
            gobject.source_remove( self.__id_cur_fade )
            self.__id_cur_fade = None

        # stop prevbin
        #FIXME: wait pipeline is NULL ?
        if self.__id_prev_fade:
            gobject.source_remove( self.__id_prev_fade )
            self.__id_prev_fade = None

        self.__prevbin.set_state( gst.STATE_NULL )
        self.__prevbin.reload_plugins()

        self.__prevbin.set_property("uri",self.__curbin.get_property("uri"))
        self.__prevbin.set_property("volume",0)
        self.__prevbin.set_clock(self.__curbin.get_clock())

        # Start the new bin
        new_state,state,status = self.__curbin.get_state()
        state_ret = self.__prevbin.set_state(state)
        timeout = 4
        state = None
        while state_ret == gst.STATE_CHANGE_ASYNC and timeout > 0:
            state_ret,state,pending_state = self.__prevbin.get_state(1 * gst.SECOND)
            timeout -= 1
              
        if state_ret != gst.STATE_CHANGE_SUCCESS :

            print "W:BinPlayWrapper:Failed to reload pipeline, use the old one (WHY:STATE)"
            self.__prevbin.set_state(gst.STATE_NULL)

        else:

            p = self.__curbin.query_position(gst.FORMAT_TIME)
            """
            event = gst.event_new_seek(
                1.0, gst.FORMAT_TIME, gst.SEEK_FLAG_FLUSH | gst.SEEK_FLAG_ACCURATE,
                gst.SEEK_TYPE_SET, p, gst.SEEK_TYPE_NONE, 0)


            if not self.__prevbin.send_event(event):
            """
            if not self.__prevbin.seek_simple(p[1],gst.SEEK_FLAG_FLUSH | gst.SEEK_FLAG_ACCURATE,p[0]):

                print "W:BinPlayWrapper:Failed to reload pipeline, use the old one (WHY: seek)"
                self.__prevbin.set_state(gst.STATE_NULL)

            else:

                self.__prevbin.set_new_stream_time(0L)

                #Disconnect the message of old pipeline
                if self.__cur_message_id:
                    bus = self.__curbin.get_bus()
                    bus.add_signal_watch()
                    bus.disconnect(self.__cur_message_id)
                
                #Connect message of the new pipeline
                bus = self.__prevbin.get_bus()
                bus.add_signal_watch()
                self.__cur_message_id = bus.connect('message', self.__on_message)

                #Set volume at the same of the old bin 
                self.__prevbin.set_property("volume",self.__curbin.get_property("volume"))
                
                #self.__id_reload_stop = gobject.timeout_add(500,self.__curbin.set_state,gst.STATE_NULL)
                self.__curbin.set_state(gst.STATE_NULL)
                
                self.__curbin.reload_plugins()

                #swicth cur and prev pipeline
                tmp = self.__curbin
                self.__curbin = self.__prevbin
                self.__prevbin = tmp

