# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###


import traceback
import gobject
import gtk

import stock

from hal import hal
from widget.browser import Browser
from widget.song_view import SortableSongView, PlaylistSongView
from widget.song_menu import SongMenu, SongMenuManager
from library import ListenDB, ListenDBQuery
from source import Source, SourceItem, PlaylistSourceItem
from song import Song

from helper import SignalCollector,Dispatcher, SignalContainer
from widget.misc import SimpleMenu

from logger import Logger

class DeviceNotSupportedException(Exception):
    pass

class DeviceWrapperInitException(Exception):
    pass

class DeviceSongView(SortableSongView):
    def __init__(self,conf_prefix):
        super(DeviceSongView,self).__init__(conf_prefix)
        self.set_menu( SongMenu() )

class DeviceBrowser(Browser):
    def __init__(self,udi):
        
        tree_song = DeviceSongView("device")
        super(DeviceBrowser,self).__init__(ListenDBQuery("device_"+udi),tree_song,"device",True,True)
        
        self.menu = SongMenu(for_browser=True)

        self.set_menu(self.menu)

        if ListenDB.isloaded():
            self.__on_db_loaded(ListenDB)
        else:
            self.autoconnect(ListenDB,"loaded",self.__on_db_loaded)

    def __on_db_loaded(self,db):
        self.connect_to_db()

class DevicePodcastSongView(SortableSongView):
    def __init__(self,conf_prefix, udi):
        kargs = {
            "#date":(_("Date"),gobject.TYPE_STRING),
            "title":(_("Title"),gobject.TYPE_STRING),
            "album":(_("Feed"),gobject.TYPE_STRING),
            "#duration":(_("Length"),gobject.TYPE_STRING),
            "#playcount":(_("Play count"),gobject.TYPE_STRING),
            "#lastplayed":(_("Last Played"),gobject.TYPE_STRING),
                      }    
        super(DevicePodcastSongView,self).__init__(conf_prefix,**kargs)

        for item in [ "play", "enqueue", "delete" ]:
            SongMenuManager.allow_item_for_type(item, "device_"+udi+"_podcast")

        dw = ListenDB.get_wrapper("device_"+udi+"_podcast")
        if "writable" in dw.capability:
            SongMenuManager.allow_item_for_type("delete", "device_"+udi+"_podcast")

        self.set_menu( SongMenu() )

class DevicePodcastBrowser(Browser):
    def __init__(self,udi):
        tree_song = DevicePodcastSongView("device_podcast", udi)
        super(DevicePodcastBrowser,self).__init__(ListenDBQuery("device_" + udi + "_podcast"),tree_song,"device_podcast",True,True)
        
        self.menu = SongMenu(for_browser=True)

        self.empty_menu = SimpleMenu()
        self.empty_menu.show_all()

        self.set_menu(self.menu,self.empty_menu)

        if ListenDB.isloaded():
            self.__on_db_loaded(ListenDB)
        else:
            self.autoconnect(ListenDB,"loaded",self.__on_db_loaded)

    def get_cols(self,all=False):
        cols = [("col0","album",_("Feed"), _("Feeds"))]
        return cols

    def __on_db_loaded(self,db):
        self.connect_to_db()


class DevicePlaylistSongView(PlaylistSongView):
    pass

class DevicePlaylistSourceItem(PlaylistSourceItem):
    is_droppage = False
    editable = False

    _playlist_songview = DevicePlaylistSongView
    _autoplaylist_songview = None

    def __init__(self,udi,pl):
        PlaylistSourceItem.__init__(self,pl)
        self.config_code = udi
        self.__udi = udi
        dw = ListenDB.get_wrapper("device_"+self.__udi)
        if "playlist" in dw.capability and "writable" in dw.capability:
            self.is_droppage = True
            self.editable = True

class DevicePodcastSourceItem(SourceItem):
    config_code = "device"
    def __init__(self,udi,widget_klass):
        self.__udi = udi
        self.widget = widget_klass(self.__udi)

    label = _("Podcasts")
    stock = stock.SRC_PODCAST_IPOD

    label = property(lambda self: ListenDB.get_wrapper("device_"+self.__udi+"_podcast").get_podcast_name())
    stock = property(lambda self: ListenDB.get_wrapper("device_"+self.__udi+"_podcast").get_podcast_stock())

    def on_drag_song_received(self,songs):
        dw = ListenDB.get_wrapper("device_"+self.__udi+"_podcast")
        if "writable" in dw.capability:
            [ ListenDB.create_song(song,"device_"+self.__udi+"_podcast") for song in songs if song.get_type().endswith("podcast") ]
            return True
        else:
            return False

class DeviceSourceItem(SourceItem):
    config_code = "device"
    def __init__(self, udi, widget_klass):
        self.__udi = udi
        self.widget = widget_klass(self.__udi)

    label = property(lambda self: ListenDB.get_wrapper("device_"+self.__udi).get_name())
    stock = property(lambda self: ListenDB.get_wrapper("device_"+self.__udi).get_stock())

    def get_menu(self):
        dw = ListenDB.get_wrapper("device_"+self.__udi)
        menu = gtk.Menu()     
        if "playlist" in dw.capability:
            #menu.append(gtk.SeparatorMenuItem())
            item = gtk.ImageMenuItem(stock.SRC_PLAYLIST)
            item.connect("activate",self.__new_playlist)
            menu.append(item)
        if "property" in dw.capability:
            item = gtk.ImageMenuItem(gtk.STOCK_PROPERTIES)
            item.connect("activate",dw.show_property)
            menu.append(item)

        item = gtk.ImageMenuItem(stock.EJECT)
        item.connect("activate",self.eject)
        menu.append(item)

        menu.show_all()
        return menu

    def eject(self,*args):
        ListenDB.get_wrapper("device_"+self.__udi).save()
        hal.eject(self.__udi)

    def __new_playlist(self,*args,**kwargs):
        ListenDB.create_playlist("device_"+self.__udi,_("New playlist"))

    def on_drag_song_received(self,songs):
        dw = ListenDB.get_wrapper("device_"+self.__udi)
        types = dw.get_authorized_types()
        if "writable" in dw.capability:
            for song in songs:
                if song.get_type() in types:
                    newtype = "device_"+self.__udi
                    if "podcast" in song.get_type():
                        newtype = "device_"+self.__udi+"_podcast"
                    ListenDB.create_song(song, newtype)
            return True
        else:
            return False

class DeviceWrapper(SignalContainer,Logger):
    # capability of the device, can be writeable, playlist, podcast, autoplaylist
    capability = []

    def __init__(self,udi):
        if not self.check(udi):
            raise DeviceNotSupportedException
        self.udi = udi
        SignalContainer.__init__(self)
        self.__type = "device_"+udi
        
    def save(self):
        pass

    def get_type(self):
        return self.__type

    def show_property(self, menuitem):
        pass

    def get_authorized_types():
        return []

    def set_playlist_name(self,pl,name):
        raise NotImplementedError()

    def get_name(self):
        return _("Unknown")

    def get_podcast_name(self):
        return _("Podcasts")

    def get_stock(self):
        return stock.SRC_UNKNOWN

    def get_podcast_stock(self):
        return stock.SRC_UNKNOWN

    def playlist_update(self,pl,uris,callback,*args_cb):
        raise NotImplementedError()

    def add_playlist(self,type, pl,callback,*args_cb):
        raise NotImplementedError()

    def del_playlist(self,type, pl,callback,*args_cb):
        raise NotImplementedError()

    def add_song(self,song,callback,*args_cb):
        raise NotImplementedError()

    def remove_song(self,song,callback,*args):
        raise NotImplementedError()

    def check(self,udi):
        raise NotImplementedError()

    def destroy(self):
        self.autodisconnect_all()

    def load_data(self):
        pass

class DeviceSource(Source):
    categorie = "device"
    __index__ = 0
    DEVICE_WRAPPER = None
    def __init__(self):
        self.__class__.__index__ += 1

        self.items = []
        self.__device_items = {}

        # Connect to bd only to wait there are loaded
        if ListenDB.isloaded():
            self.connect_to_db()
        else:
            self.autoconnect(ListenDB,"loaded",self.__db_loaded)

    def __db_loaded(self,db):
        self.connect_to_db()
    
    def connect_to_db(self):
        self.autoconnect(hal,"volume-mounted",self.mount)
        self.autoconnect(hal,"volume-umounted",self.umount)

        self.autoconnect(ListenDB,"playlist-added",self.__add_playlist)
        self.autoconnect(ListenDB,"playlist-removed",self.__remove_playlist)

        for udi in hal.get_mounted_devices_udi():
            gobject.idle_add(self.mount,hal,udi)
 
    def __add_playlist(self,db,type,pls):
        udi = type.replace("device_","")
        if self.__device_items.has_key(udi):
            [ self.__device_items[udi].append(DevicePlaylistSourceItem(udi,pl)) for pl in pls ]
            self.build_source_items()

    def __remove_playlist(self,db,type,pls):
        udi = type.replace("device_","")
        if self.__device_items.has_key(udi):

            to_delete = [ index for index,item in enumerate(self.__device_items[udi]) \
                if isinstance(item,DevicePlaylistSourceItem) and item.pl in pls ]
            [ self.__device_items[udi].pop(index) for index in to_delete ]

            self.build_source_items()

    def mount(self,hal,udi):
        if not self.__device_items.has_key(udi):
            try: dw = self.DEVICE_WRAPPER(udi)
            except DeviceNotSupportedException:
                return
            except DeviceWrapperInitException:
                self.logwarn("failed to load device %s with %s",udi,self.DEVICE_WRAPPER.__name__)
                return
            
            self.loginfo("%s loaded by plugin %s",udi,self.DEVICE_WRAPPER.__name__)
            ListenDB.register_wrapper("device_"+udi,dw)
            if "podcast" in dw.capability:
                ListenDB.register_wrapper("device_"+udi+"_podcast",dw)

            dw.load_data()

            def song_menu_action(song_menu_manager,id_menu,songs):
                if id_menu == "device_add_to_"+udi:
                    [ ListenDB.create_song(song,"device_"+udi) for song in songs ]
                if id_menu == "device_add_to_"+udi+"_podcast":
                    [ ListenDB.create_song(song,"device_"+udi+"_podcast") for song in songs ]

            dw.autoconnect(SongMenuManager,"action",song_menu_action)

            for item in ["play","enqueue","lyrics","wikipedia_album","wikipedia_artist","lastfm","delete"]:
                SongMenuManager.allow_item_for_type(item ,"device_"+udi)
            
            # Check if add_song are implemented
            if dw.add_song.im_func != DeviceWrapper.add_song.im_func:
                icons = gtk.IconFactory()
                icons.add("device_add_to_"+udi,gtk.icon_factory_lookup_default(gtk.STOCK_ADD))
                icons.add_default()
                gtk.stock_add([("device_add_to_"+udi,_("Add to %s")%dw.get_name(),0,0,"")])
   
                SongMenuManager.register_item("device_add_to_"+udi,200,"device_add_to_"+udi,sep_after=True,sep_before=True)
                SongMenuManager.allow_item_for_type("device_add_to_"+udi,"local")

            if dw.add_song.im_func != DeviceWrapper.add_song.im_func and "podcast" in dw.capability:
                icons = gtk.IconFactory()
                icons.add("device_add_to_"+udi+"_podcast",gtk.icon_factory_lookup_default(gtk.STOCK_ADD))
                icons.add_default()
                gtk.stock_add([("device_add_to_"+udi+"_podcast",_("Add podcast to %s")%dw.get_podcast_name(),0,0,"")])
   
                SongMenuManager.register_item("device_add_to_"+udi+"_podcast",200,"device_add_to_"+udi+"_podcast",sep_after=True,sep_before=True)
                SongMenuManager.allow_item_for_type("device_add_to_"+udi+"podcast","podcast")

            self.__device_items[udi] = [ DeviceSourceItem(udi,DeviceBrowser) ]
            if "podcast" in dw.capability:
                self.__device_items[udi].append(DevicePodcastSourceItem(udi, DevicePodcastBrowser) )

            self.build_source_items()

    def umount (self,hal,udi):
        # remove widget
        if not self.__device_items.has_key(udi): return
        self.unload_udi(udi)
        # Force Source list refresh
        self.build_source_items()

    def unload_udi(self, udi):
        if isinstance(self.__device_items[udi],DeviceSourceItem):
            self.__device_items[udi].widget.save_config()
        del self.__device_items[udi]

        dw = ListenDB.get_wrapper("device_"+udi)

        # remove SongMenuManager ref
        SongMenuManager.unload_items_contain(udi)

        # Clean library
        ListenDB.unregister_wrapper("device_"+udi)
        if "podcast" in dw.capability:
            ListenDB.unregister_wrapper("device_"+udi+"_podcast")

        # Destroy wrapper
        dw.destroy()

    def build_source_items(self):
        self.items = []
        for udi,items in self.__device_items.iteritems():
            self.items.extend(items)
        Dispatcher.update_source()

    def destroy(self):
        for udi, items in self.__device_items.items():
            self.unload_udi(udi)
       
        self.__device_items = {}
        self.items = []
        self.build_source_items()
        super(DeviceSource,self).destroy()

