# -*- coding: utf-8 -*-
# vim: ts=4 
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###



import gobject
import os
from config import config
from threading import Condition, Thread
import utils
import time
from random import shuffle

from xdg_support import get_xdg_config_file
import vfs

from song import Song, TAG_KEYS
from logger import Logger

from parse import Query

class MissingUriTag(Exception):
    pass

BROWSER_KEYS = ["genre", "artist", "album"]

AUTOSAVE_TIMEOUT = 1000*60*5 # 5min
SIGNAL_DB_QUERY_FIRED=50

import gtk

class SanitizeWindow(gtk.Dialog):
    """ This object are always used in a thread
    Take care of gdk threading
    """
    def __init__(self):
        title = _("Checking and Updating database")
        gtk.Dialog.__init__(self,title,None,
                 gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT, tuple())
#                     (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,)
#                 )

        self.set_position(gtk.WIN_POS_CENTER_ON_PARENT)
        self.set_resize_mode(False)
        self.set_border_width(6)
        self.set_modal(True)
        self.set_property("skip-taskbar-hint",True)
        self.set_has_separator(False)

        self.label_heading = gtk.Label("<span size=\"large\"><b>"+title+"</b></span>")
        self.label_heading.set_alignment(0,0.5)
        self.label_heading.set_use_markup(True)

        self.box_contenu = gtk.VBox(False,12)
        self.box_contenu.pack_start(self.label_heading,False,False)

        hbox = gtk.HBox(False,12)

        hbox.pack_start(gtk.image_new_from_stock(gtk.STOCK_REFRESH, gtk.ICON_SIZE_BUTTON), False, False)
        hbox.pack_start(self.box_contenu,True,True)
        hbox.set_border_width(6)

        self.vbox.pack_start(hbox,False,False)
        self.vbox.set_spacing(6)
        self.set_resizable(False)
            
        self.progress = gtk.ProgressBar()
        self.box_contenu.pack_start(self.progress,True,True)

        self.nb_items = 0
        self.update_count = 0
    
    def run(self):
        self.show_all()
        ret = gtk.Dialog.run(self)
        if ret ==  gtk.RESPONSE_CANCEL:
            # Sadly exit user don't want update database
            gtk.main_quit()
        
    def set_nb_items(self, n):
        self.nb_items = n

    def update(self):
        if self.nb_items == 0:
            self.progress.pulse()
        else:
            self.update_count += 1
            fraction = float(self.update_count)/float(self.nb_items)
            if fraction > 1.0: fraction = 1.0
            elif fraction < 0.0: fraction = 0.0
            self.progress.set_fraction(fraction)
            self.progress.set_text("%d/%d items updated"%(int(self.update_count), int(self.nb_items)) )

    def close(self):
        self.destroy()

class ListenDatabase(gobject.GObject,Logger):
    __gsignals__ = {
        "loaded" : (gobject.SIGNAL_RUN_LAST,
                    gobject.TYPE_NONE,
                    ()),
        "changed" : (gobject.SIGNAL_RUN_LAST,
                gobject.TYPE_NONE,
                (str, gobject.TYPE_PYOBJECT)),
        "quick-changed" : (gobject.SIGNAL_RUN_LAST,
                gobject.TYPE_NONE,
                (str, gobject.TYPE_PYOBJECT)),
        "simple-changed" : (gobject.SIGNAL_RUN_LAST,
                gobject.TYPE_NONE,
                (gobject.TYPE_PYOBJECT,)),
        "added"  : (gobject.SIGNAL_RUN_LAST,
                gobject.TYPE_NONE,
                (str, gobject.TYPE_PYOBJECT)),
        "removed"  : (gobject.SIGNAL_RUN_LAST,
                gobject.TYPE_NONE,
                (str, gobject.TYPE_PYOBJECT)),
        "playlist-added"  : (gobject.SIGNAL_RUN_LAST,
                gobject.TYPE_NONE,
                (str, gobject.TYPE_PYOBJECT)),
        "playlist-removed"  : (gobject.SIGNAL_RUN_LAST,
                gobject.TYPE_NONE,
                (str, gobject.TYPE_PYOBJECT))

    }

    def __init__(self):
        gobject.GObject.__init__(self)
        self.__db_wrapper = {}
        self.__db_wrapper_load = {}
        self.__reset_queued_signal()
        self.__condition = Condition()
        self.__db_operation_lock = Condition()

        self.__tree = {}
        self.__songs = {}
        self.__playlists = {}
        self.__songs_by_type = {}
        self.__hiddens = set()
        self.__song_types = []
        self.__song_types_capability = {}
        self.__playlist_types = []
        self._dirty = False
        self.__is_loaded = False
        self.__force_sanity_check = False

        self.__SAVE_DATA_TYPE = ["local", "podcast", "webradio", "lastfmradio" ]

        #gobject.idle_add(self.__load)


    def __reset_queued_signal(self):
        self.__queued_signal ={ 
            "added" : {},
            "removed" : {},
            "changed" : {},
            "quick-changed" : {},
            "playlist-added" : {},
            "playlist-removed" : {}
            }

    def set_dirty(self):
        self._dirty = True

    def isloaded(self):
        return self.__is_loaded
    
    @utils.threaded
    def async_save(self):
        self.save()
    
    def save_data_type(self, type):
        self.__SAVE_DATA_TYPE.append(type)

    def save(self):
        if not self._dirty: return True

        # Quickly copy obj ref before pickle it
        self.__db_operation_lock.acquire()

        songs = self.__songs.values()
        hiddens = self.__hiddens.copy()
        playlists = self.__playlists["local"].copy()

        self.__db_operation_lock.release()

        objs = [ song.get_dict() for song in songs if song.get_type() in self.__SAVE_DATA_TYPE and song not in hiddens]

        utils.save_db(objs, get_xdg_config_file("songs.db") )
        utils.save_db([ pl.get_pickle_obj() for pl in playlists ] , get_xdg_config_file("playlists.db"))

        self.loginfo("%d songs saved and %d playlists saved ", len(objs), len(playlists))
        self._dirty = False

    def set_force_sanity_check(self):
        self.__force_sanity_check = True

    @utils.threaded
    def load(self):
        self.loginfo("Loading library...")

        try:
            db_objs = utils.load_db( get_xdg_config_file("songs.db") )
        except:
            self.logexception("Failed to load library")
            db_objs = []
        try:
            pls_objs = utils.load_db(get_xdg_config_file("playlists.db"))
        except :
            self.logexception("Failed load playlists")
            pls_objs = []

        if self.__force_sanity_check:
            gtk.gdk.threads_enter()
            update_window = SanitizeWindow()
            update_window.set_nb_items(len(db_objs) + len(pls_objs))
            update_window.show_all()
            gtk.gdk.threads_leave()

        if db_objs:
            for obj in db_objs:
                try: stype = obj["==song_type=="]
                except KeyError: 
                    self.logerror("Song with no type found, %s", obj.get("uri"))
                    continue

                if stype not in self.__song_types: 
                    self.logwarn("Song type %s not exist, force registration", stype)
                    self.register_type(stype)

                s = Song(obj)
                s.set_type(stype)
                if self.__force_sanity_check:
                    s["uri"] = vfs.realuri(s.get("uri"))
                if s in self.__hiddens: 
                    self.__hiddens.remove(s)
                    del self.__songs[s.get("uri")]
                if not self.__force_sanity_check or not self.__songs.has_key(s.get("uri")):
                    self.add([s], self.__force_sanity_check)
                    if self.__force_sanity_check:
                        gtk.gdk.threads_enter()
                        update_window.update()
                        gtk.gdk.threads_leave()

        if pls_objs:
            for pl_obj in pls_objs:
                auto, name, infos = pl_obj
                if auto:
                    self.create_autoplaylist("local", "local", name, infos)
                else:
                    if self.__force_sanity_check:
                        infos = map(vfs.realuri, infos)
                    self.create_playlist("local", name, infos)
                if self.__force_sanity_check:
                    gtk.gdk.threads_enter()
                    update_window.update()
                    gtk.gdk.threads_leave()

        if self.__force_sanity_check:
            self.save()
            gtk.gdk.threads_enter()
            update_window.close()
            gtk.gdk.threads_leave()
        self._dirty = False

        self.__reset_queued_signal()
        gobject.timeout_add(AUTOSAVE_TIMEOUT , self.async_save)
        gobject.timeout_add(SIGNAL_DB_QUERY_FIRED*20 , self.__fire_queued_signal)
 
        self.loginfo("%s songs loaded in %d types ", len(self.__songs), len(self.__song_types))
        for type in self.__songs_by_type.keys():
            self.loginfo(" - %s > %d songs", type, len(self.__songs_by_type[type]) )

        #gobject.timeout_add(DELAY_POST_LOAD, self.__delay_post_load)

        # When signal are emited all widget like (playlist , browser, ...) will query the database to get information
        # So, wait listen start (gtk loop start)
        self.loginfo("Finish loading library")
        gobject.idle_add(self.__delay_post_load)

    def __delay_post_load(self):
        self.__is_loaded = True
        self.emit("loaded")
    
    def __fire_queued_signal(self):
        self.__condition.acquire()
        try:
            for type, songs in self.__queued_signal["removed"].iteritems():
                self.emit("removed", type, songs)

            for type, songs in self.__queued_signal["added"].iteritems():
                self.emit("added", type, songs)

            for type, infos in self.__queued_signal["changed"].iteritems():
                self.emit("changed", type, infos)
                self.emit("simple-changed", [ i[0] for i in infos] )

            for type, infos in self.__queued_signal["quick-changed"].iteritems():
                self.emit("quick-changed", type, infos)
                #FIXME: fire simple-changed or not !
                # Not for now only podcast browser use this feature
                #self.emit("simple-changed", [ i[0] for i in infos] )

            for type, playlists in self.__queued_signal["playlist-added"].iteritems():
                self.emit("playlist-added", type, playlists)

            for type, playlists in self.__queued_signal["playlist-removed"].iteritems():
                self.emit("playlist-removed", type, playlists)
        except: 
            self.logexception("Failed fire queued signal")

        self.__reset_queued_signal()
        self.__condition.release()
        return True

    def song_has_capability(self, song, capability):
        type = song.get_type()
        if self.__song_types_capability.has_key(type) and \
                capability in self.__song_types_capability[type]:
            return True
        else:
            return False

    def register_type(self, name, capability=[]):
        if name not in self.__song_types:
            self.__song_types.append(name)
            self.__song_types_capability[name] = capability
            self.__songs_by_type[name]=set()

    def unregister_type(self, name):
        if name in self.__song_types:
            self.__db_operation_lock.acquire()
            [ self.__songs.pop(song.get("uri")) for song in self.__songs_by_type[name] ]
            self.__db_operation_lock.release()
            self.__song_types.remove(name)
            del self.__song_types_capability[name]
            del self.__songs_by_type[name]
        else:
            self.logwarn("Unregister a unknown type %s", name)

    def register_wrapper(self, type, dw):
        if not self.__db_wrapper.has_key(type):
            self.register_type(type)
            self.register_playlist_type(type)
            self.__db_wrapper[type] = dw
            self.__db_wrapper_load[type] = False
        else:
            self.logwarn("Register already existing wrapper \"%s\"",type)

    def unregister_wrapper(self, name):
        self.unregister_type(name)
        self.unregister_playlist_type(name)
        try: 
            del self.__db_wrapper[name]
            del self.__db_wrapper_load[name]
        except KeyError:
             self.logwarn("Unregister a unknown type wrapper \"%s\"",name)

    def register_playlist_type(self, name):
        if name not in self.__playlist_types:
            self.__playlist_types.append(name)
            self.__playlists[name] = set()

    def unregister_playlist_type(self, name):
        if name in self.__playlist_types:
            # FIXME: free uris in playlist
            self.__playlist_types.remove(name)
            del self.__playlists[name]
        else:
            self.logwarn("W:ListenDB:Unregister a unknown playlist type \"%s\"", name)
 
    def get_wrapper(self, type):
        try: return self.__db_wrapper[type]
        except KeyError: return None

    def set_db_wrapper_load(self,name,mode):
        self.__db_wrapper_load[name] = mode

    def get_db_wrapper_load(self,name):
        try: return self.__db_wrapper_load[name]
        except KeyError: return False

    def request(self,type, string):
        try: 
            filter = Query(string).search
        except Query.error:
            self.loginfo("Request: Query error %s", string )
            return None
        else:
            if not self.__songs_by_type.has_key(type):
                self.loginfo("Request: type %s not exist", type)
                return None
            return [ song for song in self.__songs_by_type[type] if not song.get("hidden") and filter(song) ]

    def change(self, songs):
        """ prefer use set_property for a better ui update """
        self.logdeprecated("ListenDB.change() are deprecated and won't work correctly with device wrapper")
        self.remove(songs)
        self.add(songs)

    def add(self, songs, sanitarize=True):
        for song in songs:
            if sanitarize: song.sanitarize()
            type = song.get_type()
            if not self.__db_wrapper.has_key(type) or self.__db_wrapper_load[type]:
                self.__add_cb(song)
            else:
                dw = self.__db_wrapper[type]
                try: dw.add_song(song, self.__add_cb)
                except NotImplementedError(): continue

    def __add_cb(self, song):
        self._dirty = True
        type = song.get_type()
        uri = song.get("uri")
        self.__db_operation_lock.acquire()
        self.__songs[uri] = song
        self.__songs_by_type.setdefault(type, set())
        self.__songs_by_type[type].add(song)

        self.__condition.acquire()
        self.__queued_signal["added"].setdefault(type, set())
        self.__queued_signal["added"][type].add(song)
        self.__condition.release()
        self.__db_operation_lock.release()

    def remove(self, songs):
        self._dirty = True
        if not isinstance(songs,(list, set)): songs = [songs]
        for song in songs:
            type = song.get_type()
            if song in self.__hiddens:
                self.__db_operation_lock.acquire()
                self.__hiddens.remove()
                self.__db_operation_lock.release()
            if not self.__db_wrapper.has_key(type) or self.__db_wrapper_load[type]:
                self.__remove_cb(song)
            else:
                dw = self.__db_wrapper[type]
                try: dw.remove_song(song, self.__remove_cb)
                except NotImplementedError(): continue

    def __remove_cb(self, song):
        type = song.get_type()
        self.__db_operation_lock.acquire()
        try: del self.__songs[song.get("uri")]
        except KeyError: pass
        self.__songs_by_type[type].remove(song)
        self.__condition.acquire()
        self.__queued_signal["removed"].setdefault(type, set())
        self.__queued_signal["removed"][type].add(song)
        self.__condition.release()
        self.__db_operation_lock.release()
    
    def set_property(self, song, keys_values, write_to_file=False, use_quick_update=False):
        if not song: return False
        ret = True
        self._dirty = True
        type = song.get_type()
        old_keys_values = {}
        mod_keys = keys_values.keys()
        [ old_keys_values.update({key:song.get_sortable(key)}) for key in song.keys() if key in mod_keys]
        [ song.update({key:value}) for key, value in keys_values.items() if value is not None]
        for key in [ key for key, value in keys_values.items() if value is None ]: del song[key]
        song.sanitarize()
        if write_to_file: ret = song.write_to_file()
        self.__condition.acquire()
        if use_quick_update:
            self.__queued_signal["quick-changed"].setdefault(type, [])
            self.__queued_signal["quick-changed"][type].append((song, old_keys_values))
        else:
            self.__queued_signal["changed"].setdefault(type, [])
            self.__queued_signal["changed"][type].append((song, old_keys_values))
        self.__condition.release()
        return ret
        
    def del_property(self, song, keys):
        if not song: return False
        self._dirty = True
        type = song.get_type()
        old_keys_values = {}
        [ old_keys_values.update({key:song.get_sortable(key)}) for key in song.keys() if key in keys]
        for key in keys: del song[key]
        song.sanitarize()
        self.__condition.acquire()
        self.__queued_signal["changed"].setdefault(type, [])
        self.__queued_signal["changed"][type].append((song, old_keys_values))
        self.__condition.release()
        
    def reload_song_from_file(self, song):
        s = Song({"uri":song.get("uri")})
        s["uri"] = vfs.realuri(s.get("uri"))
        s.read_from_file()
        s.sanitarize()
        new_tags = {}
        for key in TAG_KEYS.values()+["#size", "#mtime", "#ctime"]:
            if s.has_key(key) and not song.has_key(key):
                new_tags[key] = s.get(key)
            elif not s.has_key(key) and song.has_key(key):
                new_tags[key] = s.get(key)
            elif s.get(key) != song.get(key):
                new_tags[key] = s.get(key)
        self.set_property(song, new_tags)
    
    def create_song(self, tags, type, read_from_file=False):
        self._dirty = True
        try: uri = tags["uri"]
        except KeyError: raise MissingUriTag()
        tags["uri"] = vfs.realuri(tags["uri"])
        return self.__create_song(tags, type, read_from_file=read_from_file)

    def __create_song(self, tags, type, hidden=False, read_from_file=False):
        s = Song(tags)
        s.set_type(type)
        if read_from_file: s.read_from_file()
        if hidden:
            s.sanitarize()
            self.__db_operation_lock.acquire()
            self.__songs[tags["uri"]] = s
            self.__db_operation_lock.release()
            self.__hiddens.add(s)
        else:
            self.add([s])
        return s

    def get_or_create_song(self, tags, type, hidden=False, read_from_file=False):
        self._dirty = True
        try: uri = vfs.realuri(tags["uri"])
        except KeyError: raise MissingUriTag()
        try: 
            s = self.__songs[uri]
            if s.get_type() != type: raise KeyError
        except KeyError: 
            s = self.__create_song(tags, type, hidden, read_from_file)
        else:
            if read_from_file: s.read_from_file()
            if s in self.__hiddens and not hidden : 
                self.__hiddens.remove(s)
                self.__db_operation_lock.acquire()
                del self.__songs[uri]
                self.__db_operation_lock.release()
                self.add([s])
            self.set_property(s, tags)
        return s
    
    def has_uri(self, uri):
        uri = vfs.realuri(uri)
        return self.__songs.has_key(uri)
    
    def full_erase(self, type):
        for song in list(self.__songs_by_type[type]):
            self.remove(song)

    def get_all_uris(self):
        return self.__songs.keys()

    def get_song(self, uri):
        uri = vfs.realuri(uri)
        try:
            return self.__songs[uri]
        except KeyError:
            # FIXME: check if make this in get_or_create_song is better
            # Need check call of get_or_create_song in all class
            if not uri: return None
            if uri.startswith("file://"):
                return self.get_or_create_song({"uri":uri}, "unknown_local", True, read_from_file=True)
            else:
                return self.get_or_create_song({"uri":uri}, "unknown", True, read_from_file=True)

    def get_songs(self, type):
        try: return set(self.__songs_by_type[type])
        except KeyError: 
            self.logwarn("get_songs:type %s unknown return empty set()",type)
            return set()

    def get_songs_from_uris(self, uris):
        return [ self.__songs[vfs.realuri(uri)] for uri in uris ]

    def create_playlist(self, type, name, items=[]):
        if items and isinstance(items[0],Song): 
            items = [ song.get("uri") for song in items ]
        pl = Playlist(type, name, items)
        self.add_playlist(type,pl)
        return pl

    def create_autoplaylist(self, type, stype, name, infos=None):
        #TODO: add db_wrapper support
        pl = AutoPlaylist(stype, name, infos)
        self.add_playlist(type, pl)
        return pl

    def get_playlists(self, type):
        return self.__playlists[type]

    def add_playlist(self, type, pl):
        if not self.__db_wrapper.has_key(type) or self.__db_wrapper_load[type]:
            self.__add_playlist_cb(type,pl)
        else:
            dw = self.__db_wrapper[type]
            try: dw.add_playlist(type, pl, self.__add_playlist_cb)
            except NotImplementedError(): return None
    
    def __add_playlist_cb(self, type, pl):
        self.__playlists[type].add(pl)
        self.__condition.acquire()
        self.__queued_signal["playlist-added"].setdefault(type, [])
        self.__queued_signal["playlist-added"][type].append(pl)
        self._dirty = True
        self.__condition.release()
            
    def del_playlist(self, type, pl):
        if not self.__db_wrapper.has_key(type) or self.__db_wrapper_load[type]:
            self.__del_playlist_cb(type,pl)
        else:
            dw = self.__db_wrapper[type]
            try: dw.del_playlist(type,pl, self.__del_playlist_cb)
            except NotImplementedError(): return None


    def __del_playlist_cb(self, type, pl):
        self.__playlists[type].discard(pl)
        self.__condition.acquire()
        self.__queued_signal["playlist-removed"].setdefault(type, [])
        self.__queued_signal["playlist-removed"][type].append(pl)
        self._dirty = True
        self.__condition.release()

    def get_random_song(self, type):
        # FIXME: need move this in source.local
        songs = list(self.__songs_by_type["local"])
        shuffle(songs)
        if songs:
            return songs[0]
        else:
            return None

class ListenDBQuery(gobject.GObject,Logger):
    __gsignals__ = {
        "full-update" : (gobject.SIGNAL_RUN_LAST,
                gobject.TYPE_NONE,
                ()),
        "update-tag" : (gobject.SIGNAL_RUN_LAST,
                gobject.TYPE_NONE,
                (str, gobject.TYPE_PYOBJECT)),
        "added"  : (gobject.SIGNAL_RUN_LAST,
                gobject.TYPE_NONE,
                (gobject.TYPE_PYOBJECT,)),
        "removed"  : (gobject.SIGNAL_RUN_LAST,
                gobject.TYPE_NONE,
                (gobject.TYPE_PYOBJECT,)),
        "quick-update" : (gobject.SIGNAL_RUN_LAST,
                gobject.TYPE_NONE,
                (gobject.TYPE_PYOBJECT,))
    }

    def __init__(self, type):
        gobject.GObject.__init__(self)
        self.__tree = ({}, set())
        self._type = type
        self._query_func = None
        self._query_string = ""
        self.__cache_func_song_tuple = None
        self.__cache_song_tuple = {}

        ListenDB.connect("added", self._db_entry_added)
        ListenDB.connect("removed", self._db_entry_removed)
        ListenDB.connect("changed", self._db_entry_changed)
        ListenDB.connect("quick-changed", self._db_entry_changed, True)
        
        self.__condition = Condition()
        self.__condition_query = Condition()
        self.__query_id = 0
        self.__reset_signal_queue()
        gobject.timeout_add(SIGNAL_DB_QUERY_FIRED, self.__fire_queued_signal)

    def get_type(self):
        return self._type
    
    def __reset_signal_queue(self):
        """
            self.__condition need to be acquired before call this function
        """
        self.__signal_to_fire={
                    "added-songs":set(),
                    "removed-songs":set(),
                    "update-genre":set(),
                    "update-artist":set(),
                    "update-album":set(),
                    "quick-update-songs":set()
                    }

    def __fire_queued_signal(self):
        self.__condition.acquire()

        try:
            if self.__signal_to_fire["update-genre"]: 
                self.logdebug("update-tag:genre %s",len(self.__signal_to_fire["update-genre"]))
                self.emit("update-tag", "genre", self.__signal_to_fire["update-genre"])
            if self.__signal_to_fire["update-artist"]: 
                self.logdebug("update-tag:artist %s",len(self.__signal_to_fire["update-artist"]))
                self.emit("update-tag", "artist", self.__signal_to_fire["update-artist"])
            if self.__signal_to_fire["update-album"]: 
                self.logdebug("update-tag:album %s",len(self.__signal_to_fire["update-album"]))
                self.emit("update-tag", "album", self.__signal_to_fire["update-album"])
            if self.__signal_to_fire["removed-songs"]: 
                self.logdebug("removed-songs: %s",len(self.__signal_to_fire["removed-songs"]))
                self.emit("removed", self.__signal_to_fire["removed-songs"])
            if self.__signal_to_fire["added-songs"]: 
                self.logdebug("added-songs: %s",len(self.__signal_to_fire["added-songs"]))
                self.emit("added", self.__signal_to_fire["added-songs"])
            if self.__signal_to_fire["quick-update-songs"]: 
                #self.logdebug("quick-update: %s",len(self.__signal_to_fire["quick-update-songs"]))
                self.emit("quick-update", self.__signal_to_fire["quick-update-songs"])
        except: 
            self.logexception("Failed fire queued signal")

        self.__reset_signal_queue()
        self.__condition.release()
        return True

    def set_cache_func_song_tuple(self, func):
        self.__cache_func_song_tuple = func

    def get_song_tuple_from_cache(self, song):
        return self.__cache_song_tuple[song.get("uri")]
    
    def _get_all_songs(self):
        return ListenDB.get_songs(self._type)

    def _set_song_tuple_cache(self, song):
        uri = song.get("uri")
        if not self.__cache_song_tuple.has_key(uri):
            self.__cache_song_tuple.update({uri:self.__cache_func_song_tuple(song)})

    def _filter(self, song, query_func = None):
        if query_func is None:
            query_func = self._query_func
        if query_func is None:
            return not song.get("hidden")
        else:
            return not song.get("hidden") and query_func(song)

#    @utils.profiling
    @utils.threaded
    def set_query(self, string = ""):

        self.__condition_query.acquire()
        self.__query_id += 1
        myid = self.__query_id
        self.__condition_query.release()

        if not isinstance(string,unicode): string = string.decode("utf8")
        string = string.strip()


        deb = time.time()
        self.loginfo("Begin query %s", string)
        
        if string:
            try: 
                query_func = Query(string).search
            except Query.error :
                self.loginfo("Query error %s, %d song cached (%f)", string, len(self.__tree[1]), (time.time()-deb) )
                return
            self.empty()
            [ self.__add_cache(song) for song in self._get_all_songs() if self._filter(song,query_func) ]
        else:
            query_func = None
            self.empty()
            [ self.__add_cache(song) for song in self._get_all_songs() if not song.get("hidden") ]
        
        self.loginfo("End query %s, %d song cached (%f)", string, len(self.__tree[1]), (time.time()-deb) )

        #Ensure signal emitted in main loop
        self.__condition_query.acquire()
        if myid == self.__query_id:
            self._query_func = query_func
            self._query_string = string 
            def fire(): self.emit("full-update")
            gobject.idle_add(fire)
        self.__condition_query.release()


    def empty(self):
        self.__tree = ({}, set())

    def _db_entry_added(self, db, type, songs):
        if type != self._type: return

        for song in songs:
            if self._filter(song):
                self.__add_cache(song)

                genre, artist, album = self.__get_info(song)

                self.__condition.acquire()
                self.__signal_to_fire["update-genre"].add(genre)
                self.__signal_to_fire["update-artist"].add(artist)
                self.__signal_to_fire["update-album"].add(album)
                self.__signal_to_fire["added-songs"].add(song)
                if self.__cache_func_song_tuple:
                    self.__cache_song_tuple[song.get("uri")]= self.__cache_func_song_tuple(song)
                self.__condition.release()

    def _db_entry_removed(self, db, type, songs):
        if type != self._type: return 
        for song in songs:
            # FIXME: normally not matched song is not in the tree cache 
            # But a bug somewhere, do that arrive, provisoirie add check in the root tree
            if self._filter(song) and song in self.__tree[1]:
                self.__delete_cache(song)

                genre, artist, album = self.__get_info(song)

                self.__condition.acquire()
                self.__signal_to_fire["update-genre"].add(genre)
                self.__signal_to_fire["update-artist"].add(artist)
                self.__signal_to_fire["update-album"].add(album)
                self.__signal_to_fire["removed-songs"].add(song)
                self.__condition.release()

                if self.__cache_func_song_tuple:
                    del self.__cache_song_tuple[song.get("uri")]

    def _db_entry_changed(self, db, type, infos, use_quick_update_change=False):
        """ use_quick_update_change must be used only if song model don't need to be resorted 
        (useful when podcast progress need update)
        """
        if type != self._type: return 

        for info in infos:
            song, old_keys_values = info
            if old_keys_values.has_key("genre") or \
                    old_keys_values.has_key("artist") or \
                    old_keys_values.has_key("album") or \
                    old_keys_values.has_key("hidden"):
                # force full change il one of this tag have been changed
                use_quick_update_change = False
            
            # Check if it really necessary to reorder the view
            if not use_quick_update_change:
                use_quick_update_change = True
                for tag, old_value in old_keys_values.iteritems():
                    new_value = song.get_sortable(tag)
                    #self.loginfo("check tag%s oldvalue: %s newvalue: %s",tag, old_value, new_value)
                    if old_value != new_value:
                        use_quick_update_change = False
                        break

            if not old_keys_values.has_key("genre"): genre = song.get_sortable("genre")
            else: genre = old_keys_values.get("genre")
            if not old_keys_values.has_key("artist"): artist = song.get_sortable("artist")
            else: artist = old_keys_values.get("artist")
            if not old_keys_values.has_key("album"): album = song.get_sortable("album")
            else: album = old_keys_values.get("album")

            #self.loginfo("use quick update: %s",use_quick_update_change )
            if song in self.__tree[1]:
                self.__condition.acquire()
                if old_keys_values.has_key("genre"):
                    self.__signal_to_fire["update-genre"].add(old_keys_values["genre"])
                if old_keys_values.has_key("artist"):
                    self.__signal_to_fire["update-artist"].add(old_keys_values["artist"])
                if old_keys_values.has_key("album"):
                    self.__signal_to_fire["update-album"].add(old_keys_values["album"])
                if not use_quick_update_change:
                    self.__signal_to_fire["removed-songs"].add(song)
                self.__condition.release()

            try: self.__delete_cache(song,(genre, artist, album))
            except: pass 

            if self.__cache_func_song_tuple:
                try: del self.__cache_song_tuple[song.get("uri")]
                except KeyError: pass 

            if self._filter(song):
                self.__add_cache(song)
                if self.__cache_func_song_tuple:
                    self.__cache_song_tuple[song.get("uri")]= self.__cache_func_song_tuple(song)

                genre, artist, album = self.__get_info(song)

                self.__condition.acquire()

                if old_keys_values.has_key("genre"):
                    self.__signal_to_fire["update-genre"].add(genre)
                if old_keys_values.has_key("artist"):
                    self.__signal_to_fire["update-artist"].add(artist)
                if old_keys_values.has_key("album"):
                    self.__signal_to_fire["update-album"].add(album)

                if use_quick_update_change:
                    self.__signal_to_fire["quick-update-songs"].add(song)
                else:
                    self.__signal_to_fire["added-songs"].add(song)
                self.__condition.release()

    def __add_cache(self, song):
        if self.__cache_func_song_tuple:
            self._set_song_tuple_cache(song)

        genre2, artist2, album = self.__get_info(song)
        sgenre, sartist, salbum = self.__get_str_info(song)

        self.__tree[1].add(song)
        for genre in [genre2, "###ALL###"]:
            for artist in [artist2, "###ALL###" ]:
                    self.__tree[0].setdefault(genre,({}, set(), sgenre))
                    self.__tree[0][genre][1].add(song)
                    self.__tree[0][genre][0].setdefault(artist,({}, set(), sartist))
                    self.__tree[0][genre][0][artist][1].add(song)
                    self.__tree[0][genre][0][artist][0].setdefault(album,({}, set(), salbum))
                    self.__tree[0][genre][0][artist][0][album][1].add(song)

    def __delete_cache(self, song, old_values=False):
        if old_values: genre2, artist2, album = old_values
        else: genre2, artist2, album = self.__get_info(song)
        try: self.__tree[1].remove(song)
        except (KeyError, ValueError):  return 
        for genre in [genre2, "###ALL###"]:
            self.__tree[0][genre][1].remove(song)
            for artist in [artist2, "###ALL###" ]:
                self.__tree[0][genre][0][artist][1].remove(song)
                self.__tree[0][genre][0][artist][0][album][1].remove(song)

                if len(self.__tree[0][genre][0][artist][0][album][1]) == 0:
                    del self.__tree[0][genre][0][artist][0][album]
                    if len(self.__tree[0][genre][0][artist][1]) == 0:
                        del self.__tree[0][genre][0][artist]
            if len(self.__tree[0][genre][1]) == 0:
                del self.__tree[0][genre]

    def __get_str_info(self, song):
        return song.get_str("genre"), song.get_str("artist"),  song.get_str("album")

    def __get_info(self, song):
        return song.get_sortable("genre"), song.get_sortable("artist"),  song.get_sortable("album")
       
    def get_random_song(self):
        songs = list(self.__tree[1])
        shuffle(songs)
        return songs[0]

    def get_songs(self, genres=[], artists=[], albums=[]):
        songs = set()
        if not genres and not artists and not albums: 
            songs.update(self.__tree[1].copy())
        else:
            #if not genres: tmp_genres = self.__tree[0].keys()
            if not genres: tmp_genres = ["###ALL###"]
            else: tmp_genres = genres
            for genre in tmp_genres:
                if not artists:
                    if not albums: 
                        try: songs.update(self.__tree[0][genre][1].copy())
                        except KeyError: continue
                        continue
                    else:
                        tmp_artists = ["###ALL###"]
                        #try: tmp_artists = self.__tree[0][genre][0].keys()
                        #except KeyError: continue
                else: tmp_artists = artists 
                for artist in tmp_artists:
                    if not albums:
                        try: songs.update(self.__tree[0][genre][0][artist][1].copy())
                        except KeyError: continue
                        continue
                    for album in albums:
                        try: 
                            songs.update(self.__tree[0][genre][0][artist][0][album][1].copy())
                        except KeyError: continue
        return songs

    def get_info(self, key_request, genres=[], artists=[], key_values=None, extended_info=False):
        if key_request not in ["genre", "artist", "album"]: raise NotImplementedError("Only genre, artist, album are support for key_request")
        infos = {}
        if key_request == "genre":
            try:
                if key_values is None: mydict = self.__tree[0].copy()
                else: 
                    mydict = {}
                    for value in key_values:
                        try: mydict[value] = self.__tree[0][value]
                        except KeyError: pass

                for key, info in mydict.iteritems():
                    if key == "###ALL###": continue
                    if extended_info:
                        title, playcount, duration, songs = infos.get(key,(info[2], 0, 0, []))
                        playcount += sum(s.get("#playcount", 0) for s in info[1])
                        duration += sum(s.get("#duration", 0) for s in info[1])
                        songs.extend(info[1])
                        infos[key] = (info[2], playcount, duration, songs)
                    else:
                        nb = len(info[1])
                        if nb > 0: infos[key] = (info[2], infos.setdefault(key,(info[2], 0))[1] + nb)
            except KeyError: pass
        else:
            tmp_genres = set(genres) & set(self.__tree[0].keys())
            if not tmp_genres: tmp_genres = ["###ALL###"] #self.__tree[0].keys()
            for genre in tmp_genres:
                if key_request == "artist":
                    try:
                        if key_values is None: mydict = self.__tree[0][genre][0].copy()
                        else:
                            mydict = {}
                            for value in key_values:
                                try: mydict[value] = self.__tree[0][genre][0][value]
                                except KeyError: pass

                        for key, info in mydict.iteritems():
                            if key == "###ALL###": continue
                            if extended_info:
                                title, playcount, duration, songs = infos.get(key,(info[2], 0, 0, []))
                                playcount += sum(s.get("#playcount", 0) for s in info[1])
                                duration += sum(s.get("#duration", 0) for s in info[1])
                                songs.extend(info[1])
                                infos[key] = (info[2], playcount, duration, songs)
                            else:
                                nb = len(info[1])
                                if nb > 0: infos[key] = (info[2], infos.setdefault(key,(info[2], 0))[1] + nb)
                    except KeyError: pass
                else:
                    try: tmp_artists = set(artists) & set(self.__tree[0][genre][0].keys())
                    except KeyError: continue
                    if not tmp_artists: tmp_artists = ["###ALL###"]
                    for artist in tmp_artists:
                        if key_request == "album":
                            try:
                                if key_values is None: mydict = self.__tree[0][genre][0][artist][0].copy()
                                else:
                                    mydict = {}
                                    for value in key_values:
                                        try: mydict[value] = self.__tree[0][genre][0][artist][0][value]
                                        except KeyError: pass

                                for key, info in mydict.iteritems():
                                    if key == "###ALL###": continue
                                    if extended_info:
                                        title, playcount, duration, songs = infos.get(key,(info[2], 0, 0, []))
                                        playcount += sum(s.get("#playcount", 0) for s in info[1])
                                        duration += sum(s.get("#duration", 0) for s in info[1])
                                        songs.extend(info[1])
                                        infos[key] = (info[2], playcount, duration, songs)
                                    else:
                                        nb = len(info[1])
                                        if nb > 0: infos[key] = (info[2], infos.setdefault(key,(info[2], 0))[1] + nb)
                            except KeyError: pass

        if key_values:
            [ infos.update({key:None}) for key in key_values if not infos.has_key(key) ]

        return infos

class AutoPlaylist(ListenDBQuery):
    def __init__(self, type, name, obj=None):
        ListenDBQuery.__init__(self, type)

        self.name = name
        self._smart_filter = None

        self.rules = {
                "string":"&()",
                "has_limit":0,
                "limit":0,
                "limit_type":"title",
                "order_by":"title",
                "inverse_order_by":False
                }
        if obj and isinstance(obj,dict): 
            if not obj.has_key("string"):
                """
                limit_type:
                0 = "title"
                1 = "mo"
                2 = "go"
                3 = "duration"
                """
                raise Exception("Unsupported AutoPlaylist rules update")
            self.rules = obj

        self.__cached_songs = set()
        if ListenDB.isloaded():
            self.prepare_rules()
            #self.reload()
        else:
            ListenDB.connect("loaded", self.__on_db_loaded)

    def set_name(self,name):
        self.name = name 

    def get_name(self):
        return self.name

    def __on_db_loaded(self, db):
        self.reload()
 
    def reload(self):
        self.prepare_rules()
        self.set_query(self._query_string)

    def get_pickle_obj(self):
        return (True, self.name, self.rules)

    def _db_entry_added(self, db, type, songs):
        if type != self._type: return

        songs = [ s for s in songs if self._smart_filter(s) ]
        self.__cached_songs = self.__cached_songs.union(songs)

        super(AutoPlaylist, self)._db_entry_added(db, type, songs)

    def _db_entry_changed(self, db, type, infos, use_quick_update_change=False):
        if type != self._type: return

        songs = [ s for s,__ in infos if self._smart_filter(s) ]
        del_songs = [ s for s in songs if not self._smart_filter(s) ]
        self.__cached_songs = self.__cached_songs.union(songs)
        self.__cached_songs = self.__cached_songs.difference(del_songs)

        super(AutoPlaylist, self)._db_entry_changed(db, type, infos, use_quick_update_change=use_quick_update_change)

    def _db_entry_removed(self, db, type, songs):
        if type != self._type: return 
        self.__cached_songs = self.__cached_songs.difference(songs)
        super(AutoPlaylist, self)._db_entry_removed(db, type, songs)


    def prepare_rules(self):
        ListenDB.set_dirty()

        try: filter = Query(self.rules["string"]).search
        except Query.error: 
            self.logexception("Automated playlist query failed: %s",self.rules["string"])
            return 

        self._smart_filter = filter

        songs = ListenDB.get_songs(self._type)
        tmp = [(s.get(self.rules["order_by"]), s.sort_key, s.get("#size"), s.get("#duration"), s) for s in songs if not self._smart_filter or self._smart_filter(s)]
        tmp.sort()

        if self.rules["inverse_order_by"]:
            tmp.reverse()
            
        songs = []
        self.logdebug("smart playlist %s rules: %s", self.name, self.rules)
        self.logdebug("smart playlist %s without limit done (len: %d)", self.name, len(tmp))
        if self.rules["has_limit"]:
            if self.rules["limit_type"] == "title":
                tmp = tmp[:self.rules["limit"]]
                songs = [s for sort_field, sort_key, size, duration, s in tmp]
                
            elif self.rules["limit_type"] == "duration":
                total_duration = 0
                for order_field, sort_key, size, duration, s in tmp:
                    total_duration += duration   
                    print total_duration 
                    if total_duration < self.rules["limit"]*60*1000:
                        songs.append(s)
                    else:
                        break;
                        
            elif self.rules["limit_type"] in [ "mo", "go" ]:
                total_size = 0
                unit = {"mo":1024*1024, "go":1024*1024*1024 }[self.rules["limit_type"]]
                for order_field, sort_key, size, duration, s in tmp:
                    total_size += size
                    if total_size < self.rules["limit"]*unit:
                        songs.append(s) 
                    else:
                        break;  
            else:
                self.logerror("smart playlist %s have limit type '%s' unsupported ", self.name, self.rules["limit_type"] )
        else:
            songs = [ s for sort_field, sort_key, size, duration, s in tmp]
        
        self.__cached_songs = set(songs)
        self.loginfo("smart playlist %s prepartion done (len: %d)",self.name, len(self.__cached_songs))

    def _get_all_songs(self):
        return self.__cached_songs

    def is_auto(self):
        return True

class Playlist(gobject.GObject,Logger):
    __gsignals__ = {
        "added"  : (gobject.SIGNAL_RUN_LAST,
                gobject.TYPE_NONE,
                (gobject.TYPE_PYOBJECT,)),
        "removed"  : (gobject.SIGNAL_RUN_LAST,
                gobject.TYPE_NONE,
                (gobject.TYPE_PYOBJECT,)),
        "update"   : (gobject.SIGNAL_RUN_LAST,
                gobject.TYPE_NONE,
                ())
    }
    __plid = 0
    def __init__(self, type, name, uris=[]):
        gobject.GObject.__init__(self)
        self.name = name
        self.__uris = uris
        self.__type = type

        self.__cache_song_tuple = {}
        self.__cache_func_song_tuple = None

        self.__plid += 1
        self.id = self.__plid

        #FIXME: Disconnect this when paylist are deleted
        ListenDB.connect("removed", self.__on_db_remove)

    def get_type(self):
        return self.__type

    def get_pickle_obj(self):
        return (False, self.name, self.__uris)
    
    def update(self):
        dw = ListenDB.get_wrapper(self.__type)
        if dw:
            try:
                dw.playlist_update(self,self.__update_cb)
                return
            except NotImplementedError():
                pass
        self.__update_cb(self.__uris)

    def __update_cb(self,uris):
        if self.__uris != uris:
            self.__uris = uris
            for uri in self.__uris:
                self.__cache_song_tuple[uri]= self.__cache_func_song_tuple(ListenDB.get_song(uri))
        ListenDB.set_dirty()
        self.emit("update")

    def set_cache_func_song_tuple(self, func):
        self.__cache_func_song_tuple = func
        for uri in self.__uris:
            self.__cache_song_tuple[uri]= self.__cache_func_song_tuple(ListenDB.get_song(uri))

    def get_song_tuple_from_cache(self, song):
        return self.__cache_song_tuple[song.get("uri")]

    def append(self, song):
        self.extend([song])

    def remove_positions(self, positions):
        positions.sort(reverse=True)
        for pos in positions:
            uri = self.__uris[pos]
            del self.__uris[pos]
            if  self.__cache_func_song_tuple and uri not in self.__uris:
                del self.__cache_song_tuple[uri]
        self.update()

    def remove_position(self, pos):
        uri = self.__uris[pos]
        del self.__uris[pos]
        if  self.__cache_func_song_tuple and uri not in self.__uris:
            del self.__cache_song_tuple[uri]
        self.update()

    def __on_db_remove(self, db, type, songs):
        [ self.remove(s) for s in songs if s.get("uri") in self.__uris ]
            
    def remove(self, song):
        while True:
            try: self.__uris.remove(song.get("uri"))
            except: break
        if self.__cache_func_song_tuple:
            try:del self.__cache_song_tuple[song.get("uri")]
            except KeyError: pass
        self.update()

    def extend(self, songs):
        for song in songs:
            if self.__cache_func_song_tuple:
                self.__cache_song_tuple[song.get("uri")]= self.__cache_func_song_tuple(song)
        self.__uris.extend([ song.get("uri") for song in songs])
        self.update()
    
    def extend_insert(self, songs, pos):
        for song in songs:
            if self.__cache_func_song_tuple:
                self.__cache_song_tuple[song.get("uri")]= self.__cache_func_song_tuple(song)
            self.__uris.insert(pos, song.get("uri"))
            pos+=1
        self.update()

    def reorder(self, uris):
        self.__uris = uris
        # Just for verification
        # self.emit("update")
        ListenDB.set_dirty()

    def insert(self, song, pos ):
        if self.__cache_func_song_tuple:
            self.__cache_song_tuple[song.get("uri")]= self.__cache_func_song_tuple(song)
        self.__uris.insert(pos, song.get("uri"))
        self.update()

    def get_songs(self):
        return ListenDB.get_songs_from_uris(self.__uris)

    def is_auto(self):
        return False

    def get_name(self):
        return self.name

    def set_name(self,name):
        dw = ListenDB.get_wrapper(self.__type)
        try:
            if dw.set_playlist_name(self,name):
                self.name = name
                return True
        except:
            self.name = name
            return True
        return False

ListenDB = ListenDatabase()
#Mandatory type
ListenDB.register_type("local", ["lyrics", "wikipedia", "lastfminfo", "audioscrobbler", "current"])
ListenDB.register_type("unknown_local", ["lyrics", "wikipedia", "lastfminfo", "audioscrobbler", "current"])
ListenDB.register_type("webradio", ["nocover","gstreamer_tag_refresh"])
ListenDB.register_type("volatile_webradio", ["nocover","gstreamer_tag_refresh"])
ListenDB.register_type("unknown", [])
ListenDB.register_type("podcast", ["lyrics", "wikipedia", "lastfminfo", "current"])
ListenDB.register_type("lastfmradio", ["lyrics", "wikipedia", "lastfminfo"])

ListenDB.register_playlist_type("local")

