# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###
# This Code use module spydaap written Erik Hetzner (https://edge.launchpad.net/spydaap)
# my daap webserver code are very inspired from file skydaap.py of it code
###


import os
import sys
import re

import gobject
import gtk
import gst

try:
    import avahi, dbus
    import dbus.glib
except ImportError: dbus_imported = False
else: dbus_imported=True

from BaseHTTPServer import BaseHTTPRequestHandler, HTTPServer

from config import config

from xdg_support import get_xdg_cache_dir
from plugins.generic import GenericPlugin
from player import Player
from widget.preference import HelperConfigureDialog
from helper import Dispatcher
from logger import Logger

from library import ListenDB
from vfs import get_path_from_uri, makedirs

from hashlib import md5

import plugins.generic.spydaap as spydaap
from plugins.generic.spydaap import cache
from plugins.generic.spydaap.metadata import MetadataCacheItem
from plugins.generic.spydaap.containers import ContainerCacheItem
from plugins.generic.spydaap.daap import do

DAAP_PORT = 3689

itunes_re = '(?:[^:]*://[0-9]{1,3}\\.[0-9]{1,3}\\.[0-9]{1,3}\\.[0-9]{1,3}:[0-9]+)?'
drop_q = '(?:\\?.*)?'
urls = (
    '/',
    'server_info', #
    '/server-info',
    'server_info', #
    '/content-codes',
    'content_codes', #
    '/databases',
    'database_list', #
    '/databases/([0-9]+)/items',
    'item_list', #
    '/databases/([0-9]+)/items/([0-9]+)\\.([0-9a-z]+)',
    'item', #
    '/databases/([0-9]+)/containers',
    'container_list', #
    itunes_re + '/databases/([0-9]+)/containers/([0-9]+)/items',
    'container_item_list', #
    '/login',
    'login', #
    '/logout',
    'logout', #
    '/update',
    'update', #
    )


class ContentRangeFile(object):
    def __init__(self, parent, start, end=None, chunk=1024):
        self.parent = parent
        self.start = start
        self.end = end
        self.chunk = chunk
        self.parent.seek(self.start)
        self.read = start

    def next(self):
        to_read = self.chunk
        if (self.end != None):
            if (self.read >= self.end):
                self.parent.close()
                raise StopIteration
            if (to_read + self.read > self.end):
                to_read = self.end - self.read
            retval = self.parent.read(to_read)
            self.read = self.read + len(retval)
        else: retval = self.parent.read(to_read)
        if retval == '':
            self.parent.close()
            raise StopIteration
        else: return retval

    def __iter__(self):
        return self

class ContainerCache(cache.OrderedCache):
    def get_item_by_pid(self, pid, n=None):
        return ContainerCacheItem(self, pid, None)

    def build(self, md_cache):
        def build_do(md, id):
            d = do('dmap.listingitem',
                   [ do('dmap.itemkind', 2),
                     do('dmap.itemid', md.id),
                     do('dmap.itemname', md.get_name()),
                     do('dmap.containeritemid', id)
                     ] )
            return d
        pid_list = []
        for pl in ListenDB.get_playlists("local"):
            entries = [ md_cache.get_item_by_pid( md5( get_path_from_uri(song.get("uri")) ).hexdigest() ) 
                            for song in pl.get_songs()]

            #FIXME: For now i remove not cached song inteads of metadata
            entries = [ entrie for entrie in entries if entrie.id ]
            d = do('daap.playlistsongs',
                   [ do('dmap.status', 200),
                     do('dmap.updatetype', 0),
                     do('dmap.specifiedtotalcount', len(entries)),
                     do('dmap.returnedcount', len(entries)),
                     do('dmap.listing',
                        [ build_do (md,id) for (id, md) in enumerate(entries) ])
                     ])
            ContainerCacheItem.write_entry(self.dir, str(pl.get_name()), d, len(entries))
            pid_list.append(md5(pl.name).hexdigest())
        self.build_index(pid_list)


class MetadataCache(cache.OrderedCache):
    def get_item_by_pid(self, pid, n=None):
        return MetadataCacheItem(self, pid, n)

    def build(self, marked={}):
        songs = ListenDB.get_songs("local")
        for song in songs:
            ffn = get_path_from_uri(song.get("uri"))
            digest = md5(ffn).hexdigest()
            marked[digest] = True
            md = self.get_item_by_pid(digest)
            if (not(md.get_exists()) or \
                    (md.get_mtime() < os.stat(ffn).st_mtime)):
                m = self.parse(song)
                if m != None:
                    MetadataCacheItem.write_entry(self.dir,str( song.get_str("title")), ffn, m)
        for item in os.listdir(self.dir):
            if (len(item) == 32) and not(marked.has_key(item)):
                os.remove(os.path.join (self.dir, item))
        self.build_index()

    def parse(self,song):
        tag_map = {
        #'TIT1': 'daap.songgrouping',
        'title': 'dmap.itemname',
        #'': 'daap.songcomposer', 
        'genre': 'daap.songgenre',
        'artist': 'daap.songartist',
        'album': 'daap.songalbum',
        '#track': 'daap.songtracknumber',
        #'TBPM': 'daap.songbeatsperminute',
        #'#date': 'daap.songyear',
        '#size': 'daap.songsize',
        '#bitrate': 'daap.songbitrate',
        '#added': 'daap.songdateadded',
        '#duration': 'daap.songtime'
        }
        d = []
        for key, mkey in tag_map.iteritems():
            tag = song.get(key)
            if not tag: continue
            if key[0] == "#":
                d.append(do(mkey, int(str(tag))))
            else:
                tag = song.get(key)
                d.append(do(mkey, str(tag) ))
        d.extend([
                do("daap.songformat", "mp3"),
                do('daap.songdescription', 'MPEG Audio File'),
                ])
        return d


class DaapRequestHandler(BaseHTTPRequestHandler,Logger):
    session_id = 1
    server_revision = 1
    def do_GET(self):
        for i in range(0, len(urls), 2 ):
            res = re.match("^"+itunes_re+urls[i]+drop_q+"$",self.path)
            if res is not None:
                args = res.groups()
                self.loginfo("'%s' match '%s' call: %s%s",self.path, urls[i], urls[i+1],args)
                if hasattr(self,urls[i+1]):
                    getattr(self, urls[i+1])(*args)
                else:
                    self.send_error(404,"Not implemented yet")
                break
            else:
                pass
                #self.loginfo("'%s' not match '%s'",self.path, urls[i])
        else:
            self.send_error(404,"Not found")

    def login(self):
        mlog = do('dmap.loginresponse',
                  [ do('dmap.status', 200),
                    do('dmap.sessionid', self.session_id) ])
        return self.send(mlog.encode())

    def logout(self):
        self.send_error(204,"No Content")

    def server_info(self):
        name = "Listen Daap Service"
        msrv = do('dmap.serverinforesponse',
                  [ do('dmap.status', 200),
                    do('dmap.protocolversion', '2.0'),
                    do('daap.protocolversion', '3.0'),
                    do('dmap.itemname', name),
                    do('dmap.authenticationmethod', 0),
                    do('dmap.loginrequired', 0),
                    do('dmap.timeoutinterval', 1800),
                    do('dmap.supportsautologout', 0),
                    do('dmap.supportsupdate', 0),
                    do('dmap.supportspersistentids', 0),
                    do('dmap.supportsextensions', 0),
                    do('dmap.supportsbrowse', 0),
                    do('dmap.supportsquery', 0),
                    do('dmap.supportsindex', 0),
                    do('dmap.supportsresolve', 0),
                    do('dmap.databasescount', 1),                
                   ])
        return self.send(msrv.encode())

    def content_codes(self):
        children = [ do('dmap.status', 200) ]
        for code in spydaap.daap.dmapCodeTypes.keys():
            (name, dtype) = spydaap.daap.dmapCodeTypes[code]
            d = do('dmap.dictionary',
                   [ do('dmap.contentcodesnumber', code),
                     do('dmap.contentcodesname', name),
                     do('dmap.contentcodestype',
                        spydaap.daap.dmapReverseDataTypes[dtype])
                     ])
            children.append(d)
        mccr = do('dmap.contentcodesresponse',
                  children)
        self.send(mccr.encode())

    def update(self):
        mupd = do('dmap.updateresponse',
                  [ do('dmap.status', 200),
                    do('dmap.serverrevision', self.server_revision),
                    ])
        self.send(mupd.encode())

    def database_list(self):
        nb_items = 1
        nb_containers = 1
        d = do('daap.serverdatabases',
               [ do('dmap.status', 200),
                 do('dmap.updatetype', 0),
                 do('dmap.specifiedtotalcount', 1),
                 do('dmap.returnedcount', 1),
                 do('dmap.listing',
                    [ do('dmap.listingitem',
                         [ do('dmap.itemid', 1),
                           do('dmap.persistentid', 1),
                           do('dmap.itemname', "Musique de Listen GO!"),
                           do('dmap.itemcount', nb_items),
                           do('dmap.containercount', nb_containers)])
                      ])
                 ])
        self.send(d.encode())

    def item_list(self,database_id):
        
        def build_item(md):
            return do('dmap.listingitem',
                      [ do('dmap.itemkind', 2),
                        do('dmap.containeritemid', md.id),
                        do('dmap.itemid', md.id),
                        md.get_dmap_raw()
                        ])
        def build(f):
            children = [ build_item (md) for md in self.server.md_cache ]
            file_count = len(children)
            d = do('daap.databasesongs',
                   [ do('dmap.status', 200),
                     do('dmap.updatetype', 0),
                     do('dmap.specifiedtotalcount', file_count),
                     do('dmap.returnedcount', file_count),
                     do('dmap.listing',
                        children) ])
            f.write(d.encode())
        self.send(self.server.cache.get('item_list', build))

    def container_list(self,database):
        container_do = []
        for i, c in enumerate(self.server.container_cache):
            d = [ do('dmap.itemid', i + 1 ),
                  do('dmap.itemcount', len(c)),
                  do('dmap.containeritemid', i + 1),
                  do('dmap.itemname', c.get_name()) ]
            if c.get_name() == 'Library': # this should be better
                d.append(do('daap.baseplaylist', 1))
            else:
                d.append(do('com.apple.itunes.smart-playlist', 1))
            container_do.append(do('dmap.listingitem', d))
        d = do('daap.databaseplaylists',
               [ do('dmap.status', 200),
                 do('dmap.updatetype', 0),
                 do('dmap.specifiedtotalcount', len(container_do)),
                 do('dmap.returnedcount', len(container_do)),
                 do('dmap.listing',
                    container_do)
                 ])
        self.send(d.encode())

    def item(self,database,item,format):
        fn = self.server.md_cache.get_item_by_id(item).get_original_filename()
        self.loginfo("%s",self.headers)
        if self.headers.has_key('HTTP_RANGE'):
            rs = self.headers["HTTP_RANGE"]
            m = re.compile('bytes=([0-9]+)-([0-9]+)?').match(rs)
            (start, end) = m.groups()
            if end != None: end = int(end)
            else: end = os.stat(fn).st_size
            start = int(start)
            self.send(f, 'audio/*', (206,"Partial Content"), 
                      { "Content-Range":
                        "bytes " + str(start) + "-"
                       + str(end) + "/" + str(os.stat(fn).st_size)
                       })
        else: 
            f = open(fn)
            self.send(f, 'audio/*')

    def container_item_list(self, database_id, container_id):
        container = self.server.container_cache.get_item_by_id(container_id)
        self.send(container.get_daap_raw())


    def send(self,data,type='application/x-dmap-tagged', responses = (200,), headers = {}):
        try:
            self.send_response(*responses)
            self.send_header('Content-Type', type)
            self.send_header('DAAP-Server', 'Listen')
            #self.send_header('Expires', '-1')
            #self.send_header('Cache-Control', 'no-cache')
            #self.send_header('Accept-Ranges', 'bytes')
            #self.send_header('Content-Language', 'en_us')
            for header, value in headers.iteritems():
                self.send_header(header, value)
            if hasattr(data,"read"):
                data = data.read()
            if (hasattr(data, 'next')):
                try:
                    self.send_header("Content-Length", str(os.stat(data.name).st_size))
                except: pass
            else:
                try:
                    self.send_header("Content-Length", str(len(data)))
                except: pass
                #sys.stdout.write(data)
            self.end_headers()
            #self.logdebug("DATA to send %s",data)
            self.wfile.write(data)
            self.wfile.flush()
            self.loginfo("Finish to send data to client")
        except:
            self.logexception("Something wrong when send data to client")

class DaapServer(HTTPServer):
    def __init__(self,server_address):
        HTTPServer.__init__(self,server_address, DaapRequestHandler)

    def build_cache(self):
        cache_dir = get_xdg_cache_dir("daapcache")
        self.cache = spydaap.cache.Cache(cache_dir)
        self.md_cache = MetadataCache(os.path.join(cache_dir, "media"))
        self.container_cache = ContainerCache(os.path.join(cache_dir, "containers"))
        self.md_cache.build()
        self.container_cache.build(self.md_cache)


class DaapdAvahi(Logger):
    domain = ""
    host = ""
    serviceName = "Listen Daap Service"
    serviceType = "_daap._tcp" # See http://www.dns-sd.org/ServiceTypes.html
    servicePort = DAAP_PORT
    serviceTXT = [ "Password = false" ]
    
    rename_count = 12 # Counter so we only rename after collisions a sensible number of times
    def __init__(self, port):
        self.server = None
        self.servicePort = port
        self.group = None
        if dbus_imported:
            try:self.bus = dbus.SystemBus()
            except:self.bus=None
        if dbus_imported and self.bus:
            try:
                self.server = dbus.Interface(self.bus.get_object(avahi.DBUS_NAME, avahi.DBUS_PATH_SERVER), avahi.DBUS_INTERFACE_SERVER)
            except:
                print "No avahi support"
        else:
            print "No avahi support"
        if self.server:
            self.server.connect_to_signal( "StateChanged", self.server_state_changed )
            self.server_state_changed(self.server.GetState())

    def add_service(self):
        if self.group is None:
            self.group = dbus.Interface( 
                    self.bus.get_object( avahi.DBUS_NAME, self.server.EntryGroupNew()), 
                    avahi.DBUS_INTERFACE_ENTRY_GROUP)
            self.group.connect_to_signal('StateChanged', self.entry_group_state_changed)
        self.loginfo("Adding service '%s' of type '%s' ...", self.serviceName, self.serviceType)
        self.group.AddService(
                avahi.IF_UNSPEC,    #interface
                avahi.PROTO_UNSPEC, #protocol
                dbus.UInt32(0),                  #flags
                self.serviceName, self.serviceType,
                self.domain, self.host,
                dbus.UInt16(self.servicePort),
                avahi.string_array_to_txt_array(self.serviceTXT))
        self.group.Commit()

    def remove_service(self):
        if not self.group is None:
            self.group.Reset()

    def server_state_changed(self, state):
        if state == avahi.SERVER_COLLISION:
            self.logwarn("Server name collision")
            self.remove_service()
        elif state == avahi.SERVER_RUNNING:
            self.add_service()

    def entry_group_state_changed(self, state, error):
        if state == avahi.ENTRY_GROUP_ESTABLISHED:
            self.loginfo("Service established.")
        elif state == avahi.ENTRY_GROUP_COLLISION:
            self.rename_count = self.rename_count - 1
            if self.rename_count > 0:
                name = self.server.GetAlternativeServiceName(self.serviceName)
                self.logwarn("Service name collision, changing name to '%s' ...", name)
                self.remove_service()
                self.add_service()
            else:
                self.logerror("No suitable service name found after 12 retries, exiting.")
        elif state == avahi.ENTRY_GROUP_FAILURE:
            self.logwarn("Error in group state changed %s", error)


class DaapServerDialog(HelperConfigureDialog):
    def __init__(self,parent):
        HelperConfigureDialog.__init__(self,parent, _("DaapServer Server"))
        self.add(self.make_lentry(_("Address"),"plugins","icecast_ip","127.0.0.1"))
        self.add(self.make_lentry(_("Port"),"plugins","icecast_port","8000"))
        self.add(self.make_lentry(_("Username"),"plugins","icecast_username","source"))
        self.add(self.make_lentry(_("Password"),"plugins","icecast_password","hackme"))
        self.add(self.make_lentry(_("Stream Name"),"plugins","icecast_name","Listen DaapServer Agent"))
        self.add(self.make_lentry(_("Description"),"plugins","icecast_description","Listen DaapServer Agent"))
        self.add(self.make_lentry(_("Url"),"plugins","icecast_url","http://localhost:8000/"))
        self.add(self.make_lentry(_("Mount point"),"plugins","icecast_mount","/listen.ogg"))
        self.show_all()


class PortUsedException(Exception):
    pass

class DaapServerPlugin(GenericPlugin):
    PLUGIN_NAME="DaapServer Support"
    PLUGIN_DESC="DaapServer Support plugin"
    PLUGIN_VERSION = "0.1"
    PLUGIN_AUTHOR = "Mehdi ABAAKOUK <theli48@gmail.com>"
    PLUGIN_WEBSITE = ""

    def __init__(self):
        super(DaapServerPlugin,self).__init__()

        self.autoconnect(config, "config-changed",self.__on_config_change)
        self.daapd = None
    

        self.port = DAAP_PORT
        while DAAP_PORT + 100 > self.port :
            try: 
                self.daapd = DaapServer(("",self.port))
                break
            except :
                self.port += 1
        if DAAP_PORT + 100 <= self.port:
            raise PortUsedException("Port %d to %s already used"%(DAAP_PORT, self.port))

        if ListenDB.isloaded():
            self.__on_db_loaded(ListenDB)
        else:
            ListenDB.connect("loaded",self.__on_db_loaded)

    def __on_db_loaded(self,db):
        self.daapd.build_cache()
        self.daapdavahi = DaapdAvahi(self.port)
        self.__id_io = gobject.io_add_watch(self.daapd.fileno(), gobject.IO_IN , self.io_callback)

    def io_callback(self, fd, condition):
        self.daapd.handle_request()
        return True

    def __on_config_change(self,dispacher,section,option,value):
        if section == "plugins" and option.find("daapserver") == 0:
            pass

    def destroy(self):
        if self.daapd is not None:
            if self.__id_io:
                gobject.source_remove(self.__id_io)
                self.__id_io
            self.daapdavahi.group.Free()
            del self.daapd
            del self.daapdavahi
        super(DaapServerPlugin,self).destroy()

    @staticmethod   
    def on_configure(parent):
        pass
        #DaapServerDialog(parent)
