# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###

import os
import traceback
import gst

import gobject
import gtk
from time import time
from datetime import datetime

from const import VERSION

MUSICBRAINZ = False
try: 
    from musicbrainz2.webservice import WebService, Query, WebServiceError, TrackFilter
except: print "No musicbrainz support (musicbrainz2 missing)"
else:
    try: 
        from tunepimp import tunepimp
        dir(tunepimp.tunepimp).index("setMusicDNSClientId")
    except: print "No musicbrainz support (tunepimp missing or version < 0.5) "
    else: MUSICBRAINZ = True

import utils
import vfs
from logger import Logger

"""
Mutagen Tag support
"""




from mutagen import File as MutagenFile
from mutagen.asf import ASF
from mutagen.apev2 import APEv2File
from mutagen.flac import FLAC
from mutagen.id3 import ID3FileType
from mutagen.mp3 import MP3
from mutagen.oggflac import OggFLAC
from mutagen.oggspeex import OggSpeex
from mutagen.oggtheora import OggTheora
from mutagen.oggvorbis import OggVorbis
from mutagen.trueaudio import TrueAudio
from mutagen.wavpack import WavPack
try: from mutagen.mp4 import MP4
except: from mutagen.m4a import M4A as MP4
from mutagen.musepack import Musepack
from mutagen.monkeysaudio import MonkeysAudio
from mutagen.optimfrog import OptimFROG
from easymp3 import EasyMP3

FORMATS = [EasyMP3, TrueAudio, OggTheora, OggSpeex, OggVorbis, OggFLAC,
            FLAC, APEv2File, MP4, ID3FileType, WavPack, Musepack,
            MonkeysAudio, OptimFROG, ASF]

TAG_KEYS = {
    "title": "title",
    "artist": "artist",
    "album": "album",
    "tracknumber": "#track",
    "discnumber": "#disc",
    "genre": "genre",
    "date": "date"
}    

TAGS_KEYS_OVERRIDE = {}
TAGS_KEYS_OVERRIDE['Musepack'] = {"tracknumber":"track","date":"year"}
TAGS_KEYS_OVERRIDE['MP4'] = {
        "title":"\xa9nam",
        "artist":"\xa9ART",
        "album":"\xa9alb",
        "tracknumber":"trkn",
        "discnumber":"disk",
        "genre":"\xa9gen",
        "date":"\xa9day"
        }
TAGS_KEYS_OVERRIDE['ASF'] = {
        "title":"Title",
        "artist":"Author",
        "album":"WM/AlbumArtist",
        "tracknumber":"WM/TrackNumber",
        "discnumber":"WM/PartOfSet",
        "genre":"WM/Genre",
        "date":"WM/Year"
        }


def file_is_supported(filename):
    try:
        fileobj = file(filename, "rb")
    except:
        return False
    try:
        header = fileobj.read(128)
        results = [Kind.score(filename, fileobj, header) for Kind in FORMATS]
    finally:
        fileobj.close()
    results = zip(results, FORMATS)
    results.sort()
    score, Kind = results[-1]
    if score > 0: return True
    else: return False


"""
Some note about tag:

For podcast:
    album contains the podcast feed name
    hidden used to marque a deleted track in a podcast feed
    
    
For iradio:
    title contains the radio name
    album contains the web site of the radio
    artist contains the now_playing title

listen tag list: (need to be more clean)
"""
USED_KEYS="""
==song_type==
uri station info_supp
#track title artist album genre #duration #progress podcast_local_uri #disc
#playcount #skipcount #lastplayed #added #date date year
description descriptionrss podcast_feed_url hidden
#mtime #ctime #rate 
album_cover_url station_track_url
#progress radio_person
#bitrate
#size
#stream_offset 
""".split()

KEY_TYPES= {
    "float":"#date".split(),
    "long":"#duration".split()
    }

class Song(dict, Logger):
    def __init__(self, otherdict = None):
        dict.__init__(self)
        if otherdict:
            for key in USED_KEYS:
                default = key[0] == 0 and 0 or None
                self[key] = otherdict.get(key,None)

    def get_dict(self):
        odict = {}
        for key, value in self.iteritems():
            if value is not None:
                odict[key] = value
        return odict

    def get_type(self):
        if self.has_key("==song_type=="):
            return self["==song_type=="]
        else:
            return "unknown"
    
    def set_type(self,type):
        self["==song_type=="] = type

    def get_filter(self):
        return " ".join([
                self.get_str("artist").lower(),
                self.get_str("album").lower(),
                self.get_str("title").lower(),
                        ])

    def match(self,filter):
        if filter == None: return True
        search = self.get_filter()
        return len( [ s for s in filter if search.find(s) != -1 ] ) == len(filter)

    def get_str(self,key,xml=False):
        """
        get_str(tag,xml=False)
        get a formated version of the tag "key"
        """
        
        if key == "uri":
            value = vfs.fsdecode(vfs.unescape_string_for_display(self.get("uri")))
        elif key == "title":
            value = utils.format_tag(self.get("title"))
            if not value:
                value = vfs.fsdecode(vfs.unescape_string_for_display(self.get("uri")))
                value = vfs.get_name(value)
        elif key in ["album","genre","artist"]:
            value = utils.format_tag(self.get(key))
        elif key == "#bitrate":
            value = self.get("#bitrate")
            if value: 
                value = "%dk"%value
        elif key == "#duration":
            value = utils.duration_to_string(self.get(key))
        elif key == "#lastplayed":
            value = self.get(key)
            if value :
                value = datetime.fromtimestamp(int(value)).strftime("%x %X")[:-3]
            else:
                value = _("Never")
        elif key == "#playcount":
            value = self.get(key) or _("Never")
        elif key == "#date" or key == "#added":
            value = self.get(key)
            if value:
                value = datetime.fromtimestamp(int(value)).strftime("%x %X")[:-3]
        elif key == "#rate":
            rate = self.get("rate")
            if rate in [0,1,2,3,4,5,"0","1","2","3","4","5"]:
                value = "rate-"+str(rate)
            else:
                value = "rate-0"
        elif key == "date":
            try:
                value =  self.get("date", "")[:4]
            except (ValueError, TypeError, KeyError): 
                pass
            if not value:
                value = self.get("#date")
                if value:
                    value = datetime.fromtimestamp(int(value)).strftime("%Y")
                else:
                    value = ""
        else:
            value = None

        if not value: value = self.get(key,"")
        if isinstance(value,int) or isinstance(value,float): value = "%d"%value
        try: value = unicode(value)
        except UnicodeDecodeError: 
            print "EncError:",key,"===", value
            value = ""
        if xml: value = utils.xmlescape(value)
        return value
            
    def get_sortable(self,key):

        if key == "uri":
            value = self.get_str("uri").lower()
        elif key in ["album","genre","artist","title"]:
            value = utils.format_tag(self.get_str(key)).lower()
        elif key == "radio_person":
            value = self.get_str("radio_person")
            if value : value = int(value.split("/")[0])
            else: value = None
        elif key == "date": 
            value = self.get("#date")
            # Ensure are None and not 0
            if not value: value = None
        else:
            value = self.get(key)

        if not value and key[0] == "#" : value = 0
        elif not value: value = u'\xff\xff\xff\xff'
        if isinstance(value,str): value = unicode(value)
        return value

    def sanitarize(self):
        if "#added" not in self:
            self["#added"] = time()

    def __setitem__(self, key, value):
        if key == "#track":
            if value is not None and not isinstance(value,int) and value.rfind("/")!=-1 and value.strip()!="":
                value = value.strip()

                disc_nr = value[value.rfind("/"):]
                try: 
                    disc_nr = int(disc_nr)
                except: 
                    disc_nr = self.get("#disc")
                self["#disc"] = disc_nr
                value = value[:value.rfind("/")]
        elif key == "date":
            try: 
                self["#date"] = utils.strdate_to_time(value)
            except: 
                value = None

        if key[0] == "#":
            try:
                if key in KEY_TYPES["float"]:
                    value = float(value)
                elif key in KEY_TYPES["long"]:
                    value = long(value)
                else:
                    value = int(value)
            except:
                # WARNING: if value are wrong, it was suppressed! 
                value = None

        if value is None:
            if key in self:
                dict.__delitem__(self, key)
        else:
            dict.__setitem__(self, key, value)

    def __sort_key(self):   
        return(
                self.get_sortable("album"),
                self.get_sortable("#disc"),
                self.get_sortable("#track"),
                self.get_sortable("artist"),
                self.get_sortable("title"),
                self.get_sortable("date"),
                self.get_sortable("#bitrate"),
                self.get_sortable("uri")
                )    
    sort_key = property(__sort_key)
    
    def __browser_sort_key(self):
        return(
                self.get_sortable("album"),
                self.get_sortable("#track"),
                self.get_sortable("artist"),
                self.get_sortable("genre"),
                self.get_sortable("title"),
                self.get_sortable("uri")
                )    
#    browser_sort_key = property(__browser_sort_key)
    
    def __call__(self, key):
        return self.get(key)

    def __hash__(self):
        return hash(self.get("uri"))

    def __repr__(self):
        return "<Song %s>"%self.get("uri")

    def __cmp__(self, other):
        if not other: return -1 
        try: return cmp(self.sort_key, other.sort_key)
        except AttributeError: return -1

    def __eq__(self,other):
        try:return self.get("uri")==other.get("uri")
        except:return False

    def exists(self):
        return vfs.exists(self.get("uri"))

    def str_dump(self):
        res = "******************************************************************************\n"
        res += "Song: "+self.get_str("uri")+"\n"
        tags = [ (key,value) for key,value in self.iteritems()]
        tags.sort()
        for key,value in tags:
            if not key.startswith("~"):
                res += "-"+key+":"+str(value)+"\n"
        res += "******************************************************************************\n"
        return res

    def dump(self):
        """
        dump()

        dump fulle metadata on the file (not only tag in the file)
        """
        print self.str_dump()


    """""""""""""""""""""
        FILE READER
    """""""""""""""""""""
    def read_from_file(self):
        """
        Load metadata from the file
        """
        if self.get_scheme()=="file" and not self.exists():
            ret = False
        if self.get_scheme()=="file" and file_is_supported(self.get_path()):
            ret = self.__read_from_local_file()
        elif self.get_scheme() in ["lastfm"]:
            ret = True
        else:
            ret = self.__read_from_remote_file()
        return ret
          
    def __read_from_local_file(self):
        try: 
            
            path = self.get_path()
            self["#size"] = os.path.getsize(path)
            self["#mtime"] = os.path.getmtime(path)
            self["#ctime"] = os.path.getctime(path)

            audio = MutagenFile(self.get_path(), FORMATS)
            tag_keys_override = None

            if audio is not None:
                tag_keys_override = TAGS_KEYS_OVERRIDE.get(audio.__class__.__name__, None)
                for file_tag,tag in TAG_KEYS.iteritems():
                    
                    if tag_keys_override and tag_keys_override.has_key(file_tag):
                        file_tag = tag_keys_override[file_tag]
                        
                    if audio.has_key(file_tag) and audio[file_tag]:
                        value = audio[file_tag]
            
                        if isinstance(value,list) or isinstance(value,tuple):
                            value = value[0]
                        if isinstance(value,tuple):
                            value = value[0]
                            
                        self[tag] = unicode(value)
            
                self["#duration"] = int(audio.info.length)*1000
                try: self["#bitrate"] = int(audio.info.bitrate)
                except AttributeError: pass
            else:
                raise "W:Song:MutagenTag:No audio found"
                    
        #FIXME: Add a real traceback support
        except Exception, e:
            print "W: Error while loading ("+self.get("uri")+")\nTracback :",e
            self.last_error = _("Error while reading")+": "+self.get_filename()
            return False
        else:
            return True
        
    def __read_from_remote_file(self):
        GST_IDS = {
            "title": "title",
            "genre": "genre",
            "artist": "artist",
            "album": "album",
            "bitrate": "#bitrate",
            'track-number':"#track"
        }
        
        is_finalize = False
        is_tagged = False
        def unknown_type(*param):
            raise "W:Song:GstTag:Gst decoder: type inconnu"
        
        def finalize(pipeline):
            state_ret = pipeline.set_state(gst.STATE_NULL)
            if state_ret != gst.STATE_CHANGE_SUCCESS:
                print "Failed change to null"
            is_finalize = True
            #print "finalize"
                
            
        def message(bus,message,pipeline):
            if message.type == gst.MESSAGE_EOS:
                finalize(pipeline)
    
            elif message.type == gst.MESSAGE_TAG:
                taglist = message.parse_tag()
                for key in taglist.keys():
                    if GST_IDS.has_key(key): 
                        if key=="bitrate":
                            value = int(taglist[key]/100)
                        elif isinstance(taglist[key],long):
                            value = int(taglist[key])   
                        else:
                            value = taglist[key]
                        self[GST_IDS[key]] = value
                        #print key,":",value
                is_tagged = True
    
            elif message.type == gst.MESSAGE_ERROR:
                err, debug = message.parse_error()
                finalize(pipeline)
                raise "W:Song:GstTag:Decoder error: %s\n%s" % (err,debug)
        try:
            try: pipeline = gst.parse_launch ("gnomevfssrc location="+self.get("uri")+" ! decodebin name=decoder ! fakesink");
            except gobject.GError : 
                raise "W:Song:GstTag:Failed to build pipeline to read metadata of",self.get("uri")
        
            decoder = pipeline.get_by_name("decoder")
            decoder.connect("unknown-type",unknown_type)
            bus = pipeline.get_bus()
            bus.connect('message', message,pipeline)
            bus.add_signal_watch()
            
            state_ret = pipeline.set_state(gst.STATE_PAUSED);
            timeout = 10
            state = None
            while state_ret == gst.STATE_CHANGE_ASYNC and not is_finalize and timeout > 0:
                state_ret,state,pending_state = pipeline.get_state(1 * gst.SECOND);
                timeout -= 1
                
            if state_ret != gst.STATE_CHANGE_SUCCESS:
                finalize(pipeline)
                print "W:Song:GstTag:Failed Read Media"
            else:
                if not is_tagged:
                    bus.poll(gst.MESSAGE_TAG,5 * gst.SECOND)
                try:
                    query = gst.query_new_duration(gst.FORMAT_TIME)
                    if pipeline.query(query):
                        total = query.parse_duration()[1]
                    else: total = 0
                except gst.QueryError: total = 0
                total //= gst.MSECOND
                #print "duration",total
                self["#duration"] = total
                if not is_tagged:
                    print "W:Song:GstTag: Media found but no tag found"
                
                finalize(pipeline)

        except Exception, e:
            print "W: Error while loading ("+self.get("uri")+")\nTracback :",e
            self.last_error = _("Error while reading")+": "+self.get_filename()    
            return False
        else: return True

            
    """""""""""""""""""""
        FILE WRITER
    """""""""""""""""""""


    def write_to_file(self):
        if self.get_scheme()!="file":
            self.last_error = self.get_scheme()+" "+_("Scheme not supported")
            return False
        if not vfs.exists(self.get("uri")):
            self.last_error = self.get_filename()+_(" doesn't exist")
            return False
        if not os.access(self.get_path(),os.W_OK):
            self.last_error = self.get_filename()+_(" doesn't have enough permission")
            return False

        try: 
            audio = MutagenFile(self.get_path(), FORMATS)
            tag_keys_override = None

            if audio is not None:
                if audio.tags is None: 
                    audio.add_tags()
                tag_keys_override = TAGS_KEYS_OVERRIDE.get(audio.__class__.__name__, None)
                
                for file_tag,tag in TAG_KEYS.iteritems():
                    
                    if tag_keys_override and tag_keys_override.has_key(file_tag):
                        file_tag = tag_keys_override[file_tag]
                        
                    #FIXME: FOund when tag is a tuple
                    """
                    orig_tag = audio[file_tag]
                    try: del(audio[file_tag])
                    except KeyError:pass
                    value = unicode(self.get(tag))
                    
                    if isinstance(orig_tag, tuple):
                        audio[file_tag] = (int(value), orig_tag[1])
                    else:
                        audio[file_tag] = value
                    """
                    if self.get(tag):
                        value = unicode(self.get(tag))
                        audio[file_tag] = value
                    else:
                        try:del(audio[file_tag])
                        except KeyError:pass
            
                audio.save()
            else:
                raise "w:Song:MutagenTag:No audio found"
                
        #FIXME: Add a real traceback support
        except Exception, e:
            print traceback.format_exc()
            print "W: Error while writting ("+self.get("uri")+")\nTracback :",e
            self.last_error = _("Error while writting")+": "+self.get_filename()
            return False
        else:
            return True
    
    
    """""""""""""""""""""""""""
        MUSICBRAINZ READER
    """""""""""""""""""""""""""
    def read_from_musicbrainz(self):
        print "I:Song:Musicbrainz:Fetch metadata of %s" % self.get_filename()

        tp = tunepimp.tunepimp("listen", VERSION ,tunepimp.tpThreadAll )
        tp.setMusicDNSClientId('001bd3a391cc704bb97cbdedb33336d5'); 

        tp.addFile( self.get_path() , 0 )

        analyzed = False
        puid = None

        while tp.getNumFiles():
            ret, type, fileId, status = tp.getNotification()
            if not ret:
                continue

            tr = tp.getTrack(fileId)
            tr.lock()
            mdata = tr.getLocalMetadata()
            puid = mdata.filePUID
            if len(puid) == 0 : puid = None
                

            tr.unlock()

            """
            fileStatusEnumDict = {
                tunepimp.eMetadataRead      : "eMetadataRead",
                tunepimp.ePending           : "ePending",
                tunepimp.eUnrecognized      : "eUnrecognized",
                tunepimp.eRecognized        : "eRecognized",
                tunepimp.ePUIDLookup         : "ePUIDLookup",
                tunepimp.ePUIDCollision      : "ePUIDCollision",
                tunepimp.eFileLookup        : "eFileLookup",
                tunepimp.eUserSelection     : "eUserSelection",
                tunepimp.eVerified          : "eVerified",
                tunepimp.eSaved             : "eSaved",
                tunepimp.eDeleted           : "eDeleted",
                tunepimp.eError             : "eError",
            }
            print "Status: ",fileStatusEnumDict[status]
            """

            if status == tunepimp.ePUIDLookup:
                tr.lock()
                puid = tr.getPUID()
                tr.unlock()

            elif status in [tunepimp.eUnrecognized, tunepimp.eRecognized, tunepimp.eSaved]:
                puid = mdata.filePUID
                if not puid or len(puid) == 0 :
                    puid = tr.getPUID()
                if not puid:
                    if not analyzed:
                        tr.lock()
                        tr.setStatus(tunepimp.ePending)
                        tr.unlock()
                        tp.wake(tr)
                        analyzed = True
                    else:
                        print "I:Song:MusicBrainz:Failed get puid of %s" % self.get_filename()
                        tp.releaseTrack(tr)
                        tp.remove(fileId)
                        tr = None

            if tr:
                tp.releaseTrack(tr)
            
            if status == tunepimp.eError:
                print "E:Song:Musicbrainz: %s - %s" % ( self.get_filename() , tp.getError())
                tp.remove(fileId)
                return False


            if puid:
                try:
                    q = Query(WebService())
                    flt = TrackFilter(puid=puid)
                    result = q.getTracks(flt)
                except WebServiceError:
                    result = []

                # Use puid check ???
                if len(result) > 0:
                    result = [(result[i].getScore(), result[i]) for i in range(0,len(result)) ]
                    result.sort()
                    track = result[0][1].getTrack()
                    self["title"] = track.getTitle()
                    self["artist"] = track.getArtist().getName()
                
                    # Can do better by asking user the better album
                    releases = track.getReleases()
                    if len(releases) > 0:
                        self["album"] = releases[0].getTitle()
                        self["date"] = releases[0].getEarliestReleaseDate()

                    print "Found on musicbrainz"
                    print "I:Song:Musicbrainz:Ok %s" % self.get_filename()
                else:
                    print "I:Song:Musicbrainz:No files match for %s" % self.get_filename()
                
                tp.remove(fileId)
                return True


    """
    Some filename manipulation
    """
    def get_path(self):
        try: return vfs.get_path_from_uri(self.get("uri"))
        except : return ""

    def get_scheme(self):
        return vfs.get_scheme(self.get("uri"))

    def get_ext(self,complete=True):
        return vfs.get_ext(self.get("uri"),complete)

    def get_filename(self):
        value = self.get("uri") 
        try:
            return vfs.fsdecode(vfs.get_name(value))
        except:
            """
            FIXME: 
                i don't meet this bug again
                i my memory, i have edit some tag with muzikbrainz and without
                And 3 song have been created without uri filed!!
                And all function that use uri failed
                
                here return value for now and print tag information for debugging
            """
            return value

    """
    The lyrics filename 
    Use same as quodlibet format
    """
    lyric_uri = property(lambda self: "file://"+vfs.fsencode(
        os.path.join(os.path.expanduser("~/.lyrics"),
                     self.get_str("artist").replace('/', '')[:128],
                     self.get_str("title").replace('/', '')[:128] + '.lyric')))


