# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###


import gtk
import gobject

import stock
from config import config
from const import VERSION, DEFAULT_AUTHOR

from helper import Dispatcher, SignalContainer
from plugins import Plugin
from widget.misc import ScrolledWindow
from widget.song_view import PlaylistSongView
from widget.browser import AutoPlaylistBrowser

from library import ListenDB

from logger import Logger

class Source(Plugin):
    
    # Desciption
    PLUGIN_NAME = "No Name"
    PLUGIN_DESC = None
    PLUGIN_VERSION = VERSION
    PLUGIN_AUTHOR = DEFAULT_AUTHOR
    PLUGIN_WEBSITE = None

    load_priority = 100 # Obsolete
    display_index = 10000
    categorie = "library"
        
    items = []
        
    def get_items(self):
        return self.items
    
    def save(self):
        pass

    def run(self):
        pass
    
    def get_default_menu(self):
        return []

    def destroy(self):
        for item in self.items:
            item.destroy()

class SourceItem(Logger):
    has_top_separateur = False
    has_bottom_separateur = False
    is_droppage = False
    source_id = None
    editable = False
    selected = False
    default_selected = False
    
    """ Minimun requirement """
    stock = stock.SRC_UNKNOWN
    label = _("Unknown source")
    widget = None
    widget_klass = gtk.VBox
    config_code = None

    def __init__(self):
        if not self.widget: self.widget = self.widget_klass()

    def on_edition(self,new_text):
        return False
    
    def on_activated(self):
        pass
    
    def on_drag_data_get(self,context, selection, info):
        return False

    def on_drag_song_received(self,songs):
        return False
    
    def on_key_press(self,event):
        return False
    
    def get_menu(self):
        pass
    
    def destroy(self):
        if self.widget: 
            if isinstance(self.widget, SignalContainer):
                self.widget.autodisconnect_all()
            self.widget.destroy()

class PlaylistSourceItem(SourceItem):
    is_droppage = False
    editable = False
    config_code = "locallibrary"

    _playlist_songview = PlaylistSongView
    _autoplaylist_songview = AutoPlaylistBrowser

    def __init__(self,pl):
        self.pl = pl
        if pl.is_auto():
            self.widget = self._autoplaylist_songview("playlist",pl)
            self.stock = stock.SRC_PLAYLIST_SMART
        else:
            self.widget = ScrolledWindow(self._playlist_songview("playlist",pl))
            def save_config():
                self.widget.child.save_config()
            self.widget.save_config = save_config
            self.stock = stock.SRC_PLAYLIST

    label = property(lambda self: self.pl.get_name())

    def on_drag_data_get(self,context, selection, info):
        if not self.editable: return False

        songs = list(self.pl.get_songs())
        if self.pl.is_auto(): songs.sort()
        list_uri = list([ song.get("uri") for song in songs])
        selection.set("text/listen-songs", 8, "\n".join(list_uri))
        selection.set_uris(list_uri)
        return True
        
    def on_drag_song_received(self,songs):
        if not self.editable: return True
        self.pl.extend(songs)
        return False
    
    def on_edition(self,new_text):
        if not self.editable: return False
        if self.pl.set_name(new_text):
            self.label = new_text
            ListenDB.set_dirty()
            return True
        return False
    
    def on_key_press(self,event):
        if not self.editable: return False
        if event.keyval == gtk.gdk.keyval_from_name("Delete"):
            self.remove_playlist(None)
            return True
        elif event.keyval == gtk.gdk.keyval_from_name("<Control>r"):
            self.edit_playlist(None)
            return True
        else:
            return False
    
    def on_activated(self):
        self.play_playlist()
        
    def play_playlist(self,w=None):
        songs = list(self.pl.get_songs())
        if self.pl.is_auto(): songs.sort()
        if len(songs)>0:
            Dispatcher.cur_playlist_play(songs)
               
    def enqueue_playlist(self,w):
        songs = list(self.pl.get_songs())
        if self.pl.is_auto(): songs.sort()
        if len(songs)>0:
            Dispatcher.cur_playlist_enqueue(songs)
            
    def remove_playlist(self,w):
        ListenDB.del_playlist(self.pl.get_type(),self.pl)
        
    def edit_playlist(self,w):
        Dispatcher.edit_source()
        
    def __export_playlist(self,w):
        from widget.dialog import WindowExportPlaylist
        WindowExportPlaylist(self.pl.get_songs()).run()

    def edit_smart_playlist(self,w):
        from widget.smart_playlist import SmartPlaylistWindow
        SmartPlaylistWindow(self.pl).run()
        
    def get_menu(self):
        menu = gtk.Menu()
        item = gtk.ImageMenuItem(stock.PLAY)
        item.connect("activate",self.play_playlist)
        menu.append(item)
        item = gtk.ImageMenuItem(stock.ENQUEUE)
        item.connect("activate",self.enqueue_playlist)
        menu.append(item)
        menu.append(gtk.SeparatorMenuItem())

        if self.pl.is_auto():
            item = gtk.ImageMenuItem(stock.EDIT)
            item.connect("activate",self.edit_smart_playlist)
            menu.append(item)
            menu.append(gtk.SeparatorMenuItem())    
        
        if self.editable:
            item = gtk.ImageMenuItem(stock.RENAME)
            item.connect("activate",self.edit_playlist)
            menu.append(item)
            item = gtk.ImageMenuItem(stock.DELETE)
            item.connect("activate",self.remove_playlist)
            menu.append(item)

        item = gtk.ImageMenuItem(stock.EXPORT)
        item.connect("activate",self.__export_playlist)
        menu.append(item)
       
        menu.show_all()
        return menu

class PlaylistSource(Source):
    PLUGIN_NAME = "Playlist support"
    PLUGIN_DESC = _("Add Playlist Support")

    categorie = "playlist"

    load_priority = 0
    display_index = 60
    items = []
    
    editable = False

    #MUST FILL THIS
    _source_item_klass = None #PlaylistSourceItem
    _type = ""

    def __init__(self):
        Source.__init__(self)
        if ListenDB.isloaded():
            self.__on_db_loaded(ListenDB)
        else:
            ListenDB.connect("loaded",self.__on_db_loaded)

    def __on_db_loaded(self,db):
        items = [ (pl.name, self._source_item_klass(pl)) for pl in ListenDB.get_playlists(self._type) ]
        items.sort()
        self.items = [ item for  __, item in items ]

        ListenDB.connect("playlist-added",self.__add_playlist)
        ListenDB.connect("playlist-removed",self.__remove_playlist)
        Dispatcher.update_source()    
    
    def __add_playlist(self,db,type,pls):
        if type != self._type: return
        items = [ (item.pl.name,item) for item in self.items ]
        for pl in pls:
            items.append((pl.name,self._source_item_klass(pl)))
        items.sort()
        self.items = [ item for  __, item in items ]
        Dispatcher.update_source()
        if self.editable: gobject.idle_add(Dispatcher.edit_playlist,pls[0])


    def __remove_playlist(self,db,type,pls):
        if type != self._type: return
        self.items = [ item for item in self.items if item.pl not in pls ]
        Dispatcher.update_source()    
    


MANDATORY_SOURCE = [ "LocalSource", "LocalPlaylistSource", "AudioCDSource", "PodcastSource", "LastFmRadioSource", "LyricsSource", "WikipediaSource" ]
BLACKLIST_SOURCE = [ "PlaylistSource", "DeviceSource" ]
