# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###

import gtk
import gobject
import pango
from urllib import quote
import threading

import utils
from config import config
import stock

from xdg_support import get_xdg_pixmap_file

from lastfm_service import lastfm_info

from helper import SignalCollector
from library import ListenDB
from player import Player

from source import Source,SourceItem
from widget.misc import ScrolledWindow
from widget.song_menu import SongMenuManager

from logger import Logger
from vfs import urlopen

class LastFmBox(gtk.VBox,Logger):
    def __init__(self):
        gtk.VBox.__init__(self,False,6)

        self.song = None
        SignalCollector.connect("lastfminfo",Player,"new-song",self.new_media_cb)
        
        btn_album = gtk.Button()
        btn_album.set_relief(gtk.RELIEF_NONE)
        btn_album.connect("clicked",self.lastfm_hook,"album")
        l = gtk.Label()
        l.set_markup("<span color=\"blue\"><u>"+_("About album")+"</u></span>")
        btn_album.add(l)
        btn_album.set_tooltip_text(_("See album information at www.last.fm"))
        
        btn_artist = gtk.Button()
        btn_artist.set_relief(gtk.RELIEF_NONE)
        btn_artist.connect("clicked",self.lastfm_hook,"artist")
        l = gtk.Label()
        l.set_markup("<span color=\"blue\"><u>"+_("About artist")+"</u></span>")
        btn_artist.add(l)
        btn_artist.set_tooltip_text(_("See artist information at www.last.fm"))
        
        
        
        logo = gtk.Image()
        logo.set_from_file( get_xdg_pixmap_file("lastfm.png") )
        logo.set_alignment(1,0.5)
        
        label_box = gtk.HBox(False,6)        
        label = gtk.Label()
        label.set_use_markup(True)
        label.set_markup("<span size=\"large\"><b>"+_("Similar artists")+"</b></span>")
        label.set_alignment(0,0.5)
        label.set_size_request(-1,22)        
        label_box.pack_start(label,True,True)
        
        label = gtk.Label()
        label.set_use_markup(True)
        label.set_markup("<span size=\"large\"><b>"+_("Top albums")+"</b></span>")
        label.set_alignment(0,0.5)
        label.set_size_request(-1,22)
        label_box.pack_start(label,True,True)



        self.artists_box = gtk.VBox()
        
        self.albums_box = gtk.VBox()

        info_box = gtk.HBox(True,6)
        info_box.pack_start(self.artists_box)
        info_box.pack_start(self.albums_box)

        self.btn_more_artist = gtk.Button()
        btn_label = gtk.Label()
        btn_label.set_markup("<span color=\"blue\"><u>"+_("More similar artists")+"</u></span>")
        btn_label.set_alignment(0,0.5)
        self.btn_more_artist.add(btn_label)
        self.btn_more_artist.connect("clicked",lambda widget:utils.website("http://www.last.fm/music/%s/+similar"%utils.filter_info_song(self.song.get_str("artist")).encode("utf-8")))
        self.btn_more_artist.set_relief(gtk.RELIEF_NONE)
        
        self.btn_more_album = gtk.Button()
        btn_label = gtk.Label()
        btn_label.set_markup("<span color=\"blue\"><u>"+_("All top albums")+"</u></span>")
        btn_label.set_alignment(0,0.5)
        self.btn_more_album.add(btn_label)
        self.btn_more_album.connect("clicked",lambda widget:utils.website("http://www.last.fm/music/%s/+albums"%utils.filter_info_song(self.song.get_str("artist")).encode("utf-8")))
        self.btn_more_album.set_relief(gtk.RELIEF_NONE)
        
        bottom_box = gtk.HBox()
        bottom_box.pack_start(btn_artist,False,False)
        bottom_box.pack_start(btn_album,False,False)
        bottom_box.pack_end(logo,False,False)

        self.pack_start(label_box,False,False)
        self.pack_start(ScrolledWindow(info_box,True,gtk.SHADOW_NONE),True,True)
        self.pack_end(bottom_box,False,False)

        self.last_artist=None
        
        self.update = False
        self.current_info ={"artist":None,"album":None}

        self.condition={}
        for type in ["artist","album"]:
            self.condition[type] = threading.Condition()
            t = threading.Thread(target=self.thread,args=(type,))
            t.setDaemon(True)
            t.start()

    def lastfm_hook(self,widget,type):
        if self.song:
            def prepare(str):
                return quote(utils.filter_info_song(str).encode("utf-8"))
            url = "http://www.last.fm/music/"
            if type=="artist":
                utils.website(url+prepare(self.song.get_str("artist")))
            if type=="album" and self.song.get("album"):
                utils.website(url+prepare(self.song.get_str("artist"))+"/"+prepare(self.song.get_str("album")))
            
    def new_media_cb(self,widget,song):
        if not song: return

        if not ListenDB.song_has_capability(song,"lastfminfo") or not song.get("artist"):
            self.fail("artist")
            self.fail("album")
            return 

        if config.get("setting","offline") == "true" :
            self.set_message("artist",_("Offline mode"))
            self.set_message("album",_("Offline mode"))
            return 

        for type in ["artist","album"]:
            self.condition[type].acquire()
            self.current_info[type] = None
            self.condition[type].release()
        
        if self.last_artist == song.get_str("artist"): return 

        self.last_artist = song.get_str("artist")
        self.song = song
        self.set_message("album",_("Fetching informations"))
        self.set_message("artist",_("Fetching informations"))
        
        lastfm_info.get_related(self.last_artist,self.fetch_info,"artist",func_fail=self.fail,func_fail_args=("artist",))
        lastfm_info.get_topalbum(self.last_artist,self.fetch_info,"album",func_fail=self.fail,func_fail_args=("album",))
    
    def fail(self,type):
        self.set_message(type,_("No information available"))

    def set_message(self,type,message) :

        l = gtk.Label(message)
        l.set_alignment(0,0)

        if type=="artist":
            self.artists_box.foreach(self.artists_box.remove)
            self.artists_box.pack_start(l,True,True)
            self.artists_box.show_all()
        else:          
            self.albums_box.foreach(self.albums_box.remove)      
            self.albums_box.pack_start(l,True,True)
            self.albums_box.show_all()
        
    def fetch_info(self,items,type):
        self.condition[type].acquire()    
        self.current_info[type] = items
        self.condition[type].notify()
        self.condition[type].release()
   
    def thread(self,type):
        while True:
            
            self.condition[type].acquire()
            while not self.current_info[type] :
                self.condition[type].wait()
            
            items = self.current_info[type] 
            self.current_info[type] = None            
            self.condition[type].release()
            
            NB_RESULT = 16
            list_btn = []
            if type=="artist":
                 for item in items[:NB_RESULT] :
                    if item.get("name") and item.get("url"):
                        list_btn.append(self.make_btn_lastfm(item["name"],item["url"],item.get("match","0"),_("Score")+" : "+item.get("match","0"),item.get("image_small")))
                 if len(items)>NB_RESULT: list_btn.append(self.btn_more_artist)            
            else:
                 initial = None
                 for item in items[:NB_RESULT]:
                    if item.get("name") and item.get("url"):
                        playcount = item.get("playcount","0") or 0
                        playcount = int(playcount)
                        if initial in [None,0]:
                            initial = playcount
                            percent = 100.0
                        else:
                            percent =  playcount * 100 / initial 
                        list_btn.append(self.make_btn_lastfm(item["name"],item["url"],percent,_("Playcount")+" : %s"%playcount,item.get("image_small")))
                 if len(items)>NB_RESULT: list_btn.append(self.btn_more_album)
                 
            if self.current_info[type]==None:
                gobject.idle_add(self.fill_table,list_btn,type)    

        
    def fill_table(self,widgets,type):
        cols = {"artist":0,"album":1}
        col = cols[type]
        if type=="artist":
            self.artists_box.foreach(self.artists_box.remove)
        else:          
            self.albums_box.foreach(self.albums_box.remove)  
        for i,w in enumerate(widgets):
            if type=="artist":
                self.artists_box.pack_start(w,False,False)
            else:          
                self.albums_box.pack_start(w,False,False)
        if type=="artist":
            self.artists_box.show_all()
        else:          
            self.albums_box.show_all()
        
    def make_btn_lastfm(self,name,url,match,match_text,image_url):
        #print name,url,match
        btn = gtk.Button()
        btn.set_relief(gtk.RELIEF_NONE)
        progress = gtk.ProgressBar()
        progress.set_ellipsize(pango.ELLIPSIZE_END)
        frac = float(match)/float(100)
        progress.set_fraction(frac)
        progress.set_text(match_text)

        l = gtk.Label()
        l.set_markup(utils.xmlescape(name))
        l.set_alignment(0,0.5)
        l.set_ellipsize(pango.ELLIPSIZE_END)
        pbox = gtk.VBox(False,0)
        pbox.pack_start(l)
        pbox.pack_start(progress,False,False)

        image = None
        if image_url:
            image = gtk.Image()
            try: sock = urlopen(image_url)
            except :
                self.logexception("Failed to load image %s (not found)",image_url)
            else:
                loader = gtk.gdk.PixbufLoader()
                loader.set_size(40, 40)
                loader.connect("closed", lambda loader,image : image.set_from_pixbuf(loader.get_pixbuf()) ,image)
            
                def thread():
                    def write(data):
                        loader.write(data)
                        try:loader.close()
                        except gobject.GError:
                            self.logwarn("Failed to load image %s (Format incorrect)",image_url)
                    data = sock.read()
                    sock.close()
                    gobject.idle_add(write,data)
                    
                t = threading.Thread(target=thread)
                t.setDaemon(True)
                t.start()

        box = gtk.HBox(False,6)
        if image:
            box.pack_start(image,False,False)
        box.pack_start(pbox,True,True)
        btn.add(box)
        btn.connect("clicked",utils.url_hook,url)
        return btn


class LastFmItem(SourceItem):
    widget_klass = LastFmBox
    config_code = "lastfm"
    label = _("LastFm")
    stock = stock.SRC_LASTFM

class LastFmSource(Source):
    PLUGIN_NAME = "Lastfm Information"
    PLUGIN_DESC = _("Display Lastfm related information")

    categorie = "info"
    display_index = 4
    signal_collector_id = "lastfminfo"

    def __init__(self):
        self.items = [ LastFmItem() ]

        def song_menu_action(song_menu_manager,id_menu,songs):
            if id_menu == "lastfm" and songs:
                artist = songs[0].get("artist")
                from lastfm_player import LastFmPlayer
                uri = LastFmPlayer.make_artist_radio(artist)
                s = {
                    "station" : LastFmPlayer.get_station_name(uri),
                    "uri" : uri
                    }
                Dispatcher.cur_playlist_play([ListenDB.get_or_create_song(s,"lastfmradio",True)])

        self.autoconnect(SongMenuManager,"action",song_menu_action)
        SongMenuManager.register_item("lastfm",55,stock.LASTFM_SIMILAR,sep_before=True)
        for type in [ "local","unknown_local"]:
            SongMenuManager.allow_item_for_type("lastfm",type)

        Source.__init__(self)

    def destroy(self):
        SongMenuManager.unregister_item("lastfm")
        Source.destroy(self)
   
