# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###

import os
import gobject
import gtk
import threading
from time import time

import traceback

import stock
import utils
from config import config
import vfs
from xdg_support import get_xdg_music_dir

from source import Source,SourceItem
from widget.song_view import SortableSongView
from widget.song_menu import SongMenu
from widget.browser import SimpleBrowser
from widget.jobs_manager import Job
from cover_manager import CoverManager
from widget.dialog import WinDir, WinFile
from widget.smart_playlist import SmartPlaylistWindow

from helper import Dispatcher

from library import ListenDB, ListenDBQuery

from song import file_is_supported

START_DIRECTORY_MONITOR=50

class LocalBrowser(SimpleBrowser):
    _type = "local"
    _cover = True
    _config_code = "local"

class LocalLibrarySourceItem(SourceItem):
    default_selected = True

    widget_klass = LocalBrowser
    label = _("Music")
    config_code = "locallibrary"
    stock = stock.SRC_BIBLIO

    def on_import_folder(self,w):
        ImportFolderJob()
        
    def on_import_file(self,w):
        ImportFileJob()
        
    def on_reload_db(self,w):
        ReloadDB()
        
    def __new_playlist(self,bt):
        ListenDB.create_playlist("local","New Playlist")
    
    def __new_auto_playlist(self,bt):
        pl = ListenDB.create_autoplaylist("local","local","New Automatic Playlist")
        SmartPlaylistWindow(pl, True).run()

    def __on_download_cover(self,w):
        CoverDownloadJob()

    def get_menu(self): 
        menu = gtk.Menu()     
            
        item = gtk.ImageMenuItem(stock.IMPORT_FILE)
        item.connect("activate",self.on_import_file)
        menu.append(item)
        item = gtk.ImageMenuItem(stock.IMPORT_FOLDER)
        item.connect("activate",self.on_import_folder)
        menu.append(item)
        menu.append(gtk.SeparatorMenuItem())
        
        item = gtk.ImageMenuItem(stock.RELOAD_LIBRARY)
        item.connect("activate",self.on_reload_db)
        menu.append(item)
        
        item = gtk.MenuItem(_("Download missing cover"))
        item.connect("activate",self.__on_download_cover)
        menu.append(item)
        menu.append(gtk.SeparatorMenuItem())
        item = gtk.ImageMenuItem(stock.SRC_PLAYLIST)
        item.connect("activate",self.__new_playlist)
        menu.append(item)
        item = gtk.ImageMenuItem(stock.SRC_PLAYLIST_SMART)
        item.connect("activate",self.__new_auto_playlist)
        menu.append(item)
        menu.show_all()
        return menu

class LocalSource(Source):
    PLUGIN_NAME = "Library Browser"
    PLUGIN_DESC = _("Allow browse your music library")

    load_priority = 0
    display_index = 50

    def __init__(self):
        self.items = [LocalLibrarySourceItem()]
        if ListenDB.isloaded():
            self.__post_load()
        else:
            ListenDB.connect("loaded",self.__post_load)
        Source.__init__(self)

    def __post_load(self, *args):
        self.logdebug("post load launch")
        if config.getboolean("library","startup_deleted"):
            gobject.timeout_add(START_DIRECTORY_MONITOR, self.__check_db)
        if config.getboolean("library","startup_added"):
            dir = config.get("library","location", get_xdg_music_dir())
            dir = os.path.expanduser(dir)
            library_path = vfs.get_uri_from_path(dir)
            gobject.timeout_add(int(START_DIRECTORY_MONITOR * 1.5), self.__import_folder_real, [library_path])

    def __check_db(self):
        CheckDeletedDBJob()

    def __import_folder_real(self,dirs):
        ImportFolderJob(dirs)

    def save(self):
        self.items[0].widget.save_config()


class CoverDownloadJob(Job):
    __message = _("Download cover...")
    
    def job(self):

        album_songs = {}
        [ album_songs.update({song.get_sortable("album"):song}) for song in ListenDB.get_songs("local") ]

        total = len(album_songs)
        for i,song in enumerate(album_songs.values()):
            CoverManager.get_cover(song, True)
            progress = float(i)/float(total)
            yield _("Download cover ")+" %d/%d"%(i,total),progress,False
        del album_songs

        
    def told_way(self):
        songs = [s for s in album_songs.values() if not CoverManager.has_cover(s) ]
        total = len(songs)
        self.loginfo("%s cover to download",total)
        for i in range(0,total):
            CoverManager.get_cover(songs[i], True)
        

class ImportFolderJob(Job):
    __message = _("Loading...")
    def __init__(self,dirs=None):
        if not dirs:
            dir = WinDir().run()
            if dir: dirs = [ dir ]

        if dirs:
            self.message = _("Reading directories...")
            self.dirs = dirs
            super(ImportFolderJob,self).__init__()
        
    def job(self):
        dirs = self.dirs
        message = self.message
        
        start = time()
        
        added = []
        db_uris = set(ListenDB.get_all_uris())

        alldirs = [ vfs.get_path_from_uri(dir) for dir in dirs ]
        last_estimated = estimated = 0 
        total_dirs = len(alldirs)
        parsed_dirs = 0
        for mdir in alldirs:
            for dirpath, dirs, names in os.walk(mdir):
                [ dirs.remove(dir) for dir in dirs if dir[0] == "." ]
                [ alldirs.append(os.path.realpath(os.path.join(dirpath,dir))) for dir in dirs if os.path.islink(os.path.join(dirpath,dir)) ]

                total_dirs += len(dirs)
                parsed_dirs += 1
                valid_files = set([ os.path.join(dirpath, name) for name in names if name[0] != "." and file_is_supported(os.path.join(dirpath, name)) ])
                for file in valid_files:
                    file = os.path.realpath(file)
                    uri = vfs.get_uri_from_path(file)
                    if file not in db_uris:
                        added.append(uri)
                    elif os.path.getctime(file) > ListenDB.get_song(uri).get("#ctime"):
                        added.append(uri)

                estimated = float(parsed_dirs)/float(total_dirs)
                if max(estimated,last_estimated) == estimated:
                    last_estimated = estimated
                yield message+" (%d of %d) "%(parsed_dirs,total_dirs ), last_estimated,False

        i = 0
        total = len(added)
        finish = []
        added = set(added)
        for uri in added:
            i += 1
            tags = {"uri":uri}
            try: ListenDB.get_or_create_song(tags,"local",read_from_file=True)
            except:
                import traceback
                traceback.print_exc()
            yield _("Reading file")+" %d/%d..."%(i,total),float(i)/float(total),False
        
        end = time()
        self.loginfo("%d songs loaded in %d seconds",total, (end-start))
            
class ImportFileJob(object):
    def __init__(self):
        uri = WinFile().run()
        if uri and file_is_supported(vfs.get_path_from_uri(uri)):
            tags = {"uri":uri}
            try: ListenDB.get_or_create_song(tags,"local",read_from_file=True)
            except:
                traceback.print_exc()

class CheckDeletedDBJob(Job):
    __message = _("Check deleted files")
    def job(self):
        songs = ListenDB.get_songs("local")
        total = len(songs)
        i = 0
        for song in songs:
            i += 1
            if not song.exists():
                ListenDB.remove(song)
            yield "Check deleted files %d/%s"%(i,total),float(i)/float(total),False

class ReloadDBJob(Job):
    __message = _("Reload database")
    def job(self):
        songs = ListenDB.get_songs("local")
        total = len(songs)
        i = 0
        for song in songs:
            i += 1
            if not song.exists():
                ListenDB.remove(song)
            else:
                try: ListenDB.reload_song_from_file(song)
                except:
                    traceback.print_exc()
                    
            yield "Reload database %d/%s"%(i,total),float(i)/float(total),False

def ReloadDB():
    ReloadDBJob()
    dir = config.get("library","location", get_xdg_music_dir())
    dir = os.path.expanduser(dir)
    monitored_folder = vfs.get_uri_from_path(dir)
    if monitored_folder:
        ImportFolderJob([monitored_folder])

