# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###

import os
import gst
import gobject
import gtk
import threading
import stock
from utils import odict
from tempfile import mkstemp
from time import time

from config import config

from source import Source,SourceItem
from widget.song_view import SortableSongView
from widget.song_menu import SongMenu , SongMenuManager
from widget.misc import ScrolledWindow
from widget.preference import HelperConfigureDialog

from library import ListenDB
from helper import Dispatcher
from player import Player
from action_manager import ActionManager
from widget.menubar import MINI_ICON_SIZE
import vfs

#Configuration
mp3_quality = {
    "cbr": (64, 96, 128, 192, 256),
    "abr": (64, 96, 128, 192, 256),
    "vbr": (9, 7, 5, 3, 1),
}
mp3_mode = {
    "cbr" : (0,"bitrate"),
    "abr" : (3,"vbr-mean-bitrate"),
    "vbr" : (4,"vbr-quality")
}
ogg_quality =  (0,0.2,0.4,0.6,0.8)

FORMAT = odict()
FORMAT["mp3"] = "MP3 (lame)"
FORMAT["ogg"] = "Oggvorbis"
FORMAT["flac"] = "Flac"

MODE = odict()
MODE["cbr"] = "Constant bitrate"
MODE["abr"] = "Average bitrate"
MODE["vbr"] = "Variable bitrate"


QUALITY = odict()
QUALITY["0"] = "Very High"
QUALITY["1"] = "High"
QUALITY["2"] = "Normal"
QUALITY["3"] = "Low"
QUALITY["4"] = "Very Low"

ListenDB.save_data_type("record")
ListenDB.register_type("recorder",["lyrics","wikipedia","lastfminfo","audioscrobbler","current"])
   
class RecorderDialog(HelperConfigureDialog):
    def __init__(self,parent):
        HelperConfigureDialog.__init__(self,parent, _("Recorder"))
        self.add(self.make_label(_("Save recorded file in")))
        self.add(self.make_file("plugins","recorder_path",os.path.expanduser("~/Recorder")))
        self.add(self.make_lmenu( _("Format"),"plugins","recorder_format",FORMAT, "ogg"))
        self.add(self.make_lmenu( _("Quality (for mp3 and Ogg only)"),"plugins","recorder_quality",QUALITY, "3"))
        self.add(self.make_lmenu( _("Mode (for mp3 only)"),"plugins","recorder_mode",MODE, "cbr"))
        self.show_all()

class RecorderSource(Source):
    PLUGIN_NAME = "Recorder"
    PLUGIN_DESC = _("Record any thing in listen")

    display_index = 80

    signal_collector_id = "recorder"

    def __init__(self):
        Source.__init__(self)


        self.__cur_recorded_song = None

        self.__differed_stop_record = False
        self.__differed_start_record = False

        self.__recording_state = "STATE_NULL"
        self.__recordbin = None

        for item_menu in [ "play", "enqueue" ,"lastfm","wikipedia_artist","wikipedia_album","lyrics","delete","deletedisk","edit"]:
            SongMenuManager.allow_item_for_type(item_menu,"recorder")

        self.items = [RecorderSourceItem()]

        self.autoconnect(Player.bin,"tee-inserted",self.__on_insert)
        self.autoconnect(Player.bin,"tee-removed",self.__on_remove)
        self.__record_btn = gtk.ToggleButton()
        self.__record_btn.add(gtk.image_new_from_stock(gtk.STOCK_MEDIA_RECORD,gtk.ICON_SIZE_BUTTON))
        #self.__record_btn.set_relief(gtk.RELIEF_NONE)
        self.autoconnect(self.__record_btn,"toggled",self.__btn_clicked)
        ActionManager.register_action("recorder",10,None,self.__record_btn)
        self.autoconnect(Player,"instant-new-song",self.__on_new_song)
        self.autoconnect(Player,"play-end",self.__on_stop)
        self.autoconnect(Player,"stopped",self.__on_stop)
    
    def destroy(self):
        if self.__recordbin: Player.bin.xfade_remove_tee(self.__recordbin)
        ActionManager.unregister_action("recorder")

    @staticmethod
    def on_configure(parent):
        RecorderDialog(parent)

    def __on_stop(self,*args):
        if self.__recordbin:
            self.stop_record()

    def __on_new_song(self,player,song):
        if self.__recordbin:
            self.stop_record()
            if self.__record_btn.get_active():
                self.record()

    def __btn_clicked(self, w):
        if w.get_active():
            self.record()
        else:
            self.stop_record()

    def __on_insert(self,bin,tee,element):
        if element != self.__recordbin: return
        if self.__recording_state == "STATE_READY":
            self.__recording_state = "STATE_RECORD"

        if self.__differed_stop_record:
            self.__differed_stop_record = False
            self.stop_record()

    def __on_remove(self,bin,tee,element):
        if element != self.__recordbin: return
        if self.__recording_state == "STATE_DIRTY":
            tags = self.__cur_recorded_song.get_dict()
            tags["uri"] = self.__recordbin.get_by_name("filesink").get_property("location")
            tags["#date"] = time()
            new_song = ListenDB.get_or_create_song(tags,"recorder")
            new_song.write_to_file()

            # Cleanup recorder
            self.__cur_recorded_song = None
            self.__recordbin = None
            self.__recording_state =  "STATE_NULL"
            #print "Song created",new_song

        if self.__differed_start_record:
            self.__differed_start_record = False
            self.record()


    def record(self):
        if self.__recording_state == "STATE_DIRTY":
            self.__differed_start_record = True
            return

        if self.__recording_state != "STATE_NULL":
            return

        # FIXME: made configure dialog to change this thing
        recorder_path = config.get("plugins","recorder_path",os.path.expanduser("~/Recorder"))
        format = config.get("plugins","recorder_format", "ogg")
        quality = int(config.get("plugins","recorder_quality", "3"))
        mode = config.get("plugins","recorder_mode", "cbr")

        vfs.makedirs(recorder_path)
        outputfile = mkstemp("."+format,"listen_record_", recorder_path)

        os.close(outputfile[0])
        os.unlink(outputfile[1])

        if format=="mp3":
            encoder = gst.element_factory_make("lame")
            encoder.set_property("quality",2)
            encoder.set_property("xingheader",0)
            encoder.set_property("vbr",mp3_mode[mode][0])
            encoder.set_property(mp3_mode[mode][1],mp3_quality[mode][quality])
        elif format=="ogg":
            encoder = gst.element_factory_make("vorbisenc")
            encoder.set_property("quality",ogg_quality[quality])
        elif format=="flac":    
            encoder = gst.element_factory_make("flacenc")

        filesink = gst.element_factory_make("gnomevfssink","filesink")
        filesink.set_property("location",outputfile[1])

        self.__recordbin = gst.Bin()
        self.__recordbin.add(encoder,filesink)
        gst.element_link_many(encoder,filesink)
        self.__recordbin.add_pad( gst.GhostPad("sink",encoder.get_pad("sink")))

        self.__recording_state = "STATE_READY"
        self.__cur_recorded_song = Player.song
        self.__recordbin.set_state(gst.STATE_PAUSED)
        Player.bin.xfade_add_tee(self.__recordbin)

    def stop_record(self):
        if self.__recording_state == "STATE_READY":
            self.__differed_stop_record = True
            return

        if self.__recording_state == "STATE_RECORD":
            self.__recording_state = "STATE_DIRTY"
            Player.bin.xfade_remove_tee(self.__recordbin)

class RecorderSourceItem(SourceItem):
    has_top_separateur = False
    has_bottom_separateur = False
    is_droppage = True
    stock = gtk.STOCK_MEDIA_RECORD
    config_code = "recorder"
    label = _("Recorder")

    def __init__(self):
        self.widget = ScrolledWindow(RecorderSongView())
    
    def on_drag_data_get(self,context, selection, info):
        list_uri = [ s.get("uri") for s in ListenDB.get_songs("recorder") ]
        selection.set("text/listen-songs", 8, "\n".join(list_uri))
        selection.set_uris(list_uri)
        return True
      
    def on_activated(self):
        self.play_playlist()
        
    def play_playlist(self,w=None):
        songs = ListenDB.get_songs("recorder")
        if len(songs)>0:
            Dispatcher.cur_playlist_play(songs)
               
    def enqueue_playlist(self,w):
        songs = ListenDB.get_songs("recorder")
        if len(songs)>0:
            Dispatcher.cur_playlist_enqueue(songs)  
            
    def get_menu(self):
        menu = gtk.Menu()
        item = gtk.ImageMenuItem(stock.PLAY)
        item.connect("activate",self.play_playlist)
        menu.append(item)
        item = gtk.ImageMenuItem(stock.ENQUEUE)
        item.connect("activate",self.enqueue_playlist)
        menu.append(item)
        menu.show_all()
        return menu
        
class RecorderSongView(SortableSongView):
    def __init__(self):
        kargs = {
                "#track" : (_("#"), gobject.TYPE_STRING),
                "#date" : (_("Recorded Date"), gobject.TYPE_STRING),
                "title" : (_("Title"), gobject.TYPE_STRING),
                "genre" : (_("Genre"), gobject.TYPE_STRING),
                "artist" : (_("Artist"), gobject.TYPE_STRING),
                "album" : (_("Album"), gobject.TYPE_STRING),
                "#bitrate" : (_("Bitrate"),gobject.TYPE_STRING) ,
                "#duration" : (_("Length"),gobject.TYPE_STRING) ,
                "#playcount" : (_("Play Count"),gobject.TYPE_STRING) ,
                "#skipcount" : (_("Skip Count"),gobject.TYPE_STRING) ,
                "#lastplayed" : (_("Last Played"),gobject.TYPE_STRING)     
                }

        super(RecorderSongView,self).__init__("recorder",**kargs)
        self.menu = SongMenu()
        self.set_menu(self.menu)

        def added(db,type,songs): 
            if type == "recorder": self.get_model().append_songs(songs)

        def delete(db,type,songs): 
            if type == "recorder": self.get_model().remove_songs(songs)

		# FIXME: this function don't reorder the song !
        def change(db,songs):
            if type == "recorder" :self.get_model().change_songs(songs)

        self.autoconnect(ListenDB,"added",added)
        self.autoconnect(ListenDB,"removed",delete)
        self.autoconnect(ListenDB,"simple-changed",change)

        if ListenDB.isloaded():
            self.__on_db_loaded(ListenDB)
        else:
            self.autoconnect(ListenDB,"loaded",self.__on_db_loaded)

    def __on_db_loaded(self,db):
        self.get_model().fill(list(ListenDB.get_songs("recorder")))
       


