# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###

import os

import gobject
import gtk
import urllib
import re

import utils
from config import config
import stock

from helper import Dispatcher, SignalCollector

from library import ListenDB
from player import Player
from web_threading import WebFetchThread
from widget.webtemplate import get_template

from source import Source,SourceItem
from widget.song_menu import SongMenuManager

from logger import newLogger

mylogger = newLogger("wikipedia")

WIKIPEDIA_HISTORY_LEN = 20

try:
    import webkit
except ImportError:
    webkit = None
try: 
    from widget.mozembed_wrap import MozClient
except ImportError:
    MozClient = None
except: 
    mylogger.exception("Failed import mozembed wrapper")
    MozClient = None

backend = os.environ.get("LISTEN_BROWSER",None)
if webkit and backend == "webkit":
    USEWEBKIT = True
elif MozClient and backend == "mozilla":
    USEWEBKIT = False
else:
    if not webkit and not MozClient:
        raise ImportError("missing python-webkit (recommended) or python-mozembed")
    elif backend:
        raise ImportError('LISTEN_BROWSER support only "webkit" or "mozilla"')
    # Fallback
    elif webkit:
        USEWEBKIT = True
    elif MozClient:
        USEWEBKIT = False

mylogger.info("wikipedia use %s has browser backend",(USEWEBKIT and "webkit" or "mozilla") )

WIKI_HOTE = "%s.wikipedia.org"
WIKI_URL = WIKI_HOTE+"/wiki/"
WIKI_URL = WIKI_HOTE+"/wiki/Special:Search/"

class WikiBox(gtk.VBox):
    def __init__(self):
        gtk.VBox.__init__(self,False,6)

        self.last_action = "artist"
        self.song = None

        if USEWEBKIT:
            self.embed = webkit.WebView()
            SignalCollector.connect("wikipedia",self.embed,"navigation-requested", self.webkit_link_uri_cb)
        else:
            self.embed = MozClient()
            SignalCollector.connect("wikipedia",self.embed,"open-uri",self.moz_link_uri_cb)

        self.btn_forward = self.make_btn(gtk.STOCK_GO_FORWARD,"forward")
        self.btn_forward.set_tooltip_text(_("Forward"))
        self.btn_back = self.make_btn(gtk.STOCK_GO_BACK,"back")
        self.btn_back.set_tooltip_text(_("Back"))

            
        self.dropdown_info = gtk.combo_box_new_text()
        self.dropdown_info.append_text(_("Artist"))
        self.dropdown_info.append_text(_("Album"))
        self.dropdown_info.append_text(_("Title"))

        i = int(config.get("wikipedia","info"))
        self.dropdown_info.set_active(i)

        SignalCollector.connect("wikipedia",self.dropdown_info,"changed",self.on_change_info)


        self.btn_browser = gtk.Button()
        self.btn_browser.add(gtk.image_new_from_icon_name("stock_internet", gtk.ICON_SIZE_LARGE_TOOLBAR))
        SignalCollector.connect("wikipedia",self.btn_browser,"clicked",self.open_in_browser)
        self.btn_browser.set_tooltip_text(_("See in web browser"))

        model = gtk.ListStore(gobject.TYPE_STRING)

        self.dropdown_lang = gtk.combo_box_new_text()

        langs = config.get("wikipedia","lang").split("<###>")
        active = 0
        self.lang = "en"
        for lang in langs:
            if lang == config.get("setting","wikipedia"):
                active = langs.index(lang)
                self.lang=lang
            self.dropdown_lang.append_text("%s.wikipedia.org"%lang)

        if langs: 
            self.dropdown_lang.set_active(active)
        SignalCollector.connect("wikipedia",self.dropdown_lang,"changed",self.on_change_lang)
        self.dropdown_lang.set_tooltip_text(_("Change wikipedia language"))

        self.auto_update = gtk.CheckButton("Auto-updating")
        self.auto_update.set_property("active",True)

        label = _("See %s on %s")
        label = label.split("%s")
        while len(label) < 3:
            label.append("")
        self.label1 = gtk.Label(label[0].strip())
        self.label2 = gtk.Label(label[1].strip())
        self.label3 = gtk.Label(label[2].strip())

        hbox = gtk.HBox(False,6)
        hbox2 = gtk.HBox(False,6)

        hbox.pack_start(self.btn_browser,False,False)
        hbox.pack_start(self.btn_back,False,False)
        hbox.pack_start(self.btn_forward,False,False)
        hbox.pack_start(self.label1,False,False)
        hbox.pack_start(self.dropdown_info,False,False)
        hbox.pack_start(self.label2,False,False)
        hbox.pack_start(self.dropdown_lang,False,False)
        hbox.pack_start(self.label3,False,False)

        hbox.pack_end(self.auto_update,False,False)

        #self.pack_start(hbox2,False,False)
        self.pack_start(hbox,False,False)

        f = gtk.Frame()
        f.add(self.embed)
        self.pack_start(f)
        self.embed.show_all()
        self.site_cache={}
        
        self.current_url = None
        self.go_back_url = []
        self.go_forward_url = []
        
        self.update_btn_status()
        
        SignalCollector.connect("wikipedia",Player,"new-song",self.new_media_cb)
        SignalCollector.connect("wikipedia",config, "config-changed",self.update_lang)
        
        self.download_thread = WebFetchThread(1024,self.fail_fetch)
        #self.download_thread.dbg = True
    
    def update_lang(self,helper,section,key,value):
        if section != "wikipedia" or key != "lang": return
        
        for i in range(0,len(self.dropdown_lang.get_model())):
            self.dropdown_lang.remove_text(0)

        langs = config.get("wikipedia","lang").split("<###>")
        for lang in langs:
            if lang == config.get("setting","wikipedia"):
                active = langs.index(lang)
                self.lang=lang
            self.dropdown_lang.append_text("%s.wikipedia.org"%lang)
        self.dropdown_lang.set_active(active)

    def fail_fetch(self):
        HTML = get_template()%("<h3>"+_("Failed to retreive data")+"</h3>")
        self.render_data(HTML)
        
    def moz_link_uri_cb(self,w,uri):
        if uri.find("wikipedia.org")!=-1 or uri.find("wikimedia.org")!=-1:
            if self.current_url:
                self.go_back_url.append(self.current_url)
            self.go_forward_url = []
            self.download_uri(uri)
        else:
            utils.website(uri);
        return True

    def webkit_link_uri_cb(self, view, frame, networkRequest):
        uri = networkRequest.get_uri()
        if uri == self.current_url: return False
        if uri.find("wikipedia.org")!=-1 or uri.find("wikimedia.org")!=-1:
            if self.current_url:
                self.go_back_url.append(self.current_url)
            self.go_forward_url = []
            self.download_uri(uri)
        else:
            utils.website(uri);
        return True
        
    def download_uri(self,uri):
        self.current_url = uri
        self.download_thread.fetch_url(uri,self.render_data,True)
        
        
    def render_data(self,html,prepare_data=False):
        if not isinstance(html,str) and not isinstance(html,unicode):
            html = html.read()
        if prepare_data:
            html = self.prepare_html(html)
        if USEWEBKIT:
            self.embed.load_html_string(html, self.current_url or "" )
        else:
            self.embed.set_data(self.current_url,html)
        gobject.idle_add(self.update_btn_status)
            

    def open_in_browser(self,btn=None):
        if self.current_url:
            utils.website(self.current_url);

    def on_change_info(self,widget):
        action_map = [ "artist", "album","title" ]
        i = self.dropdown_info.get_active()
        self.fetch_information(None,action_map[i])

    def on_change_lang(self,widget):
        self.lang = self.dropdown_lang.get_active_text().replace(".wikipedia.org","")
        config.set("setting","wikipedia",self.lang)
        self.fetch_information(None,self.last_action)
        
    def force_select(self,action,song):
        if action: self.set_action(action)
        self.new_media_cb(None,song)
        
    def set_action(self,action):
        #SignalCollector.disconnect("wikipedia",self.dropdown_info,"changed",self.on_change_info)
        if action=="artist":
            self.dropdown_info.set_active(0)
        if action=="album":
            self.dropdown_info.set_active(1)
        if action=="title":
            self.dropdown_info.set_active(2)
        #SignalCollector.connect("wikipedia",self.dropdown_info,"changed",self.on_change_info)


    def make_btn(self,stocks,action,label=None,btn=None):
        if label==None:
           btn = gtk.Button()
           image = gtk.image_new_from_stock(stocks, gtk.ICON_SIZE_BUTTON)
           btn.add(image)
        else:
           btn = gtk.RadioButton(group=btn)
           image = gtk.image_new_from_stock(stocks, gtk.ICON_SIZE_BUTTON)
           label = gtk.Label(label)
           label.set_alignment(0,0.5)
           box = gtk.HBox(False,6)
           box.pack_start(image,False,False)
           box.pack_start(label,True,True)
           btn.add(box)

        SignalCollector.connect("wikipedia",btn,"clicked",self.fetch_information,action)
        return btn

    def new_media_cb(self,widget,song):
        if not self.auto_update.get_property("active"): return 
        if song == None: return

        self.song = song

        if config.get("setting","offline") == "true" :
            HTML = get_template()%("<h3>"+_("Offline mode")+"</h3>")
            self.render_data(HTML)
            return

        if not ListenDB.song_has_capability(song,"wikipedia"):
            HTML = get_template()%("<h3>"+_("Wikipedia not available for this media type")+"</h3>")
            self.render_data(HTML)
            return

        self.fetch_information(None,self.last_action)

    def update_btn_status(self):
        if len(self.go_back_url)!=0:
            self.btn_back.set_sensitive(True)
        else:
            self.btn_back.set_sensitive(False)
        if len(self.go_forward_url)!=0:
            self.btn_forward.set_sensitive(True)
        else:
            self.btn_forward.set_sensitive(False)

    def fetch_information(self,btn,action):
        action_map = { "artist":0,"album":1,"title":2 }
        if action in ["artist","album","title"]:

            config.set("wikipedia","info","%s"%action_map[action])
            self.last_action=action
            if self.song==None:
                return
            info = self.song.get_str(action)
            url = "http://"+(WIKI_URL%self.lang)+urllib.quote(utils.filter_info_song(info).encode("utf-8"))
            if str(url) == str(self.current_url):
                return
            if self.current_url:
                self.go_back_url.append(self.current_url)
            self.go_forward_url = []
            
            self.download_uri(url)

        elif action=="back":
            url = self.go_back_url.pop()
            if self.current_url:
                self.go_forward_url.append(self.current_url)
            self.download_uri(url)

        elif action=="forward":
            url = self.go_forward_url.pop()
            if self.current_url:
                self.go_back_url.append(self.current_url)
            self.download_uri(url)
            
        while len(self.go_back_url) > WIKIPEDIA_HISTORY_LEN:
            self.go_back_url.pop(0)
        while len(self.go_forward_url) > WIKIPEDIA_HISTORY_LEN:
            self.go_forward_url.pop(0)
        self.update_btn_status()

        
        

    def prepare_html(self,html):
        
        # New template by AqD - aquila_deus
        html = re.sub("id=\"p-[a-zA-Z_\-]*\"","style=\"display:none\"",html)
        html = re.sub("id='p-[a-zA-Z_\-]*'","style=\"display:none\"",html)
        html = re.sub("id=\"siteNotice*\"","style=\"display:none\"",html)
        html = re.sub("id=\"footer*\"","style=\"display:none\"",html)
        html = re.sub("id=\"content*\"","id=\"content*\" style=\"margin-left: 0.5em; margin-right: 0.5em\"",html)
        return html

        html = html[html.find("<body"):html.find("</body>")]
        #html = re.sub('<(.*)>', "", html)
        html = re.sub("class=\"(.[^>]*)\"","",html)
        html = re.sub("id=\"(.[^>]*)\"","",html)
        html = re.sub("style=\"(.[^>]*)\"","",html)
        #html = re.sub("<input(.[^>]*)>","",html)
        #print html

        html = get_template()%html
        return html

class WikipediaItem(SourceItem):
    widget_klass = WikiBox
    config_code = "wikipedia"
    label = _("Wikipedia")
    stock = stock.SRC_WIKIPEDIA
    source_id = "wikipedia"


class WikipediaSource(Source):
    PLUGIN_NAME = "Wikipedia"
    PLUGIN_DESC = _("Display Wikipedia page of the current of selected artist, album, song")

    categorie = "info"
    display_index = 3
    signal_collector_id = "wikipedia"

    def __init__(self):
        item = WikipediaItem()
        self.items = [ item ]

        def song_menu_action(song_menu_manager,id_menu,songs):
            if id_menu == "wikipedia_artist":
                Dispatcher.select_source_id("wikipedia")
                item.widget.force_select("artist",songs[0])
            elif id_menu == "wikipedia_album":
                Dispatcher.select_source_id("wikipedia")
                item.widget.force_select("album",songs[0])
 
        self.autoconnect(SongMenuManager,"action",song_menu_action)
        SongMenuManager.register_item("wikipedia_artist",60,stock.WIKI_ARTIST)
        SongMenuManager.register_item("wikipedia_album",70,stock.WIKI_ALBUM)
        for type in [ "local","unknown_local"]:
            SongMenuManager.allow_item_for_type("wikipedia_artist",type)
            SongMenuManager.allow_item_for_type("wikipedia_album",type)

        Source.__init__(self)

    def destroy(self):
        SongMenuManager.unregister_item("wikipedia_artist")
        SongMenuManager.unregister_item("wikipedia_album")
        Source.destroy(self)

