# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###


import colorsys, string
import gtk, pango
import urllib, random, threading

import utils
from config import config
import stock

from vfs import async_get_uris_from_plain_text, async_parse_uris

from song import *
from song_model import *
from widget.view import MultiDragTreeview
from widget.misc import ScrolledWindow
from widget.control import PlayerPlaylistControl
from widget.queue_manager import QueueManager
from widget.dynamic_playlist import DynamicPlaylist

from xdg_support import get_xdg_config_file
from helper import Dispatcher
from player import Player
from library import ListenDB

from logger import Logger

AUTOSAVE_TIMEOUT = 1000*60*5 # 5min

class PlayerPlaylist(MultiDragTreeview):
    __gsignals__ = {
        "playlist-modified" : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE,())
    }
    def __init__(self):
        super(PlayerPlaylist,self).__init__()


        #SongModel only used for song modification
        self.set_model(SongModel(self,bool))


        r1 = gtk.CellRendererText()
        r1.set_property("background-gdk",gtk.gdk.color_parse("#000000"))
        r1.set_property("foreground-gdk",gtk.gdk.color_parse("#FFFFFF"))
        r1.set_property("ellipsize",pango.ELLIPSIZE_END)

        r2 = gtk.CellRendererText()
        r2.set_property("background-gdk",gtk.gdk.color_parse("#000000"))
        r2.set_property("foreground-gdk",gtk.gdk.color_parse("#FFFFFF"))
        r2.set_property("xalign",1)

        c = gtk.TreeViewColumn()
        c.pack_start(r1,True)
        c.pack_end(r2,False)
        c.set_cell_data_func(r1, self.__cell_data_func)
        c.set_cell_data_func(r2, self.__cell_data_func_duration)
        c.add_attribute(r1, "background-set",1)
        c.add_attribute(r1, "foreground-set",1)
        c.add_attribute(r2, "background-set",1)
        c.add_attribute(r2, "foreground-set",1)
        c.set_expand(True)

        self.append_column(c)
        self.set_headers_visible(False)
        self.get_selection().set_mode(gtk.SELECTION_MULTIPLE)

        self.set_enable_search(True)
        self.set_rules_hint(True)
        self.set_rubber_banding(True)

        targets = [("text/listen-songs", gtk.TARGET_SAME_APP, 1),("text/uri-list", 0, 2),("text/plain", 0, 3)]
        self.enable_model_drag_dest(targets, gtk.gdk.ACTION_COPY)
        self.drag_source_set(gtk.gdk.BUTTON1_MASK, targets,gtk.gdk.ACTION_COPY)

        self.connect("row-activated",self.on_activated)
        self.connect("drag-data-received", self.on_drag_data_received)
        self.connect('drag-data-get', self.on_drag_data_get)
        self.connect("popup-menu",self.__popup_menu)
        
        self.connect("key-press-event", self.on_key_press)
        
        Dispatcher.connect("cur-playlist-enqueue",self.helper_cb,False)
        Dispatcher.connect("cur-playlist-play",self.helper_cb,True)

        ListenDB.connect("simple-changed",self.change_songs)
        
        Dispatcher.connect("update-queue",self.update_queue)
        
        self.current_item = -1
        self.total_duration = 0
        self.stop_track = -1
        self._played = []
        
        self.__queue = []
        self.build_menu()
        
    def on_key_press(self, widget, event):
        if event.keyval == gtk.gdk.keyval_from_name("Delete"):
            self.remove_selected()
            return True
        if event.keyval == gtk.gdk.keyval_from_name("q"):
            self.queue()
            return True
        
    # Just reset the model to refresh when switch on compact view ...
    def refresh(self):
        if self.get_model() and len(self.get_model())>0:
            list_song = list([row[0] for row in self.get_model() ])    
            num = self.current_item
            self.clear()
            for i in range(0,len(list_song)):
                self.add_song(list_song[i])
                if i==num:
                    self.set_current_item(num)
        
    def get_songs(self):
        songs = []
        for row in self.get_model():
            songs.append(row[0])
        return songs
        
    def change_songs(self,helper,songs):
        def update(model, path, iter,songs):
            if model[path][0] in songs:
                model[path] = (songs[songs.index(model[iter][0])],model[path][1])
        self.get_model().foreach(update,songs)
        self.total_duration = 0
        for row in self.get_model():
            song = row[0]
            self.total_duration += song.get("#duration")
        self.emit("playlist-modified")
        
    def update_queue(self,helper,queue):
        self.__queue = queue
        if self.stop_track >= 0:
            self.stop_track = queue[len(queue) - 1]
    
    def helper_cb(self,helper,songs,play):
        if play:
            self.add_song(songs[:1][0],None,True)
            for song in songs[1:]:
                self.add_song(song,None,False)
        else:
            pos = None

            if len(self.__queue)>0:
                pos = self.__queue[-1]+1

            for song in songs:
                self.add_song(song,pos,False)
                
            if len(self.__queue)>0:
                for n in range(0,len(songs)):
                    self.__queue.append(pos+n)
            
        
    def save(self, close=False):
        # Sync player status with the playlist one
        if not close: Player.save_state()

        utils.save_db(self.get_songs(), get_xdg_config_file("playlist_cur.db"))
        utils.save_db(self.__queue, get_xdg_config_file("playlist_queue.db"))
        config.set("player", "stop_track", str(self.stop_track))
        config.set("player", "selected_track", str(self.current_item))

        # Commit change
        if not close: config.write()
        return True
        
    def load(self):
        songs = utils.load_db(get_xdg_config_file("playlist_cur.db"))
        failed =False
        if songs:
            for song in songs:
                try:
                    ListenDB.get_or_create_song(song,song.get_type(),True)
                    self.add_song(song)
                except:
                    failed = True

        if not failed:
            self.__queue = utils.load_db( get_xdg_config_file("playlist_queue.db") )
            if not self.__queue or not isinstance(self.__queue, list): self.__queue = []
            self.stop_track = config.getint("player", "stop_track")
        else:
            self.__queue = []

        current_item = config.getint("player","selected_track")
        if current_item and current_item != -1 and current_item < len(self.get_model()):
            self.set_current_item(current_item)

        self.emit("playlist-modified")
        gobject.timeout_add(AUTOSAVE_TIMEOUT,self.save)
                
    def save_to_library(self,*param):
        model = self.get_model()
        ListenDB.create_playlist("local","Saved playlist",[model[i][0] for i in range(0,len(model))])
        
    def on_activated(self, treeview, path, view_column):
        self.play_selected()

    def play_selected(self,btn=None):
        model, rows = self.get_selection().get_selected_rows()
        Player.play_new(model[rows[0]][0])
        self.set_current_item(rows[0][0]) 

    #@utils.print_timing
    def random_reorder(self, *args):
        songs = self.get_songs()
        model = self.get_model()
        model.clear()
        self.total_duration = 0
        
        if self.current_item!=-1:
            current_song = songs.pop(self.current_item)    
            self.add_song(current_song)
            self.current_item = -1
            self.set_current_item(0) 

        list_index = range(0,len(songs))
        random.shuffle(list_index)

        for index in list_index:
            self.add_song(songs[index])
            
        self.__queue = [list_index.index(pos) for pos in self.__queue]
            
        self.emit("playlist-modified")

    def is_empty(self):
        return len(self.get_model()) == 0

    def is_shuffle(self):
        return config.getboolean("setting","shuffle_mode")

    def is_repeat(self):
        return config.getboolean("setting","repeat_mode")

    def shuffle(self, *args):
        if self.is_shuffle():
            config.set("setting","shuffle_mode","false")
        else:
            config.set("setting","shuffle_mode","true")

    def repeat(self,w=None):
        if self.is_repeat():
            config.set("setting","repeat_mode","false")
        else:
            config.set("setting","repeat_mode","true")

    def clear(self,w=None):
        self.current_item = -1
        self.get_model().clear()
        self.total_duration = 0
        del(self.__queue[:])
        del(self._played[:])
        self.stop_track = -1
        self.emit("playlist-modified")

    def play_uris(self, uris, pos=None, sort=True):
        self.get_toplevel().window.set_cursor(None)    
        songs = []
        for uri in uris:
            song = ListenDB.get_song(uri)
            if song :
                songs.append(song)
        if sort: songs.sort()
        play = True
        for song in songs:
            self.add_song(song, pos, play)
            play = False
            if pos is not None:
                pos += 1

    def add_uris(self, uris, pos=None, sort=True):
        print "pos",pos,"sort",sort
        self.get_toplevel().window.set_cursor(None)
        songs = []
        for uri in uris:
            song = ListenDB.get_song(uri)
            if song :
                songs.append(song)
        if sort: songs.sort()
        for song in songs:
            self.add_song(song,pos)
            if pos is not None:
                pos += 1
            
    def add_song(self, song, pos=None, play=False):
        if song == None:
            return False

        model = self.get_model()
        
        #don't play file is a file is addded and enqueue is default action
        if config.getboolean("player","click_enqueue") and ( len(model) > 0 or not Player.is_paused() ) : 
            play = False

        #Clear playlist if no enqueue when play a new file
        if not config.getboolean("player","enqueue") and play:
            self.clear()
        
        self.total_duration += song.get("#duration",0)
        if pos==None:
            model.append((song, False))
        else:
            for n in range(0,len(self.__queue)):
                if pos < self.__queue[n]:
                    self.__queue[n] += 1    
            model.insert(pos,(song, False))
            if self.current_item >= pos:
                self.current_item += 1
                
            for i in range(0,len(self.__queue)):
                if self.__queue[i] >= pos:
                    self.__queue[i] +=1
                    
        self.emit("playlist-modified")
        if play:
            if pos==None:
                self.set_current_item(len(model)-1)
            else:
                self.set_current_item(pos)
                
            #Let main loop refresh window before play the song
            gobject.idle_add(Player.play_new,song)
        return True

    def remove_selected(self,btn=None):
            model, rows = self.get_selection().get_selected_rows()
            rows.reverse()
            for row in rows:
                self.remove(row)
            self.emit("playlist-modified")
            return True

    def crop_selected(self,btn=None):
            model, rows = self.get_selection().get_selected_rows()
            list_rows_to_delete = []
            for i in range(0,len(model)):
                if (i,) not in rows:
                    list_rows_to_delete.append(i)
            list_rows_to_delete.reverse()
            for row in list_rows_to_delete:
                self.remove((row,))
            self.emit("playlist-modified")
            return

    def remove(self,pos):
        model = self.get_model()
        if pos[0]>=len(model): return
        iter = model.get_iter(pos)
        song = model[iter][0]
        self.total_duration -= song.get("#duration",0)
        model.remove(iter)
        if self.current_item == pos[0]:
            self.current_item = -1
        elif self.current_item > pos[0]:
            self.current_item -= 1
        else:
            pass

        if pos[0] in self.__queue:
            del(self.__queue[self.__queue.index(pos[0])])
        for i in range(0,len(self.__queue)):
            if self.__queue[i] >= pos[0]:
                self.__queue[i] -=1
        self.emit("playlist-modified")

        
    def __popup_menu(self,widget):
        item_queue = self.ui.get_widget("/PlaylistMenu/Queue")
        model, rows = self.get_selection().get_selected_rows()
        nb_sel_queue = len([ row for row in rows if row[0] in self.__queue])
        if nb_sel_queue == 0:
             item_queue.child.set_label(_("_Queue"))
        elif nb_sel_queue == len(rows):
             item_queue.child.set_label(_("Un_Queue"))
        else:
             item_queue.child.set_label(_("Toggle _Queue"))
        
        item_stop_track = self.ui.get_widget("/PlaylistMenu/StopAfterThisTrack") 
        if len(rows) == 1:
            if rows[0][0] == self.stop_track:
                item_stop_track.child.set_label(_("_Do not stop after this track"))
            else:
                item_stop_track.child.set_label(_("S_top after this track"))
            item_stop_track.show()
        else:
            item_stop_track.hide()
          
        self.menu.popup(None,None,None,0,gtk.get_current_event_time())
        
    """
      FUNC GESTION NAVIGATION
    """
    def queue(self,*args):
        model, rows = self.get_selection().get_selected_rows()
        for row in rows:
            if row[0] in self.__queue:
                del(self.__queue[self.__queue.index(row[0])])
                if row[0] == self.stop_track:
                    self.stop_track = -1
            else:
                self.__queue.append(row[0])
                if self.stop_track >= 0:
                    self.stop_track = -1
        #Update a row to refresh cellrenderer
        value = model.get_value(model.get_iter(rows[0][0]),1)
        model.set_value(model.get_iter(rows[0][0]),1,value)
    
    def get_next_song(self):
        if self.is_empty():
            if config.get("setting","empty_random")=="true":
                return ListenDB.get_random_song("local")
            else:
                return None
            
        model = self.get_model()
        if len(self.__queue) == 0 and \
                not self.is_repeat() and \
                not self.is_shuffle() and \
                self.current_item == len(model)-1:
            return None

        if self.current_item != -1 and self.current_item == self.stop_track:
            self.stop_track = -1
            return None

        if len(self.__queue) > 0:
            self.set_current_item(self.__queue.pop(0))
        elif self.is_shuffle():
            played = set(self._played)
            songs = set(range(len(model)))
            remaining = songs.difference(played)
            if remaining:
                self.set_current_item(random.choice(list(remaining)))
            elif self.is_repeat() and not self.is_empty():
                del(self._played[:])
                self.set_current_item(random.choice(list(songs)))
            else:
                del(self._played[:])
                return None
        elif self.current_item < len(model)-1:
            self.set_current_item( self.current_item + 1)
        else:
            self.set_current_item(0)

        return self.get_current_song()

    def get_previous_song(self):
        if Player.get_position() > 10:
            return Player.song
        
        if self.is_empty():
            if config.get("setting","empty_random")=="true":
                return ListenDB.get_random_song("local")
            else:
                return None
    
        if not self.is_repeat() and self.current_item == 0:
            return None

        if self.current_item > 0 :
            self.set_current_item( self.current_item - 1 )
        else:
            model = self.get_model()
            self.set_current_item( len(model) - 1 )

        return self.get_current_song()

    def get_current_song(self):
        return self.get_song_from_path(self.current_item)

    def get_song_from_path(self, path):
        model = self.get_model()
        iter = model.get_iter(path)
        return model.get_value(iter, 0)

    def set_current_item(self,item):
        model = self.get_model()
        if self.current_item >= 0 :
            self._played.append(self.current_item)
            model.set_value(model.get_iter(self.current_item),1,False)
        self.current_item = item
        model.set_value(model.get_iter(self.current_item),1,True)
        self.scroll_to_cell(self.current_item)


    def on_song_search(self,model, column, key, iterator):
        title = model.get_value(iterator, 0).sprint("title")
        if title.lower().find(key.lower()) == 0:
          return False
        return True

    """""""""""""""""""""""
        CELL RENDER FUNC
    """""""""""""""""""""""
    def __cell_data_func(self, column, cell, model, iter):
        song = model.get_value(iter, 0)
        text = ""
        path =  model.get_path(iter)[0]
        if path in self.__queue:
            if path == self.stop_track:
                text += "<span background=\"red\" foreground=\"white\"><b> #%d </b></span> "%(self.__queue.index(path)+1)
            else:
                text += "<span background=\"orange\" foreground=\"white\"><b> #%d </b></span> "%(self.__queue.index(path)+1)
        text += "<b>"
        if song.get("station"):
            text +=  song.get_str("station",True)+"\n"
        text +=  song.get_str("title",True)
        text += "</b>"
        if config.get("setting","compact_playlist")=="true": 
            sep =" - "
        else: 
            text += "\n"    
            sep = ""
        if song.get_str("artist")!="":
            text += sep+"<i>"+song.get_str("artist",True)+"</i>"
            sep =" - "
        if song.get_str("album")!="":
            text += sep+song.get_str("album",True)
        if config.getboolean("setting", "dynamic_playlist_color"):
            data = self.calcSongColors(song)
            rgb = colorsys.hls_to_rgb(data[0], 0.78 + data[1], 0.90)
            cell.set_property("cell-background-gdk", gtk.gdk.Color(int(rgb[0] * 65535), int(rgb[1] * 65535), int(rgb[2] * 65535), 0))
        elif cell.get_property("cell-background-set"):
            cell.set_property("cell-background", None)
        cell.set_property("markup", text)

    def __cell_data_func_duration(self, column, cell, model, iter):
        song = model.get_value(iter, 0)
        if song.get("#duration"):
            text = "<span size=\"small\">%s</span>" % song.get_str("#duration",True)
        else:
            text = ""
        if config.get("setting","compact_playlist")=="false": text += "\n "
        if config.getboolean("setting", "dynamic_playlist_color"):
            data = self.calcSongColors(song)
            rgb = colorsys.hls_to_rgb(data[0], 0.82 + data[1], 0.90)
            cell.set_property("cell-background-gdk", gtk.gdk.Color(int(rgb[0] * 65535), int(rgb[1] * 65535), int(rgb[2] * 65535), 0))
        elif cell.get_property("cell-background-set"):
            cell.set_property("cell-background", None)
        cell.set_property("markup", text)

    def calcSongColors(self, song):
        """
            return a tuple with
            first value:  hue value of song (from 0..1)
            second value: lightness variance of song 
                          (not too high, usually between -0.1 and +0.1)
        """
        if song.get_str("artist") != "":
            artistval = (float(string.upper(song.get_str("artist", True)).__hash__() % 1001)) / 1000
        else:
            artistval = 1.0
        if song.get_str("album") != "":
            albumval = (float(string.upper(song.get_str("album")).__hash__() % 17) / 100) - 0.08
        else:
            albumval = 0.0
        return [artistval, albumval]

    """""""""""""""""""""""""""
        FUNC DRAG DROP
    """""""""""""""""""""""""""
    def on_drag_data_get(self,treeview, context, selection, info, timestamp):
        model,rows = treeview.get_selection().get_selected_rows()
        if len(rows)<0: return
        list_uri = list([ model[row[0]][0].get("uri") for row in rows])
        selection.set("text/listen-songs", 8, "\n".join(list_uri))
        selection.set_uris(list_uri)

    def on_drag_data_received(self,treeview, context, x, y, selection, info, timestamp):
        #print "PLAYLIST : drop data received"
        drop_info = treeview.get_dest_row_at_pos(x, y)

        model = treeview.get_model()

        nb_element = len(model)
        if selection.target in ["text/uri-list","text/plain","text/listen-songs"]:
            if context.get_source_widget()!=None and context.get_source_widget()==self:
                position = gtk.TREE_VIEW_DROP_AFTER
                model, rows = treeview.get_selection().get_selected_rows()
                if len(rows) < 1:
                    return
                rows = [row[0] for row in rows]

                if drop_info:
                    new_pos, position = drop_info
                    new_pos = new_pos[0]
                else:
                    new_pos = len(model)-1
                

                if new_pos < rows[0]:     
                    rows.reverse()

                all_iter = [model.get_iter(i) for i in range(0,len(model))]
                i=0
                for row in rows:
                    row += i
                    iter = model.get_iter(row)
                    is_current_item = model.get_value(iter,1)
                    new_iter = model.get_iter(new_pos)
                    if position == gtk.TREE_VIEW_DROP_BEFORE or position == gtk.TREE_VIEW_DROP_INTO_OR_BEFORE:
                        model.move_before(iter, new_iter);
                    if position == gtk.TREE_VIEW_DROP_AFTER or position == gtk.TREE_VIEW_DROP_INTO_OR_AFTER:
                        model.move_after(iter, new_iter);         
                    if new_pos < rows[0]:             
                        if self.current_item>new_pos and self.current_item != -1:
                            self.current_item +=1
                        i += 1
                    else:
                        if self.current_item<new_pos and self.current_item != -1:
                            self.current_item -=1
                        i -= 1 
                    if is_current_item:
                        self.current_item = new_pos
                
                self.__queue = [model.get_path(all_iter[pos])[0] for pos in self.__queue]
                        
                        
            else:
                self.get_toplevel().window.set_cursor(gtk.gdk.Cursor(gtk.gdk.WATCH))
                pos = self.current_item
                if drop_info:
                    pos, position = drop_info
                    pos = pos[0]
                    new_pos = pos
                    if position == gtk.TREE_VIEW_DROP_AFTER:
                        new_pos += 1
                else:
                    new_pos = len(model)

                if selection.target == "text/listen-songs" and selection.data:
                    self.add_uris(selection.data.splitlines(),new_pos,False) 
                elif selection.target == "text/uri-list":
                    async_parse_uris(selection.get_uris(),True,True,self.add_uris,new_pos)
                elif selection.target == "text/plain":
                    async_get_uris_from_plain_text(selection.data,self.add_uris,new_pos)
                else:
                   self.get_toplevel().window.set_cursor(None)
                    

    def build_menu(self):
        menu_ui = """
        <ui>
            <popup name="PlaylistMenu">
              <menuitem action="Play"/>
              <menuitem action="Delete"/>
              <menuitem action="Crop"/>
              <separator/>
              <menuitem action="StopAfterThisTrack"/>
              <menuitem action="Queue"/>
              <menuitem action="ManageQueue"/>
              <separator/>
              <menuitem action="Clear"/>
            </popup>
        </ui>
        """
        #<menuitem action="Random"/>
        
        ag = gtk.ActionGroup('Listen')
        ag.add_actions([
                         ('Play', gtk.STOCK_MEDIA_PLAY, _('_Play'), None, "", self.play_selected),
                         ('Delete', gtk.STOCK_REMOVE, _('_Delete'), "", "", self.remove_selected),
                         ('Crop', gtk.STOCK_CUT, _('C_rop'), None, "", self.crop_selected),
        #                 ('Random', stock.SHUFFLE, _('_Shuffle'), None, "", self.shuffle),
                         ('Clear', gtk.STOCK_CLEAR, _('_Clear'), "<control>Delete", "", self.clear),
                         ('StopAfterThisTrack', gtk.STOCK_MEDIA_STOP, _('S_top after this track'), "", "", self.stop_after_track),
                         ('Queue', gtk.STOCK_SORT_ASCENDING, _('_Queue'), "", "", self.queue),
                         ('ManageQueue', stock.LYRICS, _('Manage Queue'), "", "", self.manage_queue)
                       ])


        self.ui = gtk.UIManager()
        self.ui.add_ui_from_string(menu_ui)
        self.ui.insert_action_group(ag, -1)
        self.menu = self.ui.get_widget("/PlaylistMenu")

    def manage_queue(self, btn=None):
        model, rows = self.get_selection().get_selected_rows()
        if len(self.__queue)<0:
            return
        else:
            songs = [model[row][0] for row in self.__queue]
            if songs:
                QueueManager(songs, self.__queue, self)

    def stop_after_track(self, btn=None):
        model, rows = self.get_selection().get_selected_rows()
        if len(rows) == 1:
            if self.stop_track == rows[0][0]:
                self.stop_track = -1
            else:
                self.stop_track = rows[0][0]
                if self.stop_track in self.__queue:
                    bufqueue = []
                    for item in range(0, self.__queue.index(rows[0][0])):
                        bufqueue.append(self.__queue[item])
                    self.__queue = bufqueue
                self.__queue.append(rows[0][0])
                #Update a row to refresh cellrenderer
                value = model.get_value(model.get_iter(rows[0][0]),1)
                model.set_value(model.get_iter(rows[0][0]),1,value)

class PlaylistUI(gtk.VBox,Logger):
    def __init__(self, control):
        super(PlaylistUI,self).__init__()
        
        # PLAYLIST 
        self.title_playlist = gtk.Label()
        self.title_playlist.set_markup("<span size=\"large\"><b>"+_("Current playling")+"</b></span>")
        self.title_playlist.set_alignment(0,0.5)
        
        
        self.playlist = PlayerPlaylist()
        self.playlist.connect("playlist-modified",self.playlist_modified)
        
        dynamic_playlist = DynamicPlaylist(self.playlist)
        
        self.playlist_control = PlayerPlaylistControl(self.playlist)
        
        bottom_box = gtk.HBox(False,6)
        bottom_box.pack_start(dynamic_playlist.get_expander(),True,True)
        bottom_box.pack_start(self.playlist_control,False,False)
        
        self.box_playlist = gtk.VBox()
        self.box_playlist.pack_start(self.title_playlist,False,False)
        self.box_playlist.pack_start(ScrolledWindow(self.playlist),True,True)
        self.box_playlist.pack_start(bottom_box,False,False)
        self.box_playlist.pack_start(dynamic_playlist.get_widget(),False,False)
        self.box_playlist.set_spacing(6)
        
        self.pack_start(control, False , False)
        self.pack_start(self.box_playlist,True,True)
        
        self.set_spacing(6)

    def playlist_modified(self,playlist):
        markup = "<span size=\"large\"><b>"+_("Now playing")+"</b>"+"</span>"
        markup += " - "
        markup += "<i>("+_("Total time")+" : "+utils.duration_to_string(playlist.total_duration,"00:00")+")</i>"
        self.title_playlist.set_markup(markup)

