# -*- coding: utf-8 -*-
# vim: ts=4
###
# written by xsdnyd <xsdnyd@gmail.com>
# Copyright (c) 2007 XSDNYD
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
###

import gtk, pango
import gtk.glade
import gobject

import os
import threading

import utils

from config import config

from xdg_support import get_xdg_data_file
from song import Song
from song_model import *

from widget.view import MultiDragTreeview
from widget.misc import ScrolledWindow
from widget.dialog import WindowMessage,WindowError
from helper import Dispatcher

class QueueTreeview(MultiDragTreeview):
    
    def __init__(self, queue):
        super(QueueTreeview,self).__init__()
        self.queue = queue
        self.set_model(SongModel(self,bool))
        r1 = gtk.CellRendererText()
        r1.set_property("background-gdk",gtk.gdk.color_parse("#000000"))
        r1.set_property("foreground-gdk",gtk.gdk.color_parse("#FFFFFF"))
        r1.set_property("ellipsize",pango.ELLIPSIZE_END)

        r2 = gtk.CellRendererText()
        r2.set_property("background-gdk",gtk.gdk.color_parse("#000000"))
        r2.set_property("foreground-gdk",gtk.gdk.color_parse("#FFFFFF"))
        r2.set_property("xalign",1)

        c = gtk.TreeViewColumn()
        c.pack_start(r1,True)
        c.pack_end(r2,False)
        c.set_cell_data_func(r1, self.__cell_data_func)
        c.set_cell_data_func(r2, self.__cell_data_func_duration)
        c.add_attribute(r1, "background-set",1)
        c.add_attribute(r1, "foreground-set",1)
        c.add_attribute(r2, "background-set",1)
        c.add_attribute(r2, "foreground-set",1)
        c.set_expand(True)

        self.append_column(c)
        self.set_headers_visible(False)
        self.get_selection().set_mode(gtk.SELECTION_MULTIPLE)
        
        self.set_rules_hint(True)
        
        targets = [("text/listen-songs", gtk.TARGET_SAME_APP, 1),("text/uri-list", 0, 2),("text/plain", 0, 3)]
        self.enable_model_drag_dest(targets, gtk.gdk.ACTION_COPY)
        self.drag_source_set(gtk.gdk.BUTTON1_MASK, targets,gtk.gdk.ACTION_COPY)

        #self.connect("row-activated",self.on_activated)
        self.connect("drag-data-received", self.on_drag_data_received)
        self.connect('drag-data-get', self.on_drag_data_get)

    def fill(self, songs=None):
        model = self.get_model()
        for s in songs:
            model.append((Song(s), False))

    """""""""""""""""""""""""""
    FUNC DRAG DROP
    """""""""""""""""""""""""""
    def on_drag_data_get(self,treeview, context, selection, info, timestamp):
        model,rows = treeview.get_selection().get_selected_rows()
        if len(rows) < 0: 
            return
        list_uri = list([model[row[0]][0].get("uri") for row in rows])
        selection.set("text/listen-songs", 8, "\n".join(list_uri))
        selection.set_uris(list_uri)

    def on_drag_data_received(self,treeview, context, x, y, selection, info, timestamp):
        drop_info = treeview.get_dest_row_at_pos(x, y)
        model = treeview.get_model()
        nb_element = len(model)
        if selection.target in ["text/uri-list","text/plain","text/listen-songs"]:
            if context.get_source_widget() != None and context.get_source_widget() == self:
                position = gtk.TREE_VIEW_DROP_AFTER
                model, rows = treeview.get_selection().get_selected_rows()
                if len(rows) < 1:
                    return
                rows = [row[0] for row in rows]
                if drop_info:
                    new_pos, position = drop_info
                    new_pos = new_pos[0]
                else:
                    new_pos = len(model) - 1
                
                if new_pos < rows[0]:     
                    rows.reverse()

                all_iter = [model.get_iter(i) for i in range(0,len(model))]
                i = 0
                for row in rows:
                    row += i
                    iter = model.get_iter(row)
                    new_iter = model.get_iter(new_pos)
                    if position == gtk.TREE_VIEW_DROP_BEFORE or position == gtk.TREE_VIEW_DROP_INTO_OR_BEFORE:
                        model.move_before(iter, new_iter)
                    if position == gtk.TREE_VIEW_DROP_AFTER or position == gtk.TREE_VIEW_DROP_INTO_OR_AFTER:
                        model.move_after(iter, new_iter)      

                    buf = self.queue[row]
                    del(self.queue[row])
                    self.queue.insert(new_pos, buf)
                    
                    if new_pos < rows[0]:             
                        i += 1
                    else:
                        i -= 1 
                
    """""""""""""""""""""""
    CELL RENDER FUNC
    """""""""""""""""""""""
    def __cell_data_func(self, column, cell, model, iter):
        song = model.get_value(iter, 0)
        text = ""
        text += "<span size=\"small\"><b>"
        text +=  song.get_str("title",True)
        text += "</b></span>"
        if config.get("setting","compact_playlist")=="true": 
            sep = " - "
        else:
            sep = ""
            text += "\n"
        text += "<span size=\"small\">"
        if song.get_str("artist")!="":
            text += sep+"<i>"+song.get_str("artist",True)+"</i>"
            sep = " - "
        if song.get_str("album")!="":
            text += sep+song.get_str("album",True)
        text += "</span>"
        cell.set_property("markup", text)

    def __cell_data_func_duration(self, column, cell, model, iter):
        song = model.get_value(iter, 0)
        text = "<span size=\"small\">%s</span>" % song.get_str("#duration",True)
        if config.get("setting","compact_playlist")=="false": text += "\n "
        cell.set_property("markup", text)


class QueueManager(object):
    def __init__(self, songs, queue, playlist):
        self.playlist = playlist
        self.new_queue = [q for q in queue]
        self.new_songs = [Song(s) for s in songs]
        self.changed = []
        self.xml = gtk.glade.XML( get_xdg_data_file("queuemanager.glade"), domain = "listen")
        gw = self.xml.get_widget
        self.win = gw("WindowQueueManager")
        gw("CancelButton").connect("clicked", self.cancel)
        gw("SaveButton").connect("clicked", self.save)
        gw("MoveUpButton").connect("clicked", self.moveup)
        gw("MoveDownButton").connect("clicked", self.movedown)
        gw("RemoveButton").connect("clicked", self.remove_selected)
        self.box = gw("SubLayout")
        self.queueview = QueueTreeview(self.new_queue)
        self.box.add(self.queueview)
        self.win.connect("destroy",self.quit)
        self.win.show()
        self.queueview.fill(self.new_songs)
        self.queueview.show()
        self.new_queue
        
    def quit(self, *args, **kargs):
        self.win.destroy()  
            
    def cancel(self, *args, **kargs):
        self.quit()
            
    def save(self, *args, **kargs):
        Dispatcher.update_queue(self.new_queue)
        self.quit()
        
    def moveup(self, *args, **kargs):
        model, rows = self.queueview.get_selection().get_selected_rows()
        if len(rows) < 1:
            return
        for row in rows:
            if row[0] < 1:
                break
            iter = model.get_iter(row[0])
            new_iter = model.get_iter(row[0] - 1)
            model.move_before(iter, new_iter)
            buf = self.new_queue[row[0]]
            del(self.new_queue[row[0]])
            self.new_queue.insert(row[0] - 1, buf)
    
    def movedown(self, *args, **kargs):
        model, rows = self.queueview.get_selection().get_selected_rows()
        if len(rows) < 1:
            return
        rows.reverse()
        for row in rows:
            if row[0] >= len(self.new_queue) - 1:
                break
            iter = model.get_iter(row[0])
            new_iter = model.get_iter(row[0] + 1)
            model.move_after(iter, new_iter)
            buf = self.new_queue[row[0]]
            del(self.new_queue[row[0]])
            self.new_queue.insert(row[0] + 1, buf)
    
    def remove_selected(self, *args, **kargs):
        model, rows = self.queueview.get_selection().get_selected_rows()
        rows.reverse()
        for row in rows:
            self.remove(row)
                
    def remove(self, pos):
        model = self.queueview.get_model()
        if pos[0] >= len(model): 
            return
        iter = model.get_iter(pos)
        song = model[iter][0]
        model.remove(iter)
        del(self.new_queue[pos[0]])
