# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###


import gtk
import gobject
import pango
import traceback

from config import config

from widget.misc import ScrolledWindow
from widget.jobs_manager import jobs_manager

from library import ListenDB
from xdg_support import get_xdg_lib_dirs
from helper import Dispatcher
from logger import Logger

from plugins import Manager
from source import Source, MANDATORY_SOURCE, BLACKLIST_SOURCE

from widget.gossip_cellrenderer_expander import GossipCellRendererExpander

TIME_BEFORE_AUTOSELECT = 500


CATEGORIE_INFO={
        "info"     : (10,_("Informations")),
        "library"  : (20,_("Library")),
        "playlist" : (30,_("Playlists")),
        "device"   : (40,_("Devices")),
        "shared"   : (50,_("Shared libraries")),
        "shop"     : (60,_("Shops")),
        }


class SourceManager(Manager):
    Kinds = [Source]

class SourceUI(gtk.VBox,Logger):
    def __init__(self):
        super(SourceUI,self).__init__(False,12)

        
        model = gtk.TreeStore(object,object,str)

        self.tree = gtk.TreeView()
        self.tree.set_model(model)
        
        self.tree.set_headers_visible(False)

        col = gtk.TreeViewColumn()

        rtc = gtk.CellRendererText()
        rtc.set_property("xpad",0)
        rtc.set_property("visible",False)
        col.pack_start(rtc,False)
        col.set_cell_data_func(rtc,self.__categorie_set_cell_data_func)

        rp = gtk.CellRendererPixbuf()
        rp.set_property("xpad",4)
        rp.set_property("ypad",1)
        rp.set_property("visible",False)
        col.pack_start(rp,False)
        col.set_cell_data_func(rp,self.__pixbuf_set_cell_data_func)

        rt = gtk.CellRendererText()
        rt.set_property("ellipsize",pango.ELLIPSIZE_END)
        rt.connect("edited",self.on_edition)
        col.pack_start(rt,True)
        col.set_cell_data_func(rt,self.__title_set_cell_data_func)

        re = GossipCellRendererExpander()
        re.set_property("visible",False)
        col.pack_end(re,False)
        col.set_cell_data_func(re,self.__expander_set_cell_data_func)
        
        self.tree.append_column(col)
        self.tree.set_property("show-expanders",False)

        self.tree.get_selection().connect("changed", self.on_selection)

        self.tree.set_enable_search(False)
        
        self.id_motion_drag = None
        self.is_in_drag_motion = False
        self.motion_drag_current_path = None
        self.real_selected_row = None
        
        targets = [("text/listen-songs", gtk.TARGET_SAME_APP, 1), ("text/uri-list", 0, 2)]
        self.tree.enable_model_drag_dest(targets, gtk.gdk.ACTION_COPY)
        self.tree.enable_model_drag_source(gtk.gdk.BUTTON1_MASK, targets, gtk.gdk.ACTION_COPY)
        self.tree.connect("drag-data-get", self.on_drag_data_get)
        self.tree.connect("drag-data-received", self.on_drag_data_received)
        self.tree.connect("row-activated", self.on_activated)
        
        """ Need found how show mouseover interaction  """
        #self.tree.connect("drag-motion",self.on_drag_motion)
        #self.tree.connect("drag-leave",self.on_drag_leave)
        
        
        self.tree.connect("key-press-event", self.on_key_press)
        self.tree.connect("popup-menu",self.__popup_menu)
        self.tree.connect('button-press-event', self.__button_press)
        
        self.box_source = gtk.VBox()        
        box = gtk.VBox(spacing=6)
        box.pack_start(self.box_source,True,True)
        box.pack_start(jobs_manager,False,False)
        
        self.pane = gtk.HPaned()
        self.pack_start(self.pane,True,True)
       
        self.pane.pack1(ScrolledWindow(self.tree), False, True)
        self.pane.pack2(box,True,True)
        self.pane.child_set_property(self.pane.get_child1(),"resize",True)
        self.pane.child_set_property(self.pane.get_child2(),"resize",True)
        
        self.pane.set_position(int(config.get("window","pos_organizer")))

        self.sm = SourceManager(  get_xdg_lib_dirs("source"),_("Source Plugins"))
        self.enable_source_obj = {}
        self.last_selected = None
        self.sm.scan()
        self.refresh()

        Dispatcher.connect("edit-source",self.on_edition_demand)
        Dispatcher.connect("edit-playlist",self.edit_playlist)

        Dispatcher.connect("select-source-id",self.select_source_id)

        Dispatcher.connect("source-updated",self.populate)
        self.sm.connect("plugin-enabled",self.refresh)
        self.sm.connect("plugin-disabled",self.refresh)
    
    def __set_cell_background(self, cell, is_group, is_active):
        style = self.tree.get_style()
        if not is_group:
            if is_active:
                color = style.bg[gtk.STATE_SELECTED]

                color.red = (color.red + (style.white).red) /2
                color.green = (color.green + (style.white).green) /2
                color.blue = (color.blue + (style.white).blue) /2
                cell.set_property("cell-background-gdk",color)
            else:
                cell.set_property("cell-background",None)
        else:
            color = style.text_aa[gtk.STATE_INSENSITIVE]
            color.red = (color.red + (style.white).red) /2
            color.green = (color.green + (style.white).green) /2
            color.blue = (color.blue + (style.white).blue) /2
            cell.set_property("cell-background-gdk",color)


    def __title_set_cell_data_func(self, col, cell, model, iter):
        is_group = model.iter_has_child(iter)
        if is_group:
            cell.set_property("markup","")
            cell.set_property("editable",False)
        else:
            path = model.get_path(iter)
            source = model[path][0]
            item = model[path][1]
            cell.set_property("markup",item.label)
            cell.set_property("editable",item.editable)

        self.__set_cell_background(cell,is_group, False)
        

    def __pixbuf_set_cell_data_func(self, col, cell, model, iter):
        is_group = model.iter_has_child(iter)
        if is_group:
            cell.set_property("visible",False)
            cell.set_property("pixbuf",None)
        else:
            path = model.get_path(iter)
            source = model[path][0]
            item = model[path][1]
            icon = self.tree.render_icon(stock_id=item.stock,size=gtk.ICON_SIZE_MENU,detail=None)
            cell.set_property("visible",True)
            cell.set_property("pixbuf",icon)
        self.__set_cell_background(cell,is_group, False)

    def __categorie_set_cell_data_func(self, col, cell, model, iter):
        is_group = model.iter_has_child(iter)
        if is_group:
            cell.set_property("visible",True)
            cell.set_property("text","  "+model[iter][2])
        else:
            cell.set_property("visible",False)
            cell.set_property("text","  ")
        self.__set_cell_background(cell,is_group, False)

    def __expander_set_cell_data_func(self, col, cell, model, iter):
        is_group = model.iter_has_child(iter)
        if is_group:
            path = model.get_path(iter)
            cell.set_property("visible",True)
            if self.tree.row_expanded(path):
                cell.set_property("expander-style",gtk.EXPANDER_EXPANDED)
            else:
                cell.set_property("expander-style",gtk.EXPANDER_COLLAPSED )
        else:
            cell.set_property("visible",False)
        self.__set_cell_background(cell,is_group, False)

    def refresh(self,*args):
        self.loginfo("=========================================================")
        self.loginfo("*** SOURCE REFRESH START ***")

        if self.sm.list_failures():
            for name, error in self.sm.list_failures().iteritems():
                self.logwarn("failed to load plugin %s: %s",name,"".join(error))
        
        for source in self.sm.list():
            name = source.__name__
            if name not in BLACKLIST_SOURCE and (self.sm.enabled(source) or name in MANDATORY_SOURCE):
                if not self.enable_source_obj.has_key(name):
                    try: inst = source()
                    except:
                        self.logexception("%s\t\tfail to load")
                    else:
                        self.loginfo("%s\t\tloaded successfully",name)
                        self.enable_source_obj[name] = inst
                else:
                    self.loginfo("%s\t\talready loaded",name)
            elif self.enable_source_obj.has_key(name):
                source = self.enable_source_obj[name]
                source.destroy()
                del self.enable_source_obj[name]
                import gc
                gc.collect()
                if len(gc.get_referrers(source)) > 1:
                    self.loginfo("%s\t\tUnload, but all object are not free",name)
                    self.loginfo(gc.get_referrers(source))
                else:
                    self.loginfo("%s\t\tUnload successfully",name)
            else:
                self.loginfo("%s\t\tUnload ..., not loaded",name)
        
        self.loginfo("*** SOURCE REFRESH END ***")
        self.loginfo("=========================================================")

        self.populate(init=True) 

    def save(self):
        for source in self.enable_source_obj.values():
            source.save()
        iter = self.tree.get_selection().get_selected()
        if not iter or not iter[1]: path=0
        else: path = self.tree.get_model().get_path(iter[1])
        config.set("source","selected_index",str(path))
        config.set("window","pos_organizer","%d"%self.pane.get_position())

    def select_source_id(self,helper,id):
        self.force_select(id)
        
    def force_select(self,id):
        model = self.tree.get_model()
        for row in model:
            rowiter = row.iterchildren()
            while True:
                try: 
                    row = rowiter.next()
                    if row[1].source_id == id:
                        self.tree.get_selection().select_path(row.path)
                        break
                except StopIteration:
                    break
            
    def populate(self,helper=None,*param,**kwargs):
        model = self.tree.get_model()   
        model.clear()

        self.last_selected = None
        default_iter_to_select = None
        iter_to_select = None
        
        for child in self.box_source.get_children():
            self.box_source.remove(child)    
        previous_item = None
        slist = [(CATEGORIE_INFO.get(s.categorie)[0], s.categorie,s.display_index,s) for s in self.enable_source_obj.values()]
        slist.sort()
        
        cur_categorie = None
        iter_cur_categorie = None
        for pos, categorie, display_index,source in slist:
            for item in source.get_items():
                if categorie != cur_categorie:
                    self.loginfo("create categorie %s for %s",CATEGORIE_INFO.get(categorie)[1],item)
                    iter_cur_categorie = model.append(None,(None,None, (CATEGORIE_INFO.get(categorie)[1] or "Unknown categorie") ))
                    cur_categorie = categorie
                
                iter = model.append(iter_cur_categorie, (source,item,""))
                
                self.box_source.pack_start(item.widget,True,True)
                item.widget.show_all()
                item.widget.set_no_show_all(True)
                item.widget.hide()
                    
                if item.default_selected:
                    default_iter_to_select = iter
                if item.selected:
                    self.last_selected = item
                    iter_to_select = iter

                previous_item = item
    
        self.tree.expand_all()
        if not self.last_selected and len(model) > 0:

            if default_iter_to_select:
                path = model.get_path(default_iter_to_select)
            else:
                path = (0, 1)

            if not model[path]: path = (path[0], path[1] + 1)
            self.tree.get_selection().select_path( path )
            if not self.tree.get_selection().path_is_selected(path):
                self.tree.get_selection().select_path( (0,1) )
                path = (0, 1)

            try: self.last_selected = model[path][1]
            except IndexError: pass
        elif iter_to_select:
            self.tree.get_selection().select_iter(iter_to_select)
                

    def on_selection(self,treeselection):  
        model,iter = treeselection.get_selected()
        if iter!=None and model[iter][0]!=None : 
            
            childs = self.box_source.get_children()    
            """
            if len(childs)>0:
                self.box_source.remove(childs[0])    
            self.box_source.pack_start(model[iter][1].widget,True,True)
            self.box_source.show_all()
            for child in childs:
                child.hide_all()
            """
            if self.last_selected:
                self.last_selected.selected = False
                self.last_selected.widget.hide()
            model[iter][1].widget.show()
            model[iter][1].selected = True

            self.last_selected = model[iter][1]
            return True
        return False
            
    def edit_playlist(self,helper,pl):
        path = None
        model = self.tree.get_model()
        for path in range(0,len(model)):
            item =  model[path][1]
            if hasattr(item,"editable") and item.editable and not item.is_hidden() and item.pl==pl:
                break;
        if path:
            self.tree.set_cursor(path, self.tree.get_column(0), True)    
        
    def on_edition_demand(self,*param):
        model,iter = self.tree.get_selection().get_selected()
        self.tree.set_cursor(model.get_path(iter), self.tree.get_column(0), True)
  
    def invalide_node(self, treeview, path):
        bin = treeview.get_bin_window()
        rect = treeview.get_background_area(path, treeview.get_columns()[0] )
        rect.x = 0
        rect.width = treeview.allocation.width
        bin.invalidate_rect(rect, True)
      
    def on_edition(self,cell,path, new_text):
        if self.tree.get_model()[path][1].on_edition(new_text):
            self.invalide_node(self.tree,path)
            #self.tree.get_model()[path][2] = new_text
            
        
    def on_activated(self,treeview, path, view_column):
        model = treeview.get_model()
        if model[path][1]:
            return model[path][1].on_activated()
        else:
            if model[path][0] == None:
                if self.tree.row_expanded(path):
                    self.tree.collapse_row(path)
                else:
                    self.tree.expand_row(path,False)
                return True
            return False
            
    def on_drag_leave(self,widget, drag_context, timestamp):
        if self.id_motion_drag!=None:
            gobject.source_remove(self.id_motion_drag)
            self.id_motion_drag=None    
            
        
    def on_drag_motion(self,widget, drag_context, x, y, timestamp):
        
        drop_info = self.tree.get_dest_row_at_pos(x, y)
        if drop_info:
            path, position = drop_info
            path = path[0]
            #if position == gtk.TREE_VIEW_DROP_AFTER:
            #    path += 1
               
            #For automatic selection when drag drop 
            if self.id_motion_drag!=None:
                gobject.source_remove(self.id_motion_drag)
                self.id_motion_drag=None
            self.id_motion_drag = gobject.timeout_add(TIME_BEFORE_AUTOSELECT,self.on_drag_motion_cb, path)
            
            if self.tree.get_model()[path][1] and self.tree.get_model()[path][1].is_droppage:
                drag_context.drag_status(gtk.gdk.ACTION_COPY,0L)
                return True
            else:
                drag_context.drag_status(gtk.gdk.ACTION_PRIVATE,0L)
                return True
                
        else:
            drag_context.drag_status(gtk.gdk.ACTION_PRIVATE,0L)    
            return True

    def on_drag_motion_cb(self, path):
        self.tree.get_selection().select_path(path)
        if self.id_motion_drag!=None:
            gobject.source_remove(self.id_motion_drag)
            self.id_motion_drag=None
        

    def on_key_press(self, widget, event):
        model, row = self.tree.get_selection().get_selected()
        if row!=None and self.tree.get_model()[row][1]:
            item = self.tree.get_model()[row][1]
            return item.on_key_press(event)
        else:
            return False
        
    def on_drag_data_get(self,treeview, context, selection, info, timestamp):
        self.is_in_drag_motion = False
        
        model, row = self.tree.get_selection().get_selected()
        if self.real_selected_row : row = self.real_selected_row
        self.real_selected_row = None
        if row!=None:
            item = self.tree.get_model()[row][1]
            return item.on_drag_data_get(context, selection, info)
        else:
            return False
        
        
    def on_drag_data_received(self,tree, context, x, y, selection, info, timestamp):
        self.is_in_drag_motion = False
            
        drop_info = tree.get_dest_row_at_pos(x, y)
        model = tree.get_model()
        #print selection.get_uris()
        if drop_info:
            path, position = drop_info
            path0 = path[0]
            path1 = path[1]
            if position == gtk.TREE_VIEW_DROP_AFTER:
                try: 
                    model[(path0,path1 + 1)]
                except:
                    path0 += 1
                    path1 = 0
                else:
                    path1 += 1
            path = (path0, path1)
        else:
            return
        try : item = model[path][1]
        except : return
        
        if selection.data and item:
            songs = []
            for uri in  selection.data.splitlines():
                song = ListenDB.get_song(uri)
                if song :
                    songs.append(song)
            item.on_drag_song_received(songs)
        
        
    def __button_press(self, w,event):
        if event.button == 3:
            x, y = map(int, [event.x, event.y])
            try: path, col, cellx, celly = self.tree.get_path_at_pos(x, y)
            except TypeError: 
                self.__popup_default_menu()
                return True
            self.tree.grab_focus()
            selection = self.tree.get_selection()
            if not selection.path_is_selected(path):
                self.tree.set_cursor(path, col, 0)
            #else:
            col.focus_cell(col.get_cell_renderers()[0])
            self.tree.emit('popup-menu')
            return True
        elif False and event.button == 1:
            x, y = map(int, [event.x, event.y])
            try: path, col, cellx, celly = self.tree.get_path_at_pos(x, y)
            except TypeError:
                return False
            model = self.tree.get_model()
            if model[path][0] == None:
                if self.tree.row_expanded(path):
                    self.tree.collapse_row(path)
                else:
                    self.tree.expand_row(path,False)
            return False

        else:
            return False

    def __popup_menu(self,w):
        model, iter = self.tree.get_selection().get_selected()
        if iter!=None and model[iter][1]:
            menu = model[iter][1].get_menu()
            if menu:
                menu.popup(None,None,None,0,gtk.get_current_event_time())
              
    def __popup_default_menu(self):
        
        menu = gtk.Menu()
        first = True

        slist = [(s.display_index,s) for s in self.enable_source_obj.values()]
        slist.sort()
        for pos,source in slist:
            items = source.get_default_menu()
            if items:
                if not first: menu.append(gtk.SeparatorMenuItem())
                else: first = False
                for item in items:
                    menu.append(item)
        menu.show_all()
        menu.popup(None,None,None,0,gtk.get_current_event_time())    

