# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
####

#@nodistribution


import shutil
from StringIO import StringIO
import hashlib
import vfs
import time
import os.path

from xml.etree.ElementTree import fromstring as XMLFromString
import xml.sax.handler

import gobject
import gtk

import utils
from logger import Logger

from library import ListenDB
from source import Source,SourceItem
from widget.browser import SimpleBrowser
from widget.song_menu import SongMenuManager
from widget.preference import HelperConfigureDialog

from config import config
from helper import Dispatcher 
import xdg_support


SONGS_LIMIT=250
KEEPALIVE=1500
TAGS_KEY = {
        "title":"title",
	    "artist":"artist",
	    "album":"album",
	    "url":"uri",
	    "track":"#track",
	    "time":"time",
	    "size":"#size",
	    "tag":"genre"
        }


class AmpacheDialog(HelperConfigureDialog):
    def __init__(self):
        HelperConfigureDialog.__init__(self, _("Ampache Server"))
        self.add(self.make_lentry(_("Address"),"plugins","ampache_url","http://localhost/ampache/"))
        self.add(self.make_lentry(_("Username"),"plugins","ampache_username",""))
        self.add(self.make_lentry(_("Password"),"plugins","ampache_password",""))
        self.show_all()

class AmpacheSongsXMLHandler(xml.sax.handler.ContentHandler):
    def __init__(self):
        xml.sax.handler.ContentHandler.__init__(self)
        
        self.current = {}

        self.markups = TAGS_KEY.keys()

        self.count = 0
    
    def startElement(self, name, attrs):
        self.__text = ""
        if name == "song":
            self.current_song = {}

    def endElement(self, name):
        if name == "song":
            ListenDB.get_or_create_song( self.current_song, "ampache" )
            self.count += 1
        elif name in self.markups:
            tag = TAGS_KEY.get(name)
            if tag[0] == "#":
                try:
                    data = int(self.__text)
                except: pass
            else:
                data = self.__text
            self.current_song[tag] = data

    def characters(self, content):
        self.__text = self.__text + content


class AmpacheConnector(Logger):
    def __init__(self):

        self.__version = "350001"
        self.sha256_password = ""
        self.xml_rpc_url = ""
        self.user = ""
    
    def get_handshake_url(self):
        """Return the handshake url for current authentification information"""

        timestamp=str(int(time.time()))
        passphrase = hashlib.sha256()
        passphrase.update(timestamp)
        passphrase.update(self.sha256_password)
        auth = { 
                "user":self.user,
                "auth":passphrase.hexdigest(),
                "timestamp":timestamp,
                "version":self.__version,
                "action":"handshake",
                }
        
        return self.__build_url(auth)
    
    def __build_url(self, infos):
        """Return a REST url where each item in dict is a attribute"""

        return "%s?%s"%(self.xml_rpc_url, "&".join([ "%s=%s"%(k,v) for k,v in infos.iteritems() ]))

    def get_xml_data(self, url):
        """Retrieve xml from the requested url"""

        self.loginfo("try connect to %s"%url)
        try:
            f = vfs.urlopen(url)
        except:
            self.logexception("Failed connect to %s"%self.xml_rpc_url)
            return None
        try: 
            data = f.read()
            root = XMLFromString(data)
        except:
            try: f.close()
            except: pass

            self.logexception("Failed connect to %s"%self.xml_rpc_url)
            return None

        f.close()
        return root

    def reconnect(self):
        """Read configuration and made the handshake on ampache server and retrieve the session id"""

        self.xml_rpc_url = config.get("plugins","ampache_url","http://localhost/ampache/") + "server/xml.server.php"
        self.user = config.get("plugins","ampache_username","")
        self.sha256_password = config.get("plugins","ampache_password_sha256","")

        if not self.sha256_password:
            return False
        if not self.xml_rpc_url:
            return False

        url = self.get_handshake_url()
        root = self.get_xml_data(url)
        if not root: return False

        error = root.find("error")
        if error is not None:
            self.loginfo("Failed login on %s: (%s) %s"%(self.xml_rpc_url, error.get("code"), error.text))
            return False
        else:
            self.__sessionid = root.find("auth").text
            self.loginfo("Login on %s succesfully, session: %s"%(self.xml_rpc_url, self.__sessionid))
            #gobject.timeout_add(KEEPALIVE, self.__keepalive)
            return True

    def __keepalive(self):
        url = self.__build_url({"action":"ping","auth":self.__sessionid})
        root = self.get_xml_data(url)
        error = root.find("error")
        if error is not None:
            self.reconnect()
            return True
        print "Keepalive"
        return False

    @utils.threaded
    def load_songs(self, force = False):
        """ Return all songs in dict """
        
        parser = xml.sax.make_parser()
        handler = AmpacheSongsXMLHandler()
        parser.setContentHandler(handler)
        
        cachefile = xdg_support.get_xdg_cache_file("ampache.xml")
            
        if not force and os.path.exists(cachefile):
            f = open(cachefile)
            try:
                self.loginfo("parse xml at %s", cachefile)
                parser.parse(f)
            except:
                try: f.close()
                except: pass
            f.close()
            print "Found %d songs"%handler.count
        
        else:
    
            if os.path.exists(cachefile):
                os.unlink(cachefile)
            total = 0
            offset = 0 
            while True:
                infos = {
                        "action":"songs",
                        "offset":offset,
                        "limit": SONGS_LIMIT,
                        "auth":self.__sessionid,
                        }
                url = self.__build_url(infos)
    
                self.loginfo("try connect to %s", url)
                try: 
                    f = vfs.urlopen(url, timeout=30)
                except:
                    self.logexception("Failed retrieve try reconnect")
                    self.reconnect()
                    continue
                
                fxml = StringIO()
                fxml2 = StringIO()
                shutil.copyfileobj(f,fxml)
                shutil.copyfileobj(fxml,fxml2)
                f.close()
                fcache = open(cachefile,"a")
                fcache.write(fxml.getvalue())
                fcache.close()
                fxml.close()
                
                try:
                    self.loginfo("parse xml at %s", url)
                    parser.parse(fxml2)
                except:
                    try: fxml2.close()
                    except: pass
                    continue
                fxml2.close()
    
                if handler.count - total != SONGS_LIMIT:
                    total = handler.count
                    break
                
                total = handler.count
                offset += SONGS_LIMIT
            print "Found %d songs"%total
            
class AmpacheSource(Source):
    PLUGIN_NAME = "Ampache Support"
    PLUGIN_DESC = _("Allow ampache browse shared music library")
                                                                                                                                                               
    display_index = 1

    categorie = "shared"

    def __init__(self):
        Source.__init__(self)
        self.__idle_reconnect_id = None

        ListenDB.register_type("ampache",["lyrics","wikipedia","lastfminfo","audioscrobbler"])
        for menu_item in ["play","enqueue","lastfm","wikipedia_artist","wikipedia_album","lyrics"]:
            SongMenuManager.allow_item_for_type(menu_item, "ampache")

        self.c = AmpacheConnector()
        self.items = [ AmpacheLibrarySourceItem(self.c) ]
        Dispatcher.update_source()
        
        self.refresh()
        self.autoconnect(config, "config-changed",self.__on_config_change)


    def __on_config_change(self,dispacher,section,option,value):
        if section == "plugins" and option.find("ampache") == 0:
            if option == "ampache_password" and value != "":
                config.set("plugins","ampache_password_sha256",hashlib.sha256(value).hexdigest())
                config.set("plugins","ampache_password","")
            elif option == "ampache_password":
                return 
                
            if self.__idle_reconnect_id is not None:
                gobject.source_remove(self.__idle_reconnect_id)
            self.__idle_reconnect_id = gobject.timeout_add(5000,self.refresh)

    def refresh(self):
        if self.__idle_reconnect_id is not None:
            gobject.source_remove(self.__idle_reconnect_id)
        self.__idle_reconnect_id = None
        
        ListenDB.full_erase("ampache")
        if self.c.reconnect():
            self.c.load_songs()

    @staticmethod   
    def on_configure():
        AmpacheDialog()

    def delete_thyself(self):
        ListenDB.unregister_type("ampache")
        for menu_item in ["play","enqueue","lastfm","wikipedia_artist","wikipedia_album","lyrics"]:
            SongMenuManager.disallow_item_for_type(menu_item, "ampache")
        super(AmpacheSource,self).delete_thyself()
        
        
class AmpacheBrowser(SimpleBrowser):
    _cover = True
    _config_code = "ampache"
    def __init__(self):
        self._type = "ampache"
        SimpleBrowser.__init__(self)

class AmpacheLibrarySourceItem(SourceItem):
    stock = gtk.STOCK_NETWORK
    config_code = "ampache"

    def __init__(self, connector):
        name = "Ampache"
        self.label = name
        self.c = connector
        self.widget = AmpacheBrowser()

    def on_refresh(self, _menuitem):
        ListenDB.full_erase("ampache")
        if self.c.reconnect():
            self.c.load_songs(True)
        
    def get_menu(self): 
        menu = gtk.Menu()     
            
        item = gtk.ImageMenuItem(gtk.STOCK_REFRESH)
        item.connect("activate",self.on_refresh)
        menu.append(item)
        menu.show_all()
        return menu

