# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###

import os
import gtk
import gobject
from xml.dom import minidom
import threading

import stock
import utils
from config import config
import vfs

from widget.browser import Browser
from widget.song_view import SortableSongView
from widget.song_menu import SongMenu, SongMenuManager
from library import ListenDB, ListenDBQuery
from source import Source, SourceItem
from song import Song
from widget.jobs_manager import Job
from widget.misc import SimpleMenu

from helper import SignalCollector
from widget.dialog import WindowNewPodcast, WindowConfirmation


IMPORT_SUBMIT = 200


class BackgroundDeletePodcastsOperation(Job):
    __message = _("Deleting podcasts...")
    def __init__(self, songs=[] , from_disk=False):
        self.songs = songs
        self.from_disk = from_disk
        self.message = _("Deleting podcasts...")
        super(BackgroundDeletePodcastsOperation, self).__init__()
 
    def job(self):
        total = len(self.songs)
        i = 0

        for song in self.songs:
            i += 1
            if self.from_disk and song.get("podcast_local_uri"):
                try:
                    if not vfs.move_to_trash(song.get("podcast_local_uri")):
                        self.logwarn("failed to move to trash %s ", song.get("podcast_local_uri"))
                except:
                    self.logexception("failed to move to trash %s ", song.get("podcast_local_uri"))
            ListenDB.set_property(song, {"hidden":True, "#progress":0 })

            yield _("Deleting podcasts") + " %d/%s" % (i, total), float(i) / float(total), False


class PodcastSongView(SortableSongView):
    def __init__(self, conf_prefix):
        kargs = {
            "#date":(_("Date"), gobject.TYPE_STRING),
            "title":(_("Title"), gobject.TYPE_STRING),
            "album":(_("Feed"), gobject.TYPE_STRING),
            "#duration":(_("Length"), gobject.TYPE_STRING),
            "#playcount":(_("Play count"), gobject.TYPE_STRING),
            "#lastplayed":(_("Last Played"), gobject.TYPE_STRING),
            "#progress":(_("Progress"), gobject.TYPE_INT)
                      }    
        super(PodcastSongView, self).__init__(conf_prefix, **kargs)
        SongMenuManager.connect("action", self.__song_menu_action)
        SongMenuManager.register_item("podcast_delete", 30, stock.DELETE, sep_after=True) #,custom_only="customplaylistmenu")
        SongMenuManager.register_item("podcast_deletedisk", 40, stock.DELETE_DISK, sep_after=True, sep_before=True)
        SongMenuManager.allow_item_for_type("podcast_delete", "podcast")
        SongMenuManager.allow_item_for_type("podcast_deletedisk", "podcast")
        self.set_menu(SongMenu())

    def __song_menu_action(self, song_menu_manager, id_menu, songs):
        if id_menu.startswith("podcast_delete"):
            if id_menu == "podcast_deletedisk":
                text = _("Are you sure to move %d podcasts to your trash ?")
                text = text % len(songs)
                w = WindowConfirmation(text)
                if w.run() == gtk.RESPONSE_ACCEPT:
                    BackgroundDeletePodcastsOperation(songs, True)
                w.destroy()
            else:
                BackgroundDeletePodcastsOperation(songs, False)
            


class PodcastBrowser(Browser):
    def __init__(self, conf_name):
        ListenDB.register_type("podcast")

        tree_song = PodcastSongView("podcast_" + conf_name)
        super(PodcastBrowser, self).__init__(ListenDBQuery("podcast"), tree_song, "podcast_" + conf_name, True, True)
        
        self.menu = SongMenu(for_browser=True)

        self.empty_menu = SimpleMenu()
        self.empty_menu.insert_item(None, self.new_podcast, stock.NEW_PODCAST)
        self.empty_menu.show_all()

        self.set_menu(self.menu, self.empty_menu)

        if ListenDB.isloaded():
            self.__on_db_loaded(ListenDB)
        else:
            self.autoconnect(ListenDB, "loaded", self.__on_db_loaded)

    def get_cols(self, all=False):
        cols = [("col0", "album", _("Feed"), _("Feeds"))]
        return cols

    def __on_db_loaded(self, db):
        self.connect_to_db()

    def new_podcast(self, *args):
        WindowNewPodcast()

def RefreshPodcasts(songs):
    uris = {}
    for song in songs:
        if not uris.has_key(song.get("podcast_feed_url")) or song.get("#daterss") > uris[song.get("podcast_feed_url")]:
            uris[song.get("podcast_feed_url")] = song.get("#daterss")
    for uri, date in uris.iteritems():
        FeedsDownloadJob(uri, date)
    

class PodcastSourceItem(SourceItem):
    label = _("Podcast")
    config_code = "podcast"
    stock = stock.SRC_PODCAST
    
    def __init__(self):
        self.widget = PodcastBrowser("local")

    def get_menu(self): 
        menu = gtk.Menu()
        def new_podcast(w):
            WindowNewPodcast()
           
        item = gtk.ImageMenuItem(stock.NEW_PODCAST)
        SignalCollector.connect("podcast_source", item, "activate", new_podcast)
        menu.append(item)
        menu.show_all()
        return menu 

class PodcastSource(Source):
    PLUGIN_NAME = "Podcast"
    PLUGIN_DESC = "Listen and download podcast"

    load_priority = 1
    display_index = 51

    signal_collector_id = "podcast_source"

    def save(self):
        self.items[0].widget.save_config()

    def __init__(self):
        Source.__init__(self)
        self.items = [PodcastSourceItem()]

        def song_menu_action(song_menu_manager, id_menu, songs):
            if id_menu == "download_podcast":
                PodcastDownloadJob(songs)
            elif id_menu == "refresh_podcast":
                RefreshPodcasts(songs)
            elif id_menu == "new_podcast":
                WindowNewPodcast()

        self.autoconnect(SongMenuManager, "action", song_menu_action)
        SongMenuManager.register_item("download_podcast", 50, stock.PODCAST_DOWNLOAD, sep_before=True)
        SongMenuManager.register_item("refresh_podcast", 60, gtk.STOCK_REFRESH, browser_only=True)
        SongMenuManager.register_item("new_podcast", 70, stock.NEW_PODCAST, sep_before=True)
        SongMenuManager.allow_item_for_type("download_podcast", "podcast")
        SongMenuManager.allow_item_for_type("new_podcast", "podcast")
        SongMenuManager.allow_item_for_type("refresh_podcast", "podcast")


    def delete_thyself(self):
        SongMenuManager.unregister_item("download_podcast")
        SongMenuManager.unregister_item("refresh_podcast")
        SongMenuManager.unregister_item("new_podcast")
        super(PodcastSource, self).delete_thyself()

    def get_default_menu(self): 
        item = gtk.ImageMenuItem(stock.NEW_PODCAST)
        self.autoconnect(item, "activate", lambda i:WindowNewPodcast())
        return [item]
 

class FeedsDownloadJob(Job):
    __message = _("Download feed...")
    def __init__(self, feed_url, start_date=None):
        self.feed_url = feed_url
        self.start_date = start_date
        super(FeedsDownloadJob, self).__init__()
    
    def job(self):
        feed_url = self.feed_url
        start_date = self.start_date
        
        """ Some node navigation """
        def get_child(node, name):
            return node.getElementsByTagName(name)
    
        def get_value(node):
            if hasattr(node, "firstChild") and node.firstChild != None:
                return node.firstChild.nodeValue
            else:
                return ""
    
        def get_first_child_value(node, node_name):
            node = get_child(node, node_name)
            if node != None and len(node) > 0:
                return get_value(node[0])
            else:
                return "";
        try:
            sock = vfs.urlopen(feed_url)
            xmldoc = minidom.parse(sock).documentElement
            root_node = xmldoc.getElementsByTagName('channel')[0]
        except:
            self.logerror("Podcast feed load failed %s", feed_url)
            return 

        album = get_first_child_value(root_node, "title")
        descriptionrss = get_first_child_value(root_node, "description")
        lastBuildDate = utils.strdate_to_time(get_first_child_value(root_node, "lastBuildDate"))
        pubDate = utils.strdate_to_time(get_first_child_value(root_node, "pubDate"))
        
        """ choose the good date prefer pubDate"""
        if pubDate == "":
            daterss = lastBuildDate
        else:
            daterss = pubDate
            
        img_node = root_node.getElementsByTagName("image")
        if img_node != None and len(img_node) > 0:
            album_cover_url = get_first_child_value(img_node[0], "url")
        else:
            album_cover_url = None
            
        """ some check """
        if album == "" or daterss == "" or (start_date and daterss <= start_date):
            self.logwarn("Incorrect or missing tags(pubDate or lastBuildDate or title)")
            return

        added = []
        items = root_node.getElementsByTagName("item")
        total = len(items)
        for i, item in enumerate(items):
            enclosure_node = get_child(item, "enclosure")
            if enclosure_node and len(enclosure_node) > 0:
                tags = {
                    "album" : album,
                    "descriptionrss" : descriptionrss,
                    "#daterss" : daterss,
                    "album_cover_url" : album_cover_url,
                    "podcast_feed_url" : feed_url,
                    "title" : get_first_child_value(item, "title"),
                    "description" : get_first_child_value(item, "description"),
                    "uri" : enclosure_node[0].getAttribute("url"),
                    "#duration" : enclosure_node[0].getAttribute("length"),
                    "#date" : utils.strdate_to_time(get_first_child_value(item, "pubDate")),
                    "#progress" :-2, # to show the not download message
                    #"link" : get_first_child_value(item,"link")
                }
                """ Only feed with tilte,date and mp3 uri """
                if tags["uri"]  and tags["title"]  and tags["#date"] \
                     and (not start_date or tags["#date"] > start_date) and not ListenDB.has_uri(tags["uri"]):
                    added.append(tags)
                    if len(added) >= IMPORT_SUBMIT:
                        gobject.idle_add(self.idle_send_song, added)
                        added = []
                        
                    
            yield _("Reading podcast") + " %d/%d..." % (i, total), float(i) / float(total), False
            
        if len(added) > 0: 
            gobject.idle_add(self.idle_send_song, added)
        feed_uris = [ item["uri"] for item in added ]
        gobject.idle_add(self.update_older_podcast, feed_url, daterss, album_cover_url, descriptionrss, feed_uris)
           
    def update_older_podcast(self, feed_url , daterss, album_cover_url, descriptionrss, feed_uris):
        # Update old podcast, the date is very important because the podcast have the hightest date is used for refresh podcast with correct date
        podcast_songs = [s for s in ListenDB.get_songs("podcast") \
                         if s.get("podcast_feed_url") == feed_url \
                            and s.get("#daterss") < daterss]
        for song in podcast_songs:
            if song.get("hidden") == "1" and song["uri"] not in feed_uris:
                ListenDB.remove(song)
            else:
                song["descriptionrss"] = descriptionrss
                song["#daterss"] = daterss
                song["album_cover_url"] = album_cover_url
                song["podcast_feed_url"] = feed_url
        songs = ListenDB.request("podcast", "podcast_feed_url = '%s'" % feed_url)
        songs_hidden = ListenDB.request("podcast", "&(podcast_feed_url = '%s', hidden = 1) " % feed_url)
        if len(songs_hidden) == len(songs):
            [ ListenDB.remove(song) for song in songs ]

    def idle_send_song(self, tagss):
        songs = []
        for tags in tagss:
            s = ListenDB.get_or_create_song(tags, "podcast")
            songs.append(s)
        songs = [ (song.get("#date"), song) for song in songs ]
        songs.sort(reverse=True)
        songs = [ song[1] for song in songs ]
        songs = songs[:int(float(config.get("podcast", "nb_download")))]
        PodcastDownloadJob(songs)


class PodcastDownloadJob(Job):
    __message = _("Download podcasts...")
    def __init__(self, songs):
        self.songs = songs

        self.cond = threading.Condition()
        
        self.loginfo("%d songs to download", len(self.songs))
        self.queue_updated = {}
        for song in self.songs:
            self.queue_updated[song] = { "#progress":-1 }
        
        super(PodcastDownloadJob, self).__init__()
        self.__cur_uri = None
        self.__cur_downlod_song = None
        self.__id_updated = gobject.timeout_add(500, self.update_song)

    def job(self):
        self.loginfo("Start download progress")
        i = 0 
        self.cond.acquire()
        total = len(self.songs)
        while len(self.songs) > 0:
            song = self.songs.pop(0)
            self.__cur_downlod_song = song
            self.cond.release()

            i += 1
            self.loginfo("Check aviability of %s", song.get("podcast_local_uri"))
            if song.get("podcast_local_uri") and vfs.get_scheme(song.get("podcast_local_uri")) == "file" and vfs.exists(song.get("podcast_local_uri")):
                #reload duration from file:
                tmp_song = Song()
                tmp_song["uri"] = song.get("podcast_local_uri")
                tmp_song.read_from_file()

                self.cond.acquire()
                self.queue_updated[song] = { 
                    "#duration":tmp_song.get("#duration", 0),
                    "#progress":100,
                    "#ctime":tmp_song.get("#ctime", 0),
                    "#mtime":tmp_song.get("#mtime", 0)
                }
                self.cond.release()

                yield _("Download podcasts...") + " (%d/%d)" % (i, total), float(i) / float(total), False
                self.cond.acquire()
                continue
            
            
            podcast_path = os.path.expanduser(config.get("podcast", "folder") + "/")
            podcast_path += song.get_str("album") + "/"
            if not os.path.isdir(os.path.dirname(podcast_path)):
                os.makedirs(os.path.dirname(podcast_path))
            filename = song.get_filename()
            for char in [":", "/"]:
                filename = filename.replace(char, "_")
            podcast_path += filename
            podcast_path = vfs.fsencode(podcast_path)
            new_uri = vfs.get_uri_from_path(podcast_path)
            
            self.cond.acquire()
            self.__cur_uri = new_uri
            self.cond.release()

            if vfs.exists(new_uri): 
                if new_uri == song.get("podcast_local_uri") or vfs.unlink(new_uri):
                    self.logwarn("Podcast already exists or can't delete the incomplete file (%s)", new_uri)
                    self.cond.acquire()
                    self.__cur_uri = None
                    self.__cur_downlod_song = None
                    self.cond.release()
                    yield _("Download podcasts...") + " (%d/%d)" % (i, total), float(i) / float(total), False
                    self.cond.acquire()
                    continue
            
            uri = song.get("uri")
            try:
                for _size_total, _size_receive, percent in vfs.download_iterator(uri, new_uri):
                    progress = (float(i - 1) * 100 + float(percent)) / (float(total) * float(100))
                    self.cond.acquire()
                    self.queue_updated[song] = { "#progress":percent }
                    self.cond.release()
                    yield _("Download podcasts") + " %d%% (%d/%d)" % (percent, i, total), progress, False
            except IOError:
                self.cond.acquire()
                self.queue_updated[self.__cur_downlod_song] = { "#progress":-2 }
                self.__cur_uri = None
                self.__cur_downlod_song = None
                continue

            # Hum, read all tag from file only for the duration.
            tmp_song = Song()
            tmp_song["uri"] = new_uri
            tmp_song.read_from_file()
            
            self.cond.acquire()
            self.queue_updated[song] = {
                "podcast_local_uri":new_uri,
                "#duration":tmp_song.get("#duration", 0),
                "#progress":100,
                "#ctime":tmp_song.get("#ctime", 0),
                "#mtime":tmp_song.get("#mtime", 0)
                }
            self.__cur_uri = None
            self.__cur_downlod_song = None
            self.cond.release()

            yield _("Download podcasts") + "... (%d/%d)" % (i, total), float(i) / float(total), False
            self.cond.acquire()

        self.cond.release()

    def on_stop(self):
        gobject.source_remove(self.__id_updated)
        self.cond.acquire()
        for song in self.songs:
            self.queue_updated[song] = { "#progress":-2 }
        if self.__cur_downlod_song:
            self.queue_updated[self.__cur_downlod_song] = { "#progress":-2 }   
        self.cond.release()
        if self.__cur_uri:
            try: vfs.unlink(self.__cur_uri)
            except: pass
        self.update_song()

    def update_song(self):
        self.cond.acquire()
        for song, tags in self.queue_updated.iteritems():
            ListenDB.set_property(song, tags, use_quick_update=not tags.has_key("#duration"))
                
        self.queue_updated = {}
        self.cond.release()
        return True

