# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###




import gtk
import gobject

try: from elementtree.ElementTree import fromstring as XMLFromString
except ImportError: 
    # Python 2.5 rename
    from xml.etree.ElementTree import fromstring as XMLFromString
import threading
from urllib import quote

from helper import Dispatcher

import utils

from lastfm_player import LastFmPlayer
from web_threading import WebFetchThread

from widget.song_view import SortableSongView
from widget.song_menu import SongMenu
from widget.misc import ScrolledWindow, MarkupLabel

from library import ListenDB

from vfs import urlopen

class LastfmSongView(SortableSongView):
    def __init__(self):
        kwargs = {
                    "station":(_("Station"),gobject.TYPE_STRING),
                }
        super(LastfmSongView,self).__init__("lastfmradio",**kwargs)

        menu = SongMenu()
        self.set_menu(menu)

        def added(db,type,songs): 
            if type == "lastfmradio": self.get_model().append_songs(songs)

        def delete(db,type,songs): 
            if type == "lastfmradio": self.get_model().remove_songs(songs)

        def change(db,songs):
            # FIXME: this function don't reorder the song !
            if type == "lastfmradio" :self.get_model().change_songs(songs)

        self.autoconnect(ListenDB,"added",added)
        self.autoconnect(ListenDB,"removed",delete)
        self.autoconnect(ListenDB,"simple-changed",change)

        if ListenDB.isloaded():
            self.__on_db_loaded(ListenDB)
        else:
            self.autoconnect(ListenDB,"loaded",self.__on_db_loaded)

    def __on_db_loaded(self,db):
        self.get_model().fill(list(ListenDB.get_songs("lastfmradio")))
       

class LastfmWidget(gtk.VBox):
    def __init__(self):
        gtk.VBox.__init__(self,False,6)

        self.download_thread = WebFetchThread(1024,self.__fail_fetch)

        self.__station_type_box = gtk.combo_box_new_text()

        self.__radios = [
        (_("Similar artist station"),
            "http://ws.audioscrobbler.com/1.0/artist/%s/similar.xml",
            LastfmSimilarView),
        (_("Music tagged as"),
            "http://ws.audioscrobbler.com/1.0/tag/%s/search.xml?showtop10=1",
            LastfmTagView),
        (_("User station"),
            "http://ws.audioscrobbler.com/1.0/user/%s/profile.xml",
            LastfmUserView),
        ]
        """
        (_("Group station"),
            "http://ws.audioscrobbler.com/1.0/group/%s/weeklychartlist.xml",
            LastfmGroupView),
        (_("Recommended station"),
            "http://ws.audioscrobbler.com/1.0/user/%s/profile.xml",
            ),
        (_("Neighbours station"),
            "http://ws.audioscrobbler.com/1.0/user/%s/neighbours.xml",
            LastfmNeighbourView),
        (_("Fans station"),
            "http://ws.audioscrobbler.com/1.0/get.php?resource=artist&document=similar&format=xml&artist=%s",
            LastfmFansView),
        """
        [ self.__station_type_box.append_text(radio[0]) for radio in self.__radios ]

        self.__station_type_box.set_active(0)


        self.__entry = gtk.Entry()
        self.__entry.connect("activate",self.__search)

        btn_search = gtk.Button()
        btn_search.add(gtk.image_new_from_stock(gtk.STOCK_FIND,gtk.ICON_SIZE_MENU))
        btn_search.connect("clicked",self.__search)

        hbox = gtk.HBox(False,6)
        hbox.pack_end(btn_search,False,False)
        hbox.pack_end(self.__entry,False,False)
        hbox.pack_end(self.__station_type_box,False,False)

        self.song_view = LastfmSongView()
        
        self.__result_box = None

        self.pack_start(hbox,False,False)
        self.pack_start(ScrolledWindow(self.song_view))
    
    def save_config(self):
        self.song_view.save_config()

    def __fail_fetch(self,*args,**kwargs):
        print "W:LastfmWidget:Failed fetch info"
        self.__set_widget(MarkupLabel(_("<i>No station found</i>")))

    def __search(self,*args,**kwargs):
        text = self.__entry.get_text()
        if text.strip() == "": return None

        index = self.__station_type_box.get_active()
        url = self.__radios[index][1]
        widget_klass = self.__radios[index][2]
        self.download_thread.fetch_url(url%quote(text.encode("utf-8")),self.__fetch_info,widget_klass)
        self.__set_widget(MarkupLabel(_("<i>Search available station...</i>")))
    
    def __fetch_info(self,buffer,widget_klass):
        try: root = XMLFromString(buffer.read())
        except:
            self.__fail_fetch()
        else:
            root = self.__build_items(root)
            if not root: self.__fail_fetch()
            else: self.__set_widget(widget_klass(root))

    def __set_widget(self,widget):
        if self.__result_box:
            self.remove(self.__result_box)
            self.__result_box.destroy()
            del self.__result_box
            self.__result_box = None
        self.pack_start(widget,False,False)
        self.reorder_child(widget,1)
        self.__result_box = widget
        widget.show_all()
        self.show_all()
 
    def __build_items(self,root_node,sub_schema=None):
        class Xmldict(dict):
            def __init__(self,tag,text="",attr={},elem={}):
                dict.__init__(elem)
                self.tag = tag
                self.text = text
                self.attr = attr

            def __setitem__(self,key,value):
                if self.has_key(key):
                    if isinstance(self[key],list):
                        self[key].append(value)
                    else:
                        dict.__setitem__(self,key,[self[key],value])
                else:
                    dict.__setitem__(self,key,value)
                    
        element = Xmldict(root_node.tag,root_node.text,root_node.attrib)
        for node in root_node.getchildren():
            element[node.tag] = self.__build_items(node)
        return element

class LastfmFansView(gtk.HBox):
    def __init__(self,root):
        gtk.HBox.__init__(self,False,6)
        
class LastfmNeighbourView(gtk.HBox):
    def __init__(self,root):
        gtk.HBox.__init__(self,False,6)
        

class LastfmRecommendedView(gtk.HBox):
    def __init__(self,root):
        gtk.HBox.__init__(self,False,6)
        
class LastfmUserView(gtk.HBox):
    def __init__(self,root):
        gtk.HBox.__init__(self,False,6)
        self.__uri = LastFmPlayer.made_user_radio(root.attr["username"])
        if root.attr.has_key("avatar"):
            image = gtk.Image()
            try: sock = urlopen(root["avatar"].text)
            except IOError,e:
                print "W:Lastfm:IOError:",e,"(",root["avatar"].text,")"
            else:
                loader = gtk.gdk.PixbufLoader()
                loader.set_size(80, 80)
                loader.connect("closed", lambda loader,image : image.set_from_pixbuf(loader.get_pixbuf()) ,image)

                def thread():
                    def write(data):
                        loader.write(data)
                        try:loader.close()
                        except gobject.GError:
                            print "W:Webinfo:Incorrect image format"
                            pass
                    data = sock.read()
                    sock.close()
                    gobject.idle_add(write,data)

                t = threading.Thread(target=thread)
                t.setDaemon(True)
                t.start()
            self.pack_start(image,False,False)

        text = "<b><big>"+LastFmPlayer.get_station_name(self.__uri)+"</big></b>\n"
        if root.has_key("realname"): text += utils.xmlescape(_("Realname"))+": "+utils.xmlescape(root["realname"].text)+"\n"
        if root.has_key("age"): text += utils.xmlescape(_("Age"))+": "+utils.xmlescape(root["age"].text)+"\n"
        if root.has_key("playcount"): text += utils.xmlescape(_("Playcount"))+": "+utils.xmlescape(root["playcount"].text)+"\n"
        l = MarkupLabel(text)
        l.set_line_wrap(True)
        self.pack_start(l)
        self.pack_start(LastfmBtnBox(),False,False)
    
    def get_uri(self):
        return self.__uri       

class LastfmGroupView(gtk.HBox):
    def __init__(self,root):
        gtk.HBox.__init__(self,False,6)
        
class LastfmSimilarView(gtk.HBox):
    def __init__(self,root):
        gtk.HBox.__init__(self,False,6)
        
        if root.attr["streamable"] == "0":
            self.pack_start(MarkupLabel(_("<i>No station found</i>")))
            self.set_spacing(0)
        else:
            self.__uri = LastFmPlayer.make_artist_radio(root.attr["artist"])

            text = "<b><big>"+LastFmPlayer.get_station_name(self.__uri)+"</big></b>\n"
            text += utils.xmlescape(_("Featured artists: "))
            text += utils.xmlescape(",".join([elem["name"].text for elem in root["artist"][:10]]))
            text += ",..."
            l = MarkupLabel(text)
            l.set_line_wrap(True)
            self.pack_start(l)

            if root.attr.has_key("picture"):
                image = gtk.Image()
                try: sock = urlopen(root.attr["picture"])
                except IOError,e:
                    print "W:Lastfm:IOError:",e,"(",root.attr["picture"],")"
                else:
                    loader = gtk.gdk.PixbufLoader()
                    loader.set_size(80, 80)
                    loader.connect("closed", lambda loader,image : image.set_from_pixbuf(loader.get_pixbuf()) ,image)

                    def thread():
                        def write(data):
                            loader.write(data)
                            try:loader.close()
                            except gobject.GError:
                                print "W:Webinfo:Incorrect image format"
                                pass
                        data = sock.read()
                        sock.close()
                        gobject.idle_add(write,data)

                    t = threading.Thread(target=thread)
                    t.setDaemon(True)
                    t.start()
                self.pack_start(image,False,False)

            self.pack_start(LastfmBtnBox(),False,False)
    
    def get_uri(self):
        return self.__uri

class LastfmTagView(gtk.HBox):
    def __init__(self,root):
        gtk.HBox.__init__(self,False,6)

        if not root.has_key("tag"):
            self.pack_start(MarkupLabel(_("<i>No station found</i>")))
            self.set_spacing(0)
        else:
            self.__tree = gtk.TreeView()
            self.__tree.insert_column_with_attributes(-1,_("Tags"),gtk.CellRendererText(),markup=0)
            
            model = gtk.ListStore(str,str)
            for elem in root["tag"]:
                if elem["streamable"].text:
                    text = elem["name"].text
                    match = elem.get("match")
                    if match and match.text:
                        text += " <i>("+match.text+")</i>"
                    model.append((elem["name"].text,text))

            self.__tree.set_model(model)
            self.pack_start(ScrolledWindow(self.__tree))
            self.pack_start(LastfmBtnBox(),False,False)


    def get_uri(self):
        model, row = self.__tree.get_selection().get_selected()
        if not row: return
        return LastFmPlayer.make_tag_radio(model[row][0])



class LastfmBtnBox(gtk.VBox):
    def __init__(self):
        gtk.VBox.__init__(self,False,3)
        btn_add = gtk.Button(stock=gtk.STOCK_ADD)
        btn_add.connect("clicked",self.__add)

        btn_play = gtk.Button(stock=gtk.STOCK_MEDIA_PLAY)
        btn_play.connect("clicked",self.__add,True)

        btn_cancel = gtk.Button(stock=gtk.STOCK_CANCEL)
        btn_cancel.connect("clicked",self.__cancel)

        self.pack_start(btn_add,False,False)
        self.pack_start(btn_play,False,False)
        self.pack_start(btn_cancel,False,False)

    def __add(self,btn,play=False):
        uri = self.get_parent().get_uri()
        if uri:
            if play: 
                song = ListenDB.get_or_create_song( { "uri":uri, "station": LastFmPlayer.get_station_name(uri) }, "lastfmradio" , hidden=True)
                Dispatcher.cur_playlist_play([song])
            else:
                song = ListenDB.get_or_create_song( { "uri":uri, "station": LastFmPlayer.get_station_name(uri) }, "lastfmradio" )
        self.__cancel()

    def __cancel(self,*args,**kwargs):
        self.get_parent().destroy()


