/***************************************************************************
                          cconfig.h  -  description
                             -------------------
    begin                : Tue May 14 2002
    copyright            : (C) 2002-2003 by Mathias Küster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CCONFIG_H
#define CCONFIG_H

/**
  *@author Mathias Küster
  *
  * This stores all of dclib's settings and also the list of bookmarked
  * and public hubs.
  *
  * The main problem with the config handling is that adding anything
  * requires adding more functions, plus writing out the config files
  * erases any previous settings, including anything not known to
  * this version of dclib.
  */

#include <dclib/dcos.h>
#include <dclib/core/cstring.h>
#include <dclib/core/cmutex.h>
#include <dclib/core/csingleton.h>
#include <dclib/core/clist.h>
#include <dclib/core/types.h>
#include <dclib/core/cstringlist.h>

#define DCGUI_CHATLOG_PATH	"chatlog"
#define DCGUI_PLUGINCONFIG_PATH	"plugin"
#define DCGUI_IMAGE_PATH	"image"
#define DCGUI_FILELIST_PATH	"filelist"

#include <map>
#include <list>
#include <vector>

class CConfig;
class CXml;

class DCConfigHubProfile {
public:
	/** */
	DCConfigHubProfile()
	{
		m_bComment     = false;
		m_bEMail       = false;
		m_bAutoConnect = false;
		m_bSSL         = false;
		m_bTag         = true;
		m_bExtHubCount = true;
	};
	/** */
	DCConfigHubProfile( DCConfigHubProfile * p )
	{
		m_sName        = p->m_sName;
		m_sPassword    = p->m_sPassword;
		m_sNick        = p->m_sNick;
		m_sComment     = p->m_sComment;
		m_sEMail       = p->m_sEMail;
		m_sRemoteEncoding = p->m_sRemoteEncoding;
		m_sSuppressedNicks = p->m_sSuppressedNicks;
		m_bComment     = p->m_bComment;
		m_bEMail       = p->m_bEMail;
		m_bAutoConnect = p->m_bAutoConnect;
		m_bSSL         = p->m_bSSL;
		m_bTag         = p->m_bTag;
		m_bExtHubCount = p->m_bExtHubCount;
	};
	/** */
	~DCConfigHubProfile() {};

	/** profile name */
	CString m_sName;
	/** password */
	CString m_sPassword;
	/** nick */
	CString m_sNick;
	/** comment */
	CString m_sComment;
	/** email */
	CString m_sEMail;
	/** Remote encoding setting, to override global setting */
	CString m_sRemoteEncoding;
	/** Regexp of nicks to not open chat windows for */
	CString m_sSuppressedNicks;
	/** use comment */
	bool m_bComment;
	/** use email */
	bool m_bEMail;
	/** auto connect on startup */
	bool m_bAutoConnect;
	/** ssl hub */
	bool m_bSSL;
	/** tag */
	bool m_bTag;
	/** If true, use H:3/2/1, if false, use H:6 */
	bool m_bExtHubCount;
};

/* small number of profiles, sorted map is fine */
typedef std::map<CString, DCConfigHubProfile*> HubProfileMap;

class DCConfigHubItem {

/* so CConfig can change m_nPosition but nothing else can */
friend class CConfig;

public:
	/** */
	DCConfigHubItem()
	{
		m_nID          = 0;
		m_nUserCount   = 0;
		m_nShared      = 0;
		m_nMinShare    = 0;
		m_nPosition    = -1;
	};
	/** */
	DCConfigHubItem( DCConfigHubItem * p )
	{
		m_nID          = p->m_nID;
		m_sProfile     = p->m_sProfile;
		m_sName        = p->m_sName;
		m_sHost        = p->m_sHost;
		m_sDescription = p->m_sDescription;
		m_nUserCount   = p->m_nUserCount;
		m_sCountry     = p->m_sCountry;
		m_sExtra       = p->m_sExtra;
		m_nShared      = p->m_nShared;
		m_nMinShare    = p->m_nMinShare;
		m_nPosition    = p->m_nPosition;
	};
	/** */
	~DCConfigHubItem() {};

	/** the hub identify like mldonkey only used by http plugin */
	unsigned long m_nID;
	/** hubname */
	CString m_sName;
	/** hub host */
	CString m_sHost;
	/** hub description */
	CString m_sDescription;
	/** user count */
	ulonglong m_nUserCount;
	/** profile name */
	CString m_sProfile;
	/** geographic location of the hub */
	CString m_sCountry;
	/** total shared on the hub */
	ulonglong m_nShared;
	/** minimum share allowed for entry */
	ulonglong m_nMinShare;
	/** any extra xml fields in the form Key=Value */
	CString m_sExtra;

private:
	/* user changeable order in bookmark list */
	int m_nPosition;
};

typedef std::map<int, DCConfigHubItem*> OrderedBookmarkMap;

class DCConfigHubListUrl {
public:
	/** */
	DCConfigHubListUrl()
	{
		bEnabled = false;
	};
	/** */
	~DCConfigHubListUrl() {};
	/** */
	CString sUrl;
	/** */
	bool bEnabled;
};

class DCConfigShareFolder {
public:
	/** */
	DCConfigShareFolder() {};
	/** */
	~DCConfigShareFolder() {};
	/** */
	CString m_sPath;
	/** */
	CString m_sAlias;
};

class CByteArray;
class DCTransferQueueObject;
class DCFileChunkObject;

class CConfig : public CSingleton<CConfig> {

public:
	/** */
	CConfig( CString configpath = CString() );
	/** */
	virtual ~CConfig();

	/** */
	int LoadDCLib();
	/** */
	int SaveDCLib();
	/** */
	void ParseDCLibConfig( CXml * xml );

	/** */
	int LoadDCTra( CStringList<CStringList<DCTransferQueueObject> > * queue, CStringList<DCFileChunkObject> * chunks );
	/** */
	int SaveDCTra( CStringList<CStringList<DCTransferQueueObject> > * queue, CStringList<DCFileChunkObject> * chunks );

	/** Get plugin folder */
	CString GetDCLibPluginPath() const;
	/** Set plugin folder */
	void SetDCLibPluginPath( CString s );

	/** */
	CString ParseVersion( CString s );
	/** */
	CString GetListenHostString();
	/** */
	CString GetTCPHostString( bool addport = true, bool crypt = false );
	/** */
	CString GetUDPHostString( bool addport = true );

	/** */
	bool GetAutoRecreateShareList() const;
	/** */
	void SetAutoRecreateShareList( bool b );

	/** */
	int GetDownloadQueueTime() const;
	/** */
	void SetDownloadQueueTime( int i );

	/** */
	CString GetEMail( bool raw = false );
	/** */
	void SetEMail( CString email );
	/** */
	CString GetNick() const;
	/** */
	void SetNick( CString nick );
	/** */
	CString GetNick( CString hubname, CString hubhost );
	/** */
	CString GetSearchNick() const;
	/** */
	void SetSearchNick( CString searchnick );
	/** */
	CString GetAwayMessage() const;
	/** */
	void SetAwayMessage( CString s );
	/** */
	CString GetFloodOpKickMessage() const;
	/** */
	void SetFloodOpKickMessage( CString s );
	/** */
	CString GetDownloadFolder() const;
	/** */
	void SetDownloadFolder( CString downloadfolder );
	/** */
	CString GetDownloadFinishedFolder() const;
	/** */
	void SetDownloadFinishedFolder( CString s );
	/** */
	bool GetTransferAutoSearch() const;
	/** */
	void SetTransferAutoSearch( bool b );
	/** */
	bool GetCreateFile() const;
	/** */
	void SetCreateFile( bool b );

	/** */
	eClientMode GetMode( bool setting = false );
	/** */
	void SetMode( eClientMode mode );
	/** */
	CString GetDescription( bool raw = false, CString hubname = CString(), CString hubhost = CString() );
	/** */
	void SetDescription( CString description );
	/** */
	bool GetDescriptionTag() const;
	/** */
	void SetDescriptionTag( bool b );
	/** */
	CString GetAwayPrefix() const;
	/** */
	void SetAwayPrefix( CString s );

	/** */
	CString GetSpeed() const;
	/** */
	void SetSpeed( CString speed );
	/** */
	int GetTCPListenPort() const;
	/** */
	void SetTCPListenPort( int port );
	/** Get listen port for fully encrypted transfers */
	int GetCryptoListenPort() const;
	/** Set listen port for fully encrypted transfers */
	void SetCryptoListenPort( int port );
	/** */
	int GetUDPListenPort() const;
	/** */
	void SetUDPListenPort( int port );
	/** */
	CString GetHost() const;
	/** */
	void SetHost( CString host );
	/** */
	CString GetListenHost() const;
	/** */
	void SetListenHost( CString host );
	/** */
	bool GetAntiSpam() const;
	/** */
	void SetAntiSpam( bool antispam );
	/** */
	bool GetExternalIP() const;
	/** */
	void SetExternalIP( bool b );
	/** */
	int GetMaxUpload() const;
	/** */
	void SetMaxUpload( int maxupload );
	/** */
	int GetUserUploadSlots() const;
	/** */
	void SetUserUploadSlots( int i );
	/** */
	virtual void SetAwayMode( eUserAwayMode mode );
	/** */
	eUserAwayMode GetAwayMode() const;
	/** */
	long GetSharedFolders( CList<DCConfigShareFolder> * list );
	/** */
	void SetSharedFolders( CList<DCConfigShareFolder> * list );
	/** */
	bool GetLogFile() const;
	/** */
	void SetLogFile( bool logfile );
	/** True if we should log finished downloads */
	bool GetLogFinishedDownloads() const;
	/** Set to true to log finished downloads */
	void SetLogFinishedDownloads( bool logdl );
	/** True if we should log finished uploads */
	bool GetLogFinishedUploads() const;
	/** Set to true to log finished uploads */
	void SetLogFinishedUploads( bool logul );
	/** True if we should log the details in the log tab */
	bool GetLogDetails() const;
	/** Set to true to log details in the log tab */
	void SetLogDetails( bool logdet );
	/** */
	CString GetLogFileName() const;
	/** */
	void SetLogFileName( CString logfile );
	/** */
	bool GetSendMessageOnActiveModeRequest() const;
	/** */
	void SetSendMessageOnActiveModeRequest( bool b );

	/** */
	CString GetLogPath() const { return sConfigPath; };
	/** */
	CString GetImagePath() const { return sConfigPath+DCGUI_IMAGE_PATH+DIRSEPARATOR; };
	/** */
	CString GetConfigPath() const { return sConfigPath; };
	/** */
	CString GetChatLogPath() const { return sConfigPath+DCGUI_CHATLOG_PATH+DIRSEPARATOR; };
	/** */
	CString GetPluginConfigPath() const { return sConfigPath+DCGUI_PLUGINCONFIG_PATH+DIRSEPARATOR; };
	/** */
	CString GetFileListPath() const { return sConfigPath+DCGUI_FILELIST_PATH+DIRSEPARATOR; };

	/** */
	int LoadDCHub();
	/** */
	int SaveDCHub();
	/** */
	int SaveDCPublicHub();
	/** */
	int SaveDCBookHub();
	/** */
	void ParseDCHubConfig( CXml * xml );
	/** */
	void ParseDCBookHubConfig( CXml * xml );
	/** */
	long GetBookmarkHubList( CList<DCConfigHubItem> * list );
	/** */
	CStringList<CString> * GetBookmarkHubServerList();
	/** */
	bool RemoveBookmarkHub( CString name, CString host, CString description );
	/** */
	bool AddBookmarkHub( CString name, CString host, CString description );
	/** */
	bool UpdateBookmarkHub( CString name, CString host, CString description );
	/**
	 * Move a bookmark in the ordered list.
	 * The bookmark at oldpos is moved to newpos and
	 * the bookmark at newpos is moved to oldpos.
	 * The bookmarks must be next to each other.
	 */
	bool MoveBookmarkHub( int oldpos, int newpos );
	/**
	 * Re-order all the bookmarks.
	 */
	bool ReorderBookmarkHubs( const std::vector<int> & newpositions );
	/** */
	bool SetBookmarkHubProfile( CString name, CString profile );
	/** */
	bool GetBookmarkHubProfile( CString name, CString host, DCConfigHubProfile * p );
	/** */
	long GetPublicHubList( CList<DCConfigHubItem> * list );
	/** */
	long GetPublicHubListSize() const;
	/** */
	CStringList<CString> * GetPublicHubServerList();
	/** */
	bool GetBookmarkHub( CString name, DCConfigHubItem * hubitem );
	/** */
	bool GetBookmarkHub( unsigned long id, DCConfigHubItem * hubitem );
	/** */
	bool GetPublicHub( CString name, DCConfigHubItem * hubitem );
	/** */
	bool GetPublicHub( unsigned long id, DCConfigHubItem * hubitem );
	/** */
	bool RemovePublicHub( CString name, CString host, CString description );
	/** */
	bool AddPublicHub( const CString & name, const CString & host, const CString & description, const ulonglong usercount, const CString country = CString(), const ulonglong shared = 0, const ulonglong minshare = 0, const CString extra = CString() );
	/** */
	void ClearPublicHubList();
	/** */
	long GetHubListUrlList( CList<DCConfigHubListUrl> * list );
	/** */
	void SetHubListUrlList( CList<DCConfigHubListUrl> * list );

	/** */
	bool AddHubProfile( DCConfigHubProfile * p );
	/** */
	bool DelHubProfile( CString name );
	/** */
	bool GetHubProfile( CString name, DCConfigHubProfile * p );
	/** */
	bool SaveHubProfile();
	/** */
	bool LoadHubProfile();
	/** */
	std::list<CString> * GetHubProfileNames();
	/** */
	int GetReloadHubListTime() const;
	/** */
	void SetReloadHubListTime( int i );
	/** */
	int GetRecreateShareListTime() const;
	/** */
	void SetRecreateShareListTime( int i );

	/** */
	bool GetHubListStoreLocal() const;
	/** */
	void SetHubListStoreLocal( bool b );
	/** */
	int GetReconnectCount() const;
	/** */
	void SetReconnectCount( int reconnectcount );
	/** */
	int GetReconnectTimeout() const;
	/** */
	void SetReconnectTimeout( int reconnecttimeout );
	/** */
	bool GetCheckPrivateAddressSpace() const;
	/** */
	void SetCheckPrivateAddressSpace( bool b );
	/** get allow private address space only */
	bool GetPrivateAddressSpaceOnly() const;
	/** set allow private address space only */
	void SetPrivateAddressSpaceOnly( bool b );

	/** */
	int GetTransferResendTimeout() const;
	/** */
	void SetTransferResendTimeout( int i );
	/** */
	int GetTransferResponseTimeout() const;
	/** */
	void SetTransferResponseTimeout( int i );

        /** */
	eCloseType GetHubOfflineTransferClose() const;
	/** set the reaction on leaving hub */
	void SetHubOfflineTransferClose( eCloseType ect );
	
	/** */
	bool GetForceMoveEnabled() const;
	/** */
	void SetForceMoveEnabled( bool forcemoveenabled );

	/** */
	ulonglong GetMaxUploadRate() const;
	/** */
	void SetMaxUploadRate( ulonglong n );
	/** */
	ulonglong GetMaxDownloadRate() const;
	/** */
	void SetMaxDownloadRate( ulonglong n );
	/** */
	bool GetDynamicUploadRate() const;
	/** */
	void SetDynamicUploadRate( bool b );
	
	/**
	 * Open extra slots if total upload rate is less than this.
	 * Rate is in bytes/second.
	 * 0 is special value to disable.
	 */
	ulonglong GetExtraSlotsRate() const;
	/**
	 * Set rate for open extra slots.
	 * Rate is in bytes/second.
	 * 0 is special value to disable.
	 */
	void SetExtraSlotsRate( ulonglong n );
	/**
	 * Get maximum number of extra slots to open when upload rate is slow.
	 * 0 will also disable this feature.
	 */
	int GetMaxExtraSlots() const;
	/**
	 * Set maximum number of extra slots to open when upload rate is slow.
	 * 0 will also disable this feature.
	 */
	void SetMaxExtraSlots( int n );

	/** */
	bool GetChatSendToOffline() const;
	/** */
	void SetChatSendToOffline( bool b );
	/** */
	bool GetChatRecvFromOffline() const;
	/** */
	void SetChatRecvFromOffline( bool b );

	/** */
	CString GetTransferCert() const;
	/** */
	void SetTransferCert( CString s );
	/** */
	CString GetTransferKey() const;
	/** */
	void SetTransferKey( CString s );
	/** Get old dclib specific ssl mode support */
	bool GetOldSSLSupport() const;
	/** Set old dclib specific ssl mode support */
	void SetOldSSLSupport( bool b );

	/** */
	bool GetDisableHashList() const;
	/** */
	void SetDisableHashList( bool b );
	
	/** */
	bool GetCompressedTransfers() const;
	/** */
	void SetCompressedTransfers( bool b );
	
	/** Get the time in seconds between auto searches */
	long GetAutoSearchInterval() const;
	/** Set the time, in seconds, between auto searches */
	void SetAutoSearchInterval( long seconds );
	
	/** Get the maximum size of file to give MiniSlots for */
	ulonglong GetSmallFileSize() const;
	/** Set the maximum size of file to give MiniSlots for */
	void SetSmallFileSize( ulonglong s );
	
	/** If true, use H:3/2/1, if false, use H:6 */
	bool GetUseExtendedHubCount() const;
	/** */
	void SetUseExtendedHubCount( bool b );
	
	/** If true, do not share files/folders starting with "."*/
	bool GetDontShareDotFiles() const;
	/** */
	void SetDontShareDotFiles( bool b );
	
	/** Get remote encoding setting */
	CString GetRemoteEncoding() const;
	/** Get remote encoding for the given hub, falling back to global setting */
	CString GetRemoteEncoding( CString hubname, CString hubhost );
	/** Get local operating system encoding*/
	CString GetLocalEncoding() const;
	/** Set remote encoding setting */
	void SetRemoteEncoding( CString name );
	
	/** If true, use IP provided by UserIP2 */
	bool GetUserIP2Enabled() const;
	/** Set if the IP provided by UserIP2 will be used */
	void SetUserIP2Enabled( bool b );
	
	/** Get ZPipe support enabled - currently ZPipe0 */
	bool GetZPipeEnabled() const;
	/** Set ZPipe support enabled - currently ZPipe0 */
	void SetZPipeEnabled( bool b );
	
	/** Get session starting value of total downloaded */
	ulonglong GetStartDownloaded() const;
	/** Get session starting value of total uploaded */
	ulonglong GetStartUploaded() const;
	/** Reset session totals to zero */
	void ResetSessionStats();
	
	/** Get read file buffer size */
	long GetReadFileBufferSize() const;
	/** Set read file buffer size */
	void SetReadFileBufferSize( long newsize );
	
	/** Get disable ADCGet when hash list is disabled */
	bool GetDisableADCGetWithoutTTH() const;
	/** Set disable ADCGet when hash list is disabled */
	void SetDisableADCGetWithoutTTH( bool b );
	/** Get disable XML list when hash list is disabled */
	bool GetDisableXMLListWithoutTTH() const;
	/** Set disable XML list when hash list is disabled */
	void SetDisableXMLListWithoutTTH( bool b );
	/** Get enable obsolete protocol extensions */
	bool GetEnableObsoleteExt() const;
	/** Set enable obsolete protocol extensions */
	void SetEnableObsoleteExt( bool b );
	
	/** Really a valknut setting, if true shows hub commands in the menus */
	bool GetEnableUserCommand() const;
	/** Really a valknut setting, if true shows hub commands in the menus */
	void SetEnableUserCommand( bool b );
	
	/** Get minimum segment size in bytes */
	ulonglong GetMinSegSize() const;
	/** Set minimum segment size in bytes */
	void SetMinSegSize( ulonglong bytes );
	
	/** */
	CString AliasToPath( CString file );

private:
	/** */
	CString sEMail;
	/** */
	CString sNick;
	/** */
	CString sSearchNick;
	/** */
	CString sAwayMessage;
	/** */
	CString sFloodOpKickMessage;
	/** */
	CString sDownloadFolder;
	/** */
	CString sDownloadFinishedFolder;
	/** */
	eClientMode eMode;
	/** */
	CString sDescription;
	/** */
	CString sAwayPrefix;
	/** */
	bool bDescriptionTag;
	/** */
	CString sSpeed;
	/** */
	int iTCPListenPort;
	/** Port for fully encrypted transfers */
	int iCryptoListenPort;
	/** */
	int iUDPListenPort;
	/** */
	CString m_sHost;
	/** */
	CString m_sHostCache;
	/** */
	CString m_sListenHost;
	/** */
	time_t m_tHostTimeout;
	/** */
	CString m_sTransferCert;
	/** */
	CString m_sTransferKey;
	/** */
	bool bAntiSpam;
	/** */
	bool bLogFile;
	/** */
	bool bLogFinishedDownloads;
	/** */
	bool bLogFinishedUploads;
	/** */
	bool bLogDetails;
	/** */
	bool bCheckPrivateAddressSpace;
	/** */
	bool bPrivateAddressSpaceOnly;
	/** */
	CString sLogFile;
	/** */
	bool m_bExternalIP;
	/** */
	bool bSendMessageOnActiveModeRequest;
	/** */
	int iMaxUpload;
	/** */
	int iUserUploadSlots;
	/** */
	ulonglong lMaxDownloadRate;
	/** */
	ulonglong lMaxUploadRate;
	/** */
	int iReconnectCount;
	/** */
	int iReconnectTimeout;
	/** */
	int iTransferResendTimeout;
	/** */
	int iTransferResponseTimeout;
	/** */
	eCloseType eHubOfflineTransferClose;
	/** */
	bool bForceMoveEnabled;
	/** */
	bool bHubListStoreLocal;
	/** */
	eUserAwayMode eAwayMode;
	/** */
	CList<DCConfigShareFolder> SharedFolders;
	/** */
	bool bAutoRecreateShareList;
	/** Bookmark hubs, in order */
	OrderedBookmarkMap * pOrderedBookmarks;
	/** */
	CStringList<DCConfigHubItem> * pBookmarkHubList;
	/** */
	CStringList<CString> * pBookmarkHubList_HostIndex;
	/** */
	CStringList<DCConfigHubItem> * pPublicHubList;
	/** */
	CStringList<CString> * pPublicHubList_HostIndex;
	/** */
	HubProfileMap * pHubProfileMap;
	/** */
	CList<DCConfigHubListUrl> pHubListUrlList;
	/** */
	int iDownloadQueueTime;
	/** Plugin folder */
	CString m_sDCLibPluginPath;
	/** */
	bool bDynamicUploadRate;
	/** */
	bool bChatSendToOffline;
	/** */	
	bool bChatRecvFromOffline;
	/** */
	bool bTransferAutoSearch;

	/** */
	unsigned long m_nPublicHubID;
	/** */
	unsigned long m_nBookHubID;
	/** */
	int m_nReloadHubListTime;
	/** */
	int m_nRecreateShareListTime;
	/** */
	bool m_bCreateFile;
	/** */
	bool m_bDisableHashList;
	/** True if compressed transfers are enabled */
	bool m_bCompressedTransfers;
	/** The time in seconds between auto searches */
	long m_nAutoSearchInterval;
	/** The maximum size of files for which MiniSlots are granted */
	ulonglong m_nSmallFileSize;
	/** If true, use H:3/2/1, if false, use H:6 */
	bool m_bUseExtendedHubCount;
	/** If true, don't share files and folders starting with "." */
	bool m_bDontShareDotFiles;
	/** The remote text encoding */
	CString m_sRemoteEncoding;
	/** The local system encoding */
	CString m_sLocalEncoding;
	/** If true, use the IP address provided by the hub via UserIP2 */
	bool m_bUserIP2Enabled;
	/** If true, enable ZPipe support - currently ZPipe0 */
	bool m_bEnableZPipe;
	/** Session starting value of total downloaded */
	ulonglong m_nStartDownloaded;
	/** Session starting value of total uploaded */
	ulonglong m_nStartUploaded;
	/** Size of each file read when uploading */
	long m_nReadFileBufferSize;
	/** Disable ADCGet support when hash list is disabled */
	bool m_bDisableADCGetWithoutTTH;
	/** Disable XML filelist support when hash list is disabled */
	bool m_bDisableXMLListWithoutTTH;
	/** Enable obsolete CHUNK BZList GetZBlock SSL */
	bool m_bEnableObsoleteExt;
	/** If true valknut will show hub commands in the menus */
	bool m_bEnableUserCommand;
	/** Enable support for dclib <= 0.3.18 ssl transfers */
	bool m_bOldSSLSupport;
	/** Miminum segment size in bytes */
	ulonglong m_nMinSegSize;
	
	/** When uploading below this total rate open extra slots */
	ulonglong m_nExtraSlotsRate;
	/** Open maximum of this many extra slots */
	int m_nMaxExtraSlots;
	
protected:
	/** */
	CMutex m_Mutex;
	/** */
	CString sConfigPath;
};

/** inline functions */

/** */
inline void CConfig::SetEMail( CString email )
{ m_Mutex.Lock(); sEMail = email; m_Mutex.UnLock(); }
/** */
inline CString CConfig::GetNick() const
{ return sNick.Replace(' ',"\xa0"); }
/** */
inline void CConfig::SetNick( CString nick )
{ m_Mutex.Lock(); sNick = nick; m_Mutex.UnLock(); }
/** */
inline CString CConfig::GetSearchNick() const
{ return sSearchNick.Replace(' ',"\xa0"); }
/** */
inline void CConfig::SetSearchNick( CString searchnick )
{ m_Mutex.Lock(); sSearchNick = searchnick; m_Mutex.UnLock(); }
/** */
inline CString CConfig::GetAwayMessage() const
{ return sAwayMessage; }
/** */
inline void CConfig::SetAwayMessage( CString s )
{ m_Mutex.Lock(); sAwayMessage = s; m_Mutex.UnLock(); }
/** */
inline CString CConfig::GetFloodOpKickMessage() const
{ return sFloodOpKickMessage; }
/** */
inline void CConfig::SetFloodOpKickMessage( CString s )
{ m_Mutex.Lock(); sFloodOpKickMessage = s; m_Mutex.UnLock(); }
/** */
inline CString CConfig::GetDownloadFolder() const
{ return sDownloadFolder; }
/** */
inline void CConfig::SetDownloadFolder( CString downloadfolder )
{ m_Mutex.Lock(); sDownloadFolder = downloadfolder; m_Mutex.UnLock(); }
/** */
inline CString CConfig::GetDownloadFinishedFolder() const
{ return sDownloadFinishedFolder; }
/** */
inline void CConfig::SetDownloadFinishedFolder( CString s )
{ m_Mutex.Lock(); sDownloadFinishedFolder = s; m_Mutex.UnLock(); }
/** */
inline void CConfig::SetMode( eClientMode mode )
{ m_Mutex.Lock(); eMode = mode; m_Mutex.UnLock(); }
/** */
inline void CConfig::SetDescription( CString description )
{ m_Mutex.Lock(); sDescription = description; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetDescriptionTag() const
{ return bDescriptionTag; }
/** */
inline void CConfig::SetDescriptionTag( bool b )
{ m_Mutex.Lock(); bDescriptionTag = b; m_Mutex.UnLock(); }
/** */
inline CString CConfig::GetSpeed() const
{ return sSpeed; }
/** */
inline void CConfig::SetSpeed( CString speed )
{ m_Mutex.Lock(); sSpeed = speed; m_Mutex.UnLock(); }
/** */
inline int CConfig::GetTCPListenPort() const
{ return iTCPListenPort; }
/** */
inline void CConfig::SetTCPListenPort( int port )
{ m_Mutex.Lock(); iTCPListenPort = port; m_Mutex.UnLock(); }
/** */
inline int CConfig::GetCryptoListenPort() const
{ return iCryptoListenPort; }
/** */
inline void CConfig::SetCryptoListenPort( int port )
{ m_Mutex.Lock(); iCryptoListenPort = port; m_Mutex.UnLock(); }
/** */
inline int CConfig::GetUDPListenPort() const
{ return iUDPListenPort; }
/** */
inline void CConfig::SetUDPListenPort( int port )
{ m_Mutex.Lock(); iUDPListenPort = port; m_Mutex.UnLock(); }
/** */
inline CString CConfig::GetHost() const
{ return m_sHost; }
/** */
inline CString CConfig::GetListenHost() const
{ return m_sListenHost; }
/** */
inline void CConfig::SetListenHost( CString host )
{ m_Mutex.Lock(); m_sListenHost = host; m_sHostCache = ""; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetAntiSpam() const
{ return bAntiSpam; }
/** */
inline void CConfig::SetAntiSpam( bool antispam )
{ m_Mutex.Lock(); bAntiSpam = antispam; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetExternalIP() const
{ return m_bExternalIP; }
/** */
inline void CConfig::SetExternalIP( bool b )
{ m_Mutex.Lock(); m_bExternalIP = b; m_Mutex.UnLock(); }
/** */
inline int CConfig::GetMaxUpload() const
{ return iMaxUpload; }
/** */
inline void CConfig::SetMaxUpload( int maxupload )
{ m_Mutex.Lock(); iMaxUpload = maxupload; m_Mutex.UnLock(); }
/** */
inline int CConfig::GetUserUploadSlots() const
{ return iUserUploadSlots; }
/** */
inline void CConfig::SetUserUploadSlots( int i )
{ m_Mutex.Lock(); iUserUploadSlots = i; m_Mutex.UnLock(); }
/** */
inline ulonglong CConfig::GetMaxUploadRate() const
{ if ( lMaxUploadRate < 512 ) return 0; return lMaxUploadRate; }
/** */
inline void CConfig::SetMaxUploadRate( ulonglong n )
{ m_Mutex.Lock(); if ( n < 512 ) n = 0; lMaxUploadRate = n; m_Mutex.UnLock(); }
/** */
inline ulonglong CConfig::GetMaxDownloadRate() const
{ return lMaxDownloadRate; }
/** */
inline void CConfig::SetMaxDownloadRate( ulonglong n )
{ m_Mutex.Lock(); lMaxDownloadRate = n; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetDynamicUploadRate() const
{ return bDynamicUploadRate; }
/** */
inline void CConfig::SetDynamicUploadRate( bool b )
{ m_Mutex.Lock(); bDynamicUploadRate = b; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetHubListStoreLocal() const
{ return bHubListStoreLocal; }
/** */
inline void CConfig::SetHubListStoreLocal( bool b )
{ m_Mutex.Lock(); bHubListStoreLocal = b; m_Mutex.UnLock(); }
/** */
inline int CConfig::GetReconnectCount() const
{ return iReconnectCount; }
/** */
inline void CConfig::SetReconnectCount( int reconnectcount )
{ m_Mutex.Lock(); iReconnectCount = reconnectcount; m_Mutex.UnLock(); }
/** */
inline int CConfig::GetReconnectTimeout() const
{ return iReconnectTimeout; }
/** */
inline void CConfig::SetReconnectTimeout( int reconnecttimeout )
{ m_Mutex.Lock(); iReconnectTimeout = reconnecttimeout; m_Mutex.UnLock(); }
/** */
inline int CConfig::GetTransferResendTimeout() const
{ return iTransferResendTimeout; }
/** */
inline void CConfig::SetTransferResendTimeout( int i )
{ m_Mutex.Lock(); iTransferResendTimeout = i; m_Mutex.UnLock(); }
/** */
inline int CConfig::GetTransferResponseTimeout() const
{ return iTransferResponseTimeout; }
/** */
inline void CConfig::SetTransferResponseTimeout( int i )
{ m_Mutex.Lock(); iTransferResponseTimeout = i; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetForceMoveEnabled() const
{ return bForceMoveEnabled; }
/** */
inline void CConfig::SetForceMoveEnabled( bool forcemoveenabled )
{ m_Mutex.Lock(); bForceMoveEnabled = forcemoveenabled; m_Mutex.UnLock(); }
/** */
inline eUserAwayMode CConfig::GetAwayMode() const
{ return eAwayMode; }
/** */
inline int CConfig::GetDownloadQueueTime() const
{ return iDownloadQueueTime; }
/** */
inline void CConfig::SetDownloadQueueTime( int i )
{ m_Mutex.Lock(); iDownloadQueueTime = i; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetSendMessageOnActiveModeRequest() const
{ return bSendMessageOnActiveModeRequest; }
/** */
inline void CConfig::SetSendMessageOnActiveModeRequest( bool b )
{ m_Mutex.Lock(); bSendMessageOnActiveModeRequest = b; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetLogFile() const
{ return bLogFile; }
/** */
inline void CConfig::SetLogFile( bool logfile )
{ m_Mutex.Lock(); bLogFile = logfile; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetLogFinishedDownloads() const
{ return bLogFinishedDownloads; }
/** */
inline void CConfig::SetLogFinishedDownloads( bool logdl )
{ m_Mutex.Lock(); bLogFinishedDownloads = logdl; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetLogFinishedUploads() const
{ return bLogFinishedUploads; }
/** */
inline void CConfig::SetLogFinishedUploads( bool logul )
{ m_Mutex.Lock(); bLogFinishedUploads = logul; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetLogDetails() const
{ return bLogDetails; }
/** */
inline void CConfig::SetLogDetails( bool logdet )
{ m_Mutex.Lock(); bLogDetails = logdet; m_Mutex.UnLock(); }
/** */
inline CString CConfig::GetLogFileName() const
{ return sLogFile; }
/** */
inline void CConfig::SetLogFileName( CString logfile )
{ m_Mutex.Lock(); sLogFile = logfile; m_Mutex.UnLock(); }

/** */
inline bool CConfig::GetCheckPrivateAddressSpace() const
{ return bCheckPrivateAddressSpace; }
/** */
inline void CConfig::SetCheckPrivateAddressSpace( bool b )
{ m_Mutex.Lock(); bCheckPrivateAddressSpace = b; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetPrivateAddressSpaceOnly() const
{ return bPrivateAddressSpaceOnly; }
/** */
inline void CConfig::SetPrivateAddressSpaceOnly( bool b )
{ m_Mutex.Lock(); bPrivateAddressSpaceOnly = b; m_Mutex.UnLock(); }
/** */
inline CString CConfig::GetDCLibPluginPath() const
{ return m_sDCLibPluginPath; }
/** */
inline void CConfig::SetDCLibPluginPath( CString s )
{ m_Mutex.Lock(); m_sDCLibPluginPath = s; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetChatSendToOffline() const
{ return bChatSendToOffline; }
/** */
inline void CConfig::SetChatSendToOffline( bool b )
{ m_Mutex.Lock(); bChatSendToOffline = b; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetChatRecvFromOffline() const
{ return bChatRecvFromOffline; }
/** */
inline void CConfig::SetChatRecvFromOffline( bool b )
{ m_Mutex.Lock(); bChatRecvFromOffline = b; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetAutoRecreateShareList() const
{ return bAutoRecreateShareList; }
/** */
inline void CConfig::SetAutoRecreateShareList( bool b )
{ m_Mutex.Lock(); bAutoRecreateShareList = b; m_Mutex.UnLock(); }
/** */
inline CString CConfig::GetTransferCert() const
{ return m_sTransferCert; }
/** */
inline void CConfig::SetTransferCert( CString s )
{ m_Mutex.Lock(); m_sTransferCert = s; m_Mutex.UnLock(); }
/** */
inline CString CConfig::GetTransferKey() const
{ return m_sTransferKey; }
/** */
inline void CConfig::SetTransferKey( CString s )
{ m_Mutex.Lock(); m_sTransferKey = s; m_Mutex.UnLock(); }
/** */
inline eCloseType CConfig::GetHubOfflineTransferClose() const
{ return eHubOfflineTransferClose; }
/** */
inline void CConfig::SetHubOfflineTransferClose( eCloseType ect )
{ m_Mutex.Lock(); eHubOfflineTransferClose = ect; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetTransferAutoSearch() const
{ return bTransferAutoSearch; }
/** */
inline void CConfig::SetTransferAutoSearch( bool b )
{ m_Mutex.Lock(); bTransferAutoSearch = b; m_Mutex.UnLock(); }
/** */
inline int CConfig::GetReloadHubListTime() const
{ return m_nReloadHubListTime; }
/** */
inline void CConfig::SetReloadHubListTime( int i )
{ m_Mutex.Lock(); m_nReloadHubListTime = i; m_Mutex.UnLock(); }
/** */
inline int CConfig::GetRecreateShareListTime() const
{ return m_nRecreateShareListTime; }
/** */
inline void CConfig::SetRecreateShareListTime( int i )
{ m_Mutex.Lock(); m_nRecreateShareListTime = i; m_Mutex.UnLock(); }
/** */
inline CString CConfig::GetAwayPrefix() const
{ return sAwayPrefix; }
/** */
inline void CConfig::SetAwayPrefix( CString s )
{ m_Mutex.Lock(); sAwayPrefix = s; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetCreateFile() const
{ return m_bCreateFile; }
/** */
inline void CConfig::SetCreateFile( bool b )
{ m_Mutex.Lock(); m_bCreateFile = b; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetDisableHashList() const
{ return m_bDisableHashList; }
/** */
inline void CConfig::SetDisableHashList( bool b )
{ m_Mutex.Lock(); m_bDisableHashList = b; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetCompressedTransfers() const
{ return m_bCompressedTransfers; }
/** */
inline void CConfig::SetCompressedTransfers( bool b )
{ m_Mutex.Lock(); m_bCompressedTransfers = b; m_Mutex.UnLock(); }
/** */
inline long CConfig::GetAutoSearchInterval() const
{ return m_nAutoSearchInterval; }
/** */
inline void CConfig::SetAutoSearchInterval( long seconds )
{ m_Mutex.Lock(); m_nAutoSearchInterval = seconds; m_Mutex.UnLock(); }
/** */
inline ulonglong CConfig::GetSmallFileSize() const
{ return m_nSmallFileSize; }
/** */
inline void CConfig::SetSmallFileSize( ulonglong s )
{ m_Mutex.Lock(); m_nSmallFileSize = s; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetUseExtendedHubCount() const
{ return m_bUseExtendedHubCount; }
/** */
inline void CConfig::SetUseExtendedHubCount( bool b )
{ m_Mutex.Lock(); m_bUseExtendedHubCount = b; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetDontShareDotFiles() const
{ return m_bDontShareDotFiles; }
/** */
inline void CConfig::SetDontShareDotFiles( bool b )
{ m_Mutex.Lock(); m_bDontShareDotFiles = b; m_Mutex.UnLock(); }
/** */
inline CString CConfig::GetRemoteEncoding() const
{ return m_sRemoteEncoding; }
/** */
inline void CConfig::SetRemoteEncoding( CString name )
{ m_Mutex.Lock(); m_sRemoteEncoding = name; m_Mutex.UnLock(); }
/** */
inline CString CConfig::GetLocalEncoding() const
{ return m_sLocalEncoding; }
/** */
inline bool CConfig::GetUserIP2Enabled() const
{ return m_bUserIP2Enabled; }
/** */
inline void CConfig::SetUserIP2Enabled( bool b )
{ m_Mutex.Lock(); m_bUserIP2Enabled = b; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetZPipeEnabled() const
{ return m_bEnableZPipe; }
/** */
inline void CConfig::SetZPipeEnabled( bool b )
{ m_Mutex.Lock(); m_bEnableZPipe = b; m_Mutex.UnLock(); }
/** */
inline ulonglong CConfig::GetStartDownloaded() const
{ return m_nStartDownloaded; }
/** */
inline ulonglong CConfig::GetStartUploaded() const
{ return m_nStartUploaded; }
/** */
inline void CConfig::ResetSessionStats()
{ m_Mutex.Lock(); m_nStartDownloaded = 0; m_nStartUploaded = 0; m_Mutex.UnLock(); }
/** */
inline long CConfig::GetReadFileBufferSize() const
{ return m_nReadFileBufferSize; }
/** */
inline void CConfig::SetReadFileBufferSize( long newsize )
{ m_Mutex.Lock(); m_nReadFileBufferSize = newsize; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetDisableADCGetWithoutTTH() const
{ return m_bDisableADCGetWithoutTTH; }
/** */
inline void CConfig::SetDisableADCGetWithoutTTH( bool b )
{ m_Mutex.Lock(); m_bDisableADCGetWithoutTTH = b; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetDisableXMLListWithoutTTH() const
{ return m_bDisableXMLListWithoutTTH; }
/** */
inline void CConfig::SetDisableXMLListWithoutTTH( bool b )
{ m_Mutex.Lock(); m_bDisableXMLListWithoutTTH = b; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetEnableObsoleteExt() const
{ return m_bEnableObsoleteExt; }
/** */
inline void CConfig::SetEnableObsoleteExt( bool b )
{ m_Mutex.Lock(); m_bEnableObsoleteExt = b; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetEnableUserCommand() const
{ return m_bEnableUserCommand; }
/** */
inline void CConfig::SetEnableUserCommand( bool b )
{ m_Mutex.Lock(); m_bEnableUserCommand = b; m_Mutex.UnLock(); }
/** */
inline bool CConfig::GetOldSSLSupport() const
{ return m_bOldSSLSupport; }
/** */
inline void CConfig::SetOldSSLSupport( bool b )
{ m_Mutex.Lock(); m_bOldSSLSupport = b; m_Mutex.UnLock(); }
/** */
inline ulonglong CConfig::GetMinSegSize() const
{ return m_nMinSegSize; }
/** */
inline void CConfig::SetMinSegSize( ulonglong bytes )
{ m_Mutex.Lock(); m_nMinSegSize = bytes; m_Mutex.UnLock(); }
/** */
inline ulonglong CConfig::GetExtraSlotsRate() const
{ return m_nExtraSlotsRate; }
/** */
inline void CConfig::SetExtraSlotsRate( ulonglong n )
{ m_Mutex.Lock(); m_nExtraSlotsRate = n; m_Mutex.UnLock(); }
/** */
inline int CConfig::GetMaxExtraSlots() const
{ return m_nMaxExtraSlots; }
/** */
inline void CConfig::SetMaxExtraSlots( int n )
{ m_Mutex.Lock(); m_nMaxExtraSlots = n; m_Mutex.UnLock(); }
#endif
