/*
 * Drizzle Client & Protocol Library
 *
 * Copyright (C) 2008 Eric Day (eday@oddments.org)
 * All rights reserved.
 *
 * Use and distribution licensed under the BSD license.  See
 * the COPYING file in this directory for full text.
 */

/**
 * @file
 * @brief Connection declarations
 */

#ifndef __DRIZZLE_CONN_H
#define __DRIZZLE_CONN_H

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @addtogroup drizzle_con Connection Handling
 * @ingroup drizzle_client
 * @ingroup drizzle_server
 * These functions provide the core functionality for connection handling.
 * @{
 */

/**
 * Initialize a connection structure.
 */
DRIZZLE_API
drizzle_con_st *drizzle_con_create(drizzle_st *drizzle, drizzle_con_st *con);

/**
 * Clone a connection structure.
 */
DRIZZLE_API
drizzle_con_st *drizzle_con_clone(drizzle_st *drizzle, drizzle_con_st *con,
                                  drizzle_con_st *from);

/**
 * Free a connection structure.
 */
DRIZZLE_API
void drizzle_con_free(drizzle_con_st *con);

/**
 * Get file descriptor for connection.
 */
DRIZZLE_API
int drizzle_con_fd(drizzle_con_st *con);

/**
 * Use given file descriptor for connction.
 */
DRIZZLE_API
drizzle_return_t drizzle_con_set_fd(drizzle_con_st *con, int fd);

/**
 * Close a connection.
 */
DRIZZLE_API
void drizzle_con_close(drizzle_con_st *con);

/**
 * Wait for I/O on connections.
 */
DRIZZLE_API
drizzle_return_t drizzle_con_wait(drizzle_st *drizzle);

/**
 * Set events to be watched for a connection.
 */
drizzle_return_t drizzle_con_set_events(drizzle_con_st *con, short events);

/**
 * Set events that are ready for a connection. This is used with the external
 * event callbacks.
 */
void drizzle_con_set_revents(drizzle_con_st *con, short revents);

/**
 * Get next connection that is ready for I/O.
 */
DRIZZLE_API
drizzle_con_st *drizzle_con_ready(drizzle_st *drizzle);

/**
 * Get the drizzle_st struct that the connection belongs to.
 */
DRIZZLE_API
drizzle_st *drizzle_con_drizzle(drizzle_con_st *con);

/**
 * Return an error string for last library error encountered.
 */
DRIZZLE_API
const char *drizzle_con_error(drizzle_con_st *con);

/**
 * Value of errno in the case of a DRIZZLE_RETURN_ERRNO return value.
 */
DRIZZLE_API
int drizzle_con_errno(drizzle_con_st *con);

/**
 * Get server defined error code for the last result read.
 */
DRIZZLE_API
uint16_t drizzle_con_error_code(drizzle_con_st *con);

/**
 * Get SQL state code for the last result read.
 */
DRIZZLE_API
const char *drizzle_con_sqlstate(drizzle_con_st *con);

/**
 * Get options for a connection.
 */
DRIZZLE_API
drizzle_con_options_t drizzle_con_options(drizzle_con_st *con);

/**
 * Set options for a connection.
 */
DRIZZLE_API
void drizzle_con_set_options(drizzle_con_st *con,
                             drizzle_con_options_t options);

/**
 * Add options for a connection.
 */
DRIZZLE_API
void drizzle_con_add_options(drizzle_con_st *con,
                             drizzle_con_options_t options);

/**
 * Remove options for a connection.
 */
DRIZZLE_API
void drizzle_con_remove_options(drizzle_con_st *con,
                                drizzle_con_options_t options);

/**
 * Get TCP host for a connection.
 */
DRIZZLE_API
const char *drizzle_con_host(drizzle_con_st *con);

/**
 * Get TCP port for a connection.
 */
DRIZZLE_API
in_port_t drizzle_con_port(drizzle_con_st *con);

/**
 * Set TCP host and port for a connection.
 */
DRIZZLE_API
void drizzle_con_set_tcp(drizzle_con_st *con, const char *host, in_port_t port);

/**
 * Get unix domain socket for a connection.
 */
DRIZZLE_API
const char *drizzle_con_uds(drizzle_con_st *con);

/**
 * Set unix domain socket for a connection.
 */
DRIZZLE_API
void drizzle_con_set_uds(drizzle_con_st *con, const char *uds);

/**
 * Get username for a connection.
 */
DRIZZLE_API
const char *drizzle_con_user(drizzle_con_st *con);

/**
 * Get password for a connection.
 */
DRIZZLE_API
const char *drizzle_con_password(drizzle_con_st *con);

/**
 * Set username and password for a connection.
 */
DRIZZLE_API
void drizzle_con_set_auth(drizzle_con_st *con, const char *user,
                          const char *password);

/**
 * Get database for a connection.
 */
DRIZZLE_API
const char *drizzle_con_db(drizzle_con_st *con);

/**
 * Set database for a connection.
 */
DRIZZLE_API
void drizzle_con_set_db(drizzle_con_st *con, const char *db);

#if 0
DRIZZLE_API
const char *drizzle_con_ssl_cipher(drizzle_con_st *con);
DRIZZLE_API
drizzle_return_t drizzle_con_ssl(drizzle_con_st *con, char *key, char *cert,
                                char *ca_file, char *ca_path, char *cipher);
#endif

/**
 * Get application data pointer for a connection.
 */
DRIZZLE_API
void *drizzle_con_data(drizzle_con_st *con);

/**
 * Set application data pointer for a connection.
 */
DRIZZLE_API
void drizzle_con_set_data(drizzle_con_st *con, const void *data);

/**
 * Set callback function when the data pointer should be freed.
 */
DRIZZLE_API
void drizzle_con_set_data_free(drizzle_con_st *con,
                               drizzle_con_data_free_fn *free_fn);

/**
 * Get protocol version for a connection.
 */
DRIZZLE_API
uint8_t drizzle_con_protocol_version(drizzle_con_st *con);

/**
 * Get server version string for a connection.
 */
DRIZZLE_API
const char *drizzle_con_server_version(drizzle_con_st *con);

/**
 * Get server version number for a connection.
 */
DRIZZLE_API
uint32_t drizzle_con_server_version_number(drizzle_con_st *con);

/**
 * Get thread ID for a connection.
 */
DRIZZLE_API
uint32_t drizzle_con_thread_id(drizzle_con_st *con);

/**
 * Get scramble buffer for a connection.
 */
DRIZZLE_API
const uint8_t *drizzle_con_scramble(drizzle_con_st *con);

/**
 * Get capabilities for a connection.
 */
DRIZZLE_API
drizzle_capabilities_t drizzle_con_capabilities(drizzle_con_st *con);

/**
 * Get charset for a connection.
 */
DRIZZLE_API
drizzle_charset_t drizzle_con_charset(drizzle_con_st *con);

/**
 * Get status for a connection.
 */
DRIZZLE_API
drizzle_con_status_t drizzle_con_status(drizzle_con_st *con);

/**
 * Get max packet size for a connection.
 */
DRIZZLE_API
uint32_t drizzle_con_max_packet_size(drizzle_con_st *con);

/** @} */

#ifdef __cplusplus
}
#endif

#endif /* __DRIZZLE_CONN_H */
