/*
 * Drizzle Client & Protocol Library
 *
 * Copyright (C) 2008 Eric Day (eday@oddments.org)
 * All rights reserved.
 *
 * Use and distribution licensed under the BSD license.  See
 * the COPYING file in this directory for full text.
 */

/**
 * @file
 * @brief Drizzle core definitions
 */

#include "common.h"

/*
 * Public definitions
 */

const char *drizzle_version(void)
{
  return PACKAGE_VERSION;
}

drizzle_st *drizzle_create(drizzle_st *drizzle)
{
  if (drizzle == NULL)
  {
    drizzle= malloc(sizeof(drizzle_st));
    if (drizzle == NULL)
      return NULL;

    memset(drizzle, 0, sizeof(drizzle_st));
    drizzle->options|= DRIZZLE_ALLOCATED;
  }
  else
    memset(drizzle, 0, sizeof(drizzle_st));

  return drizzle;
}

drizzle_st *drizzle_clone(drizzle_st *drizzle, drizzle_st *from)
{
  drizzle_con_st *con;

  drizzle= drizzle_create(drizzle);
  if (drizzle == NULL)
    return NULL;

  drizzle->options|= (from->options & ~(uint32_t)DRIZZLE_ALLOCATED);

  for (con= from->con_list; con != NULL; con= con->next)
  {
    if (drizzle_con_clone(drizzle, NULL, con) == NULL)
    {
      drizzle_free(drizzle);
      return NULL;
    }
  }

  return drizzle;
}

void drizzle_free(drizzle_st *drizzle)
{
  drizzle_con_st *con;
  drizzle_query_st *query;

  for (con= drizzle->con_list; con != NULL; con= drizzle->con_list)
    drizzle_con_free(con);

  for (query= drizzle->query_list; query != NULL; query= drizzle->query_list)
    drizzle_query_free(query);

  if (drizzle->pfds != NULL)
    free(drizzle->pfds);

  if (drizzle->options & DRIZZLE_ALLOCATED)
    free(drizzle);
}

const char *drizzle_error(drizzle_st *drizzle)
{
  return (const char *)(drizzle->last_error);
}

int drizzle_errno(drizzle_st *drizzle)
{
  return drizzle->last_errno;
}

uint16_t drizzle_error_code(drizzle_st *drizzle)
{
  return drizzle->error_code;
}
 
const char *drizzle_sqlstate(drizzle_st *drizzle)
{
  return drizzle->sqlstate;
}

drizzle_options_t drizzle_options(drizzle_st *drizzle)
{
  return drizzle->options;
}

void drizzle_set_options(drizzle_st *drizzle, drizzle_options_t options)
{
  drizzle->options= options;
}

void drizzle_add_options(drizzle_st *drizzle, drizzle_options_t options)
{
  drizzle->options|= options;
}

void drizzle_remove_options(drizzle_st *drizzle, drizzle_options_t options)
{
  drizzle->options&= ~options;
}

void drizzle_set_event_watch(drizzle_st *drizzle,
                             drizzle_event_watch_fn *event_watch,
                             void *event_watch_arg)
{
  drizzle->event_watch= event_watch;
  drizzle->event_watch_arg= event_watch_arg;
}
