/*
 * Drizzle Client & Protocol Library
 *
 * Copyright (C) 2008 Eric Day (eday@oddments.org)
 * All rights reserved.
 *
 * Use and distribution licensed under the BSD license.  See
 * the COPYING file in this directory for full text.
 */

/**
 * @file
 * @brief Private defines, typedefs, and enums
 */

#ifndef __DRIZZLE_PRIVATE_H
#define __DRIZZLE_PRIVATE_H

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @addtogroup drizzle_constants Drizzle Constants
 * @{
 */

/**
 * Macro to set error string.
 */
#define DRIZZLE_ERROR_SET(__drizzle, __function, ...) { \
  snprintf((__drizzle)->last_error, DRIZZLE_MAX_ERROR_SIZE, \
           __function ":" __VA_ARGS__); }

/* Protocol unpacking macros. */
#define DRIZZLE_GET_BYTE2(__buffer) (uint16_t)((__buffer)[0] | \
  ((__buffer)[1] << 8))
#define DRIZZLE_GET_BYTE3(__buffer) (uint32_t)((__buffer)[0] | \
  ((__buffer)[1] << 8) | ((__buffer)[2] << 16))
#define DRIZZLE_GET_BYTE4(__buffer) (uint32_t)((__buffer)[0] | \
  ((__buffer)[1] << 8) | ((__buffer)[2] << 16) | ((__buffer)[3] << 24))
#define DRIZZLE_GET_BYTE8(__buffer) ((uint64_t)(__buffer)[0] | \
  ((uint64_t)(__buffer)[1] << 8) | ((uint64_t)(__buffer)[2] << 16) | \
  ((uint64_t)(__buffer)[3] << 24) | ((uint64_t)(__buffer)[4] << 32) | \
  ((uint64_t)(__buffer)[5] << 40) | ((uint64_t)(__buffer)[6] << 48) | \
  ((uint64_t)(__buffer)[7] << 56))

/* Protocol packing macros. */
#define DRIZZLE_SET_BYTE2(__buffer, __int) { \
  (__buffer)[0]= (uint8_t)((__int) & 0xFF); \
  (__buffer)[1]= (uint8_t)(((__int) >> 8) & 0xFF); }
#define DRIZZLE_SET_BYTE3(__buffer, __int) { \
  (__buffer)[0]= (uint8_t)((__int) & 0xFF); \
  (__buffer)[1]= (uint8_t)(((__int) >> 8) & 0xFF); \
  (__buffer)[2]= (uint8_t)(((__int) >> 16) & 0xFF); }
#define DRIZZLE_SET_BYTE4(__buffer, __int) { \
  (__buffer)[0]= (uint8_t)((__int) & 0xFF); \
  (__buffer)[1]= (uint8_t)(((__int) >> 8) & 0xFF); \
  (__buffer)[2]= (uint8_t)(((__int) >> 16) & 0xFF); \
  (__buffer)[3]= (uint8_t)(((__int) >> 24) & 0xFF); }
#define DRIZZLE_SET_BYTE8(__buffer, __int) { \
  (__buffer)[0]= (uint8_t)((__int) & 0xFF); \
  (__buffer)[1]= (uint8_t)(((__int) >> 8) & 0xFF); \
  (__buffer)[2]= (uint8_t)(((__int) >> 16) & 0xFF); \
  (__buffer)[3]= (uint8_t)(((__int) >> 24) & 0xFF); \
  (__buffer)[4]= (uint8_t)(((__int) >> 32) & 0xFF); \
  (__buffer)[5]= (uint8_t)(((__int) >> 40) & 0xFF); \
  (__buffer)[6]= (uint8_t)(((__int) >> 48) & 0xFF); \
  (__buffer)[7]= (uint8_t)(((__int) >> 56) & 0xFF); }

/* Multi-byte character macros. */
#define DRIZZLE_MB_CHAR(__c) (((__c) & 0x80) != 0)
#define DRIZZLE_MB_LENGTH(__c) \
  ((uint32_t)(__c) <= 0x7f ? 1 : \
  ((uint32_t)(__c) <= 0x7ff ? 2 : \
  ((uint32_t)(__c) <= 0xd7ff ? 3 : \
  ((uint32_t)(__c) <= 0xdfff || (uint32_t)(__c) > 0x10ffff ? 0 : \
  ((uint32_t)(__c) <= 0xffff ? 3 : 4)))))

/* State stack macros. */
#define DRIZZLE_STATE_NONE(__con) ((__con)->state_current == 0)
#define DRIZZLE_STATE_PUSH(__con, __function) { \
  assert((__con)->state_current < DRIZZLE_STATE_STACK_SIZE); \
  (__con)->state_stack[(__con)->state_current]= __function; \
  (__con)->state_current++; }
#define DRIZZLE_STATE_POP(__con) (__con)->state_current--;
#define DRIZZLE_STATE_RESET(__con) (__con)->state_current= 0;

/* Debugging macros. */
/* #define VERBOSE_DEBUG */
#ifdef VERBOSE_DEBUG
#define PDEBUG(__function, __format, ...) fprintf(stderr, \
  "%20s:%-4d %30s" __format "\n", __FILE__, __LINE__, __function, __VA_ARGS__);
#else /* !VERBOSE_DEBUG */
#define PDEBUG(__function, __format, ...)
#endif /* VERBOSE_DEBUG */

/** @} */

#ifdef __cplusplus
}
#endif

#endif /* __DRIZZLE_PRIVATE_H */
