/*
 * libexplain - Explain errno values returned by libc functions
 * Copyright (C) 2009 Peter Miller
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 3 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful,but
 * WITHOUT ANY WARRANTY; without even the implied warranty
 * ofMERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNULesser
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <libexplain/ac/linux/fs.h>
#include <libexplain/ac/sys/ioctl.h>

#include <libexplain/parse_bits.h>
#include <libexplain/sizeof.h>
#include <libexplain/string_buffer.h>

#include <ioctl-scan/report.h>
#include <ioctl-scan/scan/linux_fs.h>


#ifdef BLKROGET

static void
blkroget(int fildes)
{
    int             data;

    if (ioctl(fildes, BLKROGET, &data) >= 0)
    {
        report_bool("BLKROGET", "Read only", data);
    }
}

#endif

#ifdef BLKGETSIZE

static void
blkgetsize(int fildes)
{
    unsigned long   data;

    if (ioctl(fildes, BLKGETSIZE, &data) >= 0)
    {
        report_int("BLKGETSIZE", "Size", data, "* 512 bytes");
    }
}

#endif


#ifdef BLKRAGET

static void
blkraget(int fildes)
{
    long            data;

    if (ioctl(fildes, BLKRAGET, &data) >= 0)
    {
        report_int("BLKRAGET", "Read ahead", data, "* 512 bytes");
    }
}

#endif


#ifdef BLKFRAGET

static void
blkfraget(int fildes)
{
    int             data;

    if (ioctl(fildes, BLKFRAGET, &data) >= 0)
    {
        report_int("BLKFRAGET", "blkfraget", data, "* 512 bytes");
    }
}

#endif


#ifdef BLKSECTGET

static void
blksectget(int fildes)
{
    int             data;

    if (ioctl(fildes, BLKSECTGET, &data) >= 0)
    {
        report_ushort("BLKSECTGET", "Maximum sectors", data, 0);
    }
}

#endif

#ifdef BLKSSZGET

static void
blksszget(int fildes)
{
    int             data;

    if (ioctl(fildes, BLKSSZGET, &data) >= 0)
    {
        report_int("BLKSSZGET", "Hardware sector size", data, "bytes");
    }
}

#endif

#ifdef FIGETBSZ

static void
figetbsz(int fildes)
{
    int             data;

    if (ioctl(fildes, FIGETBSZ, &data) >= 0)
    {
        report_int("FIGETBSZ", "Block size", data, "bytes");
    }
}

#endif

#ifdef BLKELVGET

static void
blkelvget(int fildes)
{
    size_t          data;

    if (ioctl(fildes, BLKELVGET, &data) >= 0)
    {
        report_size_t("BLKELVGET", "blkelvget", data, 0);
    }
}

#endif

#ifdef BLKBSZGET

static void
blkbszget(int fildes)
{
    int             data;

    if (ioctl(fildes, BLKBSZGET, &data) >= 0)
    {
        report_int("BLKBSZGET", "Logical block size", data, "bytes");
    }
}

#endif

#ifdef BLKGETSIZE64

static void
blkgetsize64(int fildes)
{
    uint64_t        data;

    if (ioctl(fildes, BLKGETSIZE64, &data) >= 0)
    {
        report_uint64("BLKGETSIZE64", "Size", data, "bytes");
    }
}

#endif

#if defined(FS_IOC_GETFLAGS) || defined(FS_IOC32_GETFLAGS)

static void
explain_buffer_file_inode_flags(explain_string_buffer_t *sb, int value)
{
    static const explain_parse_bits_table_t table[] =
    {
        { "FS_SECRM_FL", FS_SECRM_FL },
        { "FS_UNRM_FL", FS_UNRM_FL },
        { "FS_COMPR_FL", FS_COMPR_FL },
        { "FS_SYNC_FL", FS_SYNC_FL },
        { "FS_IMMUTABLE_FL", FS_IMMUTABLE_FL },
        { "FS_APPEND_FL", FS_APPEND_FL },
        { "FS_NODUMP_FL", FS_NODUMP_FL },
        { "FS_NOATIME_FL", FS_NOATIME_FL },
        { "FS_DIRTY_FL", FS_DIRTY_FL },
        { "FS_COMPRBLK_FL", FS_COMPRBLK_FL },
        { "FS_NOCOMP_FL", FS_NOCOMP_FL },
        { "FS_ECOMPR_FL", FS_ECOMPR_FL },
        { "FS_BTREE_FL", FS_BTREE_FL },
        { "FS_INDEX_FL", FS_INDEX_FL },
        { "FS_IMAGIC_FL", FS_IMAGIC_FL },
        { "FS_JOURNAL_DATA_FL", FS_JOURNAL_DATA_FL },
        { "FS_NOTAIL_FL", FS_NOTAIL_FL },
        { "FS_DIRSYNC_FL", FS_DIRSYNC_FL },
        { "FS_TOPDIR_FL", FS_TOPDIR_FL },
        { "FS_EXTENT_FL", FS_EXTENT_FL },
        { "FS_DIRECTIO_FL", FS_DIRECTIO_FL },
        { "FS_RESERVED_FL", FS_RESERVED_FL },
    };

    explain_parse_bits_print(sb, value, table, SIZEOF(table));
}

#endif

#ifdef FS_IOC_GETFLAGS

static void
fs_ioc_getflags(int fildes)
{
    long            data;

    if (ioctl(fildes, FS_IOC_GETFLAGS, &data) >= 0)
    {
        explain_string_buffer_t sb;
        char            text[1000];

        explain_string_buffer_init(&sb, text, sizeof(text));
        explain_buffer_file_inode_flags(&sb, data);
        report("FS_IOC_GETFLAGS", "File inode flags", text);
    }
}

#endif

#ifdef FS_IOC_GETVERSION

static void
fs_ioc_getversion(int fildes)
{
    long            data;

    if (ioctl(fildes, FS_IOC_GETVERSION, &data) >= 0)
    {
        report_long("FS_IOC_GETVERSION", "File inode version", data, 0);
    }
}

#endif

#ifdef FS_IOC32_GETFLAGS

static void
fs_ioc32_getflags(int fildes)
{
    int             data;

    if (ioctl(fildes, FS_IOC32_GETFLAGS, &data) >= 0)
    {
        explain_string_buffer_t sb;
        char            text[1000];

        explain_string_buffer_init(&sb, text, sizeof(text));
        explain_buffer_file_inode_flags(&sb, data);
        report("FS_IOC32_GETFLAGS", "File inode flags", text);
    }
}

#endif


#ifdef FS_IOC32_GETVERSION

static void
fs_ioc32_getversion(int fildes)
{
    int             data;

    if (ioctl(fildes, FS_IOC32_GETVERSION, &data) >= 0)
    {
        report_int("FS_IOC32_GETVERSION", "File inode version", data, 0);
    }
}

#endif


void
scan_linux_fs(int fildes)
{
#ifndef HAVE_LINUX_FS_H
    (void)fildes;
#endif
#ifdef BLKROGET
    blkroget(fildes);
#endif
#ifdef BLKGETSIZE
    blkgetsize(fildes);
#endif
#ifdef BLKRAGET
    blkraget(fildes);
#endif
#ifdef BLKFRAGET
    blkfraget(fildes);
#endif
#ifdef BLKSECTGET
    blksectget(fildes);
#endif
#ifdef BLKSSZGET
    blksszget(fildes);
#endif
#ifdef FIGETBSZ
    figetbsz(fildes);
#endif
#ifdef BLKELVGET
    blkelvget(fildes);
#endif
#ifdef BLKBSZGET
    blkbszget(fildes);
#endif
#ifdef BLKGETSIZE64
    blkgetsize64(fildes);
#endif
#ifdef FS_IOC_GETFLAGS
    fs_ioc_getflags(fildes);
#endif
#ifdef FS_IOC_GETVERSION
    fs_ioc_getversion(fildes);
#endif
#ifdef FS_IOC32_GETFLAGS
    fs_ioc32_getflags(fildes);
#endif
#ifdef FS_IOC32_GETVERSION
    fs_ioc32_getversion(fildes);
#endif
}
