#ifndef OSL_PIECE_H
#define OSL_PIECE_H
#include "osl/misc/loki.h"
#include "osl/player.h"
#include "osl/square.h"
#include "osl/ptype.h"

#include <iosfwd>
namespace osl
{
  class Piece;
  inline bool operator==(Piece l, Piece r);
  const int EMPTY_NUM=0x80;
  const int EDGE_NUM=0x40;
  /**
   * 駒.
   * 駒はptypeo(-15 - 15), 番号(0-39), ポジション(0-0xff)からなる 
   * 上位16 bitでptypeo, 8bitで番号, 8bitでポジションとする．
   * 空きマスは 黒，PTYPE_EMPTY, 番号 0x80, ポジション 0
   * 盤外は     白，PTYPE_EDGE,  番号 0x40, ポジション 0
   */
  class Piece
  {
    int piece;
    Piece(int p) : piece(p)
    {
    }
  public:
    static const int SIZE=40;
    static const Piece makeDirect(int value) { return Piece(value); }
    int intValue() const { return piece; }
    static const Piece EMPTY()  { return Piece(BLACK,PTYPE_EMPTY,EMPTY_NUM,Square::STAND()); }
    static const Piece EDGE() { return Piece(WHITE,PTYPE_EDGE,EDGE_NUM,Square::STAND()); }
    static const int BitOffsetPtype=16;
    static const int BitOffsetPromote=BitOffsetPtype+3;
    static const int BitOffsetMovePromote=BitOffsetPromote+4;
    
    Piece(Player owner, Ptype ptype, int num, Square square)
      : piece((static_cast<int>(owner)<<20)
	      +(static_cast<int>(ptype)<<BitOffsetPtype)
	      +((num)<<8)+ square.uintValue())
    {
    }
    Piece() : piece(EMPTY().piece)
    {
    }
    /**
     * 玉を作る
     */
    static const Piece
#ifdef __GNUC__
	__attribute__ ((pure))
#endif
    makeKing(Player owner, Square square);

    Ptype ptype() const {
      return static_cast<Ptype>((piece>>BitOffsetPtype)&0xf);
    }
    PtypeO ptypeO() const {
      return static_cast<PtypeO>(piece>>BitOffsetPtype);
    }

    int number() const {
      return ((piece&0xff00)>>8);
    }

    const Square square() const {
      return Square::makeDirect(piece&0xff);
    }

    Piece& operator+=(Offset offset) {
      piece += offset.intValue();
      return *this;
    }

    void setSquare(Square square) {
      piece = (piece&0xffffff00)+square.uintValue();
    }
  private:
    bool isOnBoardByOwner(Int2Type<BLACK>) const {
      return static_cast<int>(static_cast<unsigned int>(piece)&0x800000ff)>0;
    }
    /**
     * opteronでは，
     *  return static_cast<int>((piece+0x80000000)&0x800000ff)>0;
     * の方が速かった．
     */
    bool isOnBoardByOwner(Int2Type<WHITE>) const {
      return static_cast<int>((-piece)&0x800000ff)>0;
    }
  public:
    /**
     * piece がプレイヤーPの持ち物でかつボード上にある駒の場合は true.
     * 敵の駒だったり，駒台の駒だったり，Piece::EMPTY(), PIECE_EDGEの場合は false
     * @param P(template) - プレイヤー
     * @param piece - 
     */
    template<Player P>
    bool isOnBoardByOwner() const { return isOnBoardByOwner(Int2Type<P>()); }
    /**
     * isOnBoardByOwner の通常関数のバージョン.
     */
    bool isOnBoardByOwner(Player owner) const
    {
      if(owner==BLACK)
	return isOnBoardByOwner<BLACK>();
      else
	return isOnBoardByOwner<WHITE>();
    }

    /* 成る.  PROMOTE不可なpieceに適用不可 */
    const Piece promote() const {
      assert(canPromote(ptype()));
      return Piece(piece-0x80000);
    }

    /* 成りを戻す.  PROMOTE不可なpieceに適用可  */
    const Piece unpromote() const {
      return Piece((int)piece|0x80000);
    }

    /**
     * 取られたpieceを作成. unpromoteして，Squareは0に
     * 相手の持ちものにする
     */
    const Piece captured() const {
      // return (Piece)((((int)piece|0x80000)&0xffffff00)^0xfff00000);
      // をoptimizeする
      return Piece((piece&0xfff7ff00)^0xfff80000);
    }

    const Piece promoteWithMask(int promote_mask) const {
      assert(! (isPromoted() && promote_mask));
      assert(promote_mask==0 || promote_mask==(1<<23));
      return Piece(piece - (promote_mask>>(BitOffsetMovePromote-BitOffsetPromote)));
    }

    const Piece checkPromote(bool promotep) const {
      return Piece(piece - (promotep<<19));
    }

    /**
     * promoteした駒かどうかをチェックする
     */
    bool isPromoted() const { return (piece&(1<<19))==0; }

    /**
     * promoteしていないOnBoardの駒であることのチェック
     * Lance位しか使い道がない?
     */
    bool isOnBoardNotPromoted() const{
      int mask=piece&((1<<19)|0xff);
      return mask>(1<<19);
    }
    bool isPromotedNotKingGold() const {
      assert(ptype()!=KING && ptype()!=GOLD);
      return isPromoted();
    }

    bool isEmpty() const {
      return (piece&0x8000)!=0;
    }
    static bool isEmptyNum(int num) {
      return (num&0x80)!=0;
    }
    bool isEdge() const { 
      return (piece&0x4000)!=0;
    }
    static bool isEdgeNum(int num){
      assert(!isEmptyNum(num));
       return (num&0x40)!=0;
    }
    static bool isPieceNum(int num){
      return (num&0xc0)==0;
    }
    template<Ptype T>
    bool isPtype() const{
      return (piece&0xf0000)==((T)<<BitOffsetPtype);
    }
    bool isPiece() const {
      return (piece&0xc000)==0;
    }
    /**
     * pieceであることが分かっている時に，更にBlackかどうかをチェックする．
     */
    bool pieceIsBlack() const{
      assert(isPiece());
      return static_cast<int>(piece)>=0;
    }
    Player owner() const
    {
      assert(isPiece());
      return static_cast<Player>(piece>>20);
    }

  private:
    /**
     * PIECE_EMPTY 0x00008000
     * BLACK_PIECE 0x000XxxYY X>=2, YY>0
     * PIECE_EDGE  0xfff14000
     * WHITE_PIECE 0xfffXxxYY X>=2, YY>0
     */
    bool canMoveOn(Int2Type<BLACK>) const {
      return ((piece+0xe0000)&0x104000)==0;
    }
    bool canMoveOn(Int2Type<WHITE>) const {
      return piece>=0;
    }
  public:
    /** Player Pの駒が，thisの上に移動できるか?
     * @return thisが相手の駒かEMPTYならtrue
     * @param P 手番
     */
    template<Player P>
    bool canMoveOn() const { return canMoveOn(Int2Type<P>()); }

    bool canMoveOn(Player pl) const{
      if(pl==BLACK) 
	return canMoveOn<BLACK>();
      else 
	return canMoveOn<WHITE>();
    }

    bool isOnBoard() const {
      assert(square().isValid());
      return ! square().isPieceStand();
    }
  };

  inline bool operator<(Piece l, Piece r)
  {
    return l.intValue() < r.intValue();
  }
  inline bool operator==(Piece l, Piece r)
  {
    return l.intValue() == r.intValue();
  }
  inline bool operator!=(Piece l, Piece r)
  {
    return ! (l == r);
  }

  std::ostream& operator<<(std::ostream& os,const Piece piece);
}

#endif /* OSL_PIECE_H */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
