#include "osl/record/kanjiMove.h"
#include "osl/record/kanjiCode.h"
#include "osl/record/kanjiPrint.h"
#include "osl/container/moveVector.h"
#include "osl/move_generator/legalMoves.h"
#include "osl/stl/copy_if.h"
#include <boost/foreach.hpp>
#include <boost/mem_fn.hpp>
#include <boost/lambda/lambda.hpp>
#include <boost/lambda/bind.hpp>
#include <algorithm>
#include <iterator>
#include <iostream>
using namespace boost::lambda;

int moveFromX(const osl::Move& move)
{
  const osl::Square& p = move.from();
  return p.x();
}

int moveFromY(const osl::Move& move)
{
  const osl::Square& p = move.from();
  return p.y();
}

struct SortMoveFromX : 
  public std::binary_function<osl::Move, osl::Move, bool>
{
  bool operator()(const osl::Move& a, const osl::Move& b) const
  {
    const osl::Square& a_p = a.from();
    const osl::Square& b_p = b.from();
    return a_p.x() < b_p.x();
  }
};

struct SortMoveFromXDesc :
  public std::binary_function<osl::Move, osl::Move, bool>
{
  bool operator()(const osl::Move& a, const osl::Move& b) const
  {
    const osl::Square& a_p = a.from();
    const osl::Square& b_p = b.from();
    return a_p.x() > b_p.x();
  }
};

struct SortMoveFromY :
  public std::binary_function<osl::Move, osl::Move, bool>
{
  bool operator()(const osl::Move& a, const osl::Move& b) const
  {
    const osl::Square& a_p = a.from();
    const osl::Square& b_p = b.from();
    return a_p.y() < b_p.y();
  }
};

struct SortMoveFromYDesc :
  public std::binary_function<osl::Move, osl::Move, bool>
{
  bool operator()(const osl::Move& a, const osl::Move& b) const
  {
    const osl::Square& a_p = a.from();
    const osl::Square& b_p = b.from();
    return a_p.y() > b_p.y();
  }
};

struct RemoveMoveFromXOver :
  public std::unary_function<osl::Move, bool>
{
  const int min_x;
  RemoveMoveFromXOver(const int min_x)
    : min_x(min_x)
  {}

  bool operator()(const osl::Move& m) const
  {
    const osl::Square& p = m.from();
    return p.x() > min_x;
  }
};

struct RemoveMoveFromXGTE :
  public std::unary_function<osl::Move, bool>
{
  const int min_x;
  RemoveMoveFromXGTE(const int min_x)
    : min_x(min_x)
  {}

  bool operator()(const osl::Move& m) const
  {
    const osl::Square& p = m.from();
    return p.x() >= min_x;
  }
};

struct RemoveMoveFromYOver :
  public std::unary_function<osl::Move, bool>
{
  const int min_y;
  RemoveMoveFromYOver(const int min_y)
    : min_y(min_y)
  {}

  bool operator()(const osl::Move& m) const
  {
    const osl::Square& p = m.from();
    return p.y() > min_y;
  }
};

struct RemoveMoveFromYGTE :
  public std::unary_function<osl::Move, bool>
{
  const int min_y;
  RemoveMoveFromYGTE(const int min_y)
    : min_y(min_y)
  {}

  bool operator()(const osl::Move& m) const
  {
    const osl::Square& p = m.from();
    return p.y() >= min_y;
  }
};

struct RemoveMoveFromXUnder :
  public std::unary_function<osl::Move, bool>
{
  const int max_x;
  RemoveMoveFromXUnder(const int max_x)
    : max_x(max_x)
  {}

  bool operator()(const osl::Move& m) const
  {
    const osl::Square& p = m.from();
    return p.x() < max_x;
  }
};

struct RemoveMoveFromXLTE :
  public std::unary_function<osl::Move, bool>
{
  const int max_x;
  RemoveMoveFromXLTE(const int max_x)
    : max_x(max_x)
  {}

  bool operator()(const osl::Move& m) const
  {
    const osl::Square& p = m.from();
    return p.x() <= max_x;
  }
};

struct RemoveMoveFromYUnder :
  public std::unary_function<osl::Move, bool>
{
  const int max_y;
  RemoveMoveFromYUnder(const int max_y)
    : max_y(max_y)
  {}

  bool operator()(const osl::Move& m) const
  {
    const osl::Square& p = m.from();
    return p.y() < max_y;
  }
};

struct RemoveMoveFromYLTE :
  public std::unary_function<osl::Move, bool>
{
  const int max_y;
  RemoveMoveFromYLTE(const int max_y)
    : max_y(max_y)
  {}

  bool operator()(const osl::Move& m) const
  {
    const osl::Square& p = m.from();
    return p.y() <= max_y;
  }
};

struct RemoveMoveFromXEqual :
  public std::unary_function<osl::Move, bool>
{
  const int x;
  RemoveMoveFromXEqual(const int x)
    : x(x)
  {}

  bool operator()(const osl::Move& m) const
  {
    const osl::Square& p = m.from();
    return p.x() == x;
  }
};

struct RemoveMoveFromYEqual :
  public std::unary_function<osl::Move, bool>
{
  const int y;
  RemoveMoveFromYEqual(const int y)
    : y(y)
  {}

  bool operator()(const osl::Move& m) const
  {
    const osl::Square& p = m.from();
    return p.y() == y;
  }
};

osl::record::
KanjiMove::KanjiMove()
  : verbose(false)
{
  for (size_t x=1; x<=9; ++x)
  {
    for (size_t y=1; y<=9; ++y)
    {
      const std::string str = osl::record::StandardCharacters::suji[x] + 
                              osl::record::StandardCharacters::dan[y];
      str2position[str] = osl::Square(x,y);
    }
  }
  str2piece[K_PAWN]      = osl::PAWN;
  str2piece[K_PPAWN]     = osl::PPAWN;
  str2piece[K_LANCE]     = osl::LANCE;
  str2piece[K_PLANCE_D]  = osl::PLANCE;
  str2piece[K_KNIGHT]    = osl::KNIGHT;
  str2piece[K_PKNIGHT_D] = osl::PKNIGHT;
  str2piece[K_SILVER]    = osl::SILVER;
  str2piece[K_PSILVER_D] = osl::PSILVER;
  str2piece[K_GOLD]      = osl::GOLD;
  str2piece[K_BISHOP]    = osl::BISHOP;
  str2piece[K_PBISHOP]   = osl::PBISHOP;
  str2piece[K_ROOK]      = osl::ROOK;
  str2piece[K_PROOK]     = osl::PROOK;
  str2piece[K_PROOK2]    = osl::PROOK;
  str2piece[K_KING]      = osl::KING;
}

osl::record::
KanjiMove::~KanjiMove()
{
}

osl::Square osl::record::
KanjiMove::toSquare(const std::string& s) const
{
  str2position_t::const_iterator p=str2position.find(s);
  if (p == str2position.end())
    return Square();
  return p->second;
}

osl::Ptype osl::record::
KanjiMove::toPtype(const std::string& s) const
{
  str2piece_t::const_iterator p=str2piece.find(s);
  if (p == str2piece.end())
    return Ptype();
  return p->second;
}

void osl::record::
KanjiMove::selectCandidates(found_moves_t& found, 
                            std::string& str,
                            const osl::Square& to_pos,
                            const osl::Player& player) const
{
  assert(!str.empty());
  assert(found.size() >= 2);

  if ( (str.substr(0,2) == K_MIGI && player == osl::BLACK) ||
       (str.substr(0,2) == K_HIDARI && player == osl::WHITE) )
  {
    found.sort( bind(moveFromX, boost::lambda::_1) < bind(moveFromX, boost::lambda::_2) );
    const osl::Move min = found.front();
    found.remove_if( RemoveMoveFromXOver(min.from().x()) ); // list really removes
  } 
  else if ( (str.substr(0,2) == K_HIDARI && player == osl::BLACK) || 
            (str.substr(0,2) == K_MIGI   && player == osl::WHITE) )
  {
    found.sort( bind(moveFromX, boost::lambda::_1) < bind(moveFromX, boost::lambda::_2) );
    const osl::Move max = found.back();
    found.remove_if( RemoveMoveFromXUnder(max.from().x()) ); // list really removes
  }
  else if ( (str.substr(0,2) == K_SHITA && player == osl::BLACK) || 
            (str.substr(0,2) == K_UE    && player == osl::WHITE) )
  {
    found.sort( bind(moveFromY, boost::lambda::_1) < bind(moveFromY, boost::lambda::_2) );
    const osl::Move min = found.front();
    found.remove_if( RemoveMoveFromYOver(min.from().y()) ); // list really removes
  }
  else if ( (str.substr(0,2) == K_UE    && player == osl::BLACK) || 
            (str.substr(0,2) == K_SHITA && player == osl::WHITE) )
  {
    found.sort( bind(moveFromY, boost::lambda::_1) > bind(moveFromY, boost::lambda::_2) );
    const osl::Move max = found.front();
    found.remove_if( RemoveMoveFromYUnder(max.from().y()) ); // list really removes
  }
  else if (str.substr(0,2) == K_YORU)
  {
    found.remove_if( std::not1(RemoveMoveFromYEqual(to_pos.y())) ); // list really removes
  }
  else if (str.substr(0,2) == K_SUGU && player == osl::WHITE)
  {
    found.remove_if( std::not1(RemoveMoveFromXEqual(to_pos.x())) ); // or
    found.remove_if( std::not1(RemoveMoveFromYEqual(to_pos.y()-1)) ); // list really removes
  }
  else if (str.substr(0,2) == K_SUGU && player == osl::BLACK)

  {
    found.remove_if( std::not1(RemoveMoveFromXEqual(to_pos.x())) ); // or
    found.remove_if( std::not1(RemoveMoveFromYEqual(to_pos.y()+1)) ); // list really removes
  }
  else if (str.substr(0,2) == K_HIKU && player == osl::BLACK)
  {
    found.remove_if( RemoveMoveFromYGTE(to_pos.y()) ); // list really removes
  }
  else if (str.substr(0,2) == K_HIKU && player == osl::WHITE)
  {
    found.remove_if( RemoveMoveFromYLTE(to_pos.y()) ); // list really removes
  }   
  else if (str.substr(0,2) == K_YUKU && player == osl::BLACK)
  {
    found.remove_if( RemoveMoveFromYLTE(to_pos.y()) ); // list really removes
  }
  else if (str.substr(0,2) == K_YUKU && player == osl::WHITE)
  {
    found.remove_if( RemoveMoveFromYGTE(to_pos.y()) ); // list really removes
  }

  str.erase(0,2);
  assert(!found.empty());

  if (found.size() > 1)
  {
    assert(!str.empty());
    selectCandidates(found, str, to_pos, player);
  }

  assert(found.size() == 1);
  if (!str.empty())
    std::cerr << "WARNING: A single candidate is selected, but the input string still has some characters: " << str << std::endl;
}

const osl::Move osl::record::
KanjiMove::strToMove(const std::string& orig, 
                     const osl::NumEffectState& state, 
                     const osl::Move& last_move) const
{
  std::string str(orig);
  assert(orig.size() >= 4*2 || (str.substr(2,2) == K_ONAZI && orig.size() >= 3*2));
  const osl::Player player = str.substr(0,2) == K_BLACK_SIGN ? osl::BLACK : osl::WHITE;
  assert(player == state.turn());
  str.erase(0,2);

  Square to_pos;
  if (str.substr(0,2) == K_ONAZI)
  {
    to_pos = last_move.to();
    str.erase(0,2);
    if (str.substr(0,2) == K_SPACE)
      str.erase(0,2);
  }
  else
  {
    to_pos = toSquare(str.substr(0,4));
    str.erase(0,4);
  }

  Ptype ptype;
  if (str.substr(0,2) == K_NARU) // PLANCE, PKIGHT, PSILVER
  {
    ptype = toPtype(str.substr(0,4));
    str.erase(0,4);
  }
  else
  {
    ptype = toPtype(str.substr(0,2));
    str.erase(0,2);
  }

  // promote or not
  bool is_promote = false;
  if (str.size() >= 4 && str.substr(0,4) == K_FUNARI)
    str.erase(0,4);
  else if (str.size() >= 4 && str.substr(str.size()-4,4) == K_FUNARI)
    str.erase(str.size()-4,4);
  else if (str.size() >= 2 && str.substr(0,2) == K_NARU)
  {
    is_promote = true;
    str.erase(0,2);
  }
  else if (str.size() >= 2 && str.substr(str.size()-2,2) == K_NARU)
  {
    is_promote = true;
    str.erase(str.size()-2,2);
  }

  MoveVector moves;
  LegalMoves::generateWithFullUnpromotions(state, moves);
  found_moves_t found;
  BOOST_FOREACH(Move move, moves)
  {
    if (move.oldPtype()  == ptype  &&
        move.to()        == to_pos &&
        move.isPromotion() == is_promote)
    {
      /** eliminate duplicate moves */
      if (std::find(found.begin(), found.end(), move) == found.end())
        found.push_back(move);
    }
  }
  if (verbose)
  {
    std::cerr << "\n" << orig << "\n" << state;
    std::cerr << "remain: " << str  << " (" << str.size() << " bytes)\n";
    std::cerr << "promote: " << is_promote << "\n";
    std::cerr << "ptype: " << ptype << "\n";
    std::cerr << "to_position: " << to_pos << "\n";
    std::cerr << "candidates: " << found.size() << std::endl;
    if (found.size() >=2) {
      BOOST_FOREACH(const Move move, found) {
        std::cerr << "            " << move << std::endl;
      }
    }
  }
  if (found.empty()) {
    // there is no leagal move
    return Move::INVALID();
  }
  assert(!found.empty());

  // Single candidate
  if (found.size() == 1)
    return found.front();

  // Multiple candidates
  assert(found.size() >= 2);

  // drop
  if (str.substr(0,2) == K_UTSU)
  {
    found_moves_t::iterator it = 
      std::find_if(found.begin(), found.end(),
        bind(boost::mem_fn(&Move::isDrop), boost::lambda::_1)
      );
    str.erase(0,2);
    assert(str.empty());
    assert(it != found.end());
    return *it;
  }
  else
  {
    found.remove_if(
      bind(boost::mem_fn(&Move::isDrop), boost::lambda::_1)
    ); // list really removes
    if (found.size() == 1)
      return found.front();
  }

  // Multiple candidates
  assert(found.size() >= 2);
  assert(!str.empty());
  selectCandidates(found, str, to_pos, player);
  assert(found.size() == 1);
  return found.front();
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
