#include "../recursiveDoUndoMove.h"
#include "../doUndoMoveLockTest.h"
#include "osl/record/csaRecord.h"
#include "osl/record/csaString.h"
#include "osl/state/numEffectState.h"
#include "osl/effect/numSimpleEffect.tcc"

#include "osl/container/pieceVector.h"
#include "osl/effect_action/storePiece.h"
#include "osl/oslConfig.h"

#include <boost/progress.hpp>
#include <boost/foreach.hpp>
#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <iomanip>
#include <fstream>
#include <vector>

typedef osl::NumEffectState state_t;

class EffectTest : public CppUnit::TestFixture 
#if OSL_WORDSIZE == 32
		 , public osl::misc::Align16New
#endif
{
  CPPUNIT_TEST_SUITE( EffectTest );
  CPPUNIT_TEST(testMove);
  CPPUNIT_TEST(testPieceHasEffectTo);
  /*
   * foreachEffect系 
   */
  CPPUNIT_TEST(testForeachEffect);
  CPPUNIT_TEST(testForeachEffectNotBy);
  CPPUNIT_TEST(testForeachEffectOfPiece);
  CPPUNIT_TEST(testForSortedEffect);
  CPPUNIT_TEST( testForEachEffectOfPiece );
  CPPUNIT_TEST(testHasEffectFromTo);
  /*
   * hasEffect系
   */
  CPPUNIT_TEST(testHasEffectBy);
  CPPUNIT_TEST(testHasEffectByWithRemove);
  CPPUNIT_TEST(testHasEffectNotBy);
  CPPUNIT_TEST(testHasMultipleEffectBy);
  CPPUNIT_TEST(testHasEffectPiece);
  CPPUNIT_TEST(testHasEffectDir);
  CPPUNIT_TEST(testHasEffectByPtype);
  CPPUNIT_TEST(testHasEffectByNotPinned);
  CPPUNIT_TEST(testHasEffectByNotPinnedAndKing);
  CPPUNIT_TEST(testCountEffect2);
  CPPUNIT_TEST(testCountEffect);
  CPPUNIT_TEST(testCountEffectPin);
  CPPUNIT_TEST(testKingCanBeCaptured);

  CPPUNIT_TEST(testCsaFiles);

  CPPUNIT_TEST(testEffectPtype);

  CPPUNIT_TEST(testChangedEffect);
  CPPUNIT_TEST(testSelectLong);
  CPPUNIT_TEST(testEffected);

  CPPUNIT_TEST(testKingMobility);
  CPPUNIT_TEST(testLongEffectOfDirection);

  CPPUNIT_TEST(testFindCheapThreat);
  CPPUNIT_TEST(testFindThreatenedPiece);
  CPPUNIT_TEST(testWasCheckEvasion);
  CPPUNIT_TEST_SUITE_END();
private:
  osl::NumEffectState state;
  state_t eState;
public:
  EffectTest();
  void testMove();
  void testPieceHasEffectTo();

  void testForeachEffect();
  void testForeachEffectNotBy();
  void testForeachEffectOfPiece();
  void testForSortedEffect();
  void testForEachEffectOfPiece();
  void testHasEffectFromTo();

  void testHasEffectBy();
  void testHasEffectByWithRemove();
  void testHasEffectNotBy();
  void testHasMultipleEffectBy();
  void testHasEffectPiece();
  void testHasEffectDir();
  void testHasEffectByPtype();
  void testHasEffectByNotPinned();
  void testHasEffectByNotPinnedAndKing();

  void testCountEffect2();
  void testCountEffect();
  void testCountEffectPin();
  void testKingCanBeCaptured();
  void testCsaFiles();

  void testEffectPtype();
  void testChangedEffect();
  void testSelectLong();
  void testEffected();
  void testKingMobility();
  void testLongEffectOfDirection();

  void testFindCheapThreat();
  void testFindThreatenedPiece();
  void testWasCheckEvasion();
};


CPPUNIT_TEST_SUITE_REGISTRATION(EffectTest);

using namespace osl;
using namespace osl::effect_action;

EffectTest::EffectTest() :
  state(CsaString(
	  "P1-KY-KE *  *  *  *  *  *  * \n"
	  "P2 *  *  *  *  *  * -KI-OU * \n"
	  "P3 *  * -FU *  * -KI *  *  * \n"
	  "P4-FU-HI *  *  *  *  * +GI-KY\n"
	  "P5 * -FU+FU * -FU-FU-FU-FU+FU\n"
	  "P6+FU *  *  *  *  * +KE *  * \n"
	  "P7 * +FU * +FU * +FU+FU+FU-KY\n"
	  "P8+KY *  *  *  * +KI+KI *  * \n"
	  "P9 * +KE-UM * +KA *  * +KE+OU\n"
	  "P-00FU\n"
	  "P+00FU\n"
	  "P-00FU\n"
	  "P-00GI\n"
	  "P-00GI\n"
	  "P-00GI\n"
	  "P-00HI\n"
	  "+\n"
	  ).getInitialState()),eState(state)
{
#if 0
  state=CsaString(
    "P1-KY-KE *  *  *  *  *  *  * \n"
    "P2 *  *  *  *  *  * -KI-OU * \n"
    "P3 *  * -FU *  * -KI *  *  * \n"
    "P4-FU-HI *  *  *  *  * +GI-KY\n"
    "P5 * -FU+FU * -FU-FU-FU-FU+FU\n"
    "P6+FU *  *  *  *  * +KE *  * \n"
    "P7 * +FU * +FU * +FU+FU+FU-KY\n"
    "P8+KY *  *  *  * +KI+KI *  * \n"
    "P9 * +KE-UM * +KA *  * +KE+OU\n"
    "P-00FU\n"
    "P+00FU\n"
    "P-00FU\n"
    "P-00GI\n"
    "P-00GI\n"
    "P-00GI\n"
    "P-00HI\n"
    "+\n"
    ).getInitialState();
  eState=state_t(state);
#endif
}



void EffectTest::testMove()
{
}

struct StoreSquareVector{
  vector<Square> store;
  template<Player P,Ptype Type>
  void doActionPtype(Piece p,Square pos){
    store.push_back(pos);
  }
  template<Player P>
  void doAction(Piece /*p*/,Square pos){
    store.push_back(pos);
  }
  bool isMember(Square pos) const{
    return std::find(store.begin(),store.end(),pos)!=store.end();
  }
};

void EffectTest::testHasEffectFromTo()
{
  {
    NumEffectState state(CsaString(
			   "P1-KY * +UM *  *  *  * -KE-KY\n"
			   "P2 *  *  *  *  *  *  * -OU * \n"
			   "P3 *  *  *  *  * -HI * -FU-FU\n"
			   "P4-FU * -FU * -FU-KI-FU-GI * \n"
			   "P5 *  *  *  *  *  *  *  * +FU\n"
			   "P6+FU+FU+FU+KI+FU * +FU *  * \n"
			   "P7 * +KI * +FU *  * -UM *  * \n"
			   "P8 * +OU * +GI *  * -NG *  * \n"
			   "P9+KY+KE *  *  *  *  *  * -RY\n"
			   "P+00KI00GI00KY00FU\n"
			   "P-00KE00KE00FU00FU00FU00FU\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT(state.hasEffectIf(newPtypeO(BLACK,PBISHOP),
					 Square(8,2),
					 Square(3,7)));
  }
  {
    NumEffectState state(CsaString(
			   "P1+NY+TO *  *  *  * -OU-KE-KY\n"
			   "P2 *  *  *  *  * -GI-KI *  *\n"
			   "P3 * +RY *  * +UM * -KI-FU-FU\n"
			   "P4 *  * +FU-FU *  *  *  *  *\n"
			   "P5 *  * -KE * +FU *  * +FU *\n"
			   "P6-KE *  * +FU+GI-FU *  * +FU\n"
			   "P7 *  * -UM *  *  *  *  *  *\n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 * +OU * -GI *  *  *  * -NG\n"
			   "P+00HI00KI00KE00KY00FU00FU00FU00FU00FU00FU\n"
			   "P-00KI00KY00FU00FU\n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    // (8,3)の竜
    CPPUNIT_ASSERT(state.hasEffectIf(newPtypeO(BLACK,PROOK),Square(8,3),Square(8,9)));
    CPPUNIT_ASSERT(!state.hasEffectIf(newPtypeO(BLACK,PROOK),Square(8,3),Square(3,3)));
    // (4,2)の銀
    CPPUNIT_ASSERT(state.hasEffectIf(newPtypeO(WHITE,SILVER),Square(4,2),Square(5,3)));
    CPPUNIT_ASSERT(!state.hasEffectIf(newPtypeO(WHITE,SILVER),Square(4,2),Square(3,2)));
    // (8,2)の竜, 空白でも可能
    CPPUNIT_ASSERT(state.hasEffectIf(newPtypeO(BLACK,PROOK),Square(8,2),Square(7,1)));
  }
}

void EffectTest::testForEachEffectOfPiece()
{
  NumEffectState state(CsaString(
			 "P1+NY+TO *  *  *  * -OU-KE-KY\n"
			 "P2 *  *  *  *  * -GI-KI *  *\n"
			 "P3 * +RY *  * +UM * -KI-FU-FU\n"
			 "P4 *  * +FU-FU *  *  *  *  *\n"
			 "P5 *  * -KE * +FU *  * +FU *\n"
			 "P6-KE *  * +FU+GI-FU *  * +FU\n"
			 "P7 *  * -UM *  *  *  *  *  *\n"
			 "P8 *  *  *  *  *  *  *  *  * \n"
			 "P9 * +OU * -GI *  *  *  * -NG\n"
			 "P+00HI00KI00KE00KY00FU00FU00FU00FU00FU00FU\n"
			 "P-00KI00KY00FU00FU\n"
			 "P-00AL\n"
			 "+\n").getInitialState());
  {
    // 53の馬
    StoreSquareVector store;
    state.forEachEffectOfPiece<BLACK,PBISHOP,StoreSquareVector>(Square(5,3),store);
    CPPUNIT_ASSERT(store.isMember(Square(4,2)));
    CPPUNIT_ASSERT(store.isMember(Square(2,6)));
    // block
    CPPUNIT_ASSERT(!store.isMember(Square(3,1)));
    CPPUNIT_ASSERT(!store.isMember(Square(5,1)));
  }
  {
    // 53の馬
    StoreSquareVector store;
    Piece p=state.pieceAt(Square(5,3));
    state.forEachEffectOfPiece<StoreSquareVector>(p,store);
    CPPUNIT_ASSERT(store.isMember(Square(4,2)));
    CPPUNIT_ASSERT(store.isMember(Square(2,6)));
    // block
    CPPUNIT_ASSERT(!store.isMember(Square(3,1)));
    CPPUNIT_ASSERT(!store.isMember(Square(5,1)));
  }
}

void EffectTest::testPieceHasEffectTo()
{
  {
    // (7,9)の馬
    Piece p=eState.pieceAt(Square(7,9));
    CPPUNIT_ASSERT(eState.hasEffectByPiece(p,Square(3,5)));
    CPPUNIT_ASSERT(eState.hasEffectByPiece(p,Square(8,9)));
    CPPUNIT_ASSERT(eState.hasEffectByPiece(p,Square(9,7)));
    CPPUNIT_ASSERT(!eState.hasEffectByPiece(p,Square(7,9)));
    CPPUNIT_ASSERT(!eState.hasEffectByPiece(p,Square(2,4)));
    CPPUNIT_ASSERT(!eState.hasEffectByPiece(p,Square(9,9)));
  }
  {
    Piece p=eState.pieceAt(Square(2,2));
    CPPUNIT_ASSERT(eState.hasEffectByPiece(p,Square(1,3)));
    CPPUNIT_ASSERT(eState.hasEffectByPiece(p,Square(1,1)));
    CPPUNIT_ASSERT(!eState.hasEffectByPiece(p,Square(2,2)));
    CPPUNIT_ASSERT(!eState.hasEffectByPiece(p,Square(4,4)));
  }

  {
    NumEffectState state(CsaString(
			   "P1+NY+TO *  *  *  * -OU-KE-KY\n"
			   "P2 *  *  *  *  * -GI-KI *  *\n"
			   "P3 * +RY *  * +UM * -KI-FU-FU\n"
			   "P4 *  * +FU-FU *  *  *  *  *\n"
			   "P5 *  * -KE * +FU *  * +FU *\n"
			   "P6-KE *  * +FU+GI-FU *  * +FU\n"
			   "P7 *  * -UM *  *  *  *  *  *\n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 * +OU * -GI *  *  *  * -NG\n"
			   "P+00HI00KI00KE00KY00FU00FU00FU00FU00FU00FU\n"
			   "P-00KI00KY00FU00FU\n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    
    {
      // (7,7)の馬
      Piece p=state.pieceAt(Square(7,7));
      CPPUNIT_ASSERT(state.hasEffectByPiece(p,Square(9,5)));
      CPPUNIT_ASSERT(state.hasEffectByPiece(p,Square(7,6)));
      CPPUNIT_ASSERT(state.hasEffectByPiece(p,Square(6,6)));
      CPPUNIT_ASSERT(!state.hasEffectByPiece(p,Square(5,5)));
      CPPUNIT_ASSERT(!state.hasEffectByPiece(p,Square(7,5)));
      CPPUNIT_ASSERT(!state.hasEffectByPiece(p,Square(7,7)));
    }
    {
      Piece p=state.pieceAt(Square(1,1));
      CPPUNIT_ASSERT(state.hasEffectByPiece(p,Square(1,3)));
      CPPUNIT_ASSERT(!state.hasEffectByPiece(p,Square(1,1)));
      CPPUNIT_ASSERT(!state.hasEffectByPiece(p,Square(2,3)));
      CPPUNIT_ASSERT(!state.hasEffectByPiece(p,Square(1,4)));
    }
  }
}

void EffectTest::testForeachEffect()
{
  PieceVector myPieces;
  StorePiece myPiecesStore(&myPieces);
  {
    Square to=Square(2,8);
    eState.forEachEffect<BLACK,StorePiece>
      (to, myPiecesStore);
    CPPUNIT_ASSERT(myPieces.isMember(eState.pieceAt(Square(1,9))));
    CPPUNIT_ASSERT(myPieces.isMember(eState.pieceAt(Square(3,8))));
    CPPUNIT_ASSERT(!myPieces.isMember(eState.pieceAt(Square(2,8))));
  }
}

void EffectTest::testForeachEffectNotBy()
{
  PieceVector myPieces;
  StorePiece myPiecesStore(&myPieces);
  {
    Square to=Square(2,8);
    Piece p=eState.pieceAt(Square(1,9));
    eState.forEachEffectNotBy<BLACK,StorePiece>
      (to, p, myPiecesStore);
    CPPUNIT_ASSERT(!myPieces.isMember(eState.pieceAt(Square(1,9))));
    CPPUNIT_ASSERT(myPieces.isMember(eState.pieceAt(Square(3,8))));
    CPPUNIT_ASSERT(!myPieces.isMember(eState.pieceAt(Square(2,8))));
  }
}

void EffectTest::testForeachEffectOfPiece()
{
  NumEffectState state1(CsaString(
		       "P1+NY * +TO *  *  * -OU-KE-KY\n"
		       "P2 *  *  *  *  * -GI-KI *  * \n"
		       "P3 * +RY *  * +UM * -KI-FU-FU\n"
		       "P4 *  * +FU-FU *  * -FU *  * \n"
		       "P5 *  * -KE * +FU * +KY+FU * \n"
		       "P6-KE *  * +FU+GI-FU *  * +FU\n"
		       "P7 *  * -UM *  *  *  *  *  * \n"
		       "P8 *  *  *  *  *  *  *  *  * \n"
		       "P9 * +OU * -GI *  *  *  * -NG\n"
		       "P+00FU\n"
		       "P+00FU\n"
		       "P+00FU\n"
		       "P+00FU\n"
		       "P+00FU\n"
		       "P+00FU\n"
		       "P+00FU\n"
		       "P+00KE\n"
		       "P+00KI\n"
		       "P-00KI\n"
		       "P-00KY\n"
		       "P+00HI\n"
		       "+\n"
		       ).getInitialState());
  state_t ne_state(state);
  PieceVector myPieces;
  StorePiece myPiecesStore(&myPieces);
  ne_state.forEachEffectOfPiece<StorePiece>(ne_state.pieceAt(Square(3,5)),myPiecesStore);
  CPPUNIT_ASSERT(!myPieces.isMember(eState.pieceAt(Square(1,3))));
}

void EffectTest::testForSortedEffect()
{
}

void EffectTest::testHasEffectBy()
{
  // 7-3 には後手の利きがある．
  CPPUNIT_ASSERT((eState.hasEffectAt(WHITE,Square(7,3))));
  // 7-1 には後手の利きがない
  CPPUNIT_ASSERT(!(eState.hasEffectAt(WHITE,Square(7,1))));
  // 9-5 には先手の利きがある．
  CPPUNIT_ASSERT((eState.hasEffectAt(BLACK,Square(9,5))));
  // 9-8 には後手の利きがない
  CPPUNIT_ASSERT(!(eState.hasEffectAt(BLACK,Square(9,8))));
}

void EffectTest::testHasEffectByWithRemove()
{
  {
    NumEffectState state1(CsaString(
			 "P1-KY+RY *  *  *  *  *  * -KY\n"
			 "P2 *  *  *  * +UM * +NK *  * \n"
			 "P3-FU-OU-GI-FU-FU-FU *  * -FU\n"
			 "P4 *  * -FU *  *  *  *  *  * \n"
			 "P5 *  *  *  * +KA *  *  *  * \n"
			 "P6 *  *  *  *  *  *  *  *  * \n"
			 "P7+FU * +FU+FU+FU+FU+FU * +FU\n"
			 "P8 *  * -NK * +OU *  *  *  * \n"
			 "P9+KY+KE * -HI * +KI+GI * +KY\n"
			 "P-00FU\n"
			 "P+00FU\n"
			 "P+00FU\n"
			 "P-00FU\n"
			 "P-00FU\n"
			 "P+00KE\n"
			 "P-00GI\n"
			 "P-00GI\n"
			 "P+00KI\n"
			 "P-00KI\n"
			 "P-00KI\n"
			 "-\n"
			 ).getInitialState());
    NumEffectState sState(state1);
    // 8-4 には8-3を取り除くと先手の利きがある．
    CPPUNIT_ASSERT((sState.
		    hasEffectByWithRemove<BLACK>(Square(8,4),Square(8,3))));
    // 8-9 には8-3を取り除くと先手の利きがある．
    CPPUNIT_ASSERT((sState.
		    hasEffectByWithRemove<BLACK>(Square(8,9),Square(8,3))));
  }
  {
    NumEffectState state1(CsaString(
			 "P1-KY-KE+GI+KA *  * +RY * -KY\n"
			 "P2 *  * -OU * -KI * +NK *  * \n"
			 "P3-FU * -GI-FU-FU-FU *  * -FU\n"
			 "P4 *  * -FU *  *  *  *  *  * \n"
			 "P5 *  *  *  * +KA *  *  *  * \n"
			 "P6 *  *  *  *  *  *  *  *  * \n"
			 "P7+FU * +FU+FU+FU+FU+FU * +FU\n"
			 "P8 *  * -NK * +OU *  *  *  * \n"
			 "P9+KY+KE * -HI * +KI+GI * +KY\n"
			 "P+00FU\n"
			 "P+00FU\n"
			 "P+00FU\n"
			 "P-00FU\n"
			 "P-00FU\n"
			 "P-00GI\n"
			 "P-00KI\n"
			 "P-00KI\n"
			 "-\n"
			 ).getInitialState());
    NumEffectState sState(state1);
    // 8-3 には7-2を取り除くと先手の利きがある．
    CPPUNIT_ASSERT((sState.
		    hasEffectByWithRemove<BLACK>(Square(8,3),Square(7,2))));
  }
}

void EffectTest::testHasEffectNotBy()
{
  // 9-3 は後手の香車(9-1),桂馬(8-1)の利きがある．
  CPPUNIT_ASSERT((eState.hasEffectNotBy(WHITE,eState.pieceAt(Square(9,1)),Square(9,3))));
  // 9-2 は後手の香車(9-1)利きのみ
  CPPUNIT_ASSERT(!(eState.hasEffectNotBy(WHITE,eState.pieceAt(Square(9,1)),Square(9,2))));
}

void EffectTest::testHasMultipleEffectBy()
{
  // 9-3 には後手の利きが複数ある．
  CPPUNIT_ASSERT((eState.hasMultipleEffectAt(WHITE,Square(9,3))));
  // 5-7 には後手の利きが複数ない
  CPPUNIT_ASSERT(!(eState.hasMultipleEffectAt(WHITE,Square(5,7))));
  // 2-8 には先手の利きが複数ある．
  CPPUNIT_ASSERT((eState.hasMultipleEffectAt(BLACK,Square(2,8))));
  // 2-3 には先手の利きが複数ない
  CPPUNIT_ASSERT(!(eState.hasMultipleEffectAt(BLACK,Square(2,3))));
}

void EffectTest::testHasEffectPiece()
{
  {
    Piece p=eState.findCheapAttack(WHITE,Square(5,7));
    CPPUNIT_ASSERT(p==eState.pieceAt(Square(7,9)));
  }
  {
    // 2,3への利きはどちらでも良し
    Piece p=eState.findCheapAttack(WHITE,Square(2,3));
    CPPUNIT_ASSERT(p==eState.pieceAt(Square(2,2)) ||
		   p==eState.pieceAt(Square(3,2)));
  }
  {
    //利きがない場合は呼んで良い
    Piece p=eState.findCheapAttack(WHITE,Square(5,9));
    CPPUNIT_ASSERT(p==Piece::EMPTY());
  }
}

void EffectTest::testHasEffectDir()
{
  CPPUNIT_ASSERT((eState.hasEffectInDirection<LONG_DR,WHITE>(Square(9,7))));
  CPPUNIT_ASSERT((eState.hasEffectInDirection<LONG_UL,BLACK>(Square(7,7))));
  CPPUNIT_ASSERT((! eState.hasEffectInDirection<LONG_UL,BLACK>(Square(4,8))));
}

void EffectTest::testHasEffectByPtype()
{
  {
    NumEffectState state(CsaString(
			   "P1-KY-KE * -KI * -OU * -KE-KY\n"
			   "P2 * -HI *  *  *  * -KI *  * \n"
			   "P3-FU *  * -FU-FU-FU * -FU-FU\n"
			   "P4 *  * -FU * -GI * -FU-GI * \n"
			   "P5 * -FU *  *  *  *  *  *  * \n"
			   "P6 *  * +FU+FU *  *  *  *  * \n"
			   "P7+FU+FU+KE *  * +FU+FU * +FU\n"
			   "P8 *  * +KI+GI * +GI * +HI+KA\n"
			   "P9+KY *  * +OU * +KI * +KE+KY\n"
			   "P+00FU\n"
			   "P-00KA00FU\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT(state.hasEffectByPtype<BISHOP>(BLACK, Square(5,4)));
    CPPUNIT_ASSERT(! state.hasEffectByPtype<BISHOP>(WHITE, Square(5,4)));
    CPPUNIT_ASSERT(! state.hasEffectByPtype<BISHOP>(BLACK, Square(5,5)));
    CPPUNIT_ASSERT(! state.hasEffectByPtype<BISHOP>(WHITE, Square(5,5)));
  }
  {
    const NumEffectState state(CsaString(
			   "P1-OU *  *  *  *  *  *  *  * \n"
			   "P2 *  *  *  *  *  *  * +GI * \n"
			   "P3 *  *  *  *  * +NK * +NG * \n"
			   "P4 *  *  * +KE *  *  *  *  * \n"
			   "P5 *  *  *  *  * +TO *  *  * \n"
			   "P6 *  *  * +FU+FU *  *  *  * \n"
			   "P7 * +NY *  *  * +UM *  *  * \n"
			   "P8 *  *  *  *  * +KA+RY *  * \n"
			   "P9 *  * +KY *  * +HI *  * +OU\n"
			   "P-00KI00KI00KI00KI00GI00GI00KE00KE00KY00KY00FU00FU00FU00FU00FU00FU00FU00FU00FU00FU00FU00FU00FU00FU00FU\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT(state.isConsistent(true));

    CPPUNIT_ASSERT(state.hasEffectByPtypeStrict<PAWN>(BLACK, Square(6,5)));
    CPPUNIT_ASSERT(!state.hasEffectByPtypeStrict<PPAWN>(BLACK, Square(6,5)));
    CPPUNIT_ASSERT(state.hasEffectByPtypeStrict<PAWN>(BLACK, Square(5,5)));
    CPPUNIT_ASSERT(state.hasEffectByPtypeStrict<PPAWN>(BLACK, Square(5,5)));
    CPPUNIT_ASSERT(state.hasEffectByPtypeStrict<PPAWN>(BLACK, Square(4,6)));
    CPPUNIT_ASSERT(!state.hasEffectByPtypeStrict<PAWN>(BLACK, Square(4,6)));

    CPPUNIT_ASSERT(state.hasEffectByPtypeStrict<KNIGHT>(BLACK, Square(5,2)));
    CPPUNIT_ASSERT(state.hasEffectByPtypeStrict<PKNIGHT>(BLACK, Square(5,2)));
    CPPUNIT_ASSERT(state.hasEffectByPtypeStrict<KNIGHT>(BLACK, Square(7,2)));
    CPPUNIT_ASSERT(!state.hasEffectByPtypeStrict<PKNIGHT>(BLACK, Square(7,2)));
    CPPUNIT_ASSERT(!state.hasEffectByPtypeStrict<KNIGHT>(BLACK, Square(4,2)));
    CPPUNIT_ASSERT(state.hasEffectByPtypeStrict<PKNIGHT>(BLACK, Square(4,2)));
  }
}


void EffectTest::testHasEffectByNotPinned()
{
  {
    NumEffectState state(CsaString(
			   "P1-KY-KE * -KI * -OU * -KE-KY\n"
			   "P2 *  *  *  *  *  * -KI *  * \n"
			   "P3-FU *  * -FU-FU-FU * -FU-FU\n"
			   "P4 *  * -FU * -GI * -FU-GI * \n"
			   "P5 * -FU * +FU *  *  *  *  * \n"
			   "P6-KA+FU+FU-HI *  *  *  *  * \n"
			   "P7+FU+KI+KE *  * +FU+FU * +FU\n"
			   "P8 *  *  * +GI *  * +GI+HI+KA\n"
			   "P9+KY *  * +OU * +KI * +KE+KY\n"
			   "P+00FU\n"
			   "P-00FU\n"
			   "+\n").getInitialState());
    // 68銀はpinned
    CPPUNIT_ASSERT(!state.hasEffectByNotPinned(BLACK,Square(5,7)));
    // 67は銀で取れるが，pinnedな銀なのでfalse
    CPPUNIT_ASSERT(!state.hasEffectByNotPinned(BLACK,Square(6,7)));
    // 98は関係ない香車による利きあり
    CPPUNIT_ASSERT(state.hasEffectByNotPinned(BLACK,Square(9,8)));
    // 88はpinnedの金による利きのみ
    CPPUNIT_ASSERT(!state.hasEffectByNotPinned(BLACK,Square(8,8)));
    
  }
}

void EffectTest::testHasEffectByNotPinnedAndKing()
{
  {
    NumEffectState state(CsaString(
			   "P1-KY-KE * -KI * -OU * -KE-KY\n"
			   "P2 *  *  *  *  *  * -KI *  * \n"
			   "P3-FU *  * -FU-FU-FU * -FU-FU\n"
			   "P4 *  * -FU * -GI * -FU-GI * \n"
			   "P5 * -FU * +FU *  *  *  *  * \n"
			   "P6-KA+FU+FU-HI *  *  *  *  * \n"
			   "P7+FU+KI+KE *  * +FU+FU * +FU\n"
			   "P8 *  *  * +GI *  * +GI+HI+KA\n"
			   "P9+KY *  * +OU * +KI * +KE+KY\n"
			   "P+00FU\n"
			   "P-00FU\n"
			   "+\n").getInitialState());
    // 68銀はpinned
    CPPUNIT_ASSERT(!state.hasEffectByNotPinnedAndKing(BLACK,Square(5,7)));
    // 67は銀で取れるが，pinnedな銀なのでfalse
    CPPUNIT_ASSERT(!state.hasEffectByNotPinnedAndKing(BLACK,Square(6,7)));
    // 98は関係ない香車による利きあり
    CPPUNIT_ASSERT(state.hasEffectByNotPinnedAndKing(BLACK,Square(9,8)));
    // 88はpinnedの金による利きのみ
    CPPUNIT_ASSERT(!state.hasEffectByNotPinnedAndKing(BLACK,Square(8,8)));
    // 78は玉の利きのみ
    CPPUNIT_ASSERT(!state.hasEffectByNotPinnedAndKing(BLACK,Square(7,8)));
    
  }
}

static void testFile(const std::string& fileName)
{
  Record rec=CsaFile(fileName).getRecord();
  NumEffectState state=rec.getInitialState();
  state_t ne_state(state);
  const vector<osl::Move> moves=rec.getMoves();
  if (moves.empty())
    return;
  DoUndoMoveLockTest<state_t> lockTest(ne_state, moves);
  lockTest.run();

  size_t depth=0;
  if (moves.begin()->player()==BLACK)
  {
    RecursiveDoUndoMove<BLACK,state_t>
      func(ne_state, &*moves.begin(), &*moves.end(), &depth);
    ne_state.makeUnmakeMove(*moves.begin(), func);
  }
  else{
    RecursiveDoUndoMove<WHITE,state_t>
      func(ne_state, &*moves.begin(), &*moves.end(), &depth);
    ne_state.makeUnmakeMove(*moves.begin(), func);
  }
  CPPUNIT_ASSERT_EQUAL(moves.size(), depth);
}

void EffectTest::testCsaFiles()
{
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  int count=100;
  if (OslConfig::inUnitTestShort())
    count=10;
  std::string fileName;
  while ((ifs >> fileName) && (++i<count)) {
    if (fileName == "") 
      break;
    testFile(OslConfig::testCsaFile(fileName));
  }
}

class MyAction
{
  vector<Piece> pieces;
public:
  void doAction(state_t& /*state*/,Piece p,Square /*pos*/)
  {
    pieces.push_back(p);
  }
  vector<Piece> getPieces() const{
    return pieces;
  }
};
std::ostream& operator<<(std::ostream& os,MyAction const& a)
{
  vector<Piece> pieces=a.getPieces();
  os << "MyAction: " << std::endl;
  for (size_t i=0;i<pieces.size();i++)
    os << i << " : " << pieces[i] << std::endl;
  return os << std::endl;
}
  
template <Player P>
bool kingCanBeCaptured(state_t& s)
{
  const int kingIndex=Ptype_Table.getKingIndex(P);
  const Square kingSquare=s.pieceOf(kingIndex).square();
  return s.hasEffectAt(PlayerTraits<P>::opponent, kingSquare);
}
bool kingCanBeCaptured(Player P, state_t& s)
{
  if (P==BLACK)
    return kingCanBeCaptured<BLACK>(s);
  else
    return kingCanBeCaptured<WHITE>(s);
}

void EffectTest::testCountEffect2()
{
  NumEffectState state=CsaString(
    "P1-KY-KE * -KI-OU-KI-GI-KE-KY\n"
    "P2 * -HI *  *  *  *  * -KA * \n"
    "P3-FU-FU * -FU-FU-FU-FU-FU-FU\n"
    "P4 *  * -FU-GI *  *  *  *  * \n"
    "P5 *  *  *  *  *  *  *  *  * \n"
    "P6 *  * +FU *  *  *  *  *  * \n"
    "P7+FU+FU * +FU+FU+FU+FU+FU+FU\n"
    "P8 * +KA+KI *  *  *  * +HI * \n"
    "P9+KY+KE+GI * +OU+KI+GI+KE+KY\n"
    "+\n").getInitialState();
  state_t ne_state(state);
  CPPUNIT_ASSERT_EQUAL(2, std::min(2, ne_state.countEffect(BLACK, Square(7,7))));
  CPPUNIT_ASSERT_EQUAL(2, std::min(2, ne_state.countEffect(BLACK, Square(6,6))));
  CPPUNIT_ASSERT_EQUAL(1, std::min(2, ne_state.countEffect(BLACK, Square(5,6))));
  CPPUNIT_ASSERT_EQUAL(0, std::min(2, ne_state.countEffect(BLACK, Square(6,5))));
  CPPUNIT_ASSERT_EQUAL(2, std::min(2, ne_state.countEffect(WHITE, Square(7,3))));
  CPPUNIT_ASSERT_EQUAL(2, std::min(2, ne_state.countEffect(WHITE, Square(7,5))));
  CPPUNIT_ASSERT_EQUAL(1, std::min(2, ne_state.countEffect(WHITE, Square(6,5))));
  CPPUNIT_ASSERT_EQUAL(1, std::min(2, ne_state.countEffect(WHITE, Square(5,5))));
  CPPUNIT_ASSERT_EQUAL(0, std::min(2, ne_state.countEffect(WHITE, Square(4,5))));
}

void EffectTest::testCountEffect()
{
  NumEffectState state=CsaString(
    "P1-KY-KE * -KI-OU-KI-GI-KE-KY\n"
    "P2 * -HI *  *  *  *  * -KA * \n"
    "P3-FU-FU * -FU-FU-FU-FU-FU-FU\n"
    "P4 *  * -FU-GI *  *  *  *  * \n"
    "P5 *  *  *  *  *  *  *  *  * \n"
    "P6 *  * +FU *  *  *  *  *  * \n"
    "P7+FU+FU * +FU+FU+FU+FU+FU+FU\n"
    "P8 * +KA+KI *  *  *  * +HI * \n"
    "P9+KY+KE+GI * +OU+KI+GI+KE+KY\n"
    "+\n").getInitialState();
  state_t ne_state(state);
  CPPUNIT_ASSERT_EQUAL(3, ne_state.countEffect(BLACK, Square(7,7)));
  CPPUNIT_ASSERT_EQUAL(2, ne_state.countEffect(BLACK, Square(6,6)));
  CPPUNIT_ASSERT_EQUAL(1, ne_state.countEffect(BLACK, Square(5,6)));
  CPPUNIT_ASSERT_EQUAL(0, ne_state.countEffect(BLACK, Square(6,5)));
  
  CPPUNIT_ASSERT_EQUAL(2, ne_state.countEffect(WHITE, Square(7,3)));
  CPPUNIT_ASSERT_EQUAL(2, ne_state.countEffect(WHITE, Square(7,5)));
  CPPUNIT_ASSERT_EQUAL(1, ne_state.countEffect(WHITE, Square(6,5)));
  CPPUNIT_ASSERT_EQUAL(1, ne_state.countEffect(WHITE, Square(5,5)));
  CPPUNIT_ASSERT_EQUAL(0, ne_state.countEffect(WHITE, Square(4,5)));
  CPPUNIT_ASSERT_EQUAL(4, ne_state.countEffect(WHITE, Square(5,2)));
}

void EffectTest::testCountEffectPin()
{
  NumEffectState state=CsaString(
    "P1-KY * -KI * -KE *  *  * +RY\n"
    "P2 * -OU * -GI+GI *  *  *  * \n"
    "P3 * -GI * -KI *  *  *  *  * \n"
    "P4-FU+KE-FU-FU-FU * +KA * -FU\n"
    "P5 * -FU *  *  * +FU *  *  * \n"
    "P6+FU * +FU+FU+FU *  *  * +FU\n"
    "P7 * +FU *  *  * +GI *  *  * \n"
    "P8 * +OU+KI-RY *  * +FU *  * \n"
    "P9+KY *  *  *  *  *  *  * +KY\n"
    "P+00KA00KE00KY00FU00FU00FU\n"
    "P-00KI00KE00FU\n"
    "-\n").getInitialState();
  state_t ne_state(state);
  CPPUNIT_ASSERT_EQUAL(2, ne_state.countEffect(BLACK, Square(7,7)));
  PieceMask pin;
  pin.set(ne_state.pieceOnBoard(Square(7,8)).number());
  CPPUNIT_ASSERT_EQUAL(1, ne_state.countEffect(BLACK, Square(7,7), pin));
}

void EffectTest::testKingCanBeCaptured()
{
  NumEffectState state=CsaString(
    "P1+NY+TO *  *  *  * -OU-KE-KY\n"
    "P2 *  *  *  *  * -GI-KI *  *\n"
    "P3 * +RY *  * +UM * -KI-FU-FU\n"
    "P4 *  * +FU-FU *  *  *  *  *\n"
    "P5 *  * -KE * +FU *  * +FU *\n"
    "P6-KE *  * +FU+GI-FU *  * +FU\n"
    "P7 *  * -UM *  *  *  *  *  *\n"
    "P8 *  *  *  *  *  *  *  *  * \n"
    "P9 * +OU * -GI *  *  *  * -NG\n"
    "P+00HI00KI00KE00KY00FU00FU00FU00FU00FU00FU\n"
    "P-00KI00KY00FU00FU\n"
    "P-00AL\n"
    "+\n").getInitialState();
  state_t ne_state(state);
  CPPUNIT_ASSERT(! kingCanBeCaptured(BLACK,ne_state));
  CPPUNIT_ASSERT(! kingCanBeCaptured(WHITE,ne_state));
  ne_state.makeMove(Move(Square(8,1),Square(7,1),PPAWN,PTYPE_EMPTY,false,BLACK));
  CPPUNIT_ASSERT(! kingCanBeCaptured(BLACK,ne_state));
  CPPUNIT_ASSERT(! kingCanBeCaptured(WHITE,ne_state));
  ne_state.makeMove(Move(Square::STAND(),Square(8,8),PAWN,PTYPE_EMPTY,false,WHITE));
  CPPUNIT_ASSERT(kingCanBeCaptured(BLACK,ne_state));
  CPPUNIT_ASSERT(! kingCanBeCaptured(WHITE,ne_state));
  ne_state.makeMove(Move(Square(7,1),Square(8,1),PPAWN,PTYPE_EMPTY,false,BLACK));
  CPPUNIT_ASSERT(kingCanBeCaptured(BLACK,ne_state));
  CPPUNIT_ASSERT(! kingCanBeCaptured(WHITE,ne_state));
}

void EffectTest::testEffectPtype()
{
  NumEffectState state(CsaString(
			 "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
			 "P2 * -HI *  *  *  *  * -KA * \n"
			 "P3-FU-FU-FU-FU-FU-FU * -FU-FU\n"
			 "P4 *  *  *  *  *  * -FU *  * \n"
			 "P5 *  *  *  *  *  *  *  *  * \n"
			 "P6 *  * +FU *  *  *  *  *  * \n"
			 "P7+FU+FU * +FU+FU+FU+FU+FU+FU\n"
			 "P8 * +KA *  *  *  *  * +HI * \n"
			 "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
			 "+\n").getInitialState());
  CPPUNIT_ASSERT_EQUAL(Square(2,2), state.findAttackAt<BISHOP>(WHITE, Square(6,6)).square());
}
void EffectTest::testChangedEffect()
{
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  int count=3000;
  if (OslConfig::inUnitTestShort())
    count=10;
  std::string fileName;
  boost::scoped_ptr<boost::progress_display> progress;
  if (OslConfig::inUnitTest() >= 2)
    progress.reset(new boost::progress_display(count, std::cerr));
  while ((ifs >> fileName) && (++i<count)) {
    if (progress)
      ++(*progress);
    if (fileName == "") 
      break;
    Record rec=CsaFile(OslConfig::testCsaFile(fileName)).getRecord();
    NumEffectState state(rec.getInitialState());
    const vector<osl::Move> moves=rec.getMoves();
    for(size_t i=0;i<moves.size();i++){
      PieceMask before[9][9][2];
      {
	const NumEffectState& ne_state=state;
	for(int x=1;x<=9;x++)
	  for(int y=1;y<=9;y++){
	    const Square pos(x,y);
	    CPPUNIT_ASSERT_EQUAL(state.hasLongEffectAt<LANCE>(BLACK, pos),
				 state.longEffectAt<LANCE>(pos, BLACK).any());
	    CPPUNIT_ASSERT_EQUAL(state.hasLongEffectAt<LANCE>(WHITE, pos),
				 state.longEffectAt<LANCE>(pos, WHITE).any());
	    CPPUNIT_ASSERT_EQUAL(state.hasLongEffectAt<BISHOP>(BLACK, pos),
				 state.longEffectAt<BISHOP>(pos, BLACK).any());
	    CPPUNIT_ASSERT_EQUAL(state.hasLongEffectAt<BISHOP>(WHITE, pos),
				 state.longEffectAt<BISHOP>(pos, WHITE).any());
	    CPPUNIT_ASSERT_EQUAL(state.hasLongEffectAt<ROOK>(BLACK, pos),
				 state.longEffectAt<ROOK>(pos, BLACK).any());
	    CPPUNIT_ASSERT_EQUAL(state.hasLongEffectAt<ROOK>(WHITE, pos),
				 state.longEffectAt<ROOK>(pos, WHITE).any());
	    for (int z=0; z<2; ++z) {
	      const Player pl = indexToPlayer(z);
	      before[x-1][y-1][z]=
		ne_state.effectSetAt(pos)&ne_state.piecesOnBoard(pl);
	    }
	  }
      }
      state.makeMove(moves[i]);
      PieceMask after[9][9][2];
      {
	const NumEffectState& ne_state=state;
	for(int x=1;x<=9;x++)
	  for(int y=1;y<=9;y++){
	    const Square pos(x,y);
	    for (int z=0; z<2; ++z) {
	      const Player pl = indexToPlayer(z);
	      after[x-1][y-1][z]=
		ne_state.effectSetAt(pos)&ne_state.piecesOnBoard(pl);
	    }
	  }
	for(int x=1;x<=9;x++)
	  for(int y=1;y<=9;y++){
	    const Square pos(x,y);
	    for (int z=0; z<2; ++z) {
	      const Player pl = indexToPlayer(z);
	      PieceMask b=before[x-1][y-1][z];
	      PieceMask a=after[x-1][y-1][z];
	      if(a!=b || (pos==moves[i].to() && moves[i].capturePtype()!=PTYPE_EMPTY)){
		for(int num=0;num<48;num++){
		  if(b.test(num)!=a.test(num) || 
		     (pos==moves[i].to() && moves[i].capturePtype()!=PTYPE_EMPTY && (b.test(num) || a.test(num)))){
		    CPPUNIT_ASSERT(ne_state.changedPieces().test(num));
		  }
		}
	      }
	      if(ne_state.changedEffects(pl).test(pos)){
		Piece p=ne_state.pieceAt(moves[i].to());
		if(b==a){
		  if(!a.test(p.number())){
		    std::cerr << std::endl << state << ",move=" << moves[i] << ",x=" << x << ",y=" << y << std::endl;
		    std::cerr << "changed[" << pl << "]=" << std::endl << ne_state.changedEffects(pl);
		  }
		}
	      }
	      else{
		if(b!=a){
		  std::cerr << state << ",move=" << moves[i] << ",x=" << x << ",y=" << y << std::endl;
		  std::cerr << "before " << b << "\nafter " << a << "\n";
		  std::cerr << "changed[ " << pl << "]=" << std::endl << ne_state.changedEffects(pl);
		}
		CPPUNIT_ASSERT_EQUAL(b, a);
	      }
	    }
	  }
      }
    }
  }
}

void EffectTest::testSelectLong()
{
  NumEffectState state(CsaString(
			 "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
			 "P2 * -HI *  *  *  *  * -KA * \n"
			 "P3-FU-FU-FU-FU-FU-FU * -FU-FU\n"
			 "P4 *  *  *  *  *  * -FU *  * \n"
			 "P5 *  *  *  *  *  *  *  *  * \n"
			 "P6 *  * +FU *  *  *  *  *  * \n"
			 "P7+FU+FU * +FU+FU+FU+FU+FU+FU\n"
			 "P8 * +KA *  *  *  *  * +HI * \n"
			 "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
			 "+\n").getInitialState());
  CPPUNIT_ASSERT(state.longEffectAt<LANCE>(Square(9,2)).any());
  CPPUNIT_ASSERT(state.longEffectAt<ROOK>(Square(9,2)).any());
  CPPUNIT_ASSERT(state.longEffectAt(Square(9,2), WHITE).countBit()==2);
}

static void testEffectedState(NumEffectState const& state,Move move)
{
  PieceMask b_mask=state.effectedMask(BLACK);
  PieceMask w_mask=state.effectedMask(WHITE);
  for(int num=0;num<40;num++){
    Piece p=state.pieceOf(num);
    if(p.isOnBoard()){
      Square pos=p.square();
      if(b_mask.test(num)){
	if(!state.hasEffectAt(BLACK,pos)){
	  std::cerr << std::endl << state << std::endl;
	  std::cerr << "b_mask=" << b_mask << ",num=" << num << ",pos=" << pos << ",move=" << move << std::endl;
	}
	CPPUNIT_ASSERT(state.hasEffectAt(BLACK,pos));
      }
      else{
	if(state.hasEffectAt(BLACK,pos)){
	  std::cerr << std::endl << state << std::endl;
	  std::cerr << "b_mask=" << b_mask << ",num=" << num << ",pos=" << pos << ",move=" << move << std::endl;
	}
	CPPUNIT_ASSERT(!state.hasEffectAt(BLACK,pos));
      }
      if(w_mask.test(num)){
	if(!state.hasEffectAt(WHITE,pos)){
	  std::cerr << std::endl << state << std::endl;
	  std::cerr << "w_mask=" << w_mask << ",num=" << num << ",pos=" << pos << ",move=" << move << std::endl;
	}
	CPPUNIT_ASSERT(state.hasEffectAt(WHITE,pos));
      }
      else{
	if(state.hasEffectAt(WHITE,pos)){
	  std::cerr << std::endl << state << std::endl;
	  std::cerr << "w_mask=" << w_mask << ",num=" << num << ",pos=" << pos << ",move=" << move << std::endl;
	}
	CPPUNIT_ASSERT(!state.hasEffectAt(WHITE,pos));
      }
    }
  }
}

void EffectTest::testEffected()
{
  NumEffectState state=CsaString(
    "P1+NY+TO *  *  *  * -OU-KE-KY\n"
    "P2 *  *  *  *  * -GI-KI *  *\n"
    "P3 * +RY *  * +UM * -KI-FU-FU\n"
    "P4 *  * +FU-FU *  *  *  *  *\n"
    "P5 *  * -KE * +FU *  * +FU *\n"
    "P6-KE *  * +FU+GI-FU *  * +FU\n"
    "P7 *  * -UM *  *  *  *  *  *\n"
    "P8 *  *  *  *  *  *  *  *  * \n"
    "P9 * +OU * -GI *  *  *  * -NG\n"
    "P+00HI00KI00KE00KY00FU00FU00FU00FU00FU00FU\n"
    "P-00KI00KY00FU00FU\n"
    "P-00AL\n"
    "+\n").getInitialState();
  testEffectedState(state,Move::INVALID());
  { // simple move
    NumEffectState state1=state;
    Move move(Square(8,3),Square(8,2),PROOK,PTYPE_EMPTY,false,BLACK);
    state1.makeMove(move);
    testEffectedState(state1,move);
  }
  { // drop move
    NumEffectState state1=state;
    Move move(Square(8,2),ROOK,BLACK);
    state1.makeMove(move);
    testEffectedState(state1,move);
  }
  { // capture move
    NumEffectState state1=state;
    Move move(Square(5,3),Square(4,2),PBISHOP,SILVER,false,BLACK);
    state1.makeMove(move);
    testEffectedState(state1,move);
  }
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  int count=100;
  if (OslConfig::inUnitTestShort())
    count=10;
  std::string fileName;
  while ((ifs >> fileName) && (++i<count)) {
    if (fileName == "") 
      break;
    Record rec=CsaFile(OslConfig::testCsaFile(fileName)).getRecord();
    NumEffectState state(rec.getInitialState());
    const vector<osl::Move> moves=rec.getMoves();
    for(size_t i=0;i<moves.size();i++){
      state.makeMove(moves[i]);
      testEffectedState(state,moves[i]);
    }
  }
}

void EffectTest::testKingMobility()
{
  NumEffectState state=CsaString(
    "P1+NY+TO *  *  *  * -OU-KE-KY\n"
    "P2 *  *  *  *  * -GI-KI *  *\n"
    "P3 * +RY *  * +UM * -KI-FU-FU\n"
    "P4 *  * +FU-FU *  *  *  *  *\n"
    "P5 *  * -KE * +FU *  * +FU *\n"
    "P6-KE *  * +FU+GI-FU *  * +FU\n"
    "P7 *  * -UM *  *  *  *  *  *\n"
    "P8 *  *  *  *  *  *  *  *  * \n"
    "P9 * +OU * -GI *  *  *  * -NG\n"
    "P+00HI00KI00KE00KY00FU00FU00FU00FU00FU00FU\n"
    "P-00KI00KY00FU00FU\n"
    "P-00AL\n"
    "+\n").getInitialState();
  CPPUNIT_ASSERT_EQUAL(Square(8, 10), state.kingMobilityAbs(BLACK, U));
  CPPUNIT_ASSERT_EQUAL(Square(8, 3), state.kingMobilityAbs(BLACK, D));
  CPPUNIT_ASSERT_EQUAL(Square(6, 9), state.kingMobilityAbs(BLACK, L));
  CPPUNIT_ASSERT_EQUAL(Square(10, 9), state.kingMobilityAbs(BLACK, R));
  CPPUNIT_ASSERT_EQUAL(Square(5, 6), state.kingMobilityAbs(BLACK, DL));
  CPPUNIT_ASSERT_EQUAL(Square(7, 10), state.kingMobilityAbs(BLACK, UL));
  CPPUNIT_ASSERT_EQUAL(Square(10, 7), state.kingMobilityAbs(BLACK, DR));
  CPPUNIT_ASSERT_EQUAL(Square(9, 10), state.kingMobilityAbs(BLACK, UR));

  CPPUNIT_ASSERT_EQUAL(Square(3, 2), state.kingMobilityAbs(WHITE, U));
  CPPUNIT_ASSERT_EQUAL(Square(3, 0), state.kingMobilityAbs(WHITE, D));
  CPPUNIT_ASSERT_EQUAL(Square(2, 1), state.kingMobilityAbs(WHITE, L));
  CPPUNIT_ASSERT_EQUAL(Square(8, 1), state.kingMobilityAbs(WHITE, R));
  CPPUNIT_ASSERT_EQUAL(Square(4, 2), state.kingMobilityAbs(WHITE, UR));
  CPPUNIT_ASSERT_EQUAL(Square(4, 0), state.kingMobilityAbs(WHITE, DR));
  CPPUNIT_ASSERT_EQUAL(Square(1, 3), state.kingMobilityAbs(WHITE, UL));
  CPPUNIT_ASSERT_EQUAL(Square(2, 0), state.kingMobilityAbs(WHITE, DL));

  CPPUNIT_ASSERT_EQUAL(Square(8, 3), state.kingMobilityOfPlayer(BLACK, U));
  CPPUNIT_ASSERT_EQUAL(Square(8, 10), state.kingMobilityOfPlayer(BLACK, D));
  CPPUNIT_ASSERT_EQUAL(Square(10, 9), state.kingMobilityOfPlayer(BLACK, L));
  CPPUNIT_ASSERT_EQUAL(Square(6, 9), state.kingMobilityOfPlayer(BLACK, R));
  CPPUNIT_ASSERT_EQUAL(Square(5, 6), state.kingMobilityOfPlayer(BLACK, UR));
  CPPUNIT_ASSERT_EQUAL(Square(9, 10), state.kingMobilityOfPlayer(BLACK, DL));
  CPPUNIT_ASSERT_EQUAL(Square(10, 7), state.kingMobilityOfPlayer(BLACK, UL));
  CPPUNIT_ASSERT_EQUAL(Square(7, 10), state.kingMobilityOfPlayer(BLACK, DR));

  CPPUNIT_ASSERT_EQUAL(Square(3, 2), state.kingMobilityOfPlayer(WHITE, U));
  CPPUNIT_ASSERT_EQUAL(Square(3, 0), state.kingMobilityOfPlayer(WHITE, D));
  CPPUNIT_ASSERT_EQUAL(Square(2, 1), state.kingMobilityOfPlayer(WHITE, L));
  CPPUNIT_ASSERT_EQUAL(Square(8, 1), state.kingMobilityOfPlayer(WHITE, R));
  CPPUNIT_ASSERT_EQUAL(Square(4, 2), state.kingMobilityOfPlayer(WHITE, UR));
  CPPUNIT_ASSERT_EQUAL(Square(4, 0), state.kingMobilityOfPlayer(WHITE, DR));
  CPPUNIT_ASSERT_EQUAL(Square(1, 3), state.kingMobilityOfPlayer(WHITE, UL));
  CPPUNIT_ASSERT_EQUAL(Square(2, 0), state.kingMobilityOfPlayer(WHITE, DL));
}

void EffectTest::testLongEffectOfDirection()
{
  {
    NumEffectState state=CsaString(
      "P1+NY+TO *  *  *  * -OU-KE-KY\n"
      "P2 *  *  *  *  * -GI-KI *  *\n"
      "P3 * +RY *  * +UM * -KI-FU-FU\n"
      "P4 *  * +FU-FU *  *  *  *  *\n"
      "P5 *  * -KE * +FU *  * +FU *\n"
      "P6-KE *  * +FU+GI-FU *  * +FU\n"
      "P7 *  * -UM *  *  *  *  *  *\n"
      "P8 *  *  *  *  *  *  *  *  * \n"
      "P9 * +OU * -GI *  *  *  * -NG\n"
      "P+00HI00KI00KE00KY00FU00FU00FU00FU00FU00FU\n"
      "P-00KI00KY00FU00FU\n"
      "P-00AL\n"
      "+\n").getInitialState();
    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), state.findLongAttackAt(Square(5,3), UL));
    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), state.findLongAttackAt(Square(5,3), U));
    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), state.findLongAttackAt(Square(5,3), UR));
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(8,3)), state.findLongAttackAt(Square(5,3), L));
    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), state.findLongAttackAt(Square(5,3), R));
    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), state.findLongAttackAt(Square(5,3), DL));
    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), state.findLongAttackAt(Square(5,3), D));
    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), state.findLongAttackAt(Square(5,3), R));

    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), state.findLongAttackAt(Square(4,2), UL));
    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), state.findLongAttackAt(Square(4,2), U));
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(5,3)), state.findLongAttackAt(Square(4,2), UR));
    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), state.findLongAttackAt(Square(4,2), L));
    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), state.findLongAttackAt(Square(4,2), R));
    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), state.findLongAttackAt(Square(4,2), DL));
    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), state.findLongAttackAt(Square(4,2), D));
    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), state.findLongAttackAt(Square(4,2), R));

    // king?
    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), state.findLongAttackAt(Square(8,3), D));
  }
  {
    NumEffectState state=CsaString(
      "P1-KY-KE+GI+KA *  * +RY * -KY\n"
      "P2 *  * -OU * -KI * +NK *  * \n"
      "P3-FU * -GI-FU-FU-FU *  * -FU\n"
      "P4 *  * -FU *  *  *  *  *  * \n"
      "P5 *  *  *  * +KA *  *  *  * \n"
      "P6 *  *  *  *  *  *  *  *  * \n"
      "P7+FU * +FU+FU+FU+FU+FU * +FU\n"
      "P8 *  * -NK * +OU *  *  *  * \n"
      "P9+KY+KE * -HI * +KI+GI * +KY\n"
      "P+00FU00FU00FU\n"
      "P-00KI00KI00GI00FU00FU\n"
      "-\n").getInitialState();
    CPPUNIT_ASSERT(state.isConsistent());
    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), state.findLongAttackAt(Square(7,2), UR));
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(6,1)), state.findLongAttackAt(Square(7,2), DL));
    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), state.findLongAttackAt(Square(3,2), D));
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(3,1)), state.findLongAttackAt(Square(3,2), U));
  }
}

void EffectTest::testFindCheapThreat()
{
  {
    NumEffectState state(CsaString(
			   "P1-KY-KE *  *  *  *  *  * -KY\n"
			   "P2 * -HI *  *  * -KI-OU *  * \n"
			   "P3-FU *  * +TO+FU-KI *  *  * \n"
			   "P4 *  * -FU * -FU-GI * -FU-FU\n"
			   "P5 * -FU *  * -GI-KE-FU *  * \n"
			   "P6 *  * +FU *  * -FU *  * +FU\n"
			   "P7+FU+FU+KE *  *  * +FU+FU * \n"
			   "P8 *  * +KI * +GI * +KI+OU * \n"
			   "P9+KY *  *  * +HI *  * +KE+KY\n"
			   "P+00KA00KA00FU\n"
			   "P-00GI00FU\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(5,3)), 
			 state.findCheapAttack(BLACK, Square(5,2)));
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(6,3)), 
			 state.findCheapAttack(BLACK, Square(6,2)));
    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(),
			 state.findCheapAttack(BLACK, Square(5,1)));
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(2,9)), 
			 state.findCheapAttack(BLACK, Square(3,7)));
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(5,8)), 
			 state.findCheapAttack(BLACK, Square(4,9)));
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(5,8)), 
			 state.findCheapAttack(BLACK, Square(4,7)));
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(1,9)), 
			 state.findCheapAttack(BLACK, Square(1,7)));
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(5,9)), 
			 state.findCheapAttack(BLACK, Square(9,9)));
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(5,9)), 
			 state.findCheapAttack(BLACK, Square(2,9)));
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(2,8)), 
			 state.findCheapAttack(BLACK, Square(1,9)));

    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(5,4)), 
			 state.findCheapAttack(WHITE, Square(5,5)));
  }
}

void EffectTest::testFindThreatenedPiece()
{
  {
    NumEffectState state;
    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), state.findThreatenedPiece(BLACK));
    CPPUNIT_ASSERT_EQUAL(Piece::EMPTY(), state.findThreatenedPiece(WHITE));
  }
  {
    NumEffectState state(CsaString(
			   "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
			   "P2 * -HI *  *  *  *  * -KA * \n"
			   "P3-FU-FU-FU-FU-FU-FU-FU-FU * \n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7 * +FU+FU+FU+FU+FU+FU+FU+FU\n"
			   "P8 * +KA *  *  *  *  * +HI * \n"
			   "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
			   "P+00FU00FU\n"
			   "+\n").getInitialState()); 
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(1,7)), state.findThreatenedPiece(BLACK));
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(9,3)), state.findThreatenedPiece(WHITE));
  }
  {
    NumEffectState state(CsaString(
			   "P1-KY-KE-GI-KI-OU-KI-GI * -KY\n"
			   "P2 * -HI *  *  *  *  * -KA * \n"
			   "P3 * -FU-FU-FU-FU-FU-FU-FU * \n"
			   "P4 *  *  *  * +KE *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  * -KE *  *  *  * \n"
			   "P7 * +FU+FU+FU+FU+FU+FU+FU * \n"
			   "P8 * +KA *  *  *  *  * +HI * \n"
			   "P9+KY * +GI+KI+OU+KI+GI+KE+KY\n"
			   "P+00FU00FU00FU00FU\n"
			   "+\n").getInitialState()); 
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(5,4)), state.findThreatenedPiece(BLACK));
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(5,6)), state.findThreatenedPiece(WHITE));
  }
  {
    NumEffectState state(CsaString(
			   "P1-KY-KE-GI-KI-OU-KI *  * -KY\n"
			   "P2 * -HI *  *  *  *  * -KA * \n"
			   "P3 * -FU-FU-FU-FU-FU-FU-FU * \n"
			   "P4 *  *  * +GI+KE *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  * -KE-GI *  *  * \n"
			   "P7 * +FU+FU+FU+FU+FU+FU+FU * \n"
			   "P8 * +KA *  *  *  *  * +HI * \n"
			   "P9+KY *  * +KI+OU+KI+GI+KE+KY\n"
			   "P+00FU00FU00FU00FU\n"
			   "+\n").getInitialState()); 
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(6,4)), state.findThreatenedPiece(BLACK));
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(4,6)), state.findThreatenedPiece(WHITE));
  }
  {
    NumEffectState state(CsaString(
			   "P1-KY-KE-GI * -OU-KI *  * -KY\n"
			   "P2 * -HI *  *  *  *  * -KA * \n"
			   "P3 * -FU-FU-FU-FU-FU-FU-FU * \n"
			   "P4 *  * +KI+GI+KE *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  * -KE-GI-KI *  * \n"
			   "P7 * +FU+FU+FU+FU+FU+FU+FU * \n"
			   "P8 * +KA *  *  *  *  * +HI * \n"
			   "P9+KY *  * +KI+OU * +GI+KE+KY\n"
			   "P+00FU00FU00FU00FU\n"
			   "+\n").getInitialState()); 
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(7,4)), state.findThreatenedPiece(BLACK));
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(3,6)), state.findThreatenedPiece(WHITE));
  }
  {
    NumEffectState state(CsaString(
			   "P1-KY-KE-GI * -OU-KI *  * -KY\n"
			   "P2 * -HI *  *  *  *  *  *  * \n"
			   "P3 * -FU-FU-FU-FU-FU-FU-FU * \n"
			   "P4 * +KA+KI+GI+KE *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  * -KE-GI-KI-KA * \n"
			   "P7 * +FU+FU+FU+FU+FU+FU+FU * \n"
			   "P8 *  *  *  *  *  *  * +HI * \n"
			   "P9+KY *  * +KI+OU * +GI+KE+KY\n"
			   "P+00FU00FU00FU00FU\n"
			   "+\n").getInitialState()); 
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(8,4)), state.findThreatenedPiece(BLACK));
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(2,6)), state.findThreatenedPiece(WHITE));
  }
  {
    NumEffectState state(CsaString(
			   "P1-KY-KE-GI * -OU-KI *  * -KY\n"
			   "P2 *  *  *  *  *  *  *  *  * \n"
			   "P3 * -FU-FU-FU-FU-FU-FU-FU * \n"
			   "P4+HI+KA+KI+GI+KE *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  * -KE-GI-KI-KA-HI\n"
			   "P7 * +FU+FU+FU+FU+FU+FU+FU * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9+KY *  * +KI+OU * +GI+KE+KY\n"
			   "P+00FU00FU00FU00FU\n"
			   "+\n").getInitialState()); 
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(9,4)), state.findThreatenedPiece(BLACK));
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(1,6)), state.findThreatenedPiece(WHITE));
  }
  {
    NumEffectState state(CsaString(
			   "P1-KY-KE-GI * -OU-KI *  * -KY\n"
			   "P2 *  *  *  *  *  *  *  *  * \n"
			   "P3 * -FU-FU-FU-FU-FU-FU-FU * \n"
			   "P4+UM+KA+KI+GI+KE *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  * -KE-GI-KI-HI-RY\n"
			   "P7 * +FU+FU+FU+FU+FU+FU+FU * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9+KY *  * +KI+OU * +GI+KE+KY\n"
			   "P+00FU00FU00FU00FU\n"
			   "+\n").getInitialState()); 
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(9,4)), state.findThreatenedPiece(BLACK));
    CPPUNIT_ASSERT_EQUAL(state.pieceAt(Square(1,6)), state.findThreatenedPiece(WHITE));
  }
}

static void testFileEvasion(const std::string& filename)
{
  Record record=CsaFile(filename).getRecord();
  NumEffectState state(record.getInitialState());
  const vector<Move> moves=record.getMoves();
  if (moves.empty())
    return;
  bool in_check = false;
  BOOST_FOREACH(Move move, moves) {    
    state.makeMove(move);
    if (in_check != state.wasCheckEvasion(move))
      std::cerr << state << move << ' ' << in_check << "\n";
    CPPUNIT_ASSERT_EQUAL(in_check, state.wasCheckEvasion(move));
    in_check = state.inCheck();
  }
}

void EffectTest::testWasCheckEvasion()
{
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  int count=10000;
  if (OslConfig::inUnitTestShort())
    count=10;
  std::string filename;
  while ((ifs >> filename) && (++i<count)) {
    if (filename == "") 
      break;
    testFileEvasion(OslConfig::testCsaFile(filename));
  }
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
