/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include "support.h"

START_TEST (check_base64_decode_null)
{
	SmlError *error = NULL;
	char *buffer = NULL;
	unsigned int buffersize = 0;
	
	fail_unless(smlBase64Decode(NULL, &buffer, &buffersize, &error), NULL);
	fail_unless(error == NULL);
	
	fail_unless(buffer == NULL, NULL);
	fail_unless(buffersize == 0, NULL);
}
END_TEST

START_TEST (check_base64_decode_null2)
{
	SmlError *error = NULL;
	char *buffer = NULL;
	unsigned int buffersize = 0;
	
	fail_unless(smlBase64Decode(NULL, &buffer, &buffersize, &error), NULL);
	fail_unless(error == NULL);
	
	fail_unless(buffer == NULL, NULL);
	fail_unless(buffersize == 0, NULL);
}
END_TEST

START_TEST (check_base64_decode_test)
{
	char *data = "dGVzdA==";
	SmlError *error = NULL;
	char *buffer = NULL;
	unsigned int buffersize = 0;
	
	fail_unless(smlBase64Decode(data, &buffer, &buffersize, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!strcmp(buffer, "test"), NULL);
	fail_unless(buffersize == 5, NULL);
	g_free(buffer);
}
END_TEST

START_TEST (check_base64_decode_size)
{
	char *data = "dGVzdA==/&%((";
	SmlError *error = NULL;
	char *buffer = NULL;
	unsigned int buffersize = 0;
	
	fail_unless(smlBase64DecodeBinary(data, 8, &buffer, &buffersize, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(buffersize == 5, NULL);
	fail_unless(!strncmp(buffer, "test", 4), NULL);
	g_free(buffer);
}
END_TEST

START_TEST (check_base64_decode2)
{
	char *data = "JlwiKKcpJD09YD9cJ1wnKy0tLiw7Pjw=";
	SmlError *error = NULL;
	char *buffer = NULL;
	unsigned int buffersize = 0;
	
	fail_unless(smlBase64Decode(data, &buffer, &buffersize, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!strncmp(buffer, "&\\\"()$==`?\\\'\\\'+--.,;><", 4), NULL);
	fail_unless(buffersize == 24, NULL);
	g_free(buffer);
}
END_TEST

START_TEST (check_base64_decode3)
{
	char *data = "dGVzdDE=";
	SmlError *error = NULL;
	char *buffer = NULL;
	unsigned int buffersize = 0;
	
	fail_unless(smlBase64Decode(data, &buffer, &buffersize, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!strncmp(buffer, "test1", 4), NULL);
	fail_unless(buffersize == 6, NULL);
	g_free(buffer);
}
END_TEST

START_TEST (check_base64_decode4)
{
	char *data = "dGVzdDEx";
	SmlError *error = NULL;
	char *buffer = NULL;
	unsigned int buffersize = 0;
	
	fail_unless(smlBase64Decode(data, &buffer, &buffersize, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!strncmp(buffer, "test11", 4), NULL);
	fail_unless(buffersize == 7, NULL);
	g_free(buffer);
}
END_TEST

START_TEST (check_base64_invalid1)
{
	char *data = "$%$%&";
	SmlError *error = NULL;
	char *buffer = NULL;
	unsigned int buffersize = 0;
	
	fail_unless(!smlBase64Decode(data, &buffer, &buffersize, &error), NULL);
	fail_unless(error != NULL, NULL);
}
END_TEST

START_TEST (check_base64_empty)
{
	char *data = "";
	SmlError *error = NULL;
	char *buffer = (char *)1;
	unsigned int buffersize = 1;
	
	fail_unless(smlBase64Decode(data, &buffer, &buffersize, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(buffer == NULL, NULL);
	fail_unless(buffersize == 0, NULL);
}
END_TEST

START_TEST (check_base64_invalid2)
{
	char *data = "dGVz====";
	SmlError *error = NULL;
	char *buffer = NULL;
	unsigned int buffersize = 0;
	
	fail_unless(!smlBase64Decode(data, &buffer, &buffersize, &error), NULL);
	fail_unless(error != NULL, NULL);
}
END_TEST

START_TEST (check_base64_encode_null)
{
	SmlError *error = NULL;
	char *buffer = (char *)1;
	
	fail_unless(smlBase64Encode(NULL, &buffer, &error), NULL);
	fail_unless(error == NULL);
	
	fail_unless(buffer == NULL, NULL);
}
END_TEST

START_TEST (check_base64_encode_null2)
{
	SmlError *error = NULL;
	char *buffer = (char *)1;
	
	fail_unless(smlBase64EncodeBinary(NULL, 0, &buffer, &error), NULL);
	fail_unless(error == NULL);
	
	fail_unless(buffer == NULL, NULL);
}
END_TEST

START_TEST (check_base64_encode_test)
{
	char *data = "test";
	SmlError *error = NULL;
	char *buffer = NULL;
	
	fail_unless(smlBase64Encode(data, &buffer, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!strcmp(buffer, "dGVzdA=="), NULL);
	g_free(buffer);
}
END_TEST

START_TEST (check_base64_encode_test2)
{
	char *data = "test1";
	SmlError *error = NULL;
	char *buffer = NULL;
	
	fail_unless(smlBase64Encode(data, &buffer, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!strcmp(buffer, "dGVzdDE="), NULL);
	g_free(buffer);
}
END_TEST

START_TEST (check_base64_encode_test3)
{
	char *data = "test11";
	SmlError *error = NULL;
	char *buffer = NULL;
	
	fail_unless(smlBase64Encode(data, &buffer, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(!strcmp(buffer, "dGVzdDEx"), NULL);
	g_free(buffer);
}
END_TEST

Suite *base64_suite(void)
{
	Suite *s = suite_create("Base64");
	//Suite *s2 = suite_create("Base64");
	
	create_case(s, "check_base64_decode_null", check_base64_decode_null);
	create_case(s, "check_base64_decode_null2", check_base64_decode_null2);
	create_case(s, "check_base64_decode_test", check_base64_decode_test);
	create_case(s, "check_base64_decode_size", check_base64_decode_size);
	create_case(s, "check_base64_decode2", check_base64_decode2);
	create_case(s, "check_base64_decode3", check_base64_decode3);
	create_case(s, "check_base64_decode4", check_base64_decode4);
	create_case(s, "check_base64_invalid1", check_base64_invalid1);
	create_case(s, "check_base64_empty", check_base64_empty);
	create_case(s, "check_base64_invalid2", check_base64_invalid2);

	create_case(s, "check_base64_encode_null", check_base64_encode_null);
	create_case(s, "check_base64_encode_null2", check_base64_encode_null2);
	create_case(s, "check_base64_encode_test", check_base64_encode_test);
	create_case(s, "check_base64_encode_test2", check_base64_encode_test2);
	create_case(s, "check_base64_encode_test3", check_base64_encode_test3);

	return s;
}

int main(void)
{
	int nf;

	Suite *s = base64_suite();
	
	SRunner *sr;
	sr = srunner_create(s);
	srunner_run_all(sr, CK_VERBOSE);
	nf = srunner_ntests_failed(sr);
	srunner_free(sr);
	return (nf == 0) ? 0 : 1;
}
