/*
 * Copyright (C) 2009 Splitted-Desktop Systems. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT.
 * IN NO EVENT SHALL PRECISION INSIGHT AND/OR ITS SUPPLIERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "config.h"
#include "va.h"
#include "va_compat.h"
#include "va_backend.h"
#include <stdlib.h>
#include <string.h>
#include <alloca.h>
#include <assert.h>

#define CTX(dpy)        (((VADisplayContextP)dpy)->pDriverContext)
#define COMPAT_CTX(ctx) ((VACompatContextP)(ctx)->compat)
#define ASSERT		assert

/* Invoke the normal VA API entry-point. This one can be replaced with
   one of the va_compat_*() functions */
#define VA_INVOKE(FUNC, CTX, ...) \
    (CTX)->vtable.FUNC(CTX, __VA_ARGS__)

/* Invoke the real VA API entry-point. That is, the driver's function */
#define VA_INVOKE_COMPAT_ARGS(CTX, ...) \
    (COMPAT_CTX(CTX)->compat_ctx, __VA_ARGS__)
#define VA_INVOKE_COMPAT(FUNC, CTX, ...) \
    COMPAT_CTX(CTX)->vtable.FUNC VA_INVOKE_COMPAT_ARGS(CTX, __VA_ARGS__)

typedef struct VACompatContext *VACompatContextP;

typedef VAStatus (*VABufferTranslateFunc)(VACompatContextP ctx, void *dest, const void *src);

typedef enum {
    VA_DRIVER_ID_UNKNOWN,
    VA_DRIVER_ID_POULSBO,
    VA_DRIVER_ID_IEGD
} VADriverID;

typedef struct {
    VABufferID                  id;
    VABufferType                type;
    unsigned int                size;
    unsigned int                num_elements;
    unsigned char              *data;
    VABufferTranslateFunc       translate;
    VABufferID                  compat_id;
    unsigned int                compat_size;
    unsigned char              *compat_data;
    unsigned int                map_count;
} VABufferCompat;

typedef struct VAContextMap *VAContextMapP;
typedef struct VAContextMap {
    VASurfaceID                 surface;
    VAContextID                 context;
    VAContextMapP               next;
} VAContextMap;

typedef struct {
    VAStatus (*vaTerminate)(void *);
    VAStatus (*vaQueryConfigProfiles)(void *, VAProfile *, int *);
    VAStatus (*vaQueryConfigEntrypoints)(void *, VAProfile, VAEntrypoint *, int *);
    VAStatus (*vaGetConfigAttributes)(void *, VAProfile, VAEntrypoint, VAConfigAttrib *, int);
    VAStatus (*vaCreateConfig)(void *, VAProfile, VAEntrypoint, VAConfigAttrib *, int, VAConfigID *);
    VAStatus (*vaDestroyConfig)(void *, VAConfigID);
    VAStatus (*vaQueryConfigAttributes)(void *, VAConfigID, VAProfile *, VAEntrypoint *, VAConfigAttrib *, int *);
    VAStatus (*vaCreateSurfaces)(void *, int, int, int, int, VASurfaceID *);
    VAStatus (*vaDestroySurfaces)(void *, VASurfaceID *, int);
    VAStatus (*vaCreateContext)(void *, VAConfigID, int, int, int, VASurfaceID *, int, VAContextID *);
    VAStatus (*vaDestroyContext)(void *, VAContextID);
    VAStatus (*vaCreateBuffer)(void *, VAContextID, VABufferType, unsigned int, unsigned int, void *, VABufferID *);
    VAStatus (*vaBufferSetNumElements)(void *, VABufferID, unsigned int);
    VAStatus (*vaMapBuffer)(void *, VABufferID, void **);
    VAStatus (*vaUnmapBuffer)(void *, VABufferID);
    VAStatus (*vaDestroyBuffer)(void *, VABufferID);
    VAStatus (*vaBeginPicture)(void *, VAContextID, VASurfaceID);
    VAStatus (*vaRenderPicture)(void *, VAContextID, VABufferID *, int);
    VAStatus (*vaEndPicture)(void *, VAContextID);
    VAStatus (*vaSyncSurface_pre31)(void *, VAContextID, VASurfaceID);
    VAStatus (*vaQuerySurfaceStatus)(void *, VASurfaceID, VASurfaceStatus *);
    VAStatus (*vaPutSurface)(void *, VASurfaceID, unsigned long, short, short, unsigned short, unsigned short, short, short, unsigned short, unsigned short, VARectangle *, unsigned int, unsigned int);
    VAStatus (*vaQueryImageFormats)(void *, VAImageFormat *, int *);
    VAStatus (*vaCreateImage)(void *, VAImageFormat *, int, int, VAImage *);
    VAStatus (*vaDeriveImage)(void *, VASurfaceID, VAImage *);
    VAStatus (*vaDestroyImage)(void *, VAImageID);
    VAStatus (*vaSetImagePalette)(void *, VAImageID, unsigned char *);
    VAStatus (*vaGetImage)(void *, VASurfaceID, int, int, unsigned int, unsigned int, VAImageID);
    VAStatus (*vaPutImage_pre31)(void *, VASurfaceID, VAImageID, int, int, unsigned int, unsigned int, int, int);
    VAStatus (*vaPutImage2_pre31)(void *, VASurfaceID, VAImageID, int, int, unsigned int, unsigned int, int, int, unsigned int, unsigned int);
    VAStatus (*vaQuerySubpictureFormats)(void *, VAImageFormat *, unsigned int *, unsigned int *);
    VAStatus (*vaCreateSubpicture)(void *, VAImageID, VASubpictureID *);
    VAStatus (*vaDestroySubpicture)(void *, VASubpictureID);
    VAStatus (*vaSetSubpictureImage)(void *, VASubpictureID, VAImageID);
    VAStatus (*vaSetSubpictureChromakey)(void *, VASubpictureID, unsigned int, unsigned int, unsigned int);
    VAStatus (*vaSetSubpictureGlobalAlpha)(void *, VASubpictureID, float);
    VAStatus (*vaAssociateSubpicture_pre31)(void *, VASubpictureID, VASurfaceID *, int, short, short, short, short, unsigned short, unsigned short, unsigned int);
    VAStatus (*vaAssociateSubpicture2_pre31)(void *, VASubpictureID, VASurfaceID *, int, short, short, unsigned short, unsigned short, short, short, unsigned short, unsigned short, unsigned int);
    VAStatus (*vaDeassociateSubpicture)(void *, VASubpictureID, VASurfaceID *, int);
    VAStatus (*vaQueryDisplayAttributes)(void *, VADisplayAttribute *, int *);
    VAStatus (*vaGetDisplayAttributes)(void *, VADisplayAttribute *, int);
    VAStatus (*vaSetDisplayAttributes)(void *, VADisplayAttribute *, int);

    /* 0.29 hooks */
    VAStatus (*vaSetSubpicturePalette)(void *, VASubpictureID, unsigned char *);
    VAStatus (*vaDbgCopySurfaceToBuffer)(void *, VASurfaceID, void **, unsigned int *);

    /* 0.30 hooks */
    VAStatus (*vaCreateSurfaceFromCIFrame)(void *, unsigned long, VASurfaceID *);
    VAStatus (*vaCreateSurfaceFromV4L2Buf)(void *, int, struct v4l2_format *, struct v4l2_buffer *, VASurfaceID *);
    VAStatus (*vaCopySurfaceToBuffer)(void *, VASurfaceID, unsigned int *, unsigned int *, unsigned int *, unsigned int *, unsigned int *, unsigned int *, unsigned int *, void **);

    /* 0.31 hooks */
    VAStatus (*vaPutImage)(void *, VASurfaceID, VAImageID, int, int, unsigned int, unsigned int, int, int, unsigned int, unsigned int);
    VAStatus (*vaAssociateSubpicture)(void *, VASubpictureID, VASurfaceID *, int, short, short, unsigned short, unsigned short, short, short, unsigned short, unsigned short, unsigned int);
    VAStatus (*vaSyncSurface)(void *, VASurfaceID);
} VACompatDriverVTable;

typedef struct VACompatContext {
    VABufferCompat             *buffers;
    unsigned int                buffers_count_max;
    unsigned int                compat_version;
    void                       *compat_ctx;
    VACompatDriverVTable        vtable;                 /* original vtable */
    VADriverID                  driver_id;
    char                       *driver_name;
    VAContextMapP               context_map;
    VAContextMapP               last_context_map_match;
    unsigned int                skip_frame : 1;
} VACompatContext;

#define COMPAT_MAJOR 0
#define COMPAT_MINOR 29
#include "va_compat_template.h"

#define COMPAT_MAJOR 0
#define COMPAT_MINOR 30
#include "va_compat_template.h"

#define COMPAT_MAJOR 0
#define COMPAT_MINOR 31
#include "va_compat_template.h"

static inline int va_IsIntelBuffer(VADriverContextP ctx, VABufferID id)
{
    VACompatContextP compat = ctx->compat;

    if (compat->driver_id == VA_DRIVER_ID_POULSBO) {
        /* There were only Intel implementations for VA API. Besides, the
           returned buffer IDs had the following format [ 0 BASE ID ] where
           BASE is the 7-bit value 0x04 and ID an index into the heap of
           objects */
        return (id & 0x7f000000) == 0x04000000;
    }

    if (compat->driver_id == VA_DRIVER_ID_IEGD) {
        /* XXX: there is no means to differentiate the buffers, they
           are linearly generated (IEGD 10.0 build 1335) */
        return 1;
    }

    return 0;
}

static VAContextMapP va_context_map_lookup_p(
    VACompatContextP    ctx,
    VASurfaceID         surface
)
{
    VAContextMapP m = ctx->context_map;

    while (m) {
        if (m->surface == surface)
            return m;
        m = m->next;
    }
    return NULL;
}

static VAContextID va_context_map_lookup(
    VACompatContextP    ctx,
    VASurfaceID         surface
)
{
    VAContextMapP m;

    /* Lookup in cached */
    m = ctx->last_context_map_match;
    if (m && m->surface == surface)
        return m->context;

    /* Full (slow) lookup */
    m = va_context_map_lookup_p(ctx, surface);
    if (m) {
        ctx->last_context_map_match = m;
        return m->context;
    }
    return VA_INVALID_ID;
}

static void va_context_map_add(
    VACompatContextP    ctx,
    VAContextID         context,
    VASurfaceID         surface
)
{
    VAContextMapP m;

    /* Update existing entry */
    m = va_context_map_lookup_p(ctx, surface);
    if (m) {
        m->context = context;
        return;
    }

    /* Create new mapping */
    m = malloc(sizeof(*m));
    ASSERT(m);
    if (m) {
        m->surface = surface;
        m->context = context;
        m->next    = ctx->context_map;
        ctx->context_map = m;
    }
}

static void va_context_map_remove(
    VACompatContextP    ctx,
    VAContextID         context
)
{
    VAContextMapP p = NULL;
    VAContextMapP m = ctx->context_map;
    VAContextMapP d;

    while (m) {
        if (m->context == context) {
            d = m;

            /* Unlink current node */
            if (p)
                p->next = m->next;
            else
                ctx->context_map = m->next;
            m = m->next;

            /* Reset cache */
            if (ctx->last_context_map_match == d)
                ctx->last_context_map_match = NULL;

            free(d);
        }
        else {
            p = m;
            m = m->next;
        }
    }
}

static VABufferCompat *va_GetBufferCompat (
    VADriverContextP ctx,
    VABufferID id
)
{
    VACompatContextP compat = ctx->compat;
    int index;

    if (!va_IsIntelBuffer(ctx, id))
        return NULL;

    index = id & 0x00ffffff;
    if (index >= compat->buffers_count_max)
        return NULL;

    if (compat->buffers[index].id != id)
        return NULL;

    return &compat->buffers[index];
}

static VAStatus va_TranslateBufferCompat (
    VADriverContextP ctx,
    VABufferCompat *compat_buffer
)
{
    VAStatus status;
    unsigned char *src, *dest;
    int i;

    status = VA_INVOKE(vaMapBuffer, ctx, compat_buffer->id, (void **)&src);
    if (status != VA_STATUS_SUCCESS)
        return status;

    ASSERT(compat_buffer->data);
    ASSERT(src == compat_buffer->data);
    ASSERT(compat_buffer->compat_data);
    dest = compat_buffer->compat_data;
    for (i = 0; i < compat_buffer->num_elements; i++)
    {
        /* XXX: check errors */
        status = compat_buffer->translate(ctx->compat, dest, src);

        src   += compat_buffer->size;
        dest  += compat_buffer->compat_size;
    }

    return VA_INVOKE(vaUnmapBuffer, ctx, compat_buffer->id);
}

static VAStatus va_CreateBufferCompat (
    VADriverContextP ctx,
    VAContextID context,
    VABufferID id,
    VABufferType type,
    unsigned int size,
    unsigned int num_elements,
    unsigned int compat_size,
    VABufferTranslateFunc translate_func
)
{
    VACompatContextP compat = ctx->compat;
    VABufferCompat *compat_buffer;
    int index;

    if (!va_IsIntelBuffer(ctx, id))
        return VA_STATUS_ERROR_INVALID_BUFFER;
    index = id & 0x00ffffff;

    /* XXX: this allocation strategy is not really space efficient... */
    if (index >= compat->buffers_count_max)
    {
        compat->buffers = realloc(
            compat->buffers,
            ((index + 1) * sizeof(VABufferCompat))
        );
        if (compat->buffers == NULL)
            return VA_STATUS_ERROR_ALLOCATION_FAILED;
        memset(
            &compat->buffers[compat->buffers_count_max],
            0,
            (index + 1 - compat->buffers_count_max) * sizeof(compat->buffers[0])
        );
        compat->buffers_count_max = index + 1;
    }

    compat_buffer = &compat->buffers[index];
    compat_buffer->id           = id;
    compat_buffer->type         = type;
    compat_buffer->size         = size;
    compat_buffer->num_elements = num_elements;
    compat_buffer->data         = NULL;
    compat_buffer->translate    = translate_func;
    compat_buffer->compat_size  = compat_size;
    compat_buffer->compat_data  = NULL;
    compat_buffer->map_count    = 0;

    return VA_INVOKE_COMPAT(vaCreateBuffer,
                            ctx,
                            context,
                            type,
                            compat_size,
                            num_elements,
                            NULL,
                            &compat_buffer->compat_id);
}

static VAStatus va_DestroyBufferCompat (
    VADriverContextP ctx,
    VABufferID id
)
{
    VABufferCompat *compat_buffer;
    VAStatus status;

    if ((compat_buffer = va_GetBufferCompat(ctx, id)) == NULL)
        return VA_STATUS_SUCCESS;

    /* Force unmap if there were more maps than unmaps */
    if (compat_buffer->map_count > 1)
        compat_buffer->map_count = 1;
    if (compat_buffer->map_count > 1)
    {
        if ((status = VA_INVOKE(vaUnmapBuffer, ctx, id)) != VA_STATUS_SUCCESS)
            return status;
    }

    compat_buffer->id = 0;
    return VA_INVOKE_COMPAT(vaDestroyBuffer, ctx, compat_buffer->compat_id);
}

static VAStatus va_compat_CreateBuffer (
    VADriverContextP ctx,
    VAContextID context,	/* in */
    VABufferType type,		/* in */
    unsigned int size,		/* in */
    unsigned int num_elements,	/* in */
    void *data,			/* in */
    VABufferID *buf_id		/* out */
)
{
    VABufferTranslateFunc translate_func = NULL;
    unsigned int compat_size = 0;
    VAStatus status;

    status = VA_INVOKE_COMPAT(vaCreateBuffer,
                              ctx,
                              context,
                              type,
                              size,
                              num_elements,
                              data,
                              buf_id);

    if (status != VA_STATUS_SUCCESS)
        return status;

#define INIT_TRANSLATE_FUNC_(STRUCT, MAJOR, MINOR) do {                 \
    translate_func = va_compat_translate_VA##STRUCT##_##MAJOR##_##MINOR; \
    compat_size = sizeof(VA##STRUCT##_##MAJOR##_##MINOR);               \
} while (0)
#define INIT_TRANSLATE_FUNC(BUFFER, CODEC, MAJOR, MINOR) \
    INIT_TRANSLATE_FUNC_(BUFFER##Buffer##CODEC, MAJOR, MINOR)

    /* XXX: this assumes all structures have different sizes from each other */
    switch (size) {
    case sizeof(VAPictureParameterBufferH264):
        if (type == VAPictureParameterBufferType && ctx->version_major == 0)
        {
            switch (ctx->version_minor) {
            case 29: INIT_TRANSLATE_FUNC(PictureParameter,H264, 0,29); break;
            case 30: INIT_TRANSLATE_FUNC(PictureParameter,H264, 0,30); break;
            }
        }
        break;
    case sizeof(VASliceParameterBufferH264):
        if (type == VASliceParameterBufferType && ctx->version_major == 0)
        {
            switch (ctx->version_minor) {
            case 29: INIT_TRANSLATE_FUNC(SliceParameter,H264, 0,29); break;
            case 30: INIT_TRANSLATE_FUNC(SliceParameter,H264, 0,30); break;
            }
        }
        break;
    case sizeof(VAPictureParameterBufferVC1):
        if (type == VAPictureParameterBufferType && ctx->version_major == 0)
        {
            switch (ctx->version_minor) {
            case 29: INIT_TRANSLATE_FUNC(PictureParameter,VC1, 0,29); break;
            case 30: INIT_TRANSLATE_FUNC(PictureParameter,VC1, 0,30); break;
            }
        }
        break;
    case sizeof(VAPictureParameterBufferMPEG2):
        if (type == VAPictureParameterBufferType && ctx->version_major == 0)
        {
            switch (ctx->version_minor) {
            case 29: INIT_TRANSLATE_FUNC(PictureParameter,MPEG2, 0,29); break;
            case 30: INIT_TRANSLATE_FUNC(PictureParameter,MPEG2, 0,30); break;
            }
        }
        break;
    case sizeof(VASliceParameterBufferMPEG2):
        if (type == VASliceParameterBufferType && ctx->version_major == 0)
        {
            switch (ctx->version_minor) {
            case 29: INIT_TRANSLATE_FUNC(SliceParameter,MPEG2, 0,29); break;
            case 30: INIT_TRANSLATE_FUNC(SliceParameter,MPEG2, 0,30); break;
            }
        }
        break;
    case sizeof(VAPictureParameterBufferMPEG4):
        if (type == VAPictureParameterBufferType && ctx->version_major == 0)
        {
            switch (ctx->version_minor) {
            case 29: INIT_TRANSLATE_FUNC(PictureParameter,MPEG4, 0,29); break;
            case 30: INIT_TRANSLATE_FUNC(PictureParameter,MPEG4, 0,30); break;
            case 31: INIT_TRANSLATE_FUNC(PictureParameter,MPEG4, 0,31); break;
            }
        }
        break;
    case sizeof(VAEncSliceParameterBuffer):
        if (type == VAEncSliceParameterBufferType && ctx->version_major == 0)
        {
            switch (ctx->version_minor) {
            case 30: INIT_TRANSLATE_FUNC_(EncSliceParameterBuffer, 0,30); break;
            }
        }
        break;
    }

#undef INIT_TRANSLATE_FUNC

    /* Create thunk */
    if (buf_id && translate_func)
    {
        ASSERT(compat_size > 0);

        status = va_CreateBufferCompat(ctx,
                                       context,
                                       *buf_id,
                                       type,
                                       size,
                                       num_elements,
                                       compat_size,
                                       translate_func);
    }

    return status;
}

static VAStatus va_compat_BufferSetNumElements (
    VADriverContextP ctx,
    VABufferID buf_id,	/* in */
    unsigned int num_elements /* in */
)
{
    VABufferCompat *compat_buffer;
    VAStatus status;

    status = VA_INVOKE_COMPAT(vaBufferSetNumElements, ctx, buf_id, num_elements);
    if (status != VA_STATUS_SUCCESS)
        return status;

    if ((compat_buffer = va_GetBufferCompat(ctx, buf_id)) != NULL)
    {
        compat_buffer->num_elements = num_elements;
        status = VA_INVOKE_COMPAT(vaBufferSetNumElements,
                                  ctx,
                                  compat_buffer->compat_id,
                                  num_elements);
    }

    return status;
}

static VAStatus va_compat_DestroyBuffer (
    VADriverContextP ctx,
    VABufferID buffer_id
)
{
    VAStatus status;
    if ((status = va_DestroyBufferCompat(ctx, buffer_id)) != VA_STATUS_SUCCESS)
        return status;

    return VA_INVOKE_COMPAT(vaDestroyBuffer, ctx, buffer_id);
}

static VAStatus va_compat_MapBuffer (
    VADriverContextP ctx,
    VABufferID buf_id,	/* in */
    void **pbuf 	/* out */
)
{
    VABufferCompat *compat_buffer;
    VAStatus status;

    if ((compat_buffer = va_GetBufferCompat(ctx, buf_id)) == NULL)
        return VA_INVOKE_COMPAT(vaMapBuffer, ctx, buf_id, pbuf);

    if (compat_buffer->map_count++ == 0)
    {
        status = VA_INVOKE_COMPAT(vaMapBuffer, ctx, buf_id, (void **)&compat_buffer->data);
        if (status != VA_STATUS_SUCCESS)
            return status;

        status = VA_INVOKE_COMPAT(vaMapBuffer, ctx, compat_buffer->compat_id, (void **)&compat_buffer->compat_data);
        if (status != VA_STATUS_SUCCESS)
            return status;
    }

    if (pbuf)
        *pbuf = compat_buffer->data;

    return VA_STATUS_SUCCESS;
}

static VAStatus va_compat_UnmapBuffer (
    VADriverContextP ctx,
    VABufferID buf_id	/* in */
)
{
    VABufferCompat *compat_buffer;
    VAStatus status;

    if ((compat_buffer = va_GetBufferCompat(ctx, buf_id)) == NULL)
        return VA_INVOKE_COMPAT(vaUnmapBuffer, ctx, buf_id);

    if (--compat_buffer->map_count == 0)
    {
        status = VA_INVOKE_COMPAT(vaUnmapBuffer, ctx, compat_buffer->compat_id);
        if (status != VA_STATUS_SUCCESS)
            return status;
        compat_buffer->compat_data = NULL;

        status = VA_INVOKE_COMPAT(vaUnmapBuffer, ctx, compat_buffer->id);
        if (status != VA_STATUS_SUCCESS)
            return status;
        compat_buffer->data = NULL;
    }

    return VA_STATUS_SUCCESS;
}

static VAStatus va_compat_BeginPicture (
    VADriverContextP ctx,
    VAContextID context,
    VASurfaceID render_target
)
{
    VACompatContextP compat = ctx->compat;
    compat->skip_frame = 0;
    return VA_INVOKE_COMPAT(vaBeginPicture, ctx, context, render_target);
}

static VAStatus va_compat_EndPicture (
    VADriverContextP ctx,
    VAContextID context
)
{
    VACompatContextP compat = ctx->compat;
    VAStatus status = VA_INVOKE_COMPAT(vaEndPicture, ctx, context);

    /* Ignore errors if the HW decoder did not handle VC-1 skipped P-frames */
    if (compat->skip_frame && status == VA_STATUS_ERROR_UNKNOWN)
        status = VA_STATUS_SUCCESS;

    return status;
}

static VAStatus va_compat_RenderPicture (
    VADriverContextP ctx,
    VAContextID context,
    VABufferID *buffers,
    int num_buffers
)
{
    VACompatContextP compat = ctx->compat;
    VABufferCompat *compat_buffer;
    VABufferID *compat_buffer_ids;
    VAStatus status;
    int i, n;

    if ((n = num_buffers) < 1)
        n = 1;
    compat_buffer_ids = alloca(n * sizeof(compat_buffer_ids[0]));

    for (i = 0; i < num_buffers; i++)
    {
        if ((compat_buffer = va_GetBufferCompat(ctx, buffers[i])) == NULL)
            compat_buffer_ids[i] = buffers[i];
        else
        {
            status = va_TranslateBufferCompat(ctx, compat_buffer);
            if (status != VA_STATUS_SUCCESS)
                return status;
            compat_buffer_ids[i] = compat_buffer->compat_id;
        }
    }

    if (!compat->skip_frame)
    {
        status = VA_INVOKE_COMPAT(vaRenderPicture, ctx, context, compat_buffer_ids, num_buffers);
        if (status != VA_STATUS_SUCCESS)
            return status;
    }

    /* Buffers are automatically destroyed afterwards */
    for (i = 0; i < num_buffers; i++)
    {
        if ((compat_buffer = va_GetBufferCompat(ctx, buffers[i])) != NULL)
        {
            status = VA_INVOKE_COMPAT(vaDestroyBuffer, ctx, compat_buffer->id);
            if (status != VA_STATUS_SUCCESS)
                return status;
        }
    }

    return VA_STATUS_SUCCESS;
}

#define DEFINE_VTABLE_ENTRY_(RETVAL, PROC, ARGS, COMPAT_PROC, COMPAT_ARGS, COMPAT_PRE, COMPAT_POST) \
static RETVAL va_compat_##PROC ARGS                                        \
{                                                                          \
    if (COMPAT_CTX(ctx)->vtable.va##COMPAT_PROC == NULL)                   \
        return VA_STATUS_ERROR_OPERATION_FAILED;                           \
    COMPAT_PRE;                                                            \
    RETVAL ret = COMPAT_CTX(ctx)->vtable.va##COMPAT_PROC VA_INVOKE_COMPAT_ARGS COMPAT_ARGS; \
    COMPAT_POST;                                                           \
    return ret;                                                            \
}

#define DEFINE_VTABLE_ENTRY(RETVAL, PROC, DECL_ARGS, CALL_ARGS) \
DEFINE_VTABLE_ENTRY_(RETVAL, PROC, DECL_ARGS, PROC, CALL_ARGS, {}, {})

static VAStatus va_compat_Terminate(VADriverContextP ctx)
{
    if (COMPAT_CTX(ctx)->vtable.vaTerminate == NULL)
        return VA_STATUS_ERROR_OPERATION_FAILED;
    return COMPAT_CTX(ctx)->vtable.vaTerminate(COMPAT_CTX(ctx)->compat_ctx);
}

DEFINE_VTABLE_ENTRY(
    VAStatus, QueryConfigProfiles,
    (VADriverContextP ctx, VAProfile *profile_list, int *num_profiles),
    (ctx, profile_list, num_profiles))

DEFINE_VTABLE_ENTRY(
    VAStatus, QueryConfigEntrypoints,
    (VADriverContextP ctx, VAProfile profile,
     VAEntrypoint *entrypoint_list, int *num_entrypoints),
    (ctx, profile, entrypoint_list, num_entrypoints))

DEFINE_VTABLE_ENTRY(
    VAStatus, GetConfigAttributes,
    (VADriverContextP ctx, VAProfile profile, VAEntrypoint entrypoint,
     VAConfigAttrib *attrib_list, int num_attribs),
    (ctx, profile, entrypoint, attrib_list, num_attribs))

DEFINE_VTABLE_ENTRY(
    VAStatus, CreateConfig,
    (VADriverContextP ctx, VAProfile profile, VAEntrypoint entrypoint,
     VAConfigAttrib *attrib_list, int num_attribs, VAConfigID *config_id),
    (ctx, profile, entrypoint, attrib_list, num_attribs, config_id))

DEFINE_VTABLE_ENTRY(
    VAStatus, DestroyConfig,
    (VADriverContextP ctx, VAConfigID config_id),
    (ctx, config_id))

DEFINE_VTABLE_ENTRY(
    VAStatus, QueryConfigAttributes,
    (VADriverContextP ctx, VAConfigID config_id, VAProfile *profile,
     VAEntrypoint *entrypoint, VAConfigAttrib *attrib_list, int *num_attribs),
    (ctx, config_id, profile, entrypoint, attrib_list, num_attribs))

DEFINE_VTABLE_ENTRY(
    VAStatus, CreateSurfaces,
    (VADriverContextP ctx, int width, int height, int format,
     int num_surfaces, VASurfaceID *surfaces),
    (ctx, width, height, format, num_surfaces, surfaces))

DEFINE_VTABLE_ENTRY(
    VAStatus, DestroySurfaces,
    (VADriverContextP ctx, VASurfaceID *surface_list, int num_surfaces),
    (ctx, surface_list, num_surfaces))

DEFINE_VTABLE_ENTRY(
    VAStatus, CreateContext,
    (VADriverContextP ctx, VAConfigID config_id,
     int picture_width, int picture_height, int flag,
     VASurfaceID *render_targets, int num_render_targets, VAContextID *context),
    (ctx, config_id, picture_width, picture_height, flag,
     render_targets, num_render_targets, context))

DEFINE_VTABLE_ENTRY(
    VAStatus, DestroyContext,
    (VADriverContextP ctx, VAContextID context),
    (ctx, context))

DEFINE_VTABLE_ENTRY(
    VAStatus, SyncSurface,
    (VADriverContextP ctx, VASurfaceID render_target),
    (ctx, render_target))

DEFINE_VTABLE_ENTRY(
    VAStatus, QuerySurfaceStatus,
    (VADriverContextP ctx, VASurfaceID render_target, VASurfaceStatus *status),
    (ctx, render_target, status))

DEFINE_VTABLE_ENTRY_(
    VAStatus, PutSurface,
    (VADriverContextP ctx, VASurfaceID surface, Drawable draw,
     short srcx, short srcy, unsigned short srcw, unsigned short srch,
     short destx, short desty, unsigned short destw, unsigned short desth,
     VARectangle *cliprects, unsigned int number_cliprects, unsigned int flags),
    PutSurface,
    (ctx, surface, draw, srcx, srcy, srcw, srch, destx, desty, destw, desth,
     cliprects, number_cliprects, flags),
    {
        if (COMPAT_CTX(ctx)->compat_version < 30) /* 0.30 */
            flags &= (VA_TOP_FIELD|VA_BOTTOM_FIELD|VA_CLEAR_DRAWABLE);
    },
    {})

DEFINE_VTABLE_ENTRY(
    VAStatus, QueryImageFormats,
    (VADriverContextP ctx, VAImageFormat *format_list, int *num_formats),
    (ctx, format_list, num_formats))

DEFINE_VTABLE_ENTRY(
    VAStatus, CreateImage,
    (VADriverContextP ctx, VAImageFormat *format,
     int width, int height, VAImage *image),
    (ctx, format, width, height, image))

DEFINE_VTABLE_ENTRY(
    VAStatus, DeriveImage,
    (VADriverContextP ctx, VASurfaceID surface, VAImage *image),
    (ctx, surface, image))

DEFINE_VTABLE_ENTRY(
    VAStatus, DestroyImage,
    (VADriverContextP ctx, VAImageID image),
    (ctx, image))

DEFINE_VTABLE_ENTRY(
    VAStatus, SetImagePalette,
    (VADriverContextP ctx, VAImageID image, unsigned char *palette),
    (ctx, image, palette))

DEFINE_VTABLE_ENTRY(
    VAStatus, GetImage,
    (VADriverContextP ctx, VASurfaceID surface,
     int x, int y, unsigned int width, unsigned int height, VAImageID image),
    (ctx, surface, x, y, width, height, image))

DEFINE_VTABLE_ENTRY(
    VAStatus, PutImage,
    (VADriverContextP ctx, VASurfaceID surface, VAImageID image,
     int srcx, int srcy, unsigned int srcw, unsigned int srch,
     int destx, int desty, unsigned int destw, unsigned int desth),
    (ctx, surface, image, srcx, srcy, srcw, srch, destx, desty, destw, desth))

DEFINE_VTABLE_ENTRY(
    VAStatus, QuerySubpictureFormats,
    (VADriverContextP ctx, VAImageFormat *format_list,
     unsigned int *flags, unsigned int *num_formats),
    (ctx, format_list, flags, num_formats))

DEFINE_VTABLE_ENTRY(
    VAStatus, CreateSubpicture,
    (VADriverContextP ctx, VAImageID image, VASubpictureID *subpicture),
    (ctx, image, subpicture))

DEFINE_VTABLE_ENTRY(
    VAStatus, DestroySubpicture,
    (VADriverContextP ctx, VASubpictureID subpicture),
    (ctx, subpicture))

DEFINE_VTABLE_ENTRY(
    VAStatus, SetSubpictureImage,
    (VADriverContextP ctx, VASubpictureID subpicture, VAImageID image),
    (ctx, subpicture, image))

DEFINE_VTABLE_ENTRY(
    VAStatus, SetSubpictureChromakey,
    (VADriverContextP ctx, VASubpictureID subpicture,
     unsigned int chromakey_min,
     unsigned int chromakey_max,
     unsigned int chromakey_mask),
    (ctx, subpicture, chromakey_min, chromakey_max, chromakey_mask))

DEFINE_VTABLE_ENTRY(
    VAStatus, SetSubpictureGlobalAlpha,
    (VADriverContextP ctx, VASubpictureID subpicture, float global_alpha),
    (ctx, subpicture, global_alpha))

DEFINE_VTABLE_ENTRY(
    VAStatus, AssociateSubpicture,
    (VADriverContextP ctx, VASubpictureID subpicture,
     VASurfaceID *target_surfaces, int num_surfaces,
     short srcx, short srcy, unsigned short srcw, unsigned short srch,
     short destx, short desty, unsigned short destw, unsigned short desth,
     unsigned int flags),
    (ctx, subpicture, target_surfaces, num_surfaces,
     srcx, srcy, srcw, srch, destx, desty, destw, desth, flags))

DEFINE_VTABLE_ENTRY(
    VAStatus, DeassociateSubpicture,
    (VADriverContextP ctx, VASubpictureID subpicture,
     VASurfaceID *target_surfaces, int num_surfaces),
    (ctx, subpicture, target_surfaces, num_surfaces))

DEFINE_VTABLE_ENTRY(
    VAStatus, QueryDisplayAttributes,
    (VADriverContextP ctx, VADisplayAttribute *attr_list, int *num_attributes),
    (ctx, attr_list, num_attributes))

DEFINE_VTABLE_ENTRY(
    VAStatus, GetDisplayAttributes,
    (VADriverContextP ctx, VADisplayAttribute *attr_list, int num_attributes),
    (ctx, attr_list, num_attributes))

DEFINE_VTABLE_ENTRY(
    VAStatus, SetDisplayAttributes,
    (VADriverContextP ctx, VADisplayAttribute *attr_list, int num_attributes),
    (ctx, attr_list, num_attributes))

DEFINE_VTABLE_ENTRY(
    VAStatus, CreateSurfaceFromCIFrame,
    (VADriverContextP ctx, unsigned long frame_id, VASurfaceID *surface),
    (ctx, frame_id, surface))

DEFINE_VTABLE_ENTRY(
    VAStatus, CreateSurfaceFromV4L2Buf,
    (VADriverContextP ctx, int v4l2_fd,
     struct v4l2_format *v4l2_fmt, struct v4l2_buffer *v4l2_buf,
     VASurfaceID *surface),
    (ctx, v4l2_fd, v4l2_fmt, v4l2_buf, surface))

DEFINE_VTABLE_ENTRY(
    VAStatus, CopySurfaceToBuffer,
    (VADriverContextP ctx, VASurfaceID surface, unsigned int *fourcc,
     unsigned int *luma_stride,
     unsigned int *chroma_u_stride, unsigned int *chroma_v_stride,
     unsigned int *luma_offset,
     unsigned int *chroma_u_offset, unsigned int *chroma_v_offset,
     void **buffer),
    (ctx, surface, fourcc,
     luma_stride, chroma_u_stride, chroma_v_stride,
     luma_offset, chroma_u_offset, chroma_v_offset,
     buffer))

DEFINE_VTABLE_ENTRY_(
    VAStatus, SyncSurface_pre31,
    (VADriverContextP ctx, VASurfaceID render_target),
    SyncSurface_pre31,
    (ctx, va_context_map_lookup(COMPAT_CTX(ctx), render_target), render_target),
    {}, {})

DEFINE_VTABLE_ENTRY_(
    VAStatus, PutImage_pre31,
    (VADriverContextP ctx, VASurfaceID surface, VAImageID image,
     int srcx, int srcy, unsigned int srcw, unsigned int srch,
     int destx, int desty, unsigned int destw, unsigned int desth),
    PutImage2_pre31,
    (ctx, surface, image, srcx, srcy, srcw, srch, destx, desty, destw, desth),
    {}, {})

DEFINE_VTABLE_ENTRY_(
    VAStatus, AssociateSubpicture_pre31,
    (VADriverContextP ctx, VASubpictureID subpicture,
     VASurfaceID *target_surfaces, int num_surfaces,
     short srcx, short srcy, unsigned short srcw, unsigned short srch,
     short destx, short desty, unsigned short destw, unsigned short desth,
     unsigned int flags),
    AssociateSubpicture2_pre31,
    (ctx, subpicture, target_surfaces, num_surfaces,
     srcx, srcy, srcw, srch, destx, desty, destw, desth, flags),
    {}, {})

DEFINE_VTABLE_ENTRY_(
    VAStatus, CreateContext_pre31,
    (VADriverContextP ctx, VAConfigID config_id,
     int picture_width, int picture_height, int flag,
     VASurfaceID *render_targets, int num_render_targets, VAContextID *context),
    CreateContext,
    (ctx, config_id, picture_width, picture_height, flag,
     render_targets, num_render_targets, context),
    {}, {
        VACompatContextP const compat_ctx = COMPAT_CTX(ctx);
        int i;
        for (i = 0; i < num_render_targets; i++)
            va_context_map_add(compat_ctx, *context, render_targets[i]);
    })

DEFINE_VTABLE_ENTRY_(
    VAStatus, DestroyContext_pre31,
    (VADriverContextP ctx, VAContextID context),
    DestroyContext,
    (ctx, context),
    {}, { va_context_map_remove(COMPAT_CTX(ctx), context); })

#undef DEFINE_VTABLE_ENTRY
#undef DEFINE_VTABLE_ENTRY_

static void va_compat_init_VADriverVTable(VADriverContextP ctx, int compat_version)
{
#define INIT_VTABLE_(CTX, DST_PROC, SRC_PROC) \
    (CTX)->vtable.va##DST_PROC = va_compat_##SRC_PROC
#define INIT_VTABLE(CTX, PROC) \
    INIT_VTABLE_(CTX, PROC, PROC)

    INIT_VTABLE(ctx, Terminate);
    INIT_VTABLE(ctx, QueryConfigProfiles);
    INIT_VTABLE(ctx, QueryConfigEntrypoints);
    INIT_VTABLE(ctx, GetConfigAttributes);
    INIT_VTABLE(ctx, CreateConfig);
    INIT_VTABLE(ctx, DestroyConfig);
    INIT_VTABLE(ctx, QueryConfigAttributes);
    INIT_VTABLE(ctx, CreateSurfaces);
    INIT_VTABLE(ctx, DestroySurfaces);
    INIT_VTABLE(ctx, CreateContext);
    INIT_VTABLE(ctx, DestroyContext);
    INIT_VTABLE(ctx, CreateBuffer);
    INIT_VTABLE(ctx, BufferSetNumElements);
    INIT_VTABLE(ctx, MapBuffer);
    INIT_VTABLE(ctx, UnmapBuffer);
    INIT_VTABLE(ctx, DestroyBuffer);
    INIT_VTABLE(ctx, BeginPicture);
    INIT_VTABLE(ctx, RenderPicture);
    INIT_VTABLE(ctx, EndPicture);
    INIT_VTABLE(ctx, SyncSurface);
    INIT_VTABLE(ctx, QuerySurfaceStatus);
    INIT_VTABLE(ctx, PutSurface);
    INIT_VTABLE(ctx, QueryImageFormats);
    INIT_VTABLE(ctx, CreateImage);
    INIT_VTABLE(ctx, DeriveImage);
    INIT_VTABLE(ctx, DestroyImage);
    INIT_VTABLE(ctx, SetImagePalette);
    INIT_VTABLE(ctx, GetImage);
    INIT_VTABLE(ctx, PutImage);
    INIT_VTABLE(ctx, QuerySubpictureFormats);
    INIT_VTABLE(ctx, CreateSubpicture);
    INIT_VTABLE(ctx, DestroySubpicture);
    INIT_VTABLE(ctx, SetSubpictureImage);
    INIT_VTABLE(ctx, SetSubpictureChromakey);
    INIT_VTABLE(ctx, SetSubpictureGlobalAlpha);
    INIT_VTABLE(ctx, AssociateSubpicture);
    INIT_VTABLE(ctx, DeassociateSubpicture);
    INIT_VTABLE(ctx, QueryDisplayAttributes);
    INIT_VTABLE(ctx, GetDisplayAttributes);
    INIT_VTABLE(ctx, SetDisplayAttributes);
    INIT_VTABLE(ctx, CreateSurfaceFromCIFrame);
    INIT_VTABLE(ctx, CreateSurfaceFromV4L2Buf);
    INIT_VTABLE(ctx, CopySurfaceToBuffer);

    if (compat_version && compat_version < 31) {
        INIT_VTABLE_(ctx, CreateContext,        CreateContext_pre31);
        INIT_VTABLE_(ctx, DestroyContext,       DestroyContext_pre31);
        INIT_VTABLE_(ctx, SyncSurface,          SyncSurface_pre31);
        INIT_VTABLE_(ctx, PutImage,             PutImage_pre31);
        INIT_VTABLE_(ctx, AssociateSubpicture,  AssociateSubpicture_pre31);
    }

#undef INIT_VTABLE
#undef INIT_VTABLE__
}

VAStatus va_compat_init(VADisplay dpy, int compat_version, void *compat_ctx)
{
    VADisplayContextP pDisplayContext = (VADisplayContextP)dpy;
    VADriverContextP ctx = CTX(dpy);
    VADriverContextP_0_29 ctx_0_29;
    VADriverContextP_0_30 ctx_0_30;
    VACompatContextP compat;
    VAStatus status;
    char *driver_name;

    ctx->compat                         = NULL;

    if (compat_version == 0)
        return VA_STATUS_SUCCESS;

    ASSERT(compat_ctx);
    if (compat_ctx == NULL)
        return VA_STATUS_ERROR_UNKNOWN;

    driver_name = NULL;
    status = pDisplayContext->vaGetDriverName(pDisplayContext, &driver_name);
    ASSERT(status == VA_STATUS_SUCCESS);
    if (status != VA_STATUS_SUCCESS)
        return status;

    if ((compat = malloc(sizeof(*compat))) == NULL)
        return VA_STATUS_ERROR_ALLOCATION_FAILED;
    compat->buffers                     = NULL;
    compat->buffers_count_max           = 0;
    compat->compat_version              = compat_version;
    compat->compat_ctx                  = NULL;
    compat->driver_name                 = driver_name;
    compat->context_map                 = NULL;
    compat->last_context_map_match      = NULL;
    ctx->compat                         = compat;

    if (strcmp(driver_name, "psb") == 0)
        compat->driver_id = VA_DRIVER_ID_POULSBO;
    else if (strcmp(driver_name, "iegd") == 0)
        compat->driver_id = VA_DRIVER_ID_IEGD;
    else
        compat->driver_id = VA_DRIVER_ID_UNKNOWN;

    switch (compat_version) {
    case 29:
        if ((ctx_0_29 = malloc(sizeof(*ctx_0_29))) == NULL)
            return VA_STATUS_ERROR_ALLOCATION_FAILED;
        memcpy(ctx_0_29, compat_ctx, sizeof(*ctx_0_29));
        va_compat_translate_VADriverContext_0_29(compat, ctx_0_29);
        compat->compat_ctx = ctx_0_29;
        COPY_FIELD(ctx, ctx_0_29, version_major);
        COPY_FIELD(ctx, ctx_0_29, version_minor);
        COPY_FIELD(ctx, ctx_0_29, max_profiles);
        COPY_FIELD(ctx, ctx_0_29, max_entrypoints);
        COPY_FIELD(ctx, ctx_0_29, max_attributes);
        COPY_FIELD(ctx, ctx_0_29, max_image_formats);
        COPY_FIELD(ctx, ctx_0_29, max_subpic_formats);
        COPY_FIELD(ctx, ctx_0_29, max_display_attributes);
        COPY_FIELD(ctx, ctx_0_29, str_vendor);
        break;
    case 30:
        if ((ctx_0_30 = malloc(sizeof(*ctx_0_30))) == NULL)
            return VA_STATUS_ERROR_ALLOCATION_FAILED;
        memcpy(ctx_0_30, compat_ctx, sizeof(*ctx_0_30));
        va_compat_translate_VADriverContext_0_30(compat, ctx_0_30);
        compat->compat_ctx = ctx_0_30;
        COPY_FIELD(ctx, ctx_0_30, version_major);
        COPY_FIELD(ctx, ctx_0_30, version_minor);
        COPY_FIELD(ctx, ctx_0_30, max_profiles);
        COPY_FIELD(ctx, ctx_0_30, max_entrypoints);
        COPY_FIELD(ctx, ctx_0_30, max_attributes);
        COPY_FIELD(ctx, ctx_0_30, max_image_formats);
        COPY_FIELD(ctx, ctx_0_30, max_subpic_formats);
        COPY_FIELD(ctx, ctx_0_30, max_display_attributes);
        COPY_FIELD(ctx, ctx_0_30, str_vendor);
        break;
    case VA_MINOR_VERSION:
        va_compat_translate_VADriverContext(compat, compat_ctx);
        compat->compat_ctx = compat_ctx;
        break;
    default:
        ASSERT(compat_version == 0);
        return VA_STATUS_ERROR_UNKNOWN;
    }

    va_compat_init_VADriverVTable(ctx, compat_version);
    return VA_STATUS_SUCCESS;
}

VAStatus va_compat_fini(VADisplay dpy)
{
    VADriverContextP ctx = CTX(dpy);
    VACompatContextP compat = ctx->compat;
    int i;

    if (compat == NULL)
        return VA_STATUS_SUCCESS;

    if (compat->driver_name)
    {
        free(compat->driver_name);
        compat->driver_name = NULL;
    }

    if (compat->buffers)
    {
        for (i = 0; i < compat->buffers_count_max; i++)
        {
            if (compat->buffers[i].id && compat->buffers[i].size > 0)
                va_DestroyBufferCompat(ctx, compat->buffers[i].id);
        }
        free(compat->buffers);
        compat->buffers = NULL;
    }

    if (compat->compat_ctx && compat->compat_version != VA_MINOR_VERSION)
    {
        free(compat->compat_ctx);
        compat->compat_ctx = NULL;
    }

    if (compat->context_map)
    {
        VAContextMapP d, m = compat->context_map;
        while (m) {
            d = m;
            m = m->next;
            free(d);
        }
    }
    compat->last_context_map_match = NULL;

    free(compat);
    ctx->compat = NULL;
    return VA_STATUS_SUCCESS;
}
