#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <strings.h> /* strcasecmp() */
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include <assert.h>

#include "libdrm_glue.h"

#include "X11/Xlib.h"
#include "va.h"
#include "va_backend.h"

#include "va_dri.h"
#include "va_dricommon.h"

struct dri1_drawable 
{
    struct dri_drawable base;
    union dri_buffer buffer;
    int width;
    int height;
};

static int
firegl_drmOpenMinor(int minor)
{
    char buf[64];
    int fd;

    sprintf(buf, "/dev/ati/card%d", minor);
    if ((fd = open(buf, O_RDWR, 0)) >= 0)
        return fd;
    return -1;
}

static int
firegl_drmOpenByBusID(const char *busid)
{
    int i, fd;
    drmSetVersion sv;
    const char *buf;

    for (i = 0; i < DRM_MAX_MINOR; i++) {
        if ((fd = firegl_drmOpenMinor(i)) < 0)
            continue;
        sv.drm_di_major = 1;
        sv.drm_di_minor = 1;
        sv.drm_dd_major = -1;
        sv.drm_dd_minor = -1;
        libdrm_drmSetInterfaceVersion(fd, &sv);
        buf = libdrm_drmGetBusid(fd);
        if (buf && strcasecmp(buf, busid) == 0) { /* XXX: drmMatchBusID() */
            libdrm_drmFreeBusid(buf);
            return fd;
        }
        if (buf)
            libdrm_drmFreeBusid(buf);
        close(fd);
    }
    return -1;
}

static int
drm_open_once(struct dri_state *dri_state, const char *BusID, int *newlyopened)
{
    dri_state->driConnectedFlag = VA_NONE;
    dri_state->fd = libdrm_drmOpenOnce(NULL, BusID, newlyopened);
    if (dri_state->fd < 0) {
        dri_state->fd = firegl_drmOpenByBusID(BusID);
        if (dri_state->fd >= 0) {
            *newlyopened = 1;
            dri_state->driConnectedFlag |= VA_DRI_AMD;
        }
    }
    return dri_state->fd;
}

static void
drm_close_once(struct dri_state *dri_state)
{
    /* XXX: dri_state->close() doesn't seem to be called, thus this
       function is never called either */
    if (dri_state->fd < 0)
        return;
    if (dri_state->driConnectedFlag & VA_DRI_AMD)
        close(dri_state->fd);
    else
        libdrm_drmCloseOnce(dri_state->fd);
    dri_state->fd = -1;
    dri_state->driConnectedFlag = VA_NONE;
}

static struct dri_drawable * 
dri1CreateDrawable(VADriverContextP ctx, XID x_drawable)
{
    struct dri1_drawable *dri1_drawable;

    dri1_drawable = calloc(1, sizeof(*dri1_drawable));

    if (!dri1_drawable)
        return NULL;

    dri1_drawable->base.x_drawable = x_drawable;

    return &dri1_drawable->base;
}

static void 
dri1DestroyDrawable(VADriverContextP ctx, struct dri_drawable *dri_drawable)
{
    free(dri_drawable);
}

static void 
dri1SwapBuffer(VADriverContextP ctx, struct dri_drawable *dri_drawable)
{

}

static union dri_buffer *
dri1GetRenderingBuffer(VADriverContextP ctx, struct dri_drawable *dri_drawable)
{
    struct dri1_drawable *dri1_drawable = (struct dri1_drawable *)dri_drawable;

    return &dri1_drawable->buffer;
}

static void
dri1Close(VADriverContextP ctx)
{
    struct dri_state *dri_state = (struct dri_state *)ctx->dri_state;

    free_drawable_hashtable(ctx);
    VA_DRIDestroyContext(ctx->x11_dpy, ctx->x11_screen, dri_state->hwContextID);
    assert(dri_state->pSAREA != MAP_FAILED);
    libdrm_drmUnmap(dri_state->pSAREA, SAREA_MAX);
    assert(dri_state->fd >= 0);
    drm_close_once(dri_state);
    VA_DRICloseConnection(ctx->x11_dpy, ctx->x11_screen);
}

Bool 
isDRI1Connected(VADriverContextP ctx, char **driver_name)
{
    struct dri_state *dri_state = (struct dri_state *)ctx->dri_state;
    int direct_capable;
    int driver_major;
    int driver_minor;
    int driver_patch;
    int newlyopened;
    char *BusID;
    drm_magic_t magic;        

    *driver_name = NULL;
    dri_state->fd = -1;
    dri_state->pSAREA = MAP_FAILED;
    dri_state->driConnectedFlag = VA_NONE;

    if (!VA_DRIQueryDirectRenderingCapable(ctx->x11_dpy, 
                                           ctx->x11_screen, 
                                           &direct_capable))
        goto err_out0;

    if (!direct_capable)
        goto err_out0;

    if (!VA_DRIGetClientDriverName(ctx->x11_dpy, ctx->x11_screen, 
                                   &driver_major, &driver_minor,
                                   &driver_patch, driver_name))
        goto err_out0;

    if (!VA_DRIOpenConnection(ctx->x11_dpy, ctx->x11_screen, 
                              &dri_state->hSAREA, &BusID))
        goto err_out0;

    drm_open_once(dri_state, BusID, &newlyopened);
    XFree(BusID);

    if (dri_state->fd < 0)
        goto err_out1;


    if (libdrm_drmGetMagic(dri_state->fd, &magic))
        goto err_out1;

    if (newlyopened && !VA_DRIAuthConnection(ctx->x11_dpy, ctx->x11_screen, magic))
        goto err_out1;

    if (libdrm_drmMap(dri_state->fd, dri_state->hSAREA, SAREA_MAX, &dri_state->pSAREA))
        goto err_out1;

    if (!VA_DRICreateContext(ctx->x11_dpy, ctx->x11_screen,
                             DefaultVisual(ctx->x11_dpy, ctx->x11_screen),
                             &dri_state->hwContextID, &dri_state->hwContext))
        goto err_out1;

    dri_state->driConnectedFlag &= VA_DRI_AMD; /* clear flags but AMD bit */
    dri_state->driConnectedFlag |= VA_DRI1;
    dri_state->createDrawable = dri1CreateDrawable;
    dri_state->destroyDrawable = dri1DestroyDrawable;
    dri_state->swapBuffer = dri1SwapBuffer;
    dri_state->getRenderingBuffer = dri1GetRenderingBuffer;
    dri_state->close = dri1Close;

    return True;

err_out1:
    if (dri_state->pSAREA != MAP_FAILED)
        libdrm_drmUnmap(dri_state->pSAREA, SAREA_MAX);

    if (dri_state->fd >= 0)
        drm_close_once(dri_state);

    VA_DRICloseConnection(ctx->x11_dpy, ctx->x11_screen);

err_out0:
    if (*driver_name)
        XFree(*driver_name);

    dri_state->pSAREA = MAP_FAILED;
    dri_state->fd = -1;
    *driver_name = NULL;
    
    return False;
}

