/*
 * Copyright (C) 2009 Splitted-Desktop Systems. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT.
 * IN NO EVENT SHALL PRECISION INSIGHT AND/OR ITS SUPPLIERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#define _GNU_SOURCE 1
#include "libdrm_glue.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dlfcn.h>
#include <assert.h>

#define LOAD_FUNC_(NAME, RET, ARGS, FALLBACK)   \
    static RET (*lib_##NAME) ARGS;              \
    if (lib_##NAME == NULL) {                   \
        lib_##NAME = libdrm_symbol(#NAME);      \
        if (!lib_##NAME)                        \
            lib_##NAME = FALLBACK;              \
    }                                           \
    assert(lib_##NAME != NULL)

#define LOAD_FUNC(NAME, RET, ARGS)              \
    LOAD_FUNC_(NAME, RET, ARGS, NULL)

static void *libdrm_handle;
static int libdrm_handle_ok = -1;

static inline void *libdrm_symbol(const char *name)
{
    if (!libdrm_open())
        return NULL;
    return dlsym(libdrm_handle, name);
}

int libdrm_open(void)
{
    if (libdrm_handle_ok < 0) {
        libdrm_handle = dlopen("libdrm.so.2", RTLD_LOCAL|RTLD_LAZY);
        libdrm_handle_ok = libdrm_handle != NULL;
    }
    assert(libdrm_handle);
    return libdrm_handle_ok;
}

void libdrm_close(void)
{
    if (libdrm_handle)
        dlclose(libdrm_handle);
}

// Default drmOpenOnce() and drmCloseOnce() implementations based on current GIT
#define DRM_MAX_FDS 16
static struct {
    char *BusID;
    int fd;
    int refcount;
} connection[DRM_MAX_FDS];

static int nr_fds = 0;

// Default implementation for drmOpenOnce() if none exists in the library
static int
libdrm_default_drmOpenOnce(void *unused, const char *BusID, int *newlyopened)
{
    int i;
    int fd;

    for (i = 0; i < nr_fds; i++)
        if (strcmp(BusID, connection[i].BusID) == 0) {
            connection[i].refcount++;
            *newlyopened = 0;
            return connection[i].fd;
        }

    fd = libdrm_drmOpen(unused, BusID);
    if (fd <= 0 || nr_fds == DRM_MAX_FDS)
        return fd;

    connection[nr_fds].BusID = strdup(BusID);
    connection[nr_fds].fd = fd;
    connection[nr_fds].refcount = 1;
    *newlyopened = 1;

    if (0)
        fprintf(stderr, "saved connection %d for %s %d\n", 
                nr_fds, connection[nr_fds].BusID, 
                strcmp(BusID, connection[nr_fds].BusID));
    nr_fds++;
    return fd;
}

// Default implementation for drmCloseOnce() if none exists in the library
static void libdrm_default_drmCloseOnce(int fd)
{
    int i;

    for (i = 0; i < nr_fds; i++) {
        if (fd == connection[i].fd) {
            if (--connection[i].refcount == 0) {
                libdrm_drmClose(connection[i].fd);
                free(connection[i].BusID);
                if (i < --nr_fds) 
                    connection[i] = connection[nr_fds];
                return;
            }
        }
    }
}

// Determine whether the DRM kernel driver has been loaded
int libdrm_drmAvailable(void)
{
    LOAD_FUNC(drmAvailable, int, (void));
    return lib_drmAvailable();
}

// Open the DRM device
int libdrm_drmOpen(const char *name, const char *busid)
{
    LOAD_FUNC(drmOpen, int, (const char *, const char *));
    return lib_drmOpen(name, busid);
}

// Close the device
int libdrm_drmClose(int fd)
{
    LOAD_FUNC(drmClose, int, (int));
    return lib_drmClose(fd);
}

// Open the DRM device (re-use an existing connection)
int libdrm_drmOpenOnce(void *unused, const char *BusID, int *newlyopened)
{
    LOAD_FUNC_(drmOpenOnce, int, (void *, const char *, int *),
               libdrm_default_drmOpenOnce);
    return lib_drmOpenOnce(unused, BusID, newlyopened);
}

// Close the device (unref an existing connection prior to actually closing it)
void libdrm_drmCloseOnce(int fd)
{
    LOAD_FUNC_(drmCloseOnce, void, (int), libdrm_default_drmCloseOnce);
    lib_drmCloseOnce(fd);
}

// DRM connection cookie
int libdrm_drmGetMagic(int fd, drm_magic_t * magic)
{
    LOAD_FUNC(drmGetMagic, int, (int, drm_magic_t *));
    return lib_drmGetMagic(fd, magic);
}

// Issue a set-version ioctl
int libdrm_drmSetInterfaceVersion(int fd, drmSetVersion *version)
{
    LOAD_FUNC(drmSetInterfaceVersion, int, (int, drmSetVersion *));
    return lib_drmSetInterfaceVersion(fd, version);
}

// Get the bus ID of the device
char *libdrm_drmGetBusid(int fd)
{
    LOAD_FUNC(drmGetBusid, char *, (int));
    return lib_drmGetBusid(fd);
}

// Free the bus ID information
void libdrm_drmFreeBusid(const char *busid)
{
    LOAD_FUNC(drmFreeBusid, void, (const char *));
    lib_drmFreeBusid(busid);
}

// Map a region of memory
int libdrm_drmMap(int fd,
                  drm_handle_t handle,
                  drmSize size,
                  drmAddressPtr address)
{
    LOAD_FUNC(drmMap, int, (int, drm_handle_t, drmSize, drmAddressPtr));
    return lib_drmMap(fd, handle, size, address);
}

// Unmap mappings obtained with drmMap()
int libdrm_drmUnmap(drmAddress address, drmSize size)
{
    LOAD_FUNC(drmUnmap, int, (drmAddress, drmSize));
    return lib_drmUnmap(address, size);
}
