/* Copyright 2012 Canonical Ltd.  This software is licensed under the
 * GNU Affero General Public License version 3 (see the file LICENSE).
 *
 * Power parameters utilities.
 *
 * @module Y.maas.power_parameter
 */

YUI.add('maas.power_parameters', function(Y) {

Y.log('loading maas.power_parameters');
var module = Y.namespace('maas.power_parameters');

// Only used to mockup io in tests.
module._io = new Y.IO();

var LinkedContentWidget;

/**
 * A widget class used to have the content of a node dependent of the selected
 * value of a <select> tag.
 *
 */
LinkedContentWidget = function() {
    LinkedContentWidget.superclass.constructor.apply(this, arguments);
};

LinkedContentWidget.NAME = 'dynamic-widget';

Y.extend(LinkedContentWidget, Y.Widget, {

   /**
    * Initialize the widget.
    * - cfg.srcNode is the  node which will be updated when the selected
    *   value of the 'driver node' will change.
    * - cfg.driverNode is the node containing a 'select' element.  When
    *   the selected element will change, the srcNode HTML will be
    *   updated.
    * - cfg.driverEnum is an dictionary which contains all the possible values
    *   of the driverNode's select element.
    * - cfg.templatePrefix is the prefix string which will be used to fetch
    *   all the templates for each possible value of driverEnum.
    *
    * @method initializer
    */
    initializer: function(cfg) {
        this.driverEnum = cfg.driverEnum;
        this.templatePrefix = cfg.templatePrefix;
        this.initTemplates();
    },

   /**
    * Create a dictionary containing the templates for all the possible
    * values from 'this.driverEnum'.
    *
    * @method initTemplates
    */
    initTemplates: function() {
        this.templates = {};
        var driverValue;
        for (driverValue in this.driverEnum) {
            if (this.driverEnum.hasOwnProperty(driverValue)) {
                var type = this.driverEnum[driverValue];
                var template = Y.one(
                    this.templatePrefix + type).getContent();
                this.templates[type] = template;
            }
        }
    },

   /**
    * Bind the widget to events (ot name 'evnt_name') generated by the given
    * 'driverNode'.
    *
    * @method bindTo
    */
    bindTo: function(driverNode, event_name) {
        var self = this;
        Y.one(driverNode).on(event_name, function(e) {
            var newDriverValue = e.currentTarget.get('value');
            self.switchTo(newDriverValue);
        });
        var driverValue = driverNode.get('value');
        this.setVisibility(driverValue);
    },
   /**
    * Hide 'srcNode' if the value of the 'driverNode' is the empty string
    * and show it otherwise.
    *
    * @method setVisibility
    */
    setVisibility: function(driverValue) {
        if (driverValue === '') {
            this.get('srcNode').addClass('hidden');
        }
        else {
            this.get('srcNode').removeClass('hidden');
        }
    },

   /**
    * React to a new value of the driver node: update the HTML of
    * 'srcNode'.
    *
    * @method switchTo
    */
    switchTo: function(newDriverValue) {
        // Remove old fieldset if any.
        var srcNode = this.get('srcNode');
        srcNode.all('fieldset').remove();
        // Insert the template fragment corresponding to the new value
        // of the driver in 'srcNode'.
        var old_innerHTML = srcNode.get('innerHTML');
        srcNode.set(
            'innerHTML', old_innerHTML + this.templates[newDriverValue]);
        this.setVisibility(newDriverValue);
    }

});

module.LinkedContentWidget = LinkedContentWidget;

}, '0.1', {'requires': ['widget', 'io', 'maas.enums']}
);
