# test_magicicada.py
#
# Author: Natalia Bidart <natalia.bidart@gmail.com>
#
# Copyright 2010 Chicharreros
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for magicicada."""

import itertools
import logging
import os
import sys

from functools import wraps

import gobject
import gtk
import pango

from twisted.trial.unittest import TestCase

from magicicada import MagicicadaUI, CONTENT_QUEUE, META_QUEUE, \
                       UBUNTU_ONE_ROOT, syncdaemon
from magicicada.dbusiface import QueueData, FolderData, ShareData, \
                                 NOT_SYNCHED_PATH
from magicicada.helpers import NO_OP, humanize_bytes, get_data_file
from magicicada.tests.helpers import MementoHandler


# It's ok to access private data in the test suite
# pylint: disable=W0212

# Arguments number differs from overridden method
# pylint: disable=W0221

# Instance of 'A' has no 'y' member
# pylint: disable=E1101

# Instance of 'A' has no 'y' member (but some types could not be inferred)
# pylint: disable=E1103


def process_gtk_pendings():
    """Process all gtk pending events."""
    while gtk.events_pending():
        gtk.main_iteration()


def close_dialog((dialog, test)):
    """Call the 'test', close 'dialog'."""
    response = gtk.RESPONSE_CLOSE
    try:
        process_gtk_pendings()
        test()
    finally:
        dialog.response(response)
        process_gtk_pendings()
    return False  # do not be called again


def test_and_click((button, test)):
    """Call the 'test', and click 'button'."""
    try:
        test()
    finally:
        button.clicked()
    return False  # do not be called again


class FakedSyncdaemon(object):
    """A faked syncdaemon."""

    def __init__(self):
        self._meta_paths = []

        self.current_state = syncdaemon.State()
        self.meta_queue = []
        self.content_queue = []
        self.folders = []
        self.processing_meta = False

        self.on_started_callback = NO_OP
        self.on_stopped_callback = NO_OP
        self.on_connected_callback = NO_OP
        self.on_disconnected_callback = NO_OP
        self.on_online_callback = NO_OP
        self.on_offline_callback = NO_OP
        self.status_changed_callback = NO_OP
        self.content_queue_changed_callback = NO_OP
        self.meta_queue_changed_callback = NO_OP
        self.on_metadata_ready_callback = None  # mandatory
        self.shutdown = NO_OP

        self.start = lambda: setattr(self.current_state, 'is_started', True)
        self.quit = lambda: setattr(self.current_state, 'is_started', False)
        self.connect = lambda: setattr(self.current_state,
                                       'is_connected', True)
        self.disconnect = lambda: \
                          setattr(self.current_state, 'is_connected', False)
        self.get_metadata = lambda path: self._meta_paths.append(path)


class MagicicadaUITestCase(TestCase):
    """UI test cases for Magicicada UI."""

    TEST_FILE = get_data_file('tests', 'metadata-test.txt')

    def setUp(self):
        """Init."""
        self.ui = MagicicadaUI(syncdaemon_class=FakedSyncdaemon)
        self._called = False
        self.response = None
        self._set_called = lambda *args, **kwargs: \
                           setattr(self, '_called', (args, kwargs))
        self._failed_test = False

    def tearDown(self):
        """Cleanup."""
        self.ui.on_main_window_destroy(self.ui.main_window)
        self._called = False

        if self._failed_test:
            # no, I'm not raising a bool. pylint: disable=E0702
            raise self._failed_test

    def __getattribute__(self, name):
        """Overwrite the assert methods with safer ones.

        This way if a test called by gobject in the future fails, it
        makes the whole test suite fail.
        """
        if name.startswith('assert') and hasattr(TestCase, name):

            def proxy(*args, **kwargs):
                """Function that will call the real assert."""
                real_assert = getattr(TestCase, name)
                try:
                    real_assert(self, *args, **kwargs)
                except Exception, e:
                    self._failed_test = e
                    raise
            return proxy
        else:
            return TestCase.__getattribute__(self, name)

        if self._failed_test:
            # no, I'm not raising a bool. pylint: disable=E0702
            raise self._failed_test

    def __getattribute__(self, name):
        """Overwrite the assert methods with safer ones.

        This way if a test called by gobject in the future fails, it
        makes the whole test suite fail.
        """
        if name.startswith('assert') and hasattr(TestCase, name):

            def proxy(*args, **kwargs):
                """Function that will call the real assert."""
                real_assert = getattr(TestCase, name)
                try:
                    real_assert(self, *args, **kwargs)
                except Exception, e:
                    self._failed_test = e
                    raise
            return proxy
        else:
            return TestCase.__getattribute__(self, name)

    def do_start(self):
        """Simulate that start fully happened."""
        self.ui.on_start_clicked(self.ui.start)
        self.ui.on_started()

    def do_connect(self):
        """Simulate that connect fully happened."""
        self.do_start()
        self.ui.on_connect_clicked(self.ui.connect)
        self.ui.on_connected()

    def build_some_data(self, data_type, limit=5):
        """Build some data using named_tuple 'data_type'."""
        attrs = data_type._fields
        result = []
        for i in xrange(limit):
            kwargs = dict([(attr, '%s %i' % (attr, i)) for attr in attrs])
            result.append(data_type(**kwargs))
        return result

    def assert_store_correct(self, store, items, mapping, markup=None):
        """Test that 'store' has 'items' as content."""
        msg = 'amount of rows for %s must be %s (got %s).'
        self.assertEqual(len(store), len(items),
                         msg % (store, len(items), len(store)))

        # assert rows content equal to items content
        tree_iter = store.get_iter_root()
        tmp = list(reversed(items))
        do_markup = markup is not None
        msg = "column %i ('%s') must be '%s' (got '%s' instead)"
        while tree_iter is not None:
            head = tmp.pop()
            for i, field in mapping:
                actual, = store.get(tree_iter, i)
                expected = getattr(head, field)
                if store.get_column_type(i).name == 'gboolean':
                    expected = bool(expected)
                elif do_markup:
                    expected = markup(expected)
                self.assertEqual(expected, actual,
                                 msg % (i, field, expected, actual))

            tree_iter = store.iter_next(tree_iter)
            do_markup = False  # only for first row

    def assert_indicator_disabled(self, indicator):
        """Test that 'indicator' is not sensitive."""
        self.assertFalse(indicator.is_sensitive(),
                         'indicator must not be sensitive.')

    def assert_indicator_ready(self, indicator):
        """Test that 'indicator' is sensitive and green."""
        self.assertTrue(indicator.is_sensitive(),
                        'indicator must be sensitive.')
        expected = indicator.get_pixbuf()  # a test on its own
        self.assertEqual(self.ui.active_indicator, expected,
                         'indicator must have the correct pixbuf.')

    def assert_indicator_loading(self, indicator):
        """Test that 'indicator' is sensitive and loading."""
        self.assertTrue(indicator.is_sensitive(),
                        'indicator must be sensitive.')
        expected = indicator.get_animation()  # a test on its own
        self.assertEqual(self.ui.loading_animation, expected,
                         'indicator must have the correct animation.')

    def assert_widget_availability(self, widget_name, enabled=True):
        """Check button availability according to 'enabled'."""
        widget = getattr(self.ui, widget_name)
        self.assertTrue(self.ui.widget_is_visible(widget),
                        '%s should be visible' % widget_name)
        sensitive = widget.is_sensitive()
        msg = '%s should %sbe sensitive'
        self.assertTrue(sensitive if enabled else not sensitive,
                        msg % (widget_name, '' if enabled else 'not '))

    def assert_dialog_properties(self, dialog_name=None, title=None,
                                 size=(600, 300), modal=True, dialog=None):
        """The dialog 'dialog_name' has correct properties."""
        assert (dialog_name is None) ^ (dialog is None)
        if dialog is None:
            dialog = getattr(self.ui, dialog_name)
        actual = dialog.size_request()
        msg = 'size must be %s (got %s instead).'
        self.assertEquals(size, actual, msg % (size, actual))

        msg = '%s must %sbe modal.'
        self.assertEqual(modal, dialog.get_modal(),
                         msg % (dialog_name, '' if modal else 'not '))

        position = dialog.get_property('window-position')
        expected = gtk.WIN_POS_CENTER if dialog_name is not None else \
                                      gtk.WIN_POS_MOUSE
        msg = 'dialog must have %s position (got %s instead).'
        self.assertEqual(expected, position, msg % (expected, position))

        actual = dialog.get_title()
        msg = '%s title must be %s (got %s instead)'
        self.assertEqual(title, actual, msg % (dialog_name, title, actual))

        msg = '%s must not skip taskbar.'
        self.assertFalse(dialog.get_skip_taskbar_hint(), msg % dialog_name)

        self.assertEqual(dialog.get_icon(), self.ui._icon)


class MagicicadaUIBasicTestCase(MagicicadaUITestCase):
    """UI test cases for basic state."""

    def test_init_creates_sd_instance(self):
        """SyncDaemon instance is created at creation time."""
        self.assertTrue(self.ui.sd is not None)
        self.assertTrue(isinstance(self.ui.sd, FakedSyncdaemon))

    def test_destroy_shutdowns_sd_instance(self):
        """SyncDaemon instance is shutdown at destroy time."""
        self.patch(self.ui.sd, 'shutdown', self._set_called)
        self.ui.on_main_window_destroy(self.ui.main_window)
        self.assertTrue(self._called,
                        'syncdaemon.shutdown must be called at destroy time.')

    def test_main_window_is_visible(self):
        """UI can be created and main_window is visible."""
        self.assertTrue(self.ui.widget_is_visible(self.ui.main_window))

    def test_windows_have_correct_icon(self):
        """Every window has the icon set."""
        for w in self.ui.windows:
            self.assertEqual(w.get_icon(), self.ui._icon)

    def test_start_connect_are_visible(self):
        """Start and Connect buttons are visible."""
        self.assertTrue(self.ui.widget_is_visible(self.ui.start))
        self.assertTrue(self.ui.start.is_sensitive())

        self.assertTrue(self.ui.widget_is_visible(self.ui.connect))
        self.assertFalse(self.ui.connect.is_sensitive())

    def test_stop_disconnect_are_not_visible(self):
        """Start and Connect buttons are visible."""
        self.assertFalse(self.ui.widget_is_visible(self.ui.stop))
        self.assertFalse(self.ui.widget_is_visible(self.ui.disconnect))

    def test_indicators_are_non_sensitive(self):
        """Test default sensitivity for indicators."""
        self.assertFalse(self.ui.is_started.is_sensitive())
        self.assertFalse(self.ui.is_connected.is_sensitive())
        self.assertFalse(self.ui.is_online.is_sensitive())

    def test_update_is_called_at_startup(self):
        """Update is called at startup."""
        self.patch(MagicicadaUI, 'update', self._set_called)
        self.ui = MagicicadaUI(syncdaemon_class=FakedSyncdaemon)
        self.assertTrue(self._called,
                        'update was called at startup.')


class MagicicadaUIClickedTestCase(MagicicadaUITestCase):
    """UI test cases."""

    def test_on_start_clicked(self):
        """Test on_start_clicked."""
        self.ui.on_start_clicked(self.ui.start)

        self.assertTrue(self.ui.widget_is_visible(self.ui.start))
        self.assertFalse(self.ui.start.is_sensitive())
        self.assertFalse(self.ui.widget_is_visible(self.ui.stop))

        self.assert_indicator_loading(self.ui.is_started)
        self.assert_indicator_disabled(self.ui.is_connected)
        self.assert_indicator_disabled(self.ui.is_online)

    def test_on_start_clicked_starts_syncdaemon(self):
        """Test on_start_clicked."""
        self.patch(self.ui.sd, 'start', self._set_called)
        self.ui.on_start_clicked(self.ui.start)
        self.assertTrue(self._called, 'syncdaemon.start was called.')

    def test_on_connect_clicked(self):
        """Test on_connect_clicked."""
        self.do_start()  # need to be started
        self.ui.on_connect_clicked(self.ui.connect)

        self.assertTrue(self.ui.widget_is_visible(self.ui.connect))
        self.assertFalse(self.ui.connect.is_sensitive())
        self.assertFalse(self.ui.widget_is_visible(self.ui.disconnect))

        self.assert_indicator_ready(self.ui.is_started)
        self.assert_indicator_loading(self.ui.is_connected)
        self.assert_indicator_disabled(self.ui.is_online)

    def test_on_connect_clicked_connects_syncdaemon(self):
        """Test on_connect_clicked."""
        self.patch(self.ui.sd, 'connect', self._set_called)
        self.ui.on_connect_clicked(self.ui.connect)
        self.assertTrue(self._called, 'syncdaemon.connect was called.')

    def test_on_stop_clicked(self):
        """Test on_stop_clicked."""
        self.do_start()
        assert not self.ui.widget_enabled(self.ui.disconnect)
        self.patch(self.ui, 'on_disconnect_clicked', self._set_called)
        self.ui.on_stop_clicked(self.ui.stop)

        self.assertFalse(self._called, 'on_disconnect_clicked was not called.')

        self.assertFalse(self.ui.widget_is_visible(self.ui.start))
        self.assertTrue(self.ui.widget_is_visible(self.ui.stop))
        self.assertFalse(self.ui.stop.is_sensitive())

        self.assertTrue(self.ui.widget_is_visible(self.ui.connect))
        self.assertFalse(self.ui.connect.is_sensitive())
        self.assertFalse(self.ui.widget_is_visible(self.ui.disconnect))

    def test_on_stop_clicked_if_connected(self):
        """Test on_stop_clicked."""
        self.do_connect()
        self.patch(self.ui, 'on_disconnect_clicked', self._set_called)
        self.ui.on_stop_clicked(self.ui.stop)

        self.assertTrue(self._called, 'on_disconnect_clicked was called.')

    def test_on_stop_clicked_stops_syncdaemon(self):
        """Test on_stop_clicked."""
        self.patch(self.ui.sd, 'quit', self._set_called)
        self.ui.on_stop_clicked(self.ui.stop)
        self.assertTrue(self._called, 'syncdaemon.quit was called.')

    def test_on_disconnect_clicked(self):
        """Test on_disconnect_clicked."""
        self.do_connect()
        self.ui.on_disconnect_clicked(self.ui.disconnect)

        self.assertFalse(self.ui.widget_is_visible(self.ui.connect))
        self.assertTrue(self.ui.widget_is_visible(self.ui.disconnect))
        self.assertFalse(self.ui.disconnect.is_sensitive())

    def test_on_disconnect_clicked_disconnects_syncdaemon(self):
        """Test on_disconnect_clicked."""
        self.patch(self.ui.sd, 'disconnect', self._set_called)
        self.ui.on_disconnect_clicked(self.ui.disconnect)
        self.assertTrue(self._called, 'syncdaemon.disconnect was called.')


class MagicicadaUISystrayIconTestCase(MagicicadaUITestCase):
    """UI test cases for the systray icon."""

    def test_main_window_is_hid_when_icon_clicked(self):
        """Main window is hid when the systray icon is clicked."""
        self.ui.on_status_icon_activate(self.ui.status_icon)
        self.assertFalse(self.ui.widget_is_visible(self.ui.main_window),
                         'main_window should be invisible when icon clicked.')

    def test_main_window_is_shown_when_clicked_after_hidden(self):
        """Main window is shown when the icon is clicked after hidden."""
        self.ui.on_status_icon_activate(self.ui.status_icon)  # hide
        self.ui.on_status_icon_activate(self.ui.status_icon)  # show
        msg = 'main_window should be visible when icon clicked after hidden.'
        self.assertTrue(self.ui.widget_is_visible(self.ui.main_window), msg)


def skip_abstract_class(test):
    """If 'test' belongs to an abstract class, don't run it."""

    @wraps(test)
    def inner(klass):
        """Execute 'test' only if not in an abstract class."""
        if klass.name is not None:
            test(klass)
    return inner


class _MagicicadaUIQueueTestCase(MagicicadaUITestCase):
    """Abstratc UI test cases for queue tree views."""

    name = None

    def setUp(self):
        """Init."""
        super(_MagicicadaUIQueueTestCase, self).setUp()
        if self.name is None:
            return
        self.sd_changed = getattr(self.ui.sd,
                                  '%s_queue_changed_callback' % self.name)
        self.ui_changed = getattr(self.ui,
                                  'on_%s_queue_changed' % self.name)
        self.set_sd_queue = lambda q: \
                            setattr(self.ui.sd, '%s_queue' % self.name, q)
        self.queue_store = getattr(self.ui, '%sq_store' % self.name)
        self.queue_view = getattr(self.ui, '%sq_view' % self.name)

    def build_some_data(self, limit=5):
        """Build some data to act as queue data."""
        kwargs = dict(data_type=QueueData, limit=limit)
        res = super(_MagicicadaUIQueueTestCase, self).build_some_data(**kwargs)
        # operation path share node
        return res

    def expected_markup(self, value):
        """Return the markup for row at index 'i'."""
        processing_meta = self.name == META_QUEUE and \
                          self.ui.sd.current_state.processing_meta
        processing_content = self.name == CONTENT_QUEUE and \
                          self.ui.sd.current_state.processing_content
        assert not (processing_meta and processing_content)
        must_highlight = self.ui.sd.current_state.is_online and \
                         (processing_meta or processing_content)
        result = self.ui.CURRENT_ROW % value \
                 if must_highlight and value is not None else value
        return result

    def assert_store_correct(self, store, items):
        """Test that 'store' has 'items' as content."""
        mapping = enumerate(['operation', 'path', 'share', 'node'])
        args = (store, items, mapping, self.expected_markup)
        super(_MagicicadaUIQueueTestCase, self).assert_store_correct(*args)

    def assert_current_processing_row_is_different(self):
        """Row being processed is highlighted."""
        items = self.build_some_data()
        self.sd_changed(items)

        item = items[0]
        attrs = type(item)._fields

        markup = self.expected_markup
        expected = tuple(markup(getattr(item, attr)) for attr in attrs)

        iter_root = self.queue_store.get_iter_root()
        actual = self.queue_store.get(iter_root, *xrange(len(attrs)))

        msg = 'first row for %s queue must be %s (got %s instead)' % \
              (self.name, expected, actual)
        self.assertEqual(expected, actual, msg)

    @skip_abstract_class
    def test_callback_is_connected(self):
        """Queue changed callback is connected."""
        self.assertEqual(self.sd_changed, self.ui_changed,
                         '%s queue callback must be set' % self.name)

    @skip_abstract_class
    def test_model_is_binded(self):
        """List store is binded."""
        actual = self.queue_view.get_model()
        msg = 'model for view %s differs from %s'
        self.assertEqual(self.queue_store, actual,
                         msg % (self.name, self.queue_store))

    @skip_abstract_class
    def test_on_queue_changed_updates_view(self):
        """On queue changed the view is updated."""
        items = self.build_some_data()
        self.sd_changed(items)
        self.assert_store_correct(self.queue_store, items)

    @skip_abstract_class
    def test_on_queue_changed_handles_none(self):
        """On queue changed handles None as items."""
        self.sd_changed(None)
        self.assert_store_correct(self.queue_store, [])

    @skip_abstract_class
    def test_on_queue_changed_handles_an_item_none(self):
        """On queue changed handles None as items."""
        items = [QueueData(operation='Test', path='', share=None, node=None)]
        self.sd_changed(items)
        self.assert_store_correct(self.queue_store, items)

    @skip_abstract_class
    def test_model_is_cleared_before_updating(self):
        """The model is cleared before upadting with a new set of data."""
        items = self.build_some_data()
        self.sd_changed(items)

        items = self.build_some_data()
        self.sd_changed(items)
        self.assertEqual(len(self.queue_store), len(items))

    @skip_abstract_class
    def test_view_is_enabled_if_disabled_on_changed(self):
        """The tree view is enabled on changed if it was disabled."""
        self.assertFalse(self.queue_view.is_sensitive(),
                         'Tree view must be disabled by default.')
        items = self.build_some_data()
        self.sd_changed(items)

        self.assertTrue(self.queue_view.is_sensitive(),
                        'Tree view must be enabled on changed.')

    @skip_abstract_class
    def test_update_is_correct_for_queue(self):
        """Correctly updates the queue state."""
        data = self.build_some_data()
        self.set_sd_queue(data)

        self.ui.update()

        self.assert_store_correct(self.queue_store, data)

    @skip_abstract_class
    def test_on_stopped_updates_queue(self):
        """On SD stoppped, the UI updates the queue state."""
        cb = 'on_%s_queue_changed' % self.name
        self.patch(self.ui, cb, self._set_called)
        self.ui.on_stopped()
        self.assertTrue(self._called, '%s was called on_stopped.' % cb)

    @skip_abstract_class
    def test_label_with_row_count(self):
        """Queue label shows the row count."""
        label = '%sq_label' % self.name
        actual = getattr(self.ui, label).get_text()
        expected = '%s Queue (%i)' % (self.name.capitalize(),
                                      len(self.queue_store))
        msg = '%s should be %s (got %s instead)'
        self.assertEqual(expected, actual, msg % (label, expected, actual))

        items = self.build_some_data()
        self.sd_changed(items)
        self.assertEqual(expected, actual, msg % (label, expected, actual))

    @skip_abstract_class
    def test_queue_is_updated_on_started_and_on_stopped(self):
        """Queue store is updated on_started."""
        items = self.build_some_data()
        self.set_sd_queue(items)
        self.do_start()
        self.assert_store_correct(self.queue_store, items)

        items = items[:len(items) / 2]
        self.set_sd_queue(items)
        self.ui.on_stop_clicked(self.ui.stop)
        self.ui.on_stopped()
        self.assert_store_correct(self.queue_store, items)

    @skip_abstract_class
    def test_current_processing_row_is_different_if_online(self):
        """Row being processed is highlighted when online."""
        self.ui.sd.current_state.set(is_online=True)

        self.ui.sd.current_state.set(queues='')
        self.assert_current_processing_row_is_different()

        self.ui.sd.current_state.set(queues='WORKING_ON_METADATA')
        self.assert_current_processing_row_is_different()

        self.ui.sd.current_state.set(queues='WORKING_ON_CONTENT')
        self.assert_current_processing_row_is_different()

        self.ui.sd.current_state.set(queues='WORKING_ON_BOTH')
        self.assert_current_processing_row_is_different()

    @skip_abstract_class
    def test_current_processing_row_is_not_different_if_offline(self):
        """Row being processed is not highlighted if offline."""
        self.ui.sd.current_state.set(is_online=False)

        self.ui.sd.current_state.set(queues='')
        self.assert_current_processing_row_is_different()

        self.ui.sd.current_state.set(queues='WORKING_ON_METADATA')
        self.assert_current_processing_row_is_different()

        self.ui.sd.current_state.set(queues='WORKING_ON_CONTENT')
        self.assert_current_processing_row_is_different()

        self.ui.sd.current_state.set(queues='WORKING_ON_BOTH')
        self.assert_current_processing_row_is_different()

    @skip_abstract_class
    def test_current_processing_row_is_not_different_on_offline(self):
        """Row being processed is not highlighted if on_offline was called."""
        self.ui.sd.current_state.set(is_online=False)
        self.ui.on_offline()

        self.assert_current_processing_row_is_different()

        self.ui.sd.current_state.set(queues='WORKING_ON_METADATA')
        self.assert_current_processing_row_is_different()

        self.ui.sd.current_state.set(queues='WORKING_ON_CONTENT')
        self.assert_current_processing_row_is_different()

        self.ui.sd.current_state.set(queues='WORKING_ON_BOTH')
        self.assert_current_processing_row_is_different()


class MagicicadaUIContentQueueTestCase(_MagicicadaUIQueueTestCase):
    """UI test cases for content queue view."""

    name = CONTENT_QUEUE


class MagicicadaUIMetaQueueTestCase(_MagicicadaUIQueueTestCase):
    """UI test cases for meta queue view."""

    name = META_QUEUE


class MagicicadaUIStatusTestCase(MagicicadaUITestCase):
    """UI test cases for the status label."""

    def setUp(self):
        """Init."""
        super(MagicicadaUIStatusTestCase, self).setUp()
        name = 'TEST'
        description = 'the status for testing'
        connection = 'Funny funny connection'
        queues = 'The queues are hilarious'
        self.kwargs = dict(name=name, description=description,
                           connection=connection, queues=queues)

    def assert_status_label_correct(self, name=None, description=None,
                                    connection=None, queues=None,
                                    expected=None):
        """Test that the status label is of the form name: description."""
        if expected is None:
            assert name is not None
            assert description is not None
            assert connection is not None
            assert queues is not None
            values = (name, description, queues, connection)
            expected = self.ui.STATUS_JOINER.join(values)

        actual = self.ui.status_label.get_text()
        msg = 'status label test must be "%s" (got "%s" instead).'
        self.assertEqual(expected, actual, msg % (expected, actual))

    def test_callback_is_connected(self):
        """Status callback is connected."""
        self.assertEqual(self.ui.sd.status_changed_callback,
                         self.ui.on_status_changed,
                         'status_changed callback must be set.')

    def test_status_label_ellipsizes(self):
        """The status label ellipsizes."""
        expected = pango.ELLIPSIZE_END
        actual = self.ui.status_label.get_ellipsize()
        self.assertEqual(expected, actual,
                         'label ellipsizes is ELLIPSIZE_END.')

    def test_on_status_changed_updates_status_label(self):
        """On status changed, the status label is updated."""
        self.ui.sd.current_state.set(is_started=True, **self.kwargs)
        # usual case, all values are defined
        self.ui.on_status_changed(**self.kwargs)
        self.assert_status_label_correct(**self.kwargs)

    def test_on_status_changed_updates_status_label_even_on_weird_cases(self):
        """On status changed, the status label is updated."""
        keywords = ('name', 'description', 'queues', 'connection')  # order
        for attr in keywords:
            old_value = self.kwargs[attr]

            # some weird cases: attr is '' or None
            for value in ('', None):
                self.kwargs[attr] = value
                self.ui.sd.current_state.set(is_started=True, **self.kwargs)
                self.ui.on_status_changed(**self.kwargs)
                others = (self.kwargs[k] for k in keywords if k != attr)
                expected = self.ui.STATUS_JOINER.join(others)
                self.assert_status_label_correct(expected=expected)

            self.kwargs[attr] = old_value

    def test_on_status_changed_updates_toolbar(self):
        """On status changed, the whole toolbar is updated."""
        self.patch(self.ui, 'update', self._set_called)
        self.ui.on_status_changed(**self.kwargs)
        self.assertEqual(self._called, ((), self.kwargs))

    def test_update_is_correct_for_status_label(self):
        """Correctly updates the status label."""
        self.ui.sd.current_state.set(**self.kwargs)
        self.ui.update()
        self.assert_status_label_correct(**self.kwargs)

    def test_on_stopped_updates_status_label(self):
        """On SD stoppped, the UI is updated."""
        self.ui.sd.current_state.set(**self.kwargs)
        self.ui.on_stopped()
        self.assert_status_label_correct(**self.kwargs)

    def test_status_label_default_if_not_started(self):
        """Status label is the default if not started."""
        self.assert_status_label_correct(expected=self.ui.STATUS['initial'])

    def test_status_label_empty_if_started_and_no_name_nor_desc(self):
        """Status label is empty if started but no name and no description."""
        self.do_start()
        self.ui.on_status_changed(name=None, description=None)
        self.assert_status_label_correct(expected='')

    def test_status_label_is_updated_on_started_and_on_stopped(self):
        """Status label is updated on_started."""
        self.ui.sd.current_state.set(**self.kwargs)
        self.do_start()
        self.assert_status_label_correct(**self.kwargs)

        self.kwargs['name'] = 'CHANGED'
        self.ui.sd.current_state.set(**self.kwargs)

        self.ui.on_stop_clicked(self.ui.stop)
        self.ui.on_stopped()
        self.assert_status_label_correct(**self.kwargs)


class MagicicadaUIConnectionTestCase(MagicicadaUITestCase):
    """UI test cases for connection buttons/indicators."""

    def assert_buttons_are_updated_correctly(self):
        """Test that buttons are correctly updated as per SD current state."""
        cs = self.ui.sd.current_state
        self.ui.update()

        self.assertEqual(self.ui.widget_enabled(self.ui.start),
                         not cs.is_started,
                         'start must be enabled if not started.')
        self.assertEqual(self.ui.widget_enabled(self.ui.stop),
                         cs.is_started,
                         'stop must be enabled if started.')

        if cs.is_started:
            self.assertEqual(self.ui.widget_enabled(self.ui.connect),
                             not cs.is_connected,
                             'connect must be enabled if not connected.')
            self.assertEqual(self.ui.widget_enabled(self.ui.disconnect),
                             cs.is_connected,
                             'disconnect must be enabled if connected.')
        else:
            self.assertFalse(self.ui.connect.is_sensitive(),
                             'connect must be disabled when %s' % cs)
            self.assertTrue(self.ui.widget_is_visible(self.ui.connect),
                            'connect must be visible when %s' % cs)

    def assert_indicators_are_updated_correctly(self):
        """Test that indicators are correctly updated as per SD state."""
        cs = self.ui.sd.current_state
        self.ui.update()
        if cs.is_started and not cs.is_connected and not cs.is_online:
            self.assert_indicator_ready(self.ui.is_started)
            self.assert_indicator_disabled(self.ui.is_connected)
            self.assert_indicator_disabled(self.ui.is_online)
        elif cs.is_started and cs.is_connected and not cs.is_online:
            self.assert_indicator_ready(self.ui.is_started)
            self.assert_indicator_ready(self.ui.is_connected)
            self.assert_indicator_loading(self.ui.is_online)
        elif cs.is_started and cs.is_connected and cs.is_online:
            self.assert_indicator_ready(self.ui.is_started)
            self.assert_indicator_ready(self.ui.is_connected)
            self.assert_indicator_ready(self.ui.is_online)
        elif not cs.is_started:
            self.assert_indicator_disabled(self.ui.is_started)
            self.assert_indicator_disabled(self.ui.is_connected)
            self.assert_indicator_disabled(self.ui.is_online)

    def test_all_disabled_at_startup(self):
        """Indicators are all disabled at startup."""
        self.assert_indicator_disabled(self.ui.is_started)
        self.assert_indicator_disabled(self.ui.is_connected)
        self.assert_indicator_disabled(self.ui.is_online)

    def test_callback_are_connected(self):
        """Connection callbacks are connected."""
        for callback in ('on_started', 'on_stopped',
                         'on_connected', 'on_disconnected',
                         'on_online', 'on_offline'):
            sd_cb = getattr(self.ui.sd, '%s_callback' % callback)
            ui_cb = getattr(self.ui, callback)
            self.assertEqual(sd_cb, ui_cb,
                             '%s callback must be set' % callback)

    def test_on_started_is_correct(self):
        """On SD started, the UI enables connect and indicator."""
        self.do_start()

        self.assertTrue(self.ui.widget_enabled(self.ui.stop))
        self.assertTrue(self.ui.widget_enabled(self.ui.connect))
        self.assert_indicator_ready(self.ui.is_started)
        self.assert_indicator_disabled(self.ui.is_connected)
        self.assert_indicator_disabled(self.ui.is_online)

    def test_on_connected_is_correct(self):
        """On SD connected, the UI enables indicator."""
        self.do_connect()

        self.assertTrue(self.ui.widget_enabled(self.ui.disconnect))
        self.assert_indicator_ready(self.ui.is_started)
        self.assert_indicator_ready(self.ui.is_connected)
        self.assert_indicator_loading(self.ui.is_online)

    def test_on_online_is_correct(self):
        """On SD online, the UI enables indicator."""
        self.do_connect()

        self.ui.on_online()

        self.assert_indicator_ready(self.ui.is_started)
        self.assert_indicator_ready(self.ui.is_connected)
        self.assert_indicator_ready(self.ui.is_online)

    def test_on_stopped_is_correct(self):
        """On SD stopped, the UI disables stop and indicator."""
        self.do_start()
        self.ui.on_stop_clicked(self.ui.stop)

        self.ui.on_stopped()

        self.assertTrue(self.ui.start.is_sensitive())
        self.assertTrue(self.ui.widget_is_visible(self.ui.start))
        self.assert_indicator_disabled(self.ui.is_started)
        self.assert_indicator_disabled(self.ui.is_connected)
        self.assert_indicator_disabled(self.ui.is_online)

    def test_on_disconnected_is_correct(self):
        """On SD disconnected, the UI disables connect and indicator."""
        self.do_connect()
        self.ui.on_disconnect_clicked(self.ui.disconnect)

        self.ui.on_disconnected()

        self.assertTrue(self.ui.connect.is_sensitive())
        self.assert_indicator_ready(self.ui.is_started)
        self.assert_indicator_disabled(self.ui.is_connected)
        self.assert_indicator_disabled(self.ui.is_online)

    def test_on_offline_is_correct(self):
        """On SD offline, the UI disables indicator."""
        self.do_connect()

        self.ui.on_offline()

        self.assert_indicator_ready(self.ui.is_started)
        self.assert_indicator_ready(self.ui.is_connected)
        self.assert_indicator_disabled(self.ui.is_online)

    def test_update_is_correct_for_buttons(self):
        """Correctly updates the buttons state."""
        for s, c, o in itertools.product((True, False), repeat=3):
            self.ui.sd.current_state.set(is_started=s, is_connected=c,
                                         is_online=o)
            self.assert_buttons_are_updated_correctly()

    def test_update_is_correct_for_indicators(self):
        """Correctly updates the indicators state."""
        for s, c, o in itertools.product((True, False), repeat=3):
            self.ui.sd.current_state.set(is_started=s, is_connected=c,
                                         is_online=o)
            self.assert_indicators_are_updated_correctly()


class _MagicicadaUIVolumeTestCase(MagicicadaUITestCase):
    """Abstract UI test cases for volumes (folders/shares)."""

    name = None
    data_type = None
    mapping = []

    def setUp(self):
        """Init."""
        super(_MagicicadaUIVolumeTestCase, self).setUp()
        if self.name is None:
            return
        self.volume = getattr(self.ui, self.name)
        self.volume_store = getattr(self.ui, '%s_store' % self.name)
        self.volume_view = getattr(self.ui, '%s_view' % self.name)
        self.volume_dialog_name = '%s_dialog' % self.name
        self.volume_dialog = getattr(self.ui, self.volume_dialog_name)
        self.on_volume_clicked = getattr(self.ui, 'on_%s_clicked' % self.name)
        self.volume_close = getattr(self.ui, '%s_close' % self.name)

    def build_some_data(self, limit=5):
        """Build some data to act as volume."""
        kwargs = dict(data_type=self.data_type, limit=limit)
        r = super(_MagicicadaUIVolumeTestCase, self).build_some_data(**kwargs)
        return r

    def assert_store_correct(self, store, items):
        """Test that 'store' has 'items' as content."""
        args = (store, items, self.mapping)
        super(_MagicicadaUIVolumeTestCase, self).assert_store_correct(*args)

    def assert_widget_availability(self, enabled=True):
        """Check volume availability according to 'enabled'."""
        s = super(_MagicicadaUIVolumeTestCase, self)
        s.assert_widget_availability(self.name, enabled)

    def assert_sort_order_correct(self, column, i, expected_order):
        """Check that sort order is correctly set."""
        msg0 = 'Store sort id must be %r (got %r instead).'
        msg1 = 'Store sort order must be %r (got %r instead).'
        msg3 = 'Column sort order must be %r (got %r instead).'

        actual_id, actual_order = self.volume_store.get_sort_column_id()

        # store sort column id and order
        self.assertEqual(i, actual_id, msg0 % (i, actual_id))
        self.assertEqual(expected_order, actual_order,
                         msg1 % (expected_order, actual_order))

        # column sort order
        actual_order = column.get_sort_order()
        self.assertEqual(expected_order, actual_order,
                         msg3 % (expected_order, actual_order))

    def assert_sort_indicator_correct(self, column):
        """Check that sort indicator is correctly set."""
        msg = 'Column %s must have sort indicator %s.'
        colname = column.get_name()
        # column sort indicator
        self.assertTrue(column.get_sort_indicator(), msg % (colname, 'on'))

        # all the other columns must not have the sort indicator on
        for other_column in self.volume_view.get_columns():
            if other_column.get_name() == colname:
                continue
            self.assertFalse(other_column.get_sort_indicator(),
                             msg % (other_column.get_name(), 'off'))

    @skip_abstract_class
    def test_volume_are_disabled_until_started(self):
        """Folders and shares are disabled until online."""
        # disabled at startup
        self.assert_widget_availability(enabled=False)

        # enabled when started
        self.do_start()
        self.assert_widget_availability(enabled=True)

    @skip_abstract_class
    def test_volume_are_enabled_until_stopped(self):
        """Folders and shares are enabled until offline."""
        self.do_connect()
        self.assert_widget_availability(enabled=True)

        self.ui.on_online()
        self.assert_widget_availability(enabled=True)

        self.ui.on_offline()
        self.assert_widget_availability(enabled=True)

        self.ui.on_disconnect_clicked(self.ui.disconnect)
        self.ui.on_disconnected()
        self.assert_widget_availability(enabled=True)

        # disabled when stopped
        self.ui.on_stop_clicked(self.ui.stop)
        self.assert_widget_availability(enabled=False)
        self.ui.on_stopped()
        self.assert_widget_availability(enabled=False)

    @skip_abstract_class
    def test_on_stopped_updates_volumes(self):
        """On SD stoppped, the UI disables the volumes."""
        self.do_connect()
        self.ui.on_stopped()
        self.assert_widget_availability(enabled=False)

    @skip_abstract_class
    def test_volume_close_emits_response_close(self):
        """Test volume close button emits RESPONSE_CLOSE when clicked."""
        self.patch(self.volume_dialog, 'response', self._set_called)
        self.volume_close.clicked()
        self.assertEqual(self._called, ((gtk.RESPONSE_CLOSE,), {}),
                         'volume close button should emit RESPONSE_CLOSE.')

    @skip_abstract_class
    def test_on_volume_clicked(self):
        """Test on_volume_clicked."""
        self.assertFalse(self.ui.widget_is_visible(self.volume_dialog),
                         '%s should not be visible.' % self.volume_dialog_name)

        def test():
            """Perform the test per se before closing the dialog."""
            self.assertTrue(self.ui.widget_is_visible(self.volume_dialog),
                            '%s should be visible.' % self.volume_dialog_name)
            self.assert_store_correct(self.volume_store, items)

        items = self.build_some_data()
        setattr(self.ui.sd, self.name, items)
        gobject.timeout_add(100, close_dialog,
                            (self.volume_dialog, test))
        self.on_volume_clicked(self.volume)

        # dialog was closed already
        self.assertFalse(self.ui.widget_is_visible(self.volume_dialog),
                         '%s should not be visible.' % self.volume_dialog_name)

    @skip_abstract_class
    def test_on_volume_clicked_twice(self):
        """Test on_volume_clicked twice."""

        def test():
            """Perform the test per se before closing the dialog."""
            self.assertTrue(self.ui.widget_is_visible(self.volume_dialog),
                            '%s should be visible.' % self.volume_dialog_name)
            self.assert_store_correct(self.volume_store, items)

        items = self.build_some_data()
        setattr(self.ui.sd, self.name, items)

        gobject.timeout_add(100, close_dialog,
                            (self.volume_dialog, test))
        self.on_volume_clicked(self.volume)

        gobject.timeout_add(100, close_dialog,
                            (self.volume_dialog, test))
        self.on_volume_clicked(self.volume)

    @skip_abstract_class
    def test_on_volume_clicked_handles_none(self):
        """On volume clicked handles None as items."""
        setattr(self.ui.sd, self.name, None)
        test = lambda: self.assert_store_correct(self.volume_store, [])
        gobject.timeout_add(100, close_dialog,
                            (self.volume_dialog, test))
        self.on_volume_clicked(self.volume)

    @skip_abstract_class
    def test_volume_dialog_properties(self):
        """The volume dialog has correct properties."""
        title = self.name.replace('_', ' ').capitalize()
        self.assert_dialog_properties(dialog_name=self.volume_dialog_name,
                                      title=title)

    @skip_abstract_class
    def test_volume_columns_not_sorted_at_start(self):
        """Test volume columns are not sorted at start."""
        msg = 'Column %s must not have the sort indicator on.'
        for col in self.volume_view.get_columns():
            self.assertFalse(col.get_sort_indicator(), msg % col.get_name())

    @skip_abstract_class
    def test_volume_columns_are_clickable(self):
        """Test volume columns are clickable."""
        msg = 'Column %s must be clickable.'
        for col in self.volume_view.get_columns():
            self.assertTrue(col.get_clickable(), msg % col.get_name())

    @skip_abstract_class
    def test_volume_columns_clicked_signal(self):
        """Test volume columns clicks signal is properly connected."""
        msg = 'Column %s must be connected to on_store_sort_column_changed.'
        for col in self.volume_view.get_columns():
            self.assertTrue(col.get_clickable(), msg % col.get_name())

    @skip_abstract_class
    def test_volume_sorting(self):
        """Test volume panel can be re-sorted."""
        for i, col in enumerate(self.volume_view.get_columns()):
            col.clicked()  # click on the column
            self.assert_sort_order_correct(col, i, gtk.SORT_ASCENDING)
            self.assert_sort_indicator_correct(col)

            col.clicked()  # click on the column, sort order must change
            self.assert_sort_order_correct(col, i, gtk.SORT_DESCENDING)

            col.clicked()  # click again, sort order must be the first one
            self.assert_sort_order_correct(col, i, gtk.SORT_ASCENDING)


class MagicicadaUIFoldersTestCase(_MagicicadaUIVolumeTestCase):
    """UI test cases for folders."""

    name = 'folders'
    data_type = FolderData
    mapping = enumerate(['path', 'suggested_path', 'subscribed',
                         'node', 'volume'])


class _MagicicadaUISharesTestCase(_MagicicadaUIVolumeTestCase):
    """UI test cases for shares_to_me."""

    data_type = ShareData
    mapping = enumerate(['name', 'other_visible_name', 'accepted',
                         'access_level', 'free_bytes', 'path',
                         'other_username', 'node_id', 'volume_id'])

    def assert_correct_free_bytes(self, free_bytes):
        """Free bytes are shown humanized."""
        attrs = self.data_type._fields
        kwargs = dict([(attr, str(attr)) for attr in attrs])
        kwargs['free_bytes'] = free_bytes
        item = self.data_type(**kwargs)
        setattr(self.ui.sd, self.name, [item])

        def test():
            """Perform the test per se before closing the dialog."""
            if kwargs['free_bytes'] is not None:
                i = int(kwargs['free_bytes'])
                kwargs['free_bytes'] = humanize_bytes(i, precision=2)
            item = self.data_type(**kwargs)
            self.assert_store_correct(self.volume_store, [item])

        gobject.timeout_add(100, close_dialog,
                            (self.volume_dialog, test))
        self.on_volume_clicked(self.volume)

    @skip_abstract_class
    def test_bytes_are_humanized(self):
        """Free bytes are shown humanized."""
        self.assert_correct_free_bytes(free_bytes=10000)

    @skip_abstract_class
    def test_bytes_are_humanized_even_when_string(self):
        """Free bytes are shown humanized even if they are string."""
        self.assert_correct_free_bytes(free_bytes='10000')

    @skip_abstract_class
    def test_bytes_are_humanized_even_when_none(self):
        """Free bytes are shown humanized even if they are None."""
        self.assert_correct_free_bytes(free_bytes=None)


class MagicicadaUISharesToMeTestCase(_MagicicadaUISharesTestCase):
    """UI test cases for shares_to_me."""

    name = 'shares_to_me'


class MagicicadaUISharesToOthersTestCase(_MagicicadaUISharesTestCase):
    """UI test cases for shares_to_others."""

    name = 'shares_to_others'


class MagicicadaUIMetadataTestCase(MagicicadaUITestCase):
    """UI test cases for metadata display."""

    name = 'metadata'

    def setUp(self):
        """Init."""
        super(MagicicadaUIMetadataTestCase, self).setUp()
        self._file_chooser_path = None
        self.metadata = dict(bla='ble', foo='bar')

        # no need for the file_chooser to actually run, it works.
        self.patch(self.ui.file_chooser, 'run',
                   lambda: gtk.FILE_CHOOSER_ACTION_OPEN)

        # store whatever file was set
        self.patch(self.ui.file_chooser, 'set_filename',
                   lambda path: setattr(self, '_file_chooser_path', path))

        # return the stored path
        self.patch(self.ui.file_chooser, 'get_filename',
                   lambda: getattr(self, '_file_chooser_path'))

    def assert_widget_availability(self, enabled=True):
        """Check button availability according to 'enabled'."""
        s = super(MagicicadaUIMetadataTestCase, self)
        s.assert_widget_availability(self.name, enabled)

    def assert_dialog_visibility(self, dialog, text_view, spinner, path=None):
        """Check the visibility for dialog, text_view and spinner."""
        if path is None:
            path = self.TEST_FILE

        msg = '%s visibility should be %s (got %s instead).'
        metadata_dialog = self.ui.metadata_dialogs[path]
        visible = self.ui.widget_is_visible(metadata_dialog)
        self.assertEqual(dialog, visible,
                         msg % ('metadata_dialog', dialog, visible))

        visible = self.ui.widget_is_visible(metadata_dialog.view)
        self.assertEqual(text_view, visible,
                         msg % ('metadata_view', text_view, visible))

        visible = self.ui.widget_is_visible(metadata_dialog.spinner)
        self.assertEqual(spinner, visible,
                         msg % ('metadata_spinner', spinner, visible))

    def assert_buffer_content(self, path, expected):
        """Check that the buffer content is correct for 'path'."""
        buff = self.ui.metadata_dialogs[path].view.get_buffer()
        self.assertTrue(buff is not None,
                        'buffer for metadata_view must not be None.')

        if isinstance(expected, dict):
            expected = '\n'.join('%s: %s' % i for i in expected.iteritems())
        actual = buff.get_text(*buff.get_bounds())
        msg = 'buffer content must be %s (got %s instead).'
        self.assertEqual(actual, expected, msg % (expected, actual))

    def test_metadata_are_disabled_until_started(self):
        """Metadata button is disabled until online."""
        # disabled at startup
        self.assert_widget_availability(enabled=False)

        # enabled when started
        self.do_start()
        self.assert_widget_availability(enabled=True)

    def test_metadata_are_enabled_until_stopped(self):
        """Metadata button is enabled until offline."""
        self.do_connect()
        self.assert_widget_availability(enabled=True)

        self.ui.on_online()
        self.assert_widget_availability(enabled=True)

        self.ui.on_offline()
        self.assert_widget_availability(enabled=True)

        self.ui.on_disconnect_clicked(self.ui.disconnect)
        self.ui.on_disconnected()
        self.assert_widget_availability(enabled=True)

        # disabled when stopped
        self.ui.on_stop_clicked(self.ui.stop)
        self.assert_widget_availability(enabled=False)
        self.ui.on_stopped()
        self.assert_widget_availability(enabled=False)

    def test_on_stopped_updates_volumes(self):
        """On SD stoppped, the UI disables the metadata button."""
        self.do_connect()
        self.ui.on_stopped()
        self.assert_widget_availability(enabled=False)

    def test_metadata_close_hides_the_dialog(self):
        """Test metadata close button emits RESPONSE_CLOSE when clicked."""
        dialog = self.ui._new_metadata_dialog(self.TEST_FILE)
        dialog.close.clicked()
        visible = self.ui.widget_is_visible(dialog)
        self.assertFalse(visible, 'metadata_dialog should not be visible.')

    def test_file_chooser_open_emits_response_ok(self):
        """Test volume close button emits RESPONSE_CLOSE when clicked."""
        self.patch(self.ui.file_chooser, 'response', self._set_called)

        self.ui.file_chooser_open.clicked()

        msg = 'file_chooser_open should emit %s (got %s instead).'
        expected = gtk.FILE_CHOOSER_ACTION_OPEN
        self.assertEqual(self._called, ((expected,), {}),
                         msg % (expected, self._called))

    def test_on_metadata_clicked(self):
        """Test on_metadata_clicked."""
        self.ui._u1_root = os.path.dirname(self.TEST_FILE)
        self.ui.file_chooser.set_filename(self.TEST_FILE)

        self.ui.on_metadata_clicked(self.ui.metadata)

        # file_chooser must be visible on metadata clicked.
        self.assertEqual(self.TEST_FILE, self.ui.file_chooser.get_filename(),
                         'filename returned by file chooser must be correct.')

        # dialog must exist now
        dialog = self.ui.metadata_dialogs[self.TEST_FILE]

        # metadata_dialog is enabled and shows the loading animation
        self.assert_dialog_visibility(dialog=True, text_view=False,
                                      spinner=True)
        self.assertIsInstance(dialog.spinner, gtk.Spinner)
        self.assertTrue(dialog.spinner.get_property('active'),
                        'metadata_spinner is active.')

        # Check that the metadata was asked to the SD
        self.assertEqual(1, len(self.ui.sd._meta_paths))
        self.assertEqual(self.ui.sd._meta_paths[0], self.TEST_FILE)
        # SD will eventually callback us with the metadata
        self.ui.on_metadata_ready(self.TEST_FILE, self.metadata)

        # metadata_dialog is enabled and shows the metadata
        self.assert_dialog_visibility(dialog=True, text_view=True,
                                      spinner=False)

        # user closes the dialog
        dialog.close.clicked()

        # dialog was closed already
        visible = self.ui.widget_is_visible(dialog)
        self.assertFalse(visible, 'metadata_dialog should not be visible.')

    def test_file_chooser_folder_is_u1root_when_visible(self):
        """File chooser folder is ~/Ubuntu One only when visible."""
        self.assertNotEqual(self.ui.file_chooser.get_current_folder(),
                            UBUNTU_ONE_ROOT,
                            "shouldn't have U1 folder before becoming visible")

        self.ui.file_chooser.show()
        self.ui.file_chooser.response(gtk.RESPONSE_CLOSE)
        process_gtk_pendings()

        self.assertEqual(self.ui.file_chooser.get_current_folder(),
                         UBUNTU_ONE_ROOT,
                         'should have U1 folder after becoming visible')

    def test_metadata_dialog_properties(self):
        """The metadata dialog has correct properties."""
        title = '%s for %s' % (self.name.replace('_', ' ').capitalize(),
                               self.TEST_FILE)
        dialog = self.ui._new_metadata_dialog(self.TEST_FILE)
        self.assert_dialog_properties(dialog=dialog, title=title, modal=False)

        actual = dialog.view.get_wrap_mode()
        msg = 'wrap mode for view must be gtk.WRAP_WORD (got %s instead).'
        self.assertEqual(gtk.WRAP_WORD, actual, msg % actual)

    def test_callback_is_connected(self):
        """Metadata ready callback is connected."""
        self.assertEqual(self.ui.sd.on_metadata_ready_callback,
                         self.ui.on_metadata_ready,
                         'on_metadata_ready_callback callback must be set.')

    def test_file_chooser_is_hidden_at_startup(self):
        """File chooser exists but is not visible."""
        self.assertFalse(self.ui.widget_is_visible(self.ui.file_chooser),
                         'file_chooser must be hidden by default.')

    def test_filename_is_used_only_if_open_clicked(self):
        """Filename is used only if user clicked open."""
        # no need for the file_chooser to actually run, it works.
        self.patch(self.ui.file_chooser, 'run', lambda: gtk.RESPONSE_CLOSE)
        self.patch(self.ui.sd, 'get_metadata', self._set_called)

        self.ui.on_metadata_clicked(self.ui.metadata)

        self.assertFalse(self._called,
                        'get_metadata should not be called if no file chosen.')

    def test_filename_is_stored_if_open_was_clicked(self):
        """Filename is stored in the metadata dicts if user clicked open."""
        self.assertEqual(self.ui.metadata_dialogs, {},
                        'no dialogs in the ui.')

        self.ui._u1_root = os.path.dirname(self.TEST_FILE)
        self.ui.file_chooser.set_filename(self.TEST_FILE)
        gobject.timeout_add(100, test_and_click,
                            (self.ui.file_chooser_open, NO_OP))
        self.ui.on_metadata_clicked(self.ui.metadata)

        self.assertEqual([self.TEST_FILE], self.ui.metadata_dialogs.keys(),
                         'metadata dict keys should be what the user choose.')

    def test_on_metadata_ready(self):
        """Callback on_metadata_ready updates the metadata_view."""
        path = 'bla'
        self.ui.metadata_dialogs = {path: self.ui._new_metadata_dialog(path)}
        self.ui.on_metadata_ready(path, self.metadata)

        self.assert_buffer_content(path, self.metadata)

    def test_on_metadata_ready_if_invalid_path(self):
        """Callback on_metadata_ready handles NOT_SYNCHED_PATH."""
        path = 'bla'
        self.ui.metadata_dialogs = {path: self.ui._new_metadata_dialog(path)}
        self.ui.on_metadata_ready(path, NOT_SYNCHED_PATH)

        self.assert_buffer_content(path, NOT_SYNCHED_PATH)

    def test_on_metadata_ready_doesnt_update_if_last_path_doesnt_match(self):
        """Callback on_metadata_ready updates the metadata_view."""
        self.patch(self.ui, '_new_metadata_dialog', self._set_called)
        path = 'bla'
        assert path not in self.ui.metadata_dialogs
        self.ui.on_metadata_ready(path, self.metadata)

        self.assertFalse(self._called,
                         'view should not be updated if key is not last one.')

    def test_two_metadata_windows(self):
        """More than one metadata window is allowed."""
        self.patch(self.ui.file_chooser, 'run',
                   lambda: gtk.FILE_CHOOSER_ACTION_OPEN)

        path1 = os.path.abspath(self.mktemp())
        open(path1, 'w').close()
        assert os.path.exists(path1)
        meta1 = {'value': 'Lorem ipsum dolor sit amet.'}

        path2 = os.path.abspath(self.mktemp())
        open(path2, 'w').close()
        assert os.path.exists(path2)
        meta2 = {'value': 'Etiam iaculis congue nisl.'}

        self.ui.file_chooser.get_filename = lambda: path1
        self.ui.on_metadata_clicked(self.ui.metadata)

        self.ui.file_chooser.get_filename = lambda: path2
        self.ui.on_metadata_clicked(self.ui.metadata)

        # Check that the UI has 2 metadata dialogs
        self.assertTrue(2, len(self.ui.metadata_dialogs))

        # Check that the metadata was asked to the SD
        self.assertEqual(2, len(self.ui.sd._meta_paths))
        self.assertEqual(self.ui.sd._meta_paths[0], path1)
        self.assertEqual(self.ui.sd._meta_paths[1], path2)

        # SD will eventually callback us with the metadata
        self.ui.on_metadata_ready(path1, meta1)
        # SD will eventually callback us with the metadata
        self.ui.on_metadata_ready(path2, meta2)

        self.assert_buffer_content(path1, meta1)
        self.assert_buffer_content(path2, meta2)

        # user closes the dialog
        self.ui.metadata_dialogs[path1].close.clicked()
        self.ui.metadata_dialogs[path2].close.clicked()


def override_input_output(input_args, output_args):
    """Call 'f' but receive fixed input and return fixed output."""

    def decorator(f):
        """The decorator per se."""

        @wraps(f)
        def inner(*args, **kwargs):
            """Feed 'f' with 'input_args' and return 'output_args'."""
            f(input_args)
            return output_args

        return inner

    return decorator


class MagicicadaLoggingTestCase(MagicicadaUITestCase):
    """UI test cases for logging."""

    def setUp(self):
        """Init."""
        super(MagicicadaLoggingTestCase, self).setUp()

        self.memento = MementoHandler()
        self.memento.setLevel(logging.DEBUG)
        logger = logging.getLogger('magicicada.ui')
        logger.addHandler(self.memento)

    def assert_function_logs(self, func, *args, **kwargs):
        """Check 'funcion' logs its inputs as DEBUG."""
        name = func.__name__
        msg = '%s must be logged as DEBUG'
        try:
            func(*args, **kwargs)
        except Exception:  # pylint: disable=E0501, W0703
            exc = sys.exc_info()
            self.assertTrue(self.memento.check_error(name),
                            'function (%s) must be logged as ERROR' % name)
            self.assertTrue(self.memento.check_error(exc),
                            'sys.exc_info (%s) must be logged as ERROR' % exc)
        self.assertTrue(self.memento.check_debug(name), msg % name)
        for arg in args:
            self.assertTrue(self.memento.check_debug(str(arg)), msg % arg)
        for key, val in kwargs.iteritems():
            arg = "'%s': %r" % (key, val)
            self.assertTrue(self.memento.check_debug(arg), msg % arg)

    def test_on_shares_clicked_logs(self):
        """Check _on_shares_clicked logs properly."""
        args = ([0, object(), 'test', {}], object())
        kwargs = dict(dialog=object())
        self.assert_function_logs(self.ui._on_shares_clicked, *args, **kwargs)

    def test_on_status_changed_logs(self):
        """Check _on_status_changed logs properly."""
        args = ('test status', 'status description', True, False, True)
        kwargs = dict(queues='bla', connection=None)
        self.assert_function_logs(self.ui.on_status_changed, *args, **kwargs)

    def test_on_queue_changed_logs(self):
        """Check _on_queue_changed logs properly."""
        args = ('meta',)
        kwargs = dict(items=[0, object(), 'test', {}], must_highlight=True)
        self.assert_function_logs(self.ui._on_queue_changed, *args, **kwargs)

    def test_on_metadata_ready_logs(self):
        """Check on_metadata_ready logs properly."""
        args = ()
        kwargs = dict(path='test', metadata=True)
        self.assert_function_logs(self.ui.on_metadata_ready, *args, **kwargs)
