/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file PolyShadingMethod.h
    \brief Definition of the Template class PolyShadingMethod.
    
    Magics Team - ECMWF 2004
    
    Started: Wed 18-Aug-2004
    
    Changes:
    
*/

#ifndef PolyShadingMethod_H
#define PolyShadingMethod_H

#include "magics.h"
#include "Factory.h"
#include "Translator.h"
#include "Polyline.h"
#include "ColourTechnique.h"
#include "LegendVisitor.h"
namespace magics {

class LevelSelection;
template <class P> class  PolyShadingMethod;


template <class P>
struct LegendEntryBuilder
{
    LegendEntryBuilder(LegendVisitor& legend, const PolyShadingMethod<P>& method, 
    	const ColourTechnique& colours) : 
    	legend_(legend), method_(method), colours_(colours)  {};
    ~LegendEntryBuilder() {};
    bool operator()(const pair<double, ColourInfo>& first, const pair<double, ColourInfo>& second) {
        Polyline* box = new Polyline();
        
        double min = first.second.level_;
        double max = second.second.level_;
        
        method_(*box, max, colours_);     
        box->setFillColour(colours_.right(min));
        box->setFilled(true);
        box->setStroke(true);     
        box->setColour(Colour("black"));     
        legend_.add(new BoxEntry(min, max, box));
        return false;
       
    }
    LegendVisitor& legend_;
    const PolyShadingMethod<P>& method_;
    const ColourTechnique& colours_;
};


template <class P>
class PolyShadingMethod {

public:
	PolyShadingMethod() {}
	virtual ~PolyShadingMethod() {}    
	virtual void set(const map<string, string>&) {}
	virtual void set(const XmlNode&) {}
	virtual bool accept(const string& node) { return magCompare(node, "area_fill"); }
	
	virtual void toxml(ostream&, int) {}
	virtual PolyShadingMethod<P>* clone() const
	{
		PolyShadingMethod<P>* object = new PolyShadingMethod<P>();
		return object;
	}
	virtual void prepare(const LevelSelection&) {}
	virtual bool shadingMode() { return true; }

	
    
    virtual void operator()(Polyline& poly, double, const ColourTechnique&) const
    {
       
      
        poly.setFilled(true);      
        poly.setStroke(false);      
        FillShadingProperties* shading = new FillShadingProperties();                      
        poly.setShading(shading);
    };
   
    virtual void visit(LegendVisitor& legend, const ColourTechnique& colour) { 
        
        Log::dev() << "Create legend information"  << "\n";
       
        std::adjacent_find(colour.begin(), colour.end(), LegendEntryBuilder<P>(legend, *this, colour));
     
        legend.last(); // Flag the last entry as being the last! To get a nice labelling in countinuous mode!!!
        
    };
    
protected:
     //! Method to print string about this class on to a stream of type ostream (virtual).
	 virtual void print(ostream&) const {} 

private:
    //! Copy constructor - No copy allowed
	PolyShadingMethod(const PolyShadingMethod<P>&);
    //! Overloaded << operator to copy - No copy allowed
	PolyShadingMethod<P>& operator=(const PolyShadingMethod<P>&);

// -- Friends
    //! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const PolyShadingMethod<P>& p)
		{ p.print(s); return s; }

};

template <class P>
class Translator<string, PolyShadingMethod<P> > { 
public:
	PolyShadingMethod<P>* operator()(const string& val ) {
		 return SimpleObjectMaker<PolyShadingMethod<P> >::create(val);
	}     
    PolyShadingMethod<P>* magics(const string& param)
    {
        PolyShadingMethod<P>* object=0;
		ParameterManager::update(param, object);
		return object;
    }
};

} // namespace magics

#endif
