/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file AxisMethod.cc
    \brief Implementation of the Template class AxisMethod.
    
    Magics Team - ECMWF 2005
    
    Started: Fri 7-Oct-2005
    
    Changes:
    
*/

#include <limits>

#include "AxisMethod.h"
#include "Transformation.h"

using namespace magics;

AxisMethod::AxisMethod() 
{
}

AxisMethod::~AxisMethod() 
{
}

/*!
 Class information are given to the output-stream.
*/		
void AxisMethod::print(ostream& out)  const
{
	out << "AxisMethod[";
	out << "]";
}

void LogarithmicAxisMethod::prepare(double min, double max,  AxisItems& items)
{

	    vector<int> factor;
	    factor.push_back(1);
	    factor.push_back(2);
	    factor.push_back(5);



	    int s1 = (min < 0) ? -1: 1;
	    int s2 = (max < 0) ? -1: 1;


	    int log1 =  s1 *((min) ? log10(s1*min) : -5);
	    int log2 =  s2 *((max) ? log10(s2*max) : -5);

	    bool reduce = (min*max) < 0;

	    std::set<double> ticks;

	    double from = std::min(min, max);
	    double to = std::max(min, max);

	    for (int i = std::min(log1, log2); i <= std::max(log1, log2); i +=1 ) {
	        for ( vector<int>::iterator f = factor.begin(); f != factor.end(); ++f) {

	        	double x;
	        	if ( !reduce || ( reduce && i > -4)  ) {
	        		x = (*f) * exp10(i);
	        		if ( from <= x  && x <= to && x != 0 )
	        			ticks.insert(x);
	        		x = -x;
	        		if ( from <= x  && x <= to && x != 0 )
	        			ticks.insert(x);
	        	}
	        	if ( !reduce || ( reduce && -i > -4   )) {
	        		x = (*f) * exp10(-i);
	        		if ( from <= x  && x <= to && x != 0 )
	        			ticks.insert(x);
	        		x = -x;
	        		if ( from <= x  && x <= to && x != 0)
	                	ticks.insert(x);
	        	}
	        }
	    }

	for ( std::set<double>::const_iterator step = ticks.begin(); step != ticks.end(); ++step) {
		items.push_back(new AxisItem(*step));
	}
}

void  HyperAxisMethod::updateX(const Transformation& transformation)
{
	hyperMin_ = transformation.getDataVectorMinX();
	
	hyperMax_ = transformation.getDataVectorMaxX();
	//hyperMin_.push_back(-180);
	//hyperMin_.push_back(-90);
	//hyperMax_.push_back(180);
	//hyperMax_.push_back(90);
}

void  HyperAxisMethod::updateY(const Transformation& transformation)
{
	hyperMin_ = transformation.getDataVectorMinY();
	hyperMax_ = transformation.getDataVectorMaxY();
	//hyperMin_.push_back(-180);
	//hyperMin_.push_back(-90);
	//hyperMax_.push_back(180);
	//hyperMax_.push_back(90);
}

void HyperAxisMethod::prepare(double min, double max, AxisItems& items)
{
	double inc;
	int nb = 7;
	double step;
	double log, ws;

	double wmax = std::max(min, max);
	double wmin = std::min(min, max);
	
	if (interval_ == INT_MAX ) {
	
		while (nb < 20) {
			step = (wmax-wmin)/nb;
			log = log10(step);
			ws = pow(10., int(log));
			inc = ceil(step/ws)*ws;
			MagLog::dev() << "Automatic method ---> increment = " << inc << " ---> try base=" << inc/ws << endl;
			
			 if ( wmax-wmin/inc > 5 && (inc/ws == 1 || inc/ws == 2 || inc/ws == 3 || inc/ws == 5 || inc/ws == 10) ) {
				MagLog::dev() << "Automatic method ---> increment " << inc << " OK! " << endl;
				break;
			}
			nb++;
			
		}
	}
	
	else {
		inc = interval_;
	}
	std::set<double> list;
	//list.insert(value(min));
	//ist.insert(value(max));
	
	
    if ( min < max ) {
		double first = floor(min/inc) *inc;
		double last  = 0.;    
		for (double val = first;  val <= max; val+=inc) {
			if (val >= min && val <=max) {
				list.insert(value(val));
        		last = val;
			}
		}
		list.insert(value(last+inc));
		
	}
	else {
		double first = floor(min/inc) *inc;    
		double last  = 0.;  
		for (double val = first;  val >= max; val-=inc) {
			if (val >= wmin && val <= wmax) {
				list.insert(value(val));       		
        		last = val;
			}
		}
		list.insert(value(last-inc));
	}
    
    double imin = hyperMin_[0];
    double imax = hyperMax_[0];
    double jmin = hyperMin_[1];
    double jmax = hyperMax_[1];;
    double iw = imax - imin;
    double jw = jmax - jmin;
    
    for (std::set<double>::iterator i = list.begin(); i != list.end(); ++i) {
    	vector<double> val;
    	val.push_back(*i);
    	double j = jmin+((*i)-imin)*(jw/iw);
    	val.push_back(j);
    	items.push_back( new AxisHyperItem(*i, val));
    }
}


void AxisMethod::prepare(double min, double max, AxisItems& items) 
{
	double inc;
	int nb = 7;
	double step;
	double log, ws;

	double wmax = std::max(min, max);
	double wmin = std::min(min, max);
	
	if (interval_ == INT_MAX ) {
	
		while (nb < 20) {
			step = (wmax-wmin)/nb;
			log = log10(step);
			ws = pow(10., int(log));
			inc = ceil(step/ws)*ws;
			MagLog::dev() << "Automatic method ---> increment = " << inc << " ---> try base=" << inc/ws << endl;
			
			 if ( wmax-wmin/inc > 5 && (inc/ws == 1 || inc/ws == 2 || inc/ws == 3 || inc/ws == 5 || inc/ws == 10) ) {
				MagLog::dev() << "Automatic method ---> increment " << inc << " OK! " << endl;
				break;
			}
			nb++;
			
		}
	}
	
	else {
		inc = interval_;
	}
	std::set<double> list;

	if ( min < max )
	{
		double first = floor(min/inc) *inc;
		double last  = 0.;    
		for (double val = first;  val <= max; val+=inc) {
			if (val >= min && val <=max) {
				list.insert(value(val));
        		last = val;
			}
		}
		list.insert(value(last+inc));
	}
	else
	{
		double first = floor(min/inc) *inc;    
		double last  = 0.;  
		for (double val = first;  val >= max; val-=inc) {
			if (val >= wmin && val <= wmax) {
				list.insert(value(val));       		
        			last = val;
			}
		}
		list.insert(value(last-inc));
	}

	// First we add the minor tich before the first one...
	std::set<double>::iterator front = list.begin();
	double first = *front - inc;
	step = inc/minor_tick_count_;
	for ( double ii = first; ii < *front; ii+=step ) {
		items.push_back(new AxisMinorTickItem(ii));
	}

	double last = std::numeric_limits<double>::max();
	for (std::set<double>::iterator i = front; i != list.end(); ++i) {
		// Add the minor Axis Items!!! 
		if ( last != std::numeric_limits<double>::max()) {
			double step = (*i-last)/minor_tick_count_;
			for ( double ii = last +step; ii < *i; ii+=step ) {
				items.push_back(new AxisMinorTickItem(ii));
			}
		}
		last = *i;
		addItem(items,*i);
	}
}

void AxisMethod::updateX(const Transformation& transformation)
{
	double min = transformation.getMinX();
	double max = transformation.getMaxX();
	setMin(min);
	setMax(max);
}

void AxisMethod::updateY(const Transformation& transformation)
{
	double min = transformation.getMinY();
	double max = transformation.getMaxY();
	setMin(min);
	setMax(max);
}

void     PositionListAxisMethod::prepare(double, double, AxisItems& items)
{
	for ( vector<double>::const_iterator tick = positions_.begin(); tick != positions_.end(); ++tick)
	{
		items.push_back(new AxisItem(*tick));
	}
}
