/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file LegendVisitor.h
    \brief Definition of the Template class LegendNode.
    
    Magics Team - ECMWF 2007
    
    Started: Tue 6-Mar-2007
    
    Changes:
    
*/

#ifndef LegendNode_H
#define LegendNode_H

#include "magics.h"

#include "SceneVisitor.h"

#include "PaperPoint.h"
#include "LegendVisitorAttributes.h"
#include "LegendBoxAttributes.h"
#include "XmlBasicNodeAttributes.h"
#include "XmlNode.h"
namespace magics {

class Arrow;
class Text;
class Symbol;
class Polyline;

	
class LegendEntry
{
public:
	LegendEntry(const string label) : label_(label), textPosition_(0.1) {}
	virtual ~LegendEntry() {}
	virtual void set(const PaperPoint&, BasicGraphicsObjectContainer&) {}
	
	// rowBox + columnBox used for a continuous legend... 
	virtual void rowBox(const PaperPoint&, BasicGraphicsObjectContainer&);
	virtual void columnBox(const PaperPoint&, BasicGraphicsObjectContainer&);
	
	LegendEntry(double label): textPosition_(0.1) { format(label); }
	LegendEntry(double min, double max): textPosition_(0.1) { format(min, max); }
	const string& label() const { return label_; }
	void setArrow(Arrow* arrow) { arrow_ = arrow; }
	string from()  { return tostring(from_); }
	 string to() { return tostring(to_); } 
    
    virtual double textRowPosition() { return -0.15; }
    virtual double textColumnPosition() { return textPosition_; }
	virtual void setTextPosition(double position) { textPosition_ = position; }
	
	void set(const LegendVisitorAttributes&); 

protected:
	string label_;
	double from_;
	double to_;
	string format_;
	MagFont font_;
	friend class LegendVisitor;
    double textPosition_;
	virtual Colour colour() { return Colour(); } 
	void format(double val)
	{
		from_ = val;
		to_   = val;
		
		label_ = tostring(val);
	}
	void format(double min, double max)
	{
		from_ = min;
		to_ = max;
		ostringstream s;
		
		s << min << "-" << max;
		label_ = s.str();
	}
    
	Arrow*  arrow_;
	
};

class EmptyEntry : public LegendEntry
{
public:
	EmptyEntry() : LegendEntry("") {}
	~EmptyEntry() {}
	virtual void rowBox(const PaperPoint&, BasicGraphicsObjectContainer&);
	virtual void columnBox(const PaperPoint&, BasicGraphicsObjectContainer&);
};

class SymbolEntry : public LegendEntry
{
public:
	SymbolEntry(const string& label, Symbol* symbol = 0 ) : 
		LegendEntry(label), symbol_(symbol) {}
	SymbolEntry(double label, Symbol* symbol = 0 ) :  
		LegendEntry(label), symbol_(symbol) { format(label); }
	SymbolEntry(double min, double max, Symbol* symbol = 0 ) : 
		LegendEntry(min, max), symbol_(symbol)
        { format(min, max); }
    void set(const PaperPoint&, BasicGraphicsObjectContainer&);
    
protected:
	Symbol* symbol_;
	Colour colour();
};
class SimpleSymbolEntry : public LegendEntry
{
public:
	SimpleSymbolEntry(const string& label, Symbol* symbol = 0 ) : 
		LegendEntry(label), symbol_(symbol) {}
	SimpleSymbolEntry(double label, Symbol* symbol = 0 ) :  
		LegendEntry(label), symbol_(symbol) { format(label); }
	SimpleSymbolEntry(double min, double max, Symbol* symbol = 0 ) : 
		LegendEntry(min, max), symbol_(symbol)
        { format(min, max); }
    void set(const PaperPoint&, BasicGraphicsObjectContainer&);
    
protected:
	Symbol* symbol_;
	Colour colour();
};

class LineEntry : public LegendEntry
{
public:
	LineEntry(const string label, Polyline* line = 0 ) : 
		LegendEntry(label), line_(line) {}
	LineEntry(double label, Polyline* line = 0 ) :  
		LegendEntry(label), line_(line) { format(label); }
	LineEntry(double min, double max, Polyline* line = 0 ) : 
		LegendEntry(min, max), line_(line)
        { format(min, max); }
    void set(const PaperPoint&, BasicGraphicsObjectContainer&);
    virtual void rowBox(const PaperPoint&, BasicGraphicsObjectContainer&) {}
	virtual void columnBox(const PaperPoint&, BasicGraphicsObjectContainer&) {}
protected:
	Polyline* line_;
	Colour colour();
};


class ArrowEntry : public LegendEntry
{
public:
	ArrowEntry(const string label, Arrow* arrow = 0 ) : 
		LegendEntry(label), arrow_(arrow) {}
	ArrowEntry(const double label, Arrow* arrow = 0 ) :  
		LegendEntry(label), arrow_(arrow) { format(label); }
	ArrowEntry(const double min, double max, Arrow* arrow = 0 ) : 
		LegendEntry(min, max), arrow_(arrow)
        { format(min, max); }
	void setArrow(Arrow* arrow) { arrow_ = arrow; }
	void set(const PaperPoint&, BasicGraphicsObjectContainer&);
protected:
	Arrow*  arrow_;
};

class BoxEntry : public LegendEntry
{
public:
	BoxEntry(const string label, Polyline* box = 0 ) : 
		LegendEntry(label), box_(box) {}
	BoxEntry(const double label, Polyline* box = 0 ) :  
		LegendEntry(label), box_(box) { format(label); }
	BoxEntry(double min, double max, Polyline* box = 0 ) : 
		LegendEntry(min, max), box_(box)
        { format(min, max); }
	void set(const PaperPoint&, BasicGraphicsObjectContainer&);
	virtual void rowBox(const PaperPoint&, BasicGraphicsObjectContainer&);
	virtual void columnBox(const PaperPoint&, BasicGraphicsObjectContainer&);
	
protected:
	Polyline*  box_;
	Colour colour();
};



class LegendVisitor: public LayoutVisitor, 
	public LegendVisitorAttributes, 
	public BasicPositionalObject,
	public VectorOfPointers<vector<LegendEntry*> > {

public:
	LegendVisitor();
	virtual ~LegendVisitor();
	virtual void getReady();
	virtual LegendVisitor* clone() { assert(false); return 0; }
	virtual Layout& layout() const { return *(const_cast<LegendVisitor*>(this)); }
    
    void set(const XmlNode& node) { LegendVisitorAttributes::set(node); }	
    void add(LegendEntry* entry) { VectorOfPointers<vector<LegendEntry*> >::push_back(entry); }

    void visit(AnimationStep&);
    void visit(BasicGraphicsObjectContainer&);
	void visit(BasicSceneObject&);
	bool positional() { return positional_; }

protected:
     //! Method to print string about this class on to a stream of type ostream (virtual).
	 virtual void print(ostream&) const; 
	
	 // Calculate the grid! 
	void grid(vector<pair<float, float> >&);
	void build();
	bool positional_;
private:
    //! Copy constructor - No copy allowed
	LegendVisitor(const LegendVisitor&);
    //! Overloaded << operator to copy - No copy allowed
	LegendVisitor& operator=(const LegendVisitor&);

// -- Friends
    //! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const LegendVisitor& p)
		{ p.print(s); return s; }

};


class XmlLegendVisitor : public LegendVisitor, public XmlBasicNodeAttributes
{
public:
	XmlLegendVisitor() {}
	~XmlLegendVisitor() {}
	void set(const XmlNode& node) {
		XmlNode view = node;
		view.name("view");
		XmlBasicNodeAttributes::set(view); 
		LegendVisitor::set(node);
	}
	void copy(const XmlLegendVisitor& other) {
		XmlBasicNodeAttributes::copy(other); 
		LegendVisitor::copy(other);
	}
	LegendVisitor* clone()
	{ 
		XmlLegendVisitor* legend = new XmlLegendVisitor(); 
		legend->copy(*this);
		legend->positional_ = true;
		return legend;
		
	}
	void getReady();	
};


class FortranPositionalLegendVisitor : public LegendVisitor, public LegendBoxAttributes
{
public:
	FortranPositionalLegendVisitor() {}
	~FortranPositionalLegendVisitor() {}
	void set(const XmlNode& node) {
		
		LegendBoxAttributes::set(node); 
		LegendVisitor::set(node);
	}
	void getReady();	
	void copy(const FortranPositionalLegendVisitor& other) {
		LegendBoxAttributes::copy(other); 
			LegendVisitor::copy(other);
		}
	LegendVisitor* clone() { 
		FortranPositionalLegendVisitor* legend = new FortranPositionalLegendVisitor(); 
		legend->copy(*this);
		legend->positional_ = true;
		return legend;
	}
};
 


class FortranAutomaticLegendVisitor : public LegendVisitor
{
public:
	FortranAutomaticLegendVisitor() {}
	~FortranAutomaticLegendVisitor() {}
	void set(const XmlNode& node) {
		LegendVisitor::set(node);
	}
	void getReady();	
	LegendVisitor* clone() { 
			FortranAutomaticLegendVisitor* legend = new FortranAutomaticLegendVisitor(); 
			legend->copy(*this);
			legend->positional_ = false;
			return legend;
	}
};


} // namespace magics
#endif
