/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file MagicsCalls.cc
    \brief Implementation of Fortran and C interface

 To use the C interface "magics_api.h" must be included.

 Changes: 13-06-2006 Added C interface (Stephan)

 \sa magics_api.h
*/

#include <Log.h>
#include <FortranMagics.h>
#include "WebFormat.h"

extern "C" {
#include <magics_api.h>
}





static FortranMagics* magics_ = 0;


/*! \brief Ensures backwards compability with Magics 6.x

  This base class checks if parameter are set which are not supported anymore 
  in Magics++ but need to be supported for backwards compability.
  
  To add a new parameter produce a new class which inherits from CompatibilityHelper.
  The produce a static object of that class: static <class_name> <para_name>;
*/
class CompatibilityHelper 
{
public :
	CompatibilityHelper(const string& name) {
		compatibility_[name] = this;
	}

	virtual ~CompatibilityHelper() {}
	template <class P>
	static bool check(const string& param, P value) 
	{
		map<string, CompatibilityHelper*>::const_iterator tool = compatibility_.find(lowerCase(param));
		if ( tool == compatibility_.end() ) return false;
		else return (*(tool->second))(value);
	}
	static bool check(const string& param, const string& value) 
	{
		map<string, CompatibilityHelper*>::const_iterator tool = compatibility_.find(lowerCase(param));
		if ( tool == compatibility_.end() ) return false;
		else return (*(tool->second))(string(value));
	}
	virtual bool operator()(int)           { return false; }
	virtual bool operator()(const string&) { return false; }
	virtual bool operator()(double)        { return false; }
	virtual bool operator()(const doublearray&)        { return false; }
	virtual bool operator()(const stringarray&)        { return false; }
	virtual bool operator()(bool)        { return false; }
protected:
	static map<string, CompatibilityHelper*> compatibility_;
};

class NoMoreGribex :  CompatibilityHelper
{
public :
	NoMoreGribex(const string& param) : CompatibilityHelper(param), parameter_(param) {}
	~NoMoreGribex() {}
	bool operator()(const string&)
	{
		Log::info() << "Compatibility issue: the parameter [" << parameter_ << "] is discontinued.\n"
		            << "              Please use the grib_api interface  instead." <<endl;
		return true;
	}

protected :
	string parameter_;
	
};


map<string, CompatibilityHelper*> CompatibilityHelper::compatibility_;

/*! \brief Prints info about old parameter
*/
class GribSubareaExtraction: public CompatibilityHelper {
public :
	GribSubareaExtraction() : CompatibilityHelper("grib_subarea_extraction") {}
	~GribSubareaExtraction() {}
	bool operator()(const string& )
	{
		Log::info() << "Compatibility issue: Parameter grib_subarea_extraction not required anymore.\n"<<endl;
		return true;
	}
};
/*! \brief Prints info about old parameter
*/
class GribFieldPosition: public CompatibilityHelper {
public :
	GribFieldPosition() : CompatibilityHelper("grib_field_position") {}
	~GribFieldPosition() {}
	bool operator()(int)
	{
		assert(magics_);
		magics_->resetGrib();
		return false;
	}
};

/*! \brief Converts ps_file_name into output_file_root_name.

  The convention to describe filenames in Magics++ has changed.
*/
class PsFileName: public CompatibilityHelper {
public :
	PsFileName() : CompatibilityHelper("ps_file_name") {}
	~PsFileName() {}
	bool operator()(const string& file)
	{
		Log::info() << "Compatibility issue: Parameter ps_file_name is deprecated.\n"
		            << "               Please use output_name instead."<<endl;
		ParameterManager::set("output_legacy_name", file);
		ParameterManager::set("output_file_minimal_width", 0);  
		return true;
	}
};

/*! \brief Converts ps_device into output_ps_device.
*/
class PsDevice: public CompatibilityHelper {
public :
	PsDevice() : CompatibilityHelper("ps_device") {}
	~PsDevice() {}
	bool operator()(const string& )
	{
		Log::info() << "Compatibility issue: Parameter ps_device was removed.\n"
		            << "               Please use other PostScript driver parameter instead."<<endl;
		return true;
	}
};

/*! \brief Converts ps_device into output_ps_device.
*/
class OutputPsDevice: public CompatibilityHelper {
public :
	OutputPsDevice() : CompatibilityHelper("output_ps_device") {}
	~OutputPsDevice() {}
	bool operator()(const string& )
	{
		Log::info() << "Compatibility issue: Parameter output_ps_device is deprecated."<<endl;
		return true;
	}
};

/*! \brief Removes ps_help
*/
class PsHelp: public CompatibilityHelper {
public :
	PsHelp() : CompatibilityHelper("ps_help") {}
	~PsHelp() {}
	bool operator()(const string& )
	{
		Log::info() << "Compatibility issue: Parameter ps_help was removed.\n"<<endl;
		return false;
	}
};

/*! \brief Removes ps_metric
*/
class PsMetric: public CompatibilityHelper {
public :
	PsMetric() : CompatibilityHelper("ps_metric") {}
	~PsMetric() {}
	bool operator()(const string& )
	{
		Log::info() << "Compatibility issue: Parameter ps_metric was removed.\n"<<endl;
		return false;
	}
};

/*! 
*/
class GraphType: public CompatibilityHelper {
public :
	GraphType() : CompatibilityHelper("graph_type") {}
	~GraphType() {}
	bool operator()(const string& type)
	{
		if ( magCompare(type, "curve") ) {
			ParameterManager::set("graph_shade", "off");
		}
		if ( magCompare(type, "bar") ) {
			ParameterManager::set("graph_shade", "on");
		}
		if ( magCompare(type, "area") ) {
			ParameterManager::set("graph_shade", "on");
		}
		return false;
	}
};

class GraphValuesConverter : public CompatibilityHelper
{
public:
	GraphValuesConverter(const string& from, const string & to) : 
		CompatibilityHelper(from), from_(from), to_(to) {}
	
	bool operator()(const doublearray& values)
	{
		Log::info() << "Compatibility issue: Parameter " << from_ << " is deprecated.\n"
		            << "               Please use " << to_ << " instead."<<endl;
		ParameterManager::set(to_, values);  
		return false;
	}
	bool operator()(const stringarray& values)
	{
		Log::info() << "Compatibility issue: Parameter " << from_ << " is deprecated.\n"
		            << "               Please use " << to_ << " instead."<<endl;
		ParameterManager::set(to_, values);  
		return false;
	}
	
protected:
	string from_;
	string to_;
};

static GraphValuesConverter graph_curve_x_values("graph_curve_x_values", "x_values");
static GraphValuesConverter graph_curve_y_values("graph_curve_y_values", "y_values");
static GraphValuesConverter graph_curve_date_x_values("graph_curve_date_x_values", "x_date_values");
static GraphValuesConverter graph_curve_date_y_values("graph_curve_date_y_values", "y_date_values");

static GraphValuesConverter graph_curve2_x_values("graph_curve2_x_values", "x2_values");
static GraphValuesConverter graph_curve2_y_values("graph_curve2_y_values", "y2_values");
static GraphValuesConverter graph_curve2_date_x_values("graph_curve2_date_x_values", "x2_date_values");
static GraphValuesConverter graph_curve2_date_y_values("graph_curve2_date_y_values", "y2_date_values");

static GraphValuesConverter graph_bar_x_values("graph_bar_x_values", "x_upper_values");
static GraphValuesConverter graph_bar_y_values("graph_bar_y_values", "y_values");
static GraphValuesConverter graph_bar_date_x_values("graph_bar_date_x_values", "x_upper_date_values");
static GraphValuesConverter graph_bar_date_y_values("graph_bar_date_y_values", "y_date_values");

static GraphValuesConverter graph_bar_x_lower_values("graph_bar_x_lower_values", "x_lower_values");
static GraphValuesConverter graph_bar_x_upper_values("graph_bar_x_upper_values", "x_upper_values");
static GraphValuesConverter graph_bar_date_x_lower_values("graph_bar_date_x_lower_values", "x_lower_date_values");
static GraphValuesConverter graph_bar_date_x_upper_values("graph_bar_date_x_upper_values", "x_upper_date_values");

static GraphValuesConverter graph_bar_y_lower_values("graph_bar_y_lower_values", "y_lower_values");
static GraphValuesConverter graph_bar_y_upper_values("graph_bar_y_upper_values", "y_upper_values");
static GraphValuesConverter graph_bar_date_y_lower_values("graph_bar_date_y_lower_values", "y_lower_date_values");
static GraphValuesConverter graph_bar_date_y_upper_values("graph_bar_date_y_upper_values", "y_upper_date_values");


/*! \brief Converts device_file_name into output_file_root_name.

  The convention to describe filenames in Magics++ has changed.
*/
class DeviceFileName: public CompatibilityHelper {
public :
	DeviceFileName() : CompatibilityHelper("device_file_name") {}
	~DeviceFileName() {}
	bool operator()(const string& file)
	{
		Log::info() << "Compatibility issue: Parameter device_file_name is deprecated.\n"
		            << "               Please use output_name instead."<<endl;
		ParameterManager::set("output_legacy_name", file);
		ParameterManager::set("output_file_minimal_width", 0);  
		return true;
	}
};

/*! \brief Converts device_width into output_width.

  The convention to describe filenames in Magics++ has changed.
*/
class DeviceWidth: public CompatibilityHelper {
public :
	DeviceWidth() : CompatibilityHelper("device_width") {}
	~DeviceWidth() {}
	bool operator()(const int width)
	{
		Log::info() << "Compatibility issue: Parameter device_width is deprecated.\n"
		               << "             Please use output_width instead."<<endl;
		ParameterManager::set("output_width", width);
		return true;
	}
};

/*! \brief Converts device_ into output_width.

  The convention to describe filenames in Magics++ has changed.
*/
class DeviceQualityLevel: public CompatibilityHelper {
public :
	DeviceQualityLevel() : CompatibilityHelper("device_quality_level") {}
	~DeviceQualityLevel() {}
	bool operator()(const int quality)
	{
		Log::info() << "Compatibility issue: Parameter device_quality_level is deprecated.\n"
		               << "             Please use output_jpg_quality instead."<<endl;
		ParameterManager::set("output_jpg_quality", quality);
		return true;
	}
};

/*! \brief Converts text_quality into text_font and text_font_style.

*/
class TextQuality: public CompatibilityHelper {
public :
	TextQuality() : CompatibilityHelper("text_quality") {}
	~TextQuality() {}
	bool operator()(const string& quality)
	{
		Log::info() << "Compatibility issue: Parameter text_quality is deprecated.\n"
		            << "               Please use text_font and text_font_style instead."<<endl;

		if(magCompare(quality,"low") )
		{
			ParameterManager::set("text_font","serif");
			ParameterManager::set("text_font_style","normal");
		}
		else if(magCompare(quality,"medium") )
		{
			ParameterManager::set("text_font", "sansserif");
			ParameterManager::set("text_font_style","normal");
		}
		else if(magCompare(quality,"high") )
		{
			ParameterManager::set("text_font","sansserif");
			ParameterManager::set("text_font_style", "bold");
		}
		else
		{
			Log::warning()<<"The setting "<<quality<<" for the parameter text_quality is not valid! Default font is used."<<endl;
			ParameterManager::set("text_font", "sansserif");
			ParameterManager::set("text_font_style", "normal");
		}

		return true;
	}
};

/*! \brief Converts text_quality into text_font and text_font_style.

*/
class LabelQuality: public CompatibilityHelper {
public :
	LabelQuality() : CompatibilityHelper("contour_label_quality") {}
	~LabelQuality() {}
	bool operator()(const string& quality)
	{
		Log::info() << "Compatibility issue: Parameter text_quality is deprecated.\n"
		            << "               Please use text_font and text_font_style instead."<<endl;

		if(magCompare(quality,"low") )
		{
			ParameterManager::set("contour_label_font", "serif");
			ParameterManager::set("contour_label_font_style", "normal");
		}
		else if(magCompare(quality,"medium") )
		{
			ParameterManager::set("contour_label_font", "sansserif");
			ParameterManager::set("contour_label_font_style", "normal");
		}
		else if(magCompare(quality,"high") )
		{
			ParameterManager::set("contour_label_font", "sansserif");
			ParameterManager::set("contour_label_font_style", "bold");
		}
		else
		{
			Log::warning()<<"The setting "<<quality<<" for the parameter contour_label_quality is not valid! Default font is used."<<endl;
			ParameterManager::set("contour_label_font", "sansserif");
			ParameterManager::set("contour_label_font_style", "normal");
		}

		return true;
	}
};

class TextReferenceCharacterHeight: public CompatibilityHelper {
public :
	TextReferenceCharacterHeight() : CompatibilityHelper("text_reference_character_height") {}
	~TextReferenceCharacterHeight() {}
	bool operator()(double height)
	{
		Log::warning() << "Compatibility issue: Parameter text_reference_character_height is deprecated.\n"
		               << "               Please use text_font_size instead. Text_font_size has been set to "<<height<<endl;
		ParameterManager::set("text_font_size", height);
		return true;
	}
};

/*! \brief Converts gd_file_name into output_file_root_name.

  The convention to describe GD filenames in Magics++ has changed between version 1.1 and 1.2.
*/
class GdFileName: public CompatibilityHelper {
public :
	GdFileName() : CompatibilityHelper("gd_file_name") {}
	~GdFileName() {}
	bool operator()(const string& file)
	{
		Log::info() << "Compatibility issue: Parameter gd_file_name is deprecated.\n"
		            << "              Please use output_name instead."<<endl;
		ParameterManager::set("output_legacy_name", file);
		ParameterManager::set("output_file_minimal_width", 0); 
		return true;
	}
};

class SubpageMapProjectionNone: public CompatibilityHelper {
public :
	SubpageMapProjectionNone() : CompatibilityHelper("subpage_map_projection") {}
	~SubpageMapProjectionNone() {}
	bool operator()(const string& projection)
	{
		string fix = projection;
		if ( magCompare(projection, "none") )
		{
			fix="cartesian";
			Log::info() << "Compatibility issue: The value [none] for Parameter subpage_map_projection is deprecated.\n"
			               << "               Please use [cartesian] instead."<<endl;
		}
		ParameterManager::set("subpage_map_projection",fix);
		return true;
	}
};

/*! \brief 'device' is not a valid parameter anymore 

   set device to gd and gf_format
*/
class DeviceCompatibilityHelper: public CompatibilityHelper {
public :
	DeviceCompatibilityHelper() : CompatibilityHelper("device") {}
	~DeviceCompatibilityHelper() {}
	bool operator()(const string& device)
	{
		Log::info() << "Compatibility issue: the parameter device is deprecated.\n"
		               << "              Please use the parameter output_format instead!"<< endl;

		if( magCompare(device, "jpeg") || magCompare(device, "jpg") || magCompare(device, "png") || magCompare(device, "gif")) {
			ParameterManager::set("output_format", device);
			return true;
		}
		else
		{
			ParameterManager::set("output_format", device);
			return true;
		}
		return false;
	}
};

class Legend: public CompatibilityHelper {
public :
	Legend(const string& legend) : CompatibilityHelper(legend) {}
	~Legend() {}
	bool operator()(const string& legend)
	{
		Translator<string, bool> translator;
		if ( translator(legend) == false )  
			return false;
		Log::info() << "Compatibility issue: The legend is turned on!\n";	
		ParameterManager::set("legend", "on");
		// Act as a Action routine!...
		magics_->plegend();
		return false;
	}
};

class WindArrowIndexHead: public CompatibilityHelper {
public :
	WindArrowIndexHead() : CompatibilityHelper("wind_arrow_head_index") {}
	~WindArrowIndexHead() {}
	bool operator()(int index)
	{
	    Log::info() << "Compatibility issue: Parameter wind_arrow_index_head does not exist anymore.\n"
	                   << "            use wind_arrow_head_shape and wind_arrow_head_ratio instead." << endl;
	    const int head_index = (int)index/10;
	    const int ratio = index % 10;
	    double head_ratio;

	    if (ratio == 1)       head_ratio = 0.3;
	    else if (ratio == 2)  head_ratio = 0.6;
	    else if (ratio == 3)  head_ratio = 1.;
	    else if (ratio == 4)  head_ratio = 1.3;
	    else if (ratio == 5)  head_ratio = 1.6;
	    else if (ratio == 6)  head_ratio = 2.0;
	    else
	    {
	    	Log::warning() << "invalid ratio " << ratio << " revert to default 1." << endl;
	    	head_ratio = 1.;
	    }

	    Log::info() << "  wind_arrow_head_index set to " << head_index
	                   << "AND wind_arrow_head_ratio set to " << head_ratio << endl;

	    ParameterManager::set("wind_arrow_head_shape", head_index);
	    ParameterManager::set("wind_arrow_head_ratio", head_ratio);
	    return true;
	}
};

static PsFileName ps_file_name;
static GribSubareaExtraction grib_subarea_extraction;
static GribFieldPosition grib_field_position;
static GdFileName gd_file_name;
static PsDevice ps_device;
static OutputPsDevice output_ps_device;
static PsHelp ps_help;
static PsMetric ps_metric;
static DeviceCompatibilityHelper device;
static DeviceFileName device_file_name;
static DeviceQualityLevel device_quality_level;
static DeviceWidth device_width;
static TextQuality text_quality;
static LabelQuality contour_label_quality;
static WindArrowIndexHead wind_arrow_index_head;
static TextReferenceCharacterHeight text_reference_character_height;
static SubpageMapProjectionNone subpage_map_projection_none;
static NoMoreGribex grib_mode("grib_mode");
static NoMoreGribex grib_product_block("grib_product_block");
static NoMoreGribex grib_grid_block("grib_grid_block");
static NoMoreGribex grib_output_field("grib_output_field");
static NoMoreGribex grib_input_type("grib_input_type");

static GraphType graph_type;
static Legend legend("legend");



extern "C" {

/* **************************************************************************

***
***  Fortran 77 interface
***

****************************************************************************/

void popen_()
{
	if (magics_ == 0) 
		magics_ = new FortranMagics();
	magics_->popen();
}

void pcoast_()
{
	magics_->pcoast();
}

void ptaylor_()
{
	magics_->ptaylor();

}
void pgrib_()
{
	magics_->pgrib();
}

void ptest_()
{
	magics_->ptest();
}

//! This function is overwritten in libMagPlusODB
void podb_()
{
	Log::warning() << "ODB support is NOT enabled! (Please try linking -lMagPlusODB)\n";
}


void pimport_()
{
	magics_->pimport();
}

void pnetcdf_()
{
	magics_->pnetcdf();
}

void pcont_()
{
	magics_->pcont();
}

void pobs_()
{
	magics_->pobs();
}

void praw_()
{
#ifdef MAGICS_NETPBM
	Log::warning() << "praw->not implemented\n";  
#else
	Log::warning() << "Netpbm NOT supported!" << endl;
#endif    
}

void pimage_()
{
	magics_->pimage();
}

void pplot_()
{
   Log::info() << "pplot has no effect ... use pimport instead" << endl;;  
    
}

void pnew_(const char* page, int l)
{
	magics_->pnew(string(page, l));
}

void ptext_()
{
    magics_->ptext();
}

void pwind_()
{
     magics_->pwind();
}


void pline_()
{
	magics_->pline();
}

void psymb_()
{
    magics_->psymb();
}

void pclose_()
{
	magics_->pclose();

	delete magics_;
	magics_ = 0;
}

void pact_(const char*, const char*, const char*, int, int, int)
{
	Log::dev() << "entry in the new pact [NOT implemented ]\n";    
}

void preset_(const char* name, int namel)
{
	try {
		ParameterManager::reset(string(name, namel));
	}
	catch (MagicsException& e)
	{
		Log::error() << e << "\n";
	}
}

void psetc_(const char* name, const char* value, int namel, int valuel)
{
	try {        
		string val = string(value, valuel);
		int index = val.find_last_not_of(" ");    
		if ( CompatibilityHelper::check(string(name, namel), val.substr(0, index+1)) ) return;
		ParameterManager::set(string(name, namel), val.substr(0, index+1));
	}
	catch (MagicsException& e)
	{
		Log::error() << e << "\n";
	}
}

void pseti_(const char* name, const int* value, int namel)
{
	try {
		if ( CompatibilityHelper::check(string(name, namel), int(*value)) ) return;
		ParameterManager::set(string(name, namel), int(*value));
	}
	catch (MagicsException& e)
	{
		Log::error() << e << "\n";
	}
}

void pset1i_(const char* name, const int* value, const int* dim, int nl)
{
	Log::dev() << "entry in the new pset1i\n";    
	intarray values;
	for ( int i = 0; i < *dim; i++) values.push_back(value[i]);        

	try {
		ParameterManager::set(string(name, nl), values);
	}
	catch (MagicsException& e)
	{
		Log::error() << e << "\n";
	}
}

void pset2i_(const char*, const int*, const int*, const int*, int)
{
	Log::warning() << "pset2i-->not yet implemented\n";    
}

void pset3i_(const char*, const int*, const int*, const int*, const int*, int)
{
	Log::warning() << "pset3i-->not yet implemented\n";      
}

void pset1c_(const char* name, const char* value, const int *dim, int namel, int l)
{
    Log::dev() << "entry in the new pset1c\n";    
    Log::dev() << "\tpset1c(\"" << string(name, namel) << ", "  << value << ", "  << *dim << ", " << l << ");"  << "\n";
    stringarray values;
    string work(value, (*dim)*l);
    int first =0;
    for ( int i = 0; i < *dim; i++) {
        // remove the  space at the end of the string
        string val = work.substr(first, l);
        int index = val.find_last_not_of(" ");
        values.push_back(val.substr(0, index+1));        
        first += l;
    }
    
    try {
    	string n(name, namel);
    	if ( CompatibilityHelper::check(n, values) ) return;
        ParameterManager::set(n, values);
    }
    catch (MagicsException& e)
    {
        Log::error() << e << "\n";
    }
}


void penqi_(const char* name, int* value, int namel)
{
	Log::dev() << "entry in the new penqi\n";    
	Log::dev() << "\tpenqi(\"" << string(name, namel) << "\", "  << *value << ");"  << "\n";
}

void penqc_(const char* name, char* value, int namel, int valuel)
{
	Log::dev() << "entry in the new penqc\n";    
	Log::dev() << "\tpenqc(\"" << string(name, namel) << "\", \"" << string(value, valuel) << "\");"  << "\n";
}

void ppie_()
{
	Log::warning() << "ppie-->not yet implemented\n";   
}


void pgraph_()
{
    magics_->pgraph();
}

void paxis_()
{
	 magics_->paxis();
}

void pgeo_()
{
	magics_->pgeo();
}

void peps_()
{
	Log::warning() << "peps-->not yet implemented\n";   
}


void pboxplot_()
{
	magics_->pboxplot();
}


void pprint_()
{
	Log::warning() << "pprint-->not yet implemented\n";
}

void pinfo_(){mag_info();}


/* **************************************************************************

***
***  C interface  ( calling Fortran 90 interface above )
***

****************************************************************************/
void mag_open()  {popen_();}
void mag_close() {pclose_();}
void mag_coast() {pcoast_();}
void mag_grib()  {pgrib_();}
void mag_test()  {ptest_();}
void mag_odb()   {podb_();}
void mag_import(){pimport_();}
void mag_netcdf(){pnetcdf_();}
void mag_cont()  {pcont_();}
void mag_obs()   {pobs_();}
void mag_raw()   {praw_();}
void mag_image() {pimage_();}
void mag_plot()  {pplot_();}
void mag_text()	 {ptext_();}
void mag_wind()  {pwind_();}
void mag_symb()  {psymb_();}
void mag_boxplot()  {pboxplot_();}

void mag_new(const char* page)
{
	string n(page);
	pnew_(n.c_str(), n.size());
}

void mag_reset(const char* name)
{
	string n(name);
	preset_(n.c_str(), n.size());	
}

void mag_setc(const char* name, const char* value)
{
	string n(name);
	string v(value);
	psetc_(n.c_str(), value, n.size(), v.size());
}

void mag_setr(const char* name, const double value)
{
	std::string n(name);

	try {
		if ( CompatibilityHelper::check(n, value) ) return;
		ParameterManager::set(n, value);
	}
	catch (MagicsException& e)
	{
		Log::error() << e << "\n";
	}
}

void mag_seti(const char* name, const int value)
{
	string n(name);
	pseti_(name, &value, n.size());
}

void mag_setp(const char* name, void* value)
{
	string n(name);
#ifdef MAGICS_CAIRO
    if ( magCompare(n, "output_cairo_drawing_context") ) {
       ParameterManager::set("output_cairo_drawing_context", (CairoPtr)value); 
    }
#endif
	
}
void mag_act(const char* a, const char* b, const char* c)
{
	string aa(a);
	string bb(b);
	string cc(c);
	pact_(a, b, c,aa.size(),bb.size(),cc.size());
}

void mag_set1r(const char* name, const double *data, const int dim1)
{
	std::string n(name);
	floatarray values;
	for ( int i = 0; i < dim1; i++) values.push_back(data[i]);        

	try {
		if ( CompatibilityHelper::check(n, values) ) return;
		ParameterManager::set(n, values);
	}
	catch (MagicsException& e)
	{
		Log::error() << e << "\n";
	}
}

void mag_set2r(const char* name, const double *data, const int dim1, const int dim2)
{
	Matrix matrix;
	for (int i = 0; i < (dim2) * (dim1); i++) matrix.push_back(data[i]);

	matrix.set(dim2, dim1);
	ParameterManager::set(string(name), matrix);
	Log::dev() << "Parameter " << string(name) << " set to " << matrix << "\n";
}

void mag_set3r(const char*, const double *, const int, const int, const int)
{
	Log::warning() << "pset3r --> not yet implemented\n";      
}

void mag_set1i(const char* name, const int *data, const int dim1)
{
	string n(name);
	pset1i_(name, data, &dim1, n.size());
}

void mag_set2i(const char* name, const int *data, const int dim1, const int dim2)
{
	string n(name);
	pset2i_(name, data, &dim1, &dim2, n.size());
}

void mag_set3i(const char* name, const int *data, const int dim1, const int dim2, const int dim3)
{
	string n(name);
	pset3i_(name, data, &dim1, &dim2, &dim3, n.size());
}

void mag_set1c(const char* name, const char** data, const int dim)
{
	string n(name);

	Log::dev() << "entry in the new mag_set1c\n";    
	Log::dev() << "\tmag_set1c("  << dim << " entries);\n";
	stringarray values;

	for ( int i = 0; i < dim; i++)
	{
		string work (data[i]);
		// remove the space at the start and end of the string
		int index1 = work.find_first_not_of(" ");
		int index2 = work.find_last_not_of(" ");
		values.push_back(work.substr(index1, index2+1));        
	}
    
	try {
		if ( CompatibilityHelper::check(n, values) ) return;
		ParameterManager::set(string(name, n.size()), values);
	}
	catch (MagicsException& e)
	{
		Log::error() << e << "\n";
	}
}

void mag_enqr(const char* name, double *value)
{
	double magics;
	ParameterManager::get(string(name),magics);
	*value=magics;
}


void mag_enqi(const char* name, int *value)
{
	string n(name);
	penqi_(name, value, n.size());
}

void mag_enqc(const char* name, char* value)
{
	string n(name);
	string v(value);
	penqc_(name, value, n.size(), v.size());
}


void mag_pie()   {ppie_();}
void mag_graph() {pgraph_();}
void mag_axis()  {paxis_();}
void mag_geo()   {pgeo_();}
void mag_eps()   {peps_();}
void mag_print() {pprint_();}
void mag_info()
{
	Log::userInfo() << "INFO:\n"
		<< "INFO: "<<getMagicsRelease()<<"\n"
		<< "INFO: "<<getMagicsConfigureInfo()<<"\n"
		<< "INFO:\n"
		<< "INFO: Machine: "<<getEnvVariable("HOSTNAME")<<" is running "<<getEnvVariable("VENDOR")<<" "<<getEnvVariable("OSTYPE")<<" "<<getEnvVariable("MACHTYPE")<<"\n"
		<< "INFO:\n"
		<< "INFO: $MAGPLUS_HOME    = "<<getEnvVariable("MAGPLUS_HOME")<<"\n"
		<< "INFO: $TMPDIR          = "<<getEnvVariable("TMPDIR")<<"\n"
		<< "INFO: $ODB_LIBS        = "<<getEnvVariable("ODB_LIBS")<<"\n"
		<< "INFO: $LD_LIBRARY_PATH = "<<getEnvVariable("LD_LIBRARY_PATH")<<"\n"
		<< "INFO:\n";
}

}// end of extern "C"


class AxisConverter : public CompatibilityHelper
{
public:
	AxisConverter(const string& from) : CompatibilityHelper(from), from_(from) {}
	
	bool operator()(double val)
	{
		ParameterManager::set(from_, val);
		update();
		return false;
	}

	void update()
	{
		string orientation;
		string type;
		string min;
		string max;
		double mind;
		double maxd;
		
		ParameterManager::get("axis_orientation", orientation); 
		ParameterManager::get("axis_type", type); 
		ParameterManager::get("axis_min_value", mind);
		ParameterManager::get("axis_max_value", maxd);
		ParameterManager::get("axis_date_min_value", min);
		ParameterManager::get("axis_date_max_value", max);

		if ( magCompare(orientation, "horizontal") )
		{
		    ParameterManager::set("x_axis_type",  type);
		    ParameterManager::set("x_min",  mind);
		    ParameterManager::set("x_max",  maxd);
		    ParameterManager::set("x_date_min",  min);
		    ParameterManager::set("x_date_max",  max);
		}
		else
		{
		    ParameterManager::set("y_axis_type",  type);
		    ParameterManager::set("y_min",  mind);
		    ParameterManager::set("y_max",  maxd);
		    ParameterManager::set("y_date_min",  min);
		    ParameterManager::set("y_date_max",  max);
		}
	}

	bool operator()(const string& value)
	{
		ParameterManager::set(from_, value);
		update();
		return false;
	}
protected:
	string from_;
	string horizontal_;
	string vertical_;
};

static AxisConverter axis_type("axis_type");
static AxisConverter axis_orientation("axis_orientation");

static AxisConverter axis_min_value("axis_min_value");
static AxisConverter axis_max_value("axis_max_value");
static AxisConverter axis_min_date_value("axis_date_min_value");
static AxisConverter axis_max_date_value("axis_date_max_value");

void execute_magml(const char* file)
{
	WebInterpretor::magml(file);
}

void execute_json(const char* file)
{
	WebInterpretor::json(file);
}
void set_param(const char* param, const char* value)
{
	WebInterpretor::set(param, value);
}
