/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)
 
 Licensed under the Apache License, Version 2.0 (the "License"); 
 you may not use this file except in compliance with the License. 
 You may obtain a copy of the License at 
 
 	http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software 
 distributed under the License is distributed on an "AS IS" BASIS, 
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 See the License for the specific language governing permissions and 
 limitations under the License.


 ******************************** LICENSE ********************************/

/*!
    \file OpenGLCellBarWidget.cc
    \brief Implementation of the OpenGLCellBarWidget class.
    \author Graphics Section, ECMWF

    Started: July 2008
*/

#include <OpenGLCellBarWidget.h>
#include <OpenGLDriver.h>

#include <MtInputEvent.h>

OpenGLCellBarWidget::OpenGLCellBarWidget(OpenGLDriver *driver,string name,int x, int y, int w, int h) :
					 OpenGLBaseWidget(driver,0,name,x,y,w,h)
					 
{
	//Default is vertical alingmetnt
	horizontalAlingment_=false;
	
	barX_=0;
	barY_=0;
	
	slider_=new OpenGLLabelWidget(driver,"label");

	sliderStatus_=false;
	
	slider_->setHeight(16);
	slider_->setWidth(16);
	
	//These will set the slider position
	setCellWidth(15);
	setCellHeight(12);	
	
	cellNum_=0;	
	actCell_=0;
	focusedCell_=-1;
	
	tickMarkBarStatus_=false;
	tickMarkTextGap_=4;
	tickMarkTextHeight_=5.;
	tickMarkBarGap_=-2;
	tickMarkBarHeight_=4;

	tickMarkColour_ = Colour(0.2,0.2,0.2);
	//cellActColour_=Colour(1.,0.,0.);
	cellActColour_=Colour(81./255.,162./255.,254./255.);
	cellColour_=Colour(0.,0.8,0.);
	cellBorderColour_=Colour(0.1,0.1,0.1);			

	colours_["cell"]=&cellColour_;
	colours_["cellAct"]=&cellActColour_;
	colours_["cellBorder"]=&cellBorderColour_;
		
	drag_=false;
}


OpenGLCellBarWidget::~OpenGLCellBarWidget()
{
	delete slider_;
	clearCells();
}

void OpenGLCellBarWidget::setGeometry()
{
	if(! horizontalAlingment_)
	{
		height_=cellWidth_*cellNum_;
		width_=cellWidth_;
	}
	else
	{
		width_=cellWidth_*cellNum_;
		if(!tickMarkBarStatus_)
		{
			height_=cellHeight_;
		}
		else
		{
			height_=cellHeight_+tickMarkTextGap_+tickMarkTextHeight_+
				tickMarkBarGap_+tickMarkBarHeight_;
		}
	}
}

void OpenGLCellBarWidget::setCellWidth(int w)
{
	cellWidth_=w; 
	if(! horizontalAlingment_)
	{
		slider_->setX(barX_+cellWidth_/2.-slider_->getWidth()/2.);
	}
	else
	{
		slider_->setX(barX_+cellWidth_/2.-slider_->getWidth()/2.);
	}
	
	setGeometry();

}	
void OpenGLCellBarWidget::setHeight(int h)
{
	height_=h;
	cellHeight_=height_-barY_;
}


void OpenGLCellBarWidget::setCellHeight(int h)
{
	cellHeight_=h;
	setGeometry();	
}	


void OpenGLCellBarWidget::setSliderPixmap(string fname,string type)
{
	slider_->setPixmap(fname,type);
}

void OpenGLCellBarWidget::clearCells()
{
	for(int i=0; i < cellNum_; i++)
	{
		delete cell_[i];
	}
	cell_.clear();
}


void OpenGLCellBarWidget::createCells(int num,int actvalindex)
{
	//Clear the current cells
	clearCells();

	//
	cellNum_=num;
	actCell_=actvalindex;
	focusedCell_=-1;
	setSlider();
	setGeometry();	

	//Allocate new cells and set the colour ;
	for(int i=0; i< cellNum_; i++)
	{
		OpenGLCellData *c=new OpenGLCellData;
		c->setColour(cellColour_);
		cell_.push_back(c);	
	}
}


void OpenGLCellBarWidget::setCellValue(int i,vector<string> &vals)
{
	assert(i >= 0 && i < cell_.size());
	cell_[i]->setValue(vals);
}

void OpenGLCellBarWidget::setCellColour(int i,Colour &col)
{
	assert(i >= 0 && i < cell_.size());
	{
		cell_[i]->setColour(col);
	}
}

bool OpenGLCellBarWidget::shiftActCell(int d)
{
	if(actCell_+d >=0 && actCell_+d < cellNum_)
	{
		actCell_+=d;
		setSlider();
		return true;
	}
	return false;
}

bool OpenGLCellBarWidget::setActCellToLast()
{
	if(actCell_ != cellNum_-1)
	{
		actCell_=cellNum_-1;
		setSlider();
		return true;
	}
	return false;
}

bool OpenGLCellBarWidget::setActCellToFirst()
{
	if(actCell_ != 0)
	{
		actCell_=0;
		setSlider();
		return true;
	}
	return false;
}

void OpenGLCellBarWidget::setSlider(int cell)
{
        if(cell >= 0 && cell < cellNum_)
	{
		if(! horizontalAlingment_)		
		{	
			slider_->setY(barY_+(cell+0.5)*cellHeight_-0.5*slider_->getHeight());			
		}	
		else
		{		
		
			slider_->setX(barX_+(cell+0.5)*cellWidth_-0.5*slider_->getWidth());
		}		
	}		
}


void OpenGLCellBarWidget::setSlider()
{
        setSlider(actCell_);
}


bool OpenGLCellBarWidget::checkSliderInBar()
{
	if(! horizontalAlingment_)
	{		
		if(slider_->getY() + slider_->getHeight() > 
		        barY_+cellNum_*cellHeight_-cellHeight_/2. + slider_->getHeight()/2.)
		{
			setSlider(cellNum_-1);
			return false;
		}
		else if(slider_->getY() < barY_+cellHeight_/2. - slider_->getHeight()/2.)
		{
			setSlider(0);
			return false;
		}
	}
	
	return true;		
}


bool OpenGLCellBarWidget::checkPointInSlider(int x,int y)
{	
	return slider_->checkPointInWidget(x,y);	
}


void OpenGLCellBarWidget::setActCell()
{		
	if(! horizontalAlingment_)
	{				
		int i=(slider_->getY()+slider_->getHeight()/2-barY_)/cellHeight_;
		if( i >= 0 && i < cellNum_)
		{
			actCell_=i;
		}
	}	
}	

int OpenGLCellBarWidget::identifyCell(int x,int y)
{
	if(! horizontalAlingment_)
	{
		int x1=barX_;
		int x2=barX_+cellWidth_;
		int y1=barY_;
		int y2=barY_+cellNum_*cellHeight_;		

		if((x-x1)*(x2-x) >0 && (y-y1)*(y2-y) >0 )
		{
			float r=static_cast<float>(y-barY_)/static_cast<float>(cellHeight_);
			if(r>=0. && r < cellNum_)
			{
				return static_cast<int>(r);
			}
		}
		return -1;
	}
	else
	{
		int x1=barX_;
		int x2=barX_+cellNum_*cellWidth_;
		int y1=barY_;
		int y2=barY_+cellHeight_;
		
		if((x-x1)*(x2-x) >0 && (y-y1)*(y2-y) >0 )
		{
			int i=(x-barX_)/cellWidth_;
			if(i>=0 && i< cellNum_)
			{
				return i;
			}
		}
		return -1;		
	}
}


bool OpenGLCellBarWidget::checkPointInCell(int x,int y)
{	
	int i;
	if((i=identifyCell(x,y)) != -1)
	{		
		actCell_=i;
		return true;		
	}	
	
	return false;	
}


void OpenGLCellBarWidget::render()
{
	float xp, yp;
	Colour col;
	Colour *pcol,*pcol1;

	glPushMatrix();
	glTranslatef(x_,y_,0.);

	glPushAttrib(GL_POLYGON_BIT);
	
	//Only front sides are rendered
	glFrontFace(GL_CCW);
	glEnable(GL_CULL_FACE);	
	glCullFace(GL_BACK);
		
	//------------------
	// Render background
	//-------------------
	
	glPolygonMode(GL_FRONT,GL_FILL);
	glColor3f(211./255.,215./255.,223./255.); //glColor3f(1.,1.,1.);
	glRectf(-1,-1,width_+1,height_+1);

	//-------------------
	// Render cells
	//-------------------

	//Fill cells		
	glPolygonMode(GL_FRONT,GL_FILL);
			
	//---------------------------
	// Vertical cellbar
	//---------------------------

	if(! horizontalAlingment_)
	{
		xp=barX_;
		yp=barY_;
	
		glShadeModel(GL_SMOOTH);

		for(int i=0; i< cellNum_;i++)
		{				
			if(sliderStatus_ == false)
			{		
				if(i==focusedCell_)
				{				
					col=cell_[i]->colour();
					col.scaleLight(0.7);			
				}
				else
				{
					col=cell_[i]->colour();
				}	
			}
			else
			{
				col=cell_[i]->colour();
			}
		
			glColor3f(col.red(),col.green(),col.blue());
			glPolygonMode(GL_FRONT,GL_FILL);
			glRectf(xp,yp,xp+cellWidth_,yp+cellHeight_);

			yp+=cellHeight_;
				
		}

		glShadeModel(GL_FLAT);
	}

	//---------------------------
	// Horizontal cellbar
	//---------------------------	

	else
	{
		//----------------------
		//Fill cells		
		//----------------------

		glPolygonMode(GL_FRONT,GL_FILL);

		xp=barX_;
		yp=barY_;
	
		int cellStart=0;
		int cellEnd=cellNum_;

		if(clipping_)
		{
			cellStart=clipArea_.x()/cellWidth_;
			assert(cellStart >=0 && cellStart < cellNum_);

			glTranslatef(cellStart*cellWidth_-clipArea_.x(),0.,0.);

			cellEnd=cellStart+(clipArea_.width()/cellWidth_)+1;
			assert(cellEnd  >=0 && cellEnd <= cellNum_);
		}

		for(int i=cellStart; i< cellEnd;i++)
		{													
			if(sliderStatus_ == false)
			{		
				col=cell_[i]->colour();
				/*if(i==actCell_)
				{
					col=cellActColour_;
				}
				else
				{
					col=cell_[i]->colour();
				}*/

				if(i==focusedCell_)
				{									
					col.scaleLight(0.7);				
				}
			}
			else
			{
				col=cell_[i]->colour();
			}

			glColor3f(col.red(),col.green(),col.blue());
			glPolygonMode(GL_FRONT,GL_FILL);
			glRectf(xp,yp,xp+cellWidth_,yp+cellHeight_);

			xp+=cellWidth_;
		}
		
		//Glow effect for the act cell
		/*if(clipping_ == false || actCell_ >= cellStart && actCell_ <= cellEnd)
		{
			pcol=&cellActColour_;
			glPolygonMode(GL_FRONT,GL_FILL);

			int dx=1;
			int dy=1;
			float dalpha=0.1;

			xp=barX_+actCell_*cellWidth_;
			yp=barY_;
			glEnable(GL_BLEND); 
			glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);	
				
			for(int i=1; i < 9; i++)
			{
				glColor4f(pcol->red(),pcol->green(),pcol->blue(),
				1.-static_cast<float>(i)*dalpha);		
				glRectf(xp-i*dx,yp-i*dy,xp+cellWidth_+i*dx,yp+cellHeight_+i*dy);
			}

			glDisable(GL_BLEND);
		}*/	


		//-------------------
		//Cells outline	
		//-------------------

		glPolygonMode(GL_FRONT,GL_LINE);
	
		glPushAttrib(GL_ENABLE_BIT | GL_LINE_BIT);	
		glLineWidth(1);

		col = cellBorderColour_;	
		glColor3f(col.red(),col.green(),col.blue()); 	

		xp=barX_;
		yp=barY_;
	
		for(int i=cellStart; i< cellEnd;i++)
		{											
			glRectf(xp,yp,xp+cellWidth_,yp+cellHeight_);
			xp+=cellWidth_;
		}

		if(sliderStatus_ == false && 
		  (clipping_ == false || 
	           (actCell_ >= cellStart && actCell_ < cellEnd)))
		{
			xp=barX_+actCell_*cellWidth_;
			yp=barY_+1;
			pcol=&cellActColour_;

			glPushAttrib(GL_LINE_BIT| GL_CURRENT_BIT);				

			glPolygonMode(GL_FRONT,GL_LINE);
			glLineWidth(2);	

			glColor3f(pcol->red(),pcol->green(),pcol->blue());
			glRectf(xp,yp,xp+cellWidth_,yp+cellHeight_-2);	

			glPopAttrib();
		}
		
		glPopAttrib();
	}
			
	//Cells outline
	
	glPolygonMode(GL_FRONT,GL_LINE);
	
	glPushAttrib(GL_ENABLE_BIT | GL_LINE_BIT);
	/*glEnable(GL_BLEND);
	//glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glBlendFunc(GL_CONSTANT_ALPHA,GL_ONE_MINUS_CONSTANT_ALPHA);

	glEnable(GL_LINE_SMOOTH);
	glHint(GL_LINE_SMOOTH_HINT,GL_NICEST);*/		

	glLineWidth(1);

	col = cellBorderColour_;	
	glColor3f(col.red(),col.green(),col.blue()); 	
	
	if(! horizontalAlingment_)
	{
		xp=barX_;
		yp=barY_;
	
		for(int i=0; i< cellNum_;i++)
		{	
			glRectf(xp,yp,xp+cellWidth_,yp+cellHeight_);
			//Render frame
			/*if(sliderStatus_ == false &&  i==actCell_)
			{							
				pcol=&cellActColour_;

				glPushAttrib(GL_LINE_BIT| GL_CURRENT_BIT);				

				glPolygonMode(GL_FRONT,GL_LINE);
				glLineWidth(2);	

				glColor3f(pcol->red(),pcol->green(),pcol->blue());
				glRectf(xp,yp,xp+cellWidth_,yp+cellHeight_);	

				glPopAttrib();

			}*/
			//glRectf(xp,yp,xp+cellWidth_,yp+cellHeight_);
			yp+=cellHeight_;

		}
			if(sliderStatus_ == false)
			{							
				xp=barX_;
				yp=barY_+actCell_*cellHeight_;
				pcol=&cellActColour_;

				glPushAttrib(GL_LINE_BIT| GL_CURRENT_BIT);				

				glPolygonMode(GL_FRONT,GL_LINE);
				glLineWidth(2);	

				glColor3f(pcol->red(),pcol->green(),pcol->blue());
				glRectf(xp,yp,xp+cellWidth_-1,yp+cellHeight_-1);	

				glPopAttrib();
			}

		
	}	

	glPopAttrib();

	//--------------------
	// Render slider
	//--------------------

	if(sliderStatus_)
	{
		slider_->render();
	}

	glPopAttrib();	
	glPopMatrix();

	rendered_=true;

}
void OpenGLCellBarWidget::getPosInWidget(int x, int y, int &rx, int &ry)
{
	if(!clipping_)
	{
		rx=x-x_;
		ry=y-y_;
	}
	else
	{	
		rx=x-x_+clipArea_.x();
		ry=y-y_;		
	}
}

//--------------------------------------------------
// Event handling
//--------------------------------------------------

void OpenGLCellBarWidget::event(MtInputEvent *event)
{
	rendered_=false;
	
	if(cellNum_<=0 ) return;
	
	int x,y;
	MtMouseEvent *mev;	

	switch(event->type())
	{
	
	case Mt::MousePressEvent:				
		mev = (MtMouseEvent *) event;
		getPosInWidget(mev->x(),mev->y(),x,y);

		if(mev->button() & Mt::LeftButton) // left  mouse button
		{			
			input(x,y);									
		}
		
		break;
	
	
	case Mt::MouseMoveEvent:
		mev = (MtMouseEvent *) event;

		if(sliderStatus_)
		{
			if(mev->button() & Mt::LeftButton)
			{
				getPosInWidget(mev->x(),mev->y(),x,y);

				if(drag_)
 				{	
					sliderDrag(x,y);	
				}
			}
			
		}
		
		if( ! (mev->button() & Mt::LeftButton) )		
		{
			getPosInWidget(mev->x(),mev->y(),x,y);
			focusedCellInput(x,y);
		}	
	
		break;	
	
	case Mt::MouseReleaseEvent:
		mev = (MtMouseEvent *) event;

		getPosInWidget(mev->x(),mev->y(),x,y);
		
		if(mev->button() & Mt::LeftButton) // left  mouse button
 		{			 						
			if(drag_)
			{
				sliderDrag_end(x, y);			
			}
 		}
		break;
	
	default:
		break;
	
	}
	
}



//----------------------------------------
// Button1 press event
//----------------------------------------

void OpenGLCellBarWidget::input(int x,int y)
{
	//If slider selected then start slider drag
	if(sliderStatus_ == true && checkPointInSlider(x,y) == true)
	{
		sliderDrag_start(x,y);
		return;
	}		
	
	//A cell is selected
	else if(checkPointInCell(x,y) == true)
	{
		executeCallback(OpenGLWidget::ValueChangedCallback,&actCell_);
	}	
}

void OpenGLCellBarWidget::sliderDrag_start(int x,int y)
{			
	drag_=true;
	
	prevX_=x;
	prevY_=y;
}

void OpenGLCellBarWidget::sliderDrag(int x,int y)
{			
	if(!horizontalAlingment_)
	{
		slider_->setY(slider_->getY()+y-prevY_);
		prevY_=y;
		
		if(checkSliderInBar()==false)
		{
			prevY_=slider_->getY();
		}			
	}
}

void OpenGLCellBarWidget::sliderDrag_end(int x,int y)
{				
	drag_=false;
	
	setActCell();
	executeCallback(OpenGLWidget::ValueChangedCallback,&actCell_);	
}


void OpenGLCellBarWidget::focusedCellInput(int x,int y)
{
	int ac=identifyCell(x,y);
	if(ac != focusedCell_)
	{
		/*if(ac != -1)
		{
			drag_=true;		
		}
		else
		{			
			drag_=true;
			//drag_=false;
		}*/

		focusedCell_=ac;
		executeCallback(OpenGLWidget::FocusChangedCallback,&focusedCell_);

		//drag_=false;
	}
}	

bool OpenGLCellBarWidget::active()
{
	return drag_;;
}	
