/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)
 
 Licensed under the Apache License, Version 2.0 (the "License"); 
 you may not use this file except in compliance with the License. 
 You may obtain a copy of the License at 
 
 	http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software 
 distributed under the License is distributed on an "AS IS" BASIS, 
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 See the License for the specific language governing permissions and 
 limitations under the License.


 ******************************** LICENSE ********************************/

/*! \file OpenGLMagnifier.h
    \brief Definition of the OpenGLMagnifier.
    \author Graphics Section, ECMWF
    
    Started: October 2007    
*/

#ifndef OpenGLMagnifier_H
#define OpenGLMagnifier_H

#include <GL/gl.h>

#include <magics.h>
#include <Log.h>
#include <PaperPoint.h>
#include <OpenGLDriverObserver.h>
#include <OpenGLBaseTool.h>
#include <OpenGLNode.h>
#include <OpenGLGui.h>

namespace magics {

class OpenGLDriver;
class OpenGLFontHandler;
class OpenGLMagnifierScale;
class OpenGLMagnifierButton;
class OpenGLTextureItem;
class OpenGLFboItem;
class OpenGLDriverManager;

class OpenGLRingData
{
public:
	OpenGLRingData(float,float, float, int);
	bool checkRadius(float r) { return r>=minRadius_ && r< maxRadius_;};
	
	float width_;
	float minRadius_;
	float maxRadius_;
	vector<float> coeffCos_;
	vector<float> coeffSin_;
	int ptNum_;	
};

class OpenGLGridValArea
{
public:
	OpenGLGridValArea() : x_(0.), y_(0.), width_(0.), height_(0.), sizeFactor_(3), dx_(0.), dy_(0.), zoom_(1.) {};

	bool checkArea(float, float,float,float);
	bool update(float,float,float,float);

	float x() {return x_;}	
	float y() {return y_;}
	float width() {return width_;}
	float height() {return height_;}
	float dx() {return dx_;}	
	float dy() {return dy_;}

private:
	float x_;
	float y_;
	float width_;
	float height_;
	float sizeFactor_;
	float dx_;
	float dy_;
	float zoom_;
};


class OpenGLMagnifier :  public OpenGLNodeVisitor, public OpenGLGui
{
friend class OpenGLMagnifierScale;
friend class OpenGLMagnifierButton;

public:
	OpenGLMagnifier(OpenGLDriver *,string);
	~OpenGLMagnifier();		
	
	void init(const int,const int,const int,const int);	
			
	void build();
	void show();
	void hide();
	
	bool active();
	
	void clear();	
			
	void unselect();

	void event(MtInputEvent*);
 
        void renderScale();
        void remagnify();

	void currentFrameChanged();

	void driver(OpenGLDriver* drv) {driver_=drv;};
	

private:
	enum OpenGLMagnifierPointType {MGP_LENS,MGP_FRAME,MGP_OUTSIDE,MGP_CELL};
	enum OpenGLMagnifierMode {MGM_TEXT, MGM_DATA};

	bool checkPointInWindow(const float,const float) const;	
	bool checkCircleInWindow(const float,const float,const float) const;
	
	bool checkPointInMagnifier(const int, const int);
	OpenGLMagnifierPointType identifyPoint(const int, const int);

	void drag_start(int,int);
	void replace(const int, const int);
	void resize(const int, const int);
	
	void render(const bool);
	void renderFrame(OpenGLRingData *);	
	
	void setFrameData();	
	
	bool visitEnter(OpenGLNode&);
	bool visitLeave(OpenGLNode&);
	void visit(OpenGLNode&);	
		
	void renderLens();
	bool visitEnter_renderLens(OpenGLNode&);
	bool visitLeave_renderLens(OpenGLNode&);
	void visit_renderLens(OpenGLNode&);

	void clearTextDpl();
	bool visitEnter_clearTextDpl(OpenGLNode&);
	bool visitLeave_clearTextDpl(OpenGLNode&);
	void visit_clearTextDpl(OpenGLNode&);
	
	bool updateGridValArea(bool forceUpdate=false);

	bool changeMagnifierMode();
	bool calcPowerScaleForData();

	float radius_;
	float minRadius_;

	OpenGLMagnifierMode magnifierMode_;
	std::map<OpenGLMagnifierMode, OpenGLMagnifierScale*> scale_;
	OpenGLMagnifierScale* actScale_;

	int actFrameIndex_;
	vector<OpenGLRingData> frames_;
		
	vector<OpenGLMagnifierButton*> button_;

	float centreX_;
	float centreY_;
		
	int minX_;
	int maxX_;
	int minY_;
	int maxY_;
	
	int refX_;
	int refY_;
	
	int prevX_;
	int prevY_;
	
	GLuint dpyListFrame_;
        GLuint dpyListScale_;
	
	OpenGLFontHandler* fontHandler_;

	bool pointInMagnifier_;

	typedef void (OpenGLMagnifier::*VisitFunctionVoidPtr)(OpenGLNode&);
	typedef bool (OpenGLMagnifier::*VisitFunctionBoolPtr)(OpenGLNode&);
	
	VisitFunctionVoidPtr visitFunc_;
	VisitFunctionBoolPtr visitEnterFunc_;
	VisitFunctionBoolPtr visitLeaveFunc_;	

	float gridValAreaX_;
	float gridValAreaY_;
	float gridValAreaWidth_;
	float gridValAreaheight_;

	OpenGLGridValArea *gridValArea_;
	bool gridValPlotEnabled_;

	stack< std::set<int> > areaIndexStack_;

	OpenGLFboItem*	fbo_;
	OpenGLTextureItem* tex_;
	bool magnifierModeChanged_;
};

}// namespace magics
#endif
