/* This file is part of the KDE project
   Copyright (C) 2006-2007 Omat Holding B.V. <info@omat.nl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include "addresslineedit.h"

#include <QMouseEvent>
#include <QKeyEvent>
#include <QTimer>
#include <QPoint>
#include <QWhatsThis>

#include <kpimutils/email.h>

#include <KColorScheme>
#include <KDebug>

//---------- Completion class --------------------//

using namespace Mailody;

Completion::Completion( KLineEdit *parent )
        : KCompletion()
{
    m_box = new KCompletionBox( parent );
}

QString Completion::makeCompletion( const QString &text )
{
    QStringList results = substringCompletion( text );
    if ( !results.isEmpty() ) {
        m_box->setItems( results );
        m_box->popup();
    } else if ( m_box->isVisible() )
        m_box->hide();

    return QString();
}

//------------- AddressLineEdit -------------------//
AddressLineEdit::AddressLineEdit( QWidget *parent )
        : KLineEdit( parent )
        , m_helpText( false )
{
    m_completion = new Completion( this );
    m_completion->setOrder( KCompletion::Sorted );
    m_completion->setIgnoreCase( true );
    connect( m_completion->box(), SIGNAL( activated( const QString& ) ),
             SLOT( slotSetAddress( const QString& ) ) );

    setCompletionObject( m_completion );
    setAutoDeleteCompletionObject( true );

    m_timer = new QTimer( this );
    m_timer->setSingleShot( true );
    connect( m_timer, SIGNAL( timeout() ), SLOT( slotShowHelp() ) );
}

void AddressLineEdit::keyPressEvent( QKeyEvent *e )
{
    if ( help() && !( e->key() == Qt::Key_Return || e->key() == Qt::Key_Enter ) ) {
        clear();
        setHelp( false );
        KLineEdit::keyPressEvent( e );
        validate();
    } else if ( help() && ( e->key() == Qt::Key_Return || e->key() == Qt::Key_Enter ) )
        emit shiftAddress();
    else if ( !help() && ( e->key() == Qt::Key_Return || e->key() == Qt::Key_Enter ) ) {
        if ( m_completion->box()->currentItem() && m_completion->boxVisible() ) {
            emit addAddress( m_completion->box()->currentItem()->text() );
            m_completion->box()->hide();
        } else
            emit addAddress( text() );
    } else {
        KLineEdit::keyPressEvent( e );
        validate();
    }
}

void AddressLineEdit::mousePressEvent( QMouseEvent *e )
{
    if ( help() ) {
        clear();
        setHelp( false );
        KLineEdit::mousePressEvent( e );
    }
}

void AddressLineEdit::validate()
{
    static bool oldResult = false;
    KPIMUtils::EmailParseResult result = KPIMUtils::isValidAddress( text() );
    if ( result == KPIMUtils::AddressOk && !oldResult ) {
        emit containsValidAddress( true );
        oldResult = true;
    } else if ( result != KPIMUtils::AddressOk && oldResult ) {
        emit containsValidAddress( false );
        oldResult = false;
    }

    if ( result != KPIMUtils::AddressOk && !help() ) {
        m_showHelp = emailParseResultToString( result );
        m_timer->start( 3000 );
    }
}

void AddressLineEdit::slotShowHelp()
{
    /* seems fragile to me. I don't seem to understand the positioning of this widget */
    QPoint globalPos( mapToGlobal( mapToParent( pos() ) ) );
    QPoint pos( globalPos.x(), globalPos.y()+5 );
    QWhatsThis::showText( pos, m_showHelp );
}

void AddressLineEdit::slotSetAddress( const QString &text )
{
    setText( text );
    validate();
}

void AddressLineEdit::setHelp( bool i )
{
    m_helpText = i;
    QPalette p;
    if ( i ) {
        p.setColor( QPalette::Normal, QPalette::Text,
                    QColor( 160,160,160 ) );
        setCursorPosition( 0 );
    } else {
        p.setColor( QPalette::Normal, QPalette::Text,
                    KColorScheme( QPalette::Active, KColorScheme::View ).foreground().color() );
    }
    setPalette( p );

}

#include "addresslineedit.moc"
