/* This file is part of the KDE project

   Copyright (C) 2006-2007 Omat Holding B.V. <info@omat.nl>
   Copyright (C) 2007 Frode M. Døving <frode@lnix.net>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/


#ifndef MESSAGEDATA_H
#define MESSAGEDATA_H

#include <QDateTime>

#include <KLocale>
#include <kio/job.h>
#include <KTemporaryFile>

#include <kmime/kmime_message.h>
#include <akonadi/item.h>
#include <akonadi/collection.h>

#include <boost/shared_ptr.hpp>
typedef boost::shared_ptr<KMime::Message> MessagePtr;

namespace Mailody
{

class ImapManager;

/**
 * @class MessageData
 * This is the main class which holds a message. It will disect the message
 * and fill the variables.
 * @author Tom Albers <tomalbers@kde.nl>
 */
class MessageData : public QObject
{
    Q_OBJECT

public:
    /**
     * Contructor. After construction the basic info to create a header
     * list is created. The other member variables will only be filled
     * after a request for the body is made.
     * @param parent parent object
     * @param name give it a name
     */
    explicit MessageData( QWidget* parent, const char* name );

    /**
     * Destructor. As there can be multiple classes using this data,
     * please call done(), instead of deleting this object. See more
     * info at Done() and setBlocked().
     */
    ~MessageData();

    bool setMsg( const Akonadi::Item&, const Akonadi::Collection& );

    enum bodyType {
        Plain = 0,
        HTML,
        Source
    };

    void requestBody( Mailody::MessageData::bodyType );

    QString vDate() const;

    /** Returns the akonadi item */
    Akonadi::Item item() const      {
        return m_item;
    }

    /** Returns the mailbox */
    Akonadi::Collection collection() const {
        return m_collection;
    }

    /** Returns the email address of the sender */
    QString sender_email() const    {
        return m_sender_email;
    }

    /** Returns the name of the sender*/
    QString sender() const          {
        return m_sender;
    }

    /** Returns the name of the sender, prepended with To: if needed*/
    QString vsender() const          {
        return m_vsender;
    }

    /** Returns the to header of the message, all recepients */
    QString to() const              {
        return m_to;
    }

    /** Returns the size of message */
    int size() const                {
        return m_size;
    }

    /** Returns the size of message for the user */
    QString vSize() const           {
        return
            KIO::convertSize( KIO::filesize_t( m_size ) );
    }

    /** Returns id of the message where this is a reply to */
    QString inreplyto() const       {
        return m_inreplyto;
    }

    /** Returns the cc header of the message, all recepients */
    QString cc() const              {
        return m_cc;
    }

    /** Returns the subject. */
    QString subject() {
        return m_subject;
    }

    /**
     * Returns a list of the attachments, attached to the message
     * in the form path,name
     */
    QHash<KUrl,QString> attachments() const {
        return m_attachments;
    }

    /** Returns the body of the message, */
    QString body() const            {
        return m_body;
    }

    /** Returns the original message */
    QString originalBody() const    {
        return m_realraw;
    }

    /** Returns the used userAgent of the message */
    QString userAgent() const       {
        return m_userAgent;
    }

    /** Returns the messageID of the message */
    QString messageID() const       {
        return m_messageID;
    }

    /** Returns the replyTo of the message */
    QString replyTo() const         {
        return m_replyTo;
    }

    /** Returns the resent from of the message (redirected messages) */
    QString resentFrom() const      {
        return m_resentFrom;
    }

    /** Returns the resent to of the message (redirected messages) */
    QString resentTo() const        {
        return m_resentTo;
    }

    /** Returns the raw version of the message. You can call this to
        get the version of the message, which you can use to print
        for example. It will always return the plain version if possible */
    QString raw() const             {
        return m_raw;
    }

    /** Returns the DataTime of the message */
    KDateTime date() const          {
        return m_date;
    }

    /** Returns the uid of the message */
    int uid() const                 {
        return m_uid;
    }

    /** Returns a list of the to addresses. See also to() */
    QStringList to_list() const    {
        return m_to_addresses;
    }

    /** Returns a list of the cc addresses. See also cc() */
    QStringList cc_list() const    {
        return m_cc_addresses;
    }

    /**
     * Returns the name of the sender, including the emailaddress
     * @param i include the company if provided
     */
    QString sender_full( bool i )  {
        return i && !m_company.isEmpty() ? m_sender_full+" ("+m_company+")"
               : m_sender_full;
    }

    /**
     * Return the contents of X-Mailody-Signature. (XMS)
     * This header is added when one saves the message
     * it is used when replacing the signature inside the composer
     * on re-edit. Not included in sent messages.
     */
    QString getXMS()        {
        return m_xms;
    }

    /** Return the contents of X-Mailody-SP. (XMSP)
     * that is the signature placement int.
     */
    int getXMSP()           {
        return m_xmsp;
    }

    /** Returns if this message is deleted */
    bool isDeleted() const;

    /**
     * Returns if this message is read (not really New, it can be old but
     * unread.
    */
    bool isNew() const;

    /**
     * Returns if a user label has been set for this message.
     * @param i the type of the label, we support 0-5, where 0 is none.
     */
    bool hasUserTag( int i ) const;

    /**
     * the sort date. see for more info the setSortDate() method
     */
    QDateTime sortDate() const {
        return m_sortDate;
    };

    /**
     * Here you can set the sort date. This date will be used when
     * sorting on date is performed. In our case, you can set this date
     * to the date of the youngest child for example. This is independent
     * of the shown date.
     */
    void setSortDate( QDateTime t ) {
        m_sortDate = t;
    };

    /**
     * Here you can indicate that you don't want this data to be deleted. When
     * done() has been called before and @p block is false, this object will
     * be deleted.
     */
    void setBlocked( bool block );

    /**
     * This will delete this object, unless setBlocked() is called with true.
     * when that is the case, the object will only be deleted after a
     * setBlocked call with parameter false.
     */
    void done();

    /**
     * this returns flags from the initial creation data. This is a very
     * fast function, but may not be accurate.
     */
    bool containsFlagsCached( const QString& flag ) {
        return m_flagsCached.contains( flag );
    };


private:
    Akonadi::Item           m_item;
    bool                    m_block;
    bool                    m_delete;
    ImapManager*            m_datamanager;
    QString                 m_subject;
    QString                 m_sender;
    QString                 m_vsender;
    QString                 m_sender_email;
    QString                 m_sender_full;
    QString                 m_company;
    QString                 m_cc;
    int                     m_size;
    QString                 m_inreplyto;
    QStringList             m_cc_addresses;
    QString                 m_to;
    QStringList             m_to_addresses;
    QHash<KUrl,QString>     m_attachments;
    QList<KTemporaryFile*>  m_attachmentPointers;
    QStringList             m_headers;
    QString                 m_body;
    KDateTime               m_date;
    QDateTime               m_sortDate;
    QString                 m_userAgent;
    QString                 m_messageID;
    QString                 m_replyTo;
    QString                 m_resentFrom;
    QString                 m_resentTo;
    QString                 m_vDate;

    QString                 m_raw;
    QString                 m_realraw;
    bodyType                m_bodyType;
    MessagePtr              m_msg;

    int                     m_uid;
    int                     m_xmsp;
    Akonadi::Collection     m_collection;
    QStringList             m_flags;
    QString                 m_flagsCached;
    QString                 m_xms;

signals:
    /*
     * Emitted when the message is ready, for your convenience, the
     * MessageData (this) is send with the signal.
      */
    void messageData( const MessageData* );

private slots:
    void slotItemChangeFlagDone( KJob* );
};

}

#endif
