/*
   Copyright (C) 2006-2007 Omat Holding B.V. <info@omat.nl>
   Copyright (C) 2007 Frode M. Døving <frode@lnix.net>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

// Own
#include "messageview.h"
#include "addressbook.h"
#include "addresslabel.h"

// Qt
#include <QApplication>
#include <QClipboard>
#include <QMenu>
#include <QScrollBar>
#include <QTextCodec>
#include <QTextDocument> // Qt::escape()
#include <QTextStream>

// KDE
#include <KFileDialog>
#include <KGlobalSettings>
#include <KHTMLView>
#include <KMessageBox>
#include <KRun>
#include <KToolInvocation>

#include <dom/html_element.h>
#include <kpimutils/linklocator.h>
#include <kio/netaccess.h>

using namespace Mailody;

TotalView::TotalView( QWidget* parent )
        : QWidget( parent ), m_msg( 0 )
{
    QGridLayout* layout = new QGridLayout( this );

    m_headers = new MessageHeaderView( this );
    layout->addWidget( m_headers, 0, 0 );
    layout->setMargin( 0 );

    m_body    = new MessageBodyView( this );
    layout->addWidget( m_body->widget(), 1, 0 );

    autoSaveAttachments();
}

TotalView::~TotalView()
{
}

void TotalView::setMsg( const Akonadi::Item& item, const Akonadi::Collection& collection )
{
    // disconnect to prevent multiple connections!
    if ( m_msg )
        delete m_msg;

    m_msg = new MessageData( this, "message" );

    connect( m_msg, SIGNAL( messageData( const MessageData* ) ),
             m_body, SLOT( slotShowMsg( const MessageData* ) ) );

    bool ok = m_msg->setMsg( item, collection );
    kDebug() << "Loaded: " << ok;
    if ( !ok )
        return;

    m_headers->setMsg( m_msg );
    m_body->setMsg( m_msg );
}

void TotalView::clearView()
{
    m_headers->clearView();
    m_body->clearView();
}

void TotalView::save( const QString& filename )
{
    QFile file( filename );
    QTextStream stream( &file );
    stream.setCodec( QTextCodec::codecForName( "UTF-8" ) );
    file.open( QIODevice::WriteOnly );
    stream << m_msg->raw().trimmed();
    file.close();
}

void TotalView::print()
{
    QString msgheadersdata;
    msgheadersdata += "<table border=1 cellspacing=0 "
                      "cellpadding=0 width=100%>";
    msgheadersdata += "<tr><td colspan=2 bgcolor=\"#dddddd\"><b><h2>"
                      + msg()->subject() + "<h2></b></td>";
    msgheadersdata += "</tr>";
    msgheadersdata += "<tr><td align=right><b>" + i18n( "From:" ) + "</b></td>";
    msgheadersdata += "    <td width=\"100%\">" + msg()->sender_full( true )
                      + "</td>";
    msgheadersdata += "</tr>";
    msgheadersdata += "<tr><td align=right><b>" + i18n( "To:" ) + "</b></td>";
    msgheadersdata += "    <td width=\"100%\">" + msg()->to() + "</td>";
    msgheadersdata += "</tr>";
    if ( !msg()->cc().isEmpty() ) {
        msgheadersdata += "<tr><td align=right><b>" + i18n( "Cc:" )
                          + "</b></td>";
        msgheadersdata += "    <td width=\"100%\">" + msg()->cc() + "</td>";
        msgheadersdata += "</tr>";
    }
    msgheadersdata += "<tr><td align=right><b>" + i18n( "Date:" ) + "</b></td>";
    msgheadersdata += "    <td width=\"100%\">" + msg()->vDate() + "</td>";
    msgheadersdata += "</tr></table>";

    KHTMLPart* newPart = new KHTMLPart( this );
    newPart->setJScriptEnabled( false );
    newPart->setJavaEnabled( false );
    newPart->setMetaRefreshEnabled( false );
    newPart->setPluginsEnabled( false );
    newPart->setOnlyLocalReferences( !m_body->externalImage() );
    m_body->fixedfont()
    ? newPart->setStandardFont( KGlobalSettings::fixedFont().family() )
    : newPart->setStandardFont( KGlobalSettings::generalFont().family() );

    newPart->begin();
    newPart->write( msgheadersdata );
    newPart->write( "<br><br>" );
    newPart->write( msg()->body() );
    newPart->end();
    newPart->view()->print();
    delete newPart;
}

// --------------------- Headers -------------------------------

MessageHeaderView::MessageHeaderView( QWidget* parent )
        : QScrollArea( parent ), m_currentMessage( 0 ),  m_storeAttachments( false )
{
    hide();
    setHorizontalScrollBarPolicy( Qt::ScrollBarAlwaysOff );
    setLineWidth( 1 );
    setFrameStyle( QFrame::Panel | QFrame::Raised );

    m_label = new AddressLabel( widget() );
    m_label->setWordWrap( true );
    setWidget( m_label );

    m_foto = new QLabel( this );

    // linkActivated only seems to work once, use custom implementation.
    connect( m_label, SIGNAL( leftMouseActivated( const QString& ) ),
             SLOT( slotLeftMouseClick( const QString& ) ) );
    connect( m_label, SIGNAL( rightMouseActivated( const QString& ) ),
             SLOT( slotRightMouseClick( const QString& ) ) );
}

MessageHeaderView::~MessageHeaderView()
{}

void MessageHeaderView::setMsg( MessageData* msg )
{
    kDebug() << endl;

    m_currentMessage = msg;
    verticalScrollBar()->setValue( 0 );
    horizontalScrollBar()->setValue( 0 );

    if ( msg && isHidden() ) {
        show(); // can be hidden by favorite website / middle click.
    }

    // keep it after the possible show(), so the first message is
    // shown correctly too.
    updateHeaders();
}

void MessageHeaderView::autoSaveAttachments()
{
    KConfigGroup config = KGlobal::config()->group( "General" );
    m_storeAttachments = config.readEntry( "storeAllAttachments", false );
    m_attachmentFolder = KUrl::fromPath( config.readEntry( "attachmentsFolder" ) );
}

void MessageHeaderView::getFreeFilename( KUrl &fileLocation )
{
    // now we see if a file with that name already exists.
    const QString path = fileLocation.path();
    bool exists = KIO::NetAccess::exists( fileLocation, KIO::NetAccess::DestinationSide, this );
    int i = 0;
    while ( exists ) {
        // it exists, so find the extension, add a digit and try again.
        int extAt = path.lastIndexOf( '.' );
        if ( extAt == -1 )
            extAt = path.length();

        const QString newFilename = path.left( extAt ) + '_' + QString::number( i ) + path.mid( extAt );
        fileLocation.setPath( newFilename );

        exists = KIO::NetAccess::exists( fileLocation, KIO::NetAccess::DestinationSide, this );
        i++;
    }
}

void MessageHeaderView::autoSaveAttachment( const KUrl &fileLocation, const QString &fileName )
{
    KUrl destination( m_attachmentFolder );
    destination.addPath( fileName );
    getFreeFilename( destination );

    kDebug() << "Filelocation" << fileLocation << "Visible Name" << fileName << "Destination" << destination;

    KIO::NetAccess::file_copy( fileLocation, destination, this );
}

void MessageHeaderView::updateHeaders()
{
    if ( !m_currentMessage )
        return;

    bool hasPhoto=false;
    /* TODO: PORT
       // copied from KMail...
    QString photoURL;
    int photoWidth = 60;
    int photoHeight = 60;

    KABC::AddresseeList addresses =
        Addressbook::instance()->findByEmail(m_currentMessage->sender_email());
    if (addresses.count()==1)
    {

        // Next part is an example of how the on hover show more info.
        // as it needs to do the same for all addresses in the headers
        // I need to think about this more (it will require an address
        // book search for every address for example. Remove next line when
        // and uncomment to see it.

        QString tip;
        PhoneNumber::List list = addresses[0].phoneNumbers();
        QValueList<PhoneNumber>::Iterator it=list.begin();
        for(; it!=list.end(); ++it)
        {
            tip.append((*it).label() + ": " + (*it).number() );
            tip.append("\n");
            kDebug() << tip << endl;
        }

        if (!tip.isEmpty())
        {
            m_sender_data->setUseTips(true);
            m_sender_data->setTipText(tip);
        }
        else
            m_sender_data->setUseTips(false);

        // stop here!

        if (addresses[0].photo().isIntern())
        {
            QImage photo = addresses[0].photo().data();
            if ( !photo.isNull() )
            {
                photoWidth = photo.width();
                photoHeight = photo.height();
                // scale below 60, otherwise it can get way too large
                if ( photoHeight > 60 ) {
                    double ratio = ( double )photoHeight / ( double )photoWidth;
                    photoHeight = 60;
                    photoWidth = (int)( 60 / ratio );
                    photo = photo.scaled( photoWidth, photoHeight );
                }

                Q3MimeSourceFactory::defaultFactory()->
                        setImage( "UID"+addresses[0].uid(), photo );

                m_foto->setText(
                        QString("<img src=\"%1\" width=\"%2\" height=\"%3\">")
                                .arg( "UID"+addresses[0].uid() )
                                .arg( photoWidth )
                                .arg( photoHeight ));
                hasPhoto = true;
            }
        }
    }
    else
        m_foto->setText(QString::null);
    // --- end

    */
    //------------- the rest
    using KPIMUtils::LinkLocator;
    const int flags = LinkLocator::PreserveSpaces |
                      LinkLocator::HighlightText |
                      LinkLocator::IgnoreUrls;

    QString text = "<table style=\"margin-right: 80px; margin-left:4px;\" ";
    text.append( "cellpadding=0 cellspacing=1 width=\""
                 + QString::number( width() ) + "\">" );
    text.append( "<tr><td colspan=2>"
                 + LinkLocator::convertToHtml( m_currentMessage->subject()
                                               .trimmed(), flags )
                 + "</td></tr>" );

    QString t = "<a href=\"email:"
                + LinkLocator::convertToHtml(
                    m_currentMessage->sender_full( false ),
                    flags ) + "\">"
                + LinkLocator::convertToHtml(
                    m_currentMessage->sender_full( true )
                    .trimmed(), flags ) + "</a>";
    text.append( "<tr><td valign=top align=right>" + i18n( "From" ) + ": </td>" );
    if ( m_currentMessage->userAgent().startsWith( "Mailody" ) )
        text.append( "<td valign=top width=\"100%\"><font color=\"#22cc22\">"
                     + t + "<td></tr>\n" );
    else
        text.append( "<td valign=top width=\"100%\">" + t + "</td></tr>" );

    QStringList to = m_currentMessage->to_list();
    QString toList;
    if ( to.count()>0 ) {
        QStringList::Iterator it=to.begin();
        for ( ; it!=to.end(); ++it ) {
            if ( !toList.isEmpty() )
                toList.append( ", " );
            toList.append( "<a href=\"email:"
                           + LinkLocator::convertToHtml(( *it ).trimmed(), flags )
                           + "\">"
                           + LinkLocator::convertToHtml(( *it ).trimmed(), flags )
                           + "</a>" );
        }
        text.append( "<tr><td align=right valign=top>" + i18n( "To" ) + ": </td>" );
        text.append( "<td valign=top width=\"100%\">" + toList + "</td></tr>" );
    }

    QStringList cc = m_currentMessage->cc_list();
    if ( cc.count()>0 ) {
        QString t;
        QStringList::Iterator it=cc.begin();
        for ( ; it!=cc.end(); ++it ) {
            if ( !t.isEmpty() )
                t.append( ", " );
            t.append( "<a href=\"email:"
                      + LinkLocator::convertToHtml(( *it ).trimmed(), flags )
                      + "\">"
                      + LinkLocator::convertToHtml(( *it ).trimmed(), flags )
                      + "</a>" );

        }
        text.append( "<tr><td valign=top align=right>"
                     + i18n( "Cc" )+": </td>" );
        text.append( "<td valign=top width=\"100%\">" + t + "</td></tr>" );
    }

    QString resentFrom = m_currentMessage->resentFrom().trimmed();
    if ( !resentFrom.isEmpty() ) {
        QString t = "<a href=\"email:"
                    + LinkLocator::convertToHtml( resentFrom, flags ) +"\">"
                    + LinkLocator::convertToHtml( resentFrom, flags ) + "</a>";

        QString t2;
        QString resentTo = m_currentMessage->resentTo().trimmed();
        if ( resentTo.isEmpty() )
            t2 = toList;
        else
            t2 = "<a href=\"email:"
                 + LinkLocator::convertToHtml( resentTo, flags ) +"\">"
                 + LinkLocator::convertToHtml( resentTo, flags ) + "</a>";

        text.append( "<tr><td valign=top align=right><nobr>" +
                     i18n( "Resent" )+": </nobr></td>" );
        text.append( "<td valign=top>"
                     + i18nc( "resent by from-address(1) to to-address(2)",
                              "by %1 to %2", t, t2 ) + "</td></tr>" );
    }

    QHash<KUrl,QString> list = m_currentMessage->attachments();
    QHash<KUrl,QString>::Iterator ita;
    if ( !list.isEmpty() ) {
        QString t;
        for ( ita = list.begin(); ita != list.end(); ++ita ) {
            if ( !t.isEmpty() )
                t.append( ", " );
            t.append( "<a href=\"attachment:"+ita.key().path()+"\">" +
                      Qt::escape( ita.value().trimmed() ) + "</a>" );

            if ( m_storeAttachments )
                autoSaveAttachment( ita.key(), ita.value() );
        }
        text.append( "<tr><td align=right valign=top>"
                     + i18n( "Attachments" )+": </td>" );
        text.append( "<td valign=top align=left>" + t + "</td></tr>" );
    }

    text.append( "</table>" );

    m_label->setText( text );
    m_label->adjustSize();

    if ( m_label->height()+6 < 100 )
        setMaximumHeight( m_label->height()+6 );
    else
        setMaximumHeight( 100 );

    if ( !hasPhoto )
        m_label->setFixedWidth( viewport()->width()-5 );
    else
        m_label->setFixedWidth( viewport()->width()-60 );
}

void MessageHeaderView::clearView()
{
    m_label->setText( "<table width=\"100%\" height=\"100%\">"
                      "<tr height=\"100%\"><td width=\"100%\">&nbsp;</td></tr></table>" );
    m_foto->setText( QString() );
    hide();
}

void MessageHeaderView::slotLeftMouseClick( const QString& to )
{
    kDebug();
    if ( !m_currentMessage )
        return;

    if ( to.startsWith( "email:" ) ) {
        Akonadi::Collection box;
        if ( m_currentMessage )
            box = m_currentMessage->collection();

        emit openComposer( box, to.mid( 6 ) );
    } else if ( to.startsWith( "attachment:" ) ) {
        QString attachment = to.mid( 11 );
        KRun *run = new KRun( attachment, this );
        run->setRunExecutables( false );
    }
}

void MessageHeaderView::slotRightMouseClick( const QString& to )
{
    kDebug() << to << endl;
    if ( to.startsWith( "email:" ) ) {
        QString address = to.mid( 6 );

        QMenu* p = new QMenu( this );
        QAction* newmsg = new QAction( i18n( "New Message to..." ), this );
        newmsg->setIcon( KIcon( "mail-message-new" ) );
        QAction* add = new QAction( i18n( "Add to KAddressBook..." ), this );
        add->setIcon( KIcon( "contact-new" ) );
        QAction* copy = new QAction( i18n( "Copy" ), this );
        copy->setIcon( KIcon( "edit-copy" ) );

        p->addAction( newmsg );
        p->addAction( add );
        p->addSeparator();
        p->addAction( copy );

        QAction* choice = p->exec( QCursor::pos() );

        if ( choice == add )
            Addressbook::instance()->add( address );
        else if ( choice == newmsg )
            slotLeftMouseClick( to );
        else if ( choice == copy ) {
            QClipboard *cb = QApplication::clipboard();
            cb->setText( address, QClipboard::Clipboard );
        }
    } else if ( to.startsWith( "attachment:" ) ) {
        QString attachment = to.mid( 11 );

        QMenu* p = new QMenu( this );
        QAction* open = new QAction( i18n( "Open" ), this );
        QAction* openwith = new QAction( i18n( "Open with..." ), this );
        QAction* save = new QAction( i18n( "Save..." ), this );
        QAction* saveAll = new QAction( i18n( "Save All..." ), this );

        p->addAction( open );
        p->addAction( openwith );
        p->addAction( save );
        p->addAction( saveAll );

        QAction* choice = p->exec( QCursor::pos() );

        KUrl url( attachment );
        if ( choice == open ) {
            KRun *run = new KRun( url, this );
            run->setRunExecutables( false );
        } else if ( choice == openwith )
            KRun::displayOpenWithDialog( KUrl::List::List( url ), this );
        else if ( choice == save )
            slotLeftMouseClick( to );
        else if ( choice == saveAll ) {
            KUrl base = KFileDialog::getExistingDirectoryUrl(
                            KUrl( "kfiledialog:///saveAttachmentsFolder" ), this );
            if ( !base.hasPath() )
                return;

            QHash<KUrl,QString> aMap = m_currentMessage->attachments();
            QHash<KUrl,QString>::Iterator ita;
            for ( ita = aMap.begin(); ita != aMap.end(); ++ita ) {
                KUrl destination( base );
                destination.addPath( ita.value() );
                getFreeFilename( destination );

                kDebug() << "copy: " << ita.key() << "->" << destination;

                KIO::NetAccess::file_copy( ita.key(), destination, this );
            }
        }
    }
}

// --------------------- Message -------------------------------

MessageBodyView::MessageBodyView( QWidget* parent )
        : KHTMLPart( parent ),
        m_currentMessage( 0 ),
        m_bodyType( MessageData::Plain ),
        m_fixedFont( false ),
        m_externalImage( false ),
        m_openDirectly( false )
{
    setOnlyLocalReferences( true );
    setJScriptEnabled( false );
    setJavaEnabled( false );
    setMetaRefreshEnabled( false );
    setPluginsEnabled( false );

    show();

    KConfigGroup config = KGlobal::config()->group( "Messageview" );
    setFontScaleFactor( config.readEntry( "zoomlevel", 80 ) );

    connect( this, SIGNAL( popupMenu( const QString&, const QPoint& ) ),
             SLOT( slotPopupMenu( const QString&, const QPoint& ) ) );
    connect( this, SIGNAL( setWindowCaption( const QString& ) ),
             SLOT( slotSetCaption( const QString& ) ) );
}

MessageBodyView::~MessageBodyView()
{
    KConfigGroup config = KGlobal::config()->group( "Messageview" );
    config.writeEntry( "zoomlevel", fontScaleFactor() );
    config.sync();
}

void MessageBodyView::slotSetCaption( const QString& title )
{
    emit pageTitle( title );
}

void MessageBodyView::setMsg( MessageData* msg )
{
    m_currentMessage = msg;
    m_currentMessage->requestBody( m_bodyType );
    m_openDirectly = false;
}


void MessageBodyView::slotShowMsg( const MessageData* msg )
{
    // kDebug() << m_currentMessage->body() << endl;
    if ( msg != m_currentMessage ) {
        kDebug() << "Message is gone...." << endl;
        return;
    }

    static_cast<QAbstractScrollArea*>( widget() )->verticalScrollBar()->setValue( 0 );
    static_cast<QAbstractScrollArea*>( widget() )->horizontalScrollBar()->setValue( 0 );

    begin();
    write( m_currentMessage->body().isEmpty() ? "&nbsp;" :
           m_currentMessage->body() );
    end();
}

void MessageBodyView::setHTML( bool i )
{
    i ? m_bodyType = MessageData::HTML : m_bodyType = MessageData::Plain;
    if ( m_currentMessage )
        m_currentMessage->requestBody( m_bodyType );
}

void MessageBodyView::setViewSource()
{
    m_bodyType = MessageData::Source;

    if ( m_currentMessage )
        m_currentMessage->requestBody( m_bodyType );
}

void MessageBodyView::setFixedFont( bool i )
{
    m_fixedFont = i;
    i ? setStandardFont( KGlobalSettings::fixedFont().family() )
    : setStandardFont( KGlobalSettings::generalFont().family() );

    // there is no update() call afaik
    if ( m_currentMessage ) {
        begin();
        write( m_currentMessage->body() );
        end();
    }
}

void MessageBodyView::setExternalImage( bool i, bool screenupdate )
{
    m_externalImage = i;
    setOnlyLocalReferences( !i );

    // there is no update() call afaik
    if ( m_currentMessage && screenupdate ) {
        begin();
        write( m_currentMessage->body() );
        end();
    }
}

void MessageBodyView::setOpenLinksDirectly( bool i )
{
    m_openDirectly = i;
}

bool MessageBodyView::urlSelected( const QString &url, int button, int,
                                   const QString&,
                                   const KParts::OpenUrlArguments&,
                                   const KParts::BrowserArguments& )
{
    // if the text of the link contains http:// or www., match that against the
    // actual url. If that does not contain it, we might have a case of
    // phishing. I know it's not waterproof, but it's a start.
    QString linkText = DOM::HTMLElement( nodeUnderMouse() ).innerText().string();
    QString urlHost  = KUrl( url ).host();
    if (( linkText.indexOf( "http" ) > -1 || linkText.indexOf( "www" ) > -1 ) &&
            linkText.indexOf( urlHost ) == -1 ) {
        kDebug() << "PHISHING alert" << endl;
        int i = KMessageBox::warningYesNo( 0,
                                           i18n( "You have clicked on a link which might not indicate "
                                                 "correctly where you are really going to. Please check "
                                                 "if you really want to view a page on this server:\n\n"
                                                 "%1\n\nDo you want to go there?", urlHost ),
                                           i18n( "Phishing Alert" ) );
        if ( i == KMessageBox::No )
            return false;
    }

    if ( url.startsWith( "mailto:" ) ) {
        Akonadi::Collection box;
        if ( m_currentMessage )
            box = m_currentMessage->collection();

        emit openComposer( box, url.mid( 7 ) );
    } else {
        if ( button == Qt::MidButton )
            emit openInTab( url );
        else if ( m_openDirectly )
            openUrl( url );
        else
            KToolInvocation::invokeBrowser( url );
    }
    return true;
}

void MessageBodyView::slotPopupMenu( const QString& url, const QPoint& point )
{
    QMenu* p = new QMenu( 0 );

    QAction* add    = new QAction( i18n( "Add to KAddressBook..." ), this );
    add->setIcon( KIcon( "contact-new" ) );
    QAction* copy   = new QAction( i18n( "Copy" ), this );
    add->setIcon( KIcon( "edit-copy" ) );
    QAction* newmsg = new QAction( i18n( "New Message to..." ), this );
    newmsg->setIcon( KIcon( "mail-message-new" ) );
    QAction* newurl = new QAction( i18n( "Open" ), this );
    QAction* newtab = new QAction( i18n( "Open in New Tab" ), this );
    QAction* zoomin = new QAction( i18n( "Zoom In" ), this );
    zoomin->setIcon( KIcon( "zoom-in" ) );
    QAction* zoomout = new QAction( i18n( "Zoom Out" ), this );
    zoomout->setIcon( KIcon( "zoom-out" ) );

    if ( url.startsWith( "mailto:" ) ) {
        p->addAction( newmsg );
        p->addAction( add );
    } else if ( !url.isNull() ) {
        p->addAction( newurl );
        p->addAction( newtab );
    }

    p->addSeparator();

    if ( hasSelection() || !url.isNull() ) {
        p->addAction( copy );
        p->addSeparator();
    }

    p->addAction( zoomin );
    p->addAction( zoomout );

    QAction* choice = p->exec( point );


    if ( choice == copy ) {
        QClipboard *cb = QApplication::clipboard();

        if ( hasSelection() ) {
            cb->setText( selectedText(), QClipboard::Clipboard );
        } else if ( url.startsWith( "mailto:" ) )
            cb->setText( url.mid( 7 ), QClipboard::Clipboard );
        else
            cb->setText( url, QClipboard::Clipboard );
    } else if ( choice == newmsg ) {
        Akonadi::Collection box;
        if ( m_currentMessage )
            box = m_currentMessage->collection();
        emit openComposer( box, url.mid( 7 ) );
    } else if ( choice == newurl )
        KToolInvocation::invokeBrowser( url );
    else if ( choice == add ) {
        Addressbook::instance()->add( url.mid( 7 ) );
    } else if ( choice == newtab )
        emit openInTab( url );
    else if ( choice == zoomin )
        increaseZoomLevel();
    else if ( choice == zoomout )
        decreaseZoomLevel();
    delete p;
}

void MessageBodyView::increaseZoomLevel()
{
    setFontScaleFactor( fontScaleFactor()+10 );
}

void MessageBodyView::decreaseZoomLevel()
{
    setFontScaleFactor( fontScaleFactor()-10 );
}

void MessageBodyView::clearView()
{
    begin();
    write( "&nbsp;" );
    end();
}

#include "messageview.moc"
