(*
    ||M||  This file is part of HELM, an Hypertextual, Electronic        
    ||A||  Library of Mathematics, developed at the Computer Science     
    ||T||  Department, University of Bologna, Italy.                     
    ||I||                                                                
    ||T||  HELM is free software; you can redistribute it and/or         
    ||A||  modify it under the terms of the GNU General Public License   
    \   /  version 2 or (at your option) any later version.      
     \ /   This software is distributed as is, NO WARRANTY.     
      V_______________________________________________________________ *)

(* $Id: nCic.ml 9058 2008-10-13 17:42:30Z tassi $ *)

open Printf

open DisambiguateTypes
open UriManager

module Ast = CicNotationPt
module NRef = NReference 

let debug_print _ = ();;
(* let debug_print s = prerr_endline (Lazy.force s);; *)

let cic_name_of_name = function
  | Ast.Ident (n, None) ->  n
  | _ -> assert false
;;

let refine_term 
 metasenv subst context uri ~coercion_db ~use_coercions term _ ~localization_tbl=
  assert (uri=None);
  debug_print (lazy (sprintf "TEST_INTERPRETATION: %s" 
    (NCicPp.ppterm ~metasenv ~subst ~context term)));
  try
    let localise t = 
      try NCicUntrusted.NCicHash.find localization_tbl t
      with Not_found -> 
        prerr_endline (NCicPp.ppterm ~metasenv ~subst ~context t);
        assert false
    in
    let metasenv, subst, term, _ = 
      NCicRefiner.typeof
        (NCicUnifHint.db ())
        ~look_for_coercion:(
          if use_coercions then 
           NCicCoercion.look_for_coercion coercion_db
          else (fun _ _ _ _ _ -> []))
        metasenv subst context term None ~localise 
    in
     Disambiguate.Ok (term, metasenv, subst, ())
  with
  | NCicRefiner.Uncertain loc_msg ->
      debug_print (lazy ("UNCERTAIN: [" ^ snd (Lazy.force loc_msg) ^ "] " ^ 
        NCicPp.ppterm ~metasenv ~subst ~context term)) ;
      Disambiguate.Uncertain loc_msg
  | NCicRefiner.RefineFailure loc_msg ->
      debug_print (lazy (sprintf "PRUNED:\nterm%s\nmessage:%s"
        (NCicPp.ppterm ~metasenv ~subst ~context term) (snd(Lazy.force loc_msg))));
      Disambiguate.Ko loc_msg
;;

  (* TODO move it to Cic *)
let find_in_context name context =
  let rec aux acc = function
    | [] -> raise Not_found
    | hd :: _ when hd = name -> acc
    | _ :: tl ->  aux (acc + 1) tl
  in
  aux 1 context

let interpretate_term_and_interpretate_term_option 
  ?(create_dummy_ids=false) 
    ~obj_context ~mk_choice ~env ~uri ~is_path ~localization_tbl 
=
  (* create_dummy_ids shouldbe used only for interpretating patterns *)
  assert (uri = None);

  let rec aux ~localize loc context = function
    | CicNotationPt.AttributedTerm (`Loc loc, term) ->
        let res = aux ~localize loc context term in
        if localize then 
         NCicUntrusted.NCicHash.add localization_tbl res loc;
       res
    | CicNotationPt.AttributedTerm (_, term) -> aux ~localize loc context term
    | CicNotationPt.Appl (CicNotationPt.Symbol (symb, i) :: args) ->
        let cic_args = List.map (aux ~localize loc context) args in
        Disambiguate.resolve ~mk_choice ~env (Symbol (symb, i)) (`Args cic_args)
    | CicNotationPt.Appl terms ->
       NCic.Appl (List.map (aux ~localize loc context) terms)
    | CicNotationPt.Binder (binder_kind, (var, typ), body) ->
        let cic_type = aux_option ~localize loc context `Type typ in
        let cic_name = cic_name_of_name var  in
        let cic_body = aux ~localize loc (cic_name :: context) body in
        (match binder_kind with
        | `Lambda -> NCic.Lambda (cic_name, cic_type, cic_body)
        | `Pi
        | `Forall -> NCic.Prod (cic_name, cic_type, cic_body)
        | `Exists ->
            Disambiguate.resolve ~env ~mk_choice (Symbol ("exists", 0))
              (`Args [ cic_type; NCic.Lambda (cic_name, cic_type, cic_body) ]))
    | CicNotationPt.Case (term, indty_ident, outtype, branches) ->
        let cic_term = aux ~localize loc context term in
        let cic_outtype = aux_option ~localize loc context `Term outtype in
        let do_branch ((_, _, args), term) =
         let rec do_branch' context = function
           | [] -> aux ~localize loc context term
           | (name, typ) :: tl ->
               let cic_name = cic_name_of_name name in
               let cic_body = do_branch' (cic_name :: context) tl in
               let typ =
                 match typ with
                 | None -> NCic.Implicit `Type
                 | Some typ -> aux ~localize loc context typ
               in
               NCic.Lambda (cic_name, typ, cic_body)
         in
          do_branch' context args
        in
        if create_dummy_ids then
         let branches =
          List.map
           (function
               Ast.Wildcard,term -> ("wildcard",None,[]), term
             | Ast.Pattern _,_ ->
                raise (DisambiguateTypes.Invalid_choice 
                 (lazy (loc, "Syntax error: the left hand side of a "^
                   "branch pattern must be \"_\"")))
           ) branches
         in
         (*
          NCic.MutCase (ref, cic_outtype, cic_term,
            (List.map do_branch branches))
          *) ignore branches; assert false (* patterns not implemented yet *)
        else
         let indtype_ref =
          match indty_ident with
          | Some (indty_ident, _) ->
             (match Disambiguate.resolve ~env ~mk_choice 
                (Id indty_ident) (`Args []) with
              | NCic.Const (NReference.Ref (_,NReference.Ind _) as r) -> r
              | NCic.Implicit _ ->
                 raise (Disambiguate.Try_again 
                  (lazy "The type of the term to be matched is still unknown"))
              | t ->
                raise (DisambiguateTypes.Invalid_choice 
                  (lazy (loc,"The type of the term to be matched "^
                          "is not (co)inductive: " ^ NCicPp.ppterm 
                          ~metasenv:[] ~subst:[] ~context:[] t))))
          | None ->
              let rec fst_constructor =
                function
                   (Ast.Pattern (head, _, _), _) :: _ -> head
                 | (Ast.Wildcard, _) :: tl -> fst_constructor tl
                 | [] -> raise (Invalid_choice (lazy (loc,"The type "^
                     "of the term to be matched cannot be determined "^
                     "because it is an inductive type without constructors "^
                     "or because all patterns use wildcards")))
              in
(*
              DisambiguateTypes.Environment.iter
                  (fun k v ->
                      prerr_endline
                        (DisambiguateTypes.string_of_domain_item k ^ " => " ^
                        description_of_alias v)) env; 
*)
              (match Disambiguate.resolve ~env ~mk_choice
                (Id (fst_constructor branches)) (`Args []) with
              | NCic.Const (NReference.Ref (_,NReference.Con _) as r) -> 
                   let b,_,_,_,_ = NCicEnvironment.get_checked_indtys r in
                   NReference.mk_indty b r
              | NCic.Implicit _ ->
                 raise (Disambiguate.Try_again 
                  (lazy "The type of the term to be matched is still unknown"))
              | t ->
                raise (DisambiguateTypes.Invalid_choice 
                  (lazy (loc, 
                  "The type of the term to be matched is not (co)inductive: " 
                  ^ NCicPp.ppterm ~metasenv:[] ~subst:[] ~context:[] t))))
         in
         let _,leftsno,itl,_,indtyp_no =
          NCicEnvironment.get_checked_indtys indtype_ref in
         let _,_,_,cl =
          try
           List.nth itl indtyp_no
          with _ -> assert false in
         let rec count_prod t =
                 match NCicReduction.whd ~subst:[] [] t with
               NCic.Prod (_, _, t) -> 1 + (count_prod t)
             | _ -> 0 
         in 
         let rec sort branches cl =
          match cl with
             [] ->
              let rec analyze unused unrecognized useless =
               function
                  [] ->
                   if unrecognized != [] then
                    raise (DisambiguateTypes.Invalid_choice
                     (lazy
                       (loc,"Unrecognized constructors: " ^
                        String.concat " " unrecognized)))
                   else if useless > 0 then
                    raise (DisambiguateTypes.Invalid_choice
                     (lazy
                       (loc,"The last " ^ string_of_int useless ^
                        "case" ^ if useless > 1 then "s are" else " is" ^
                        " unused")))
                   else
                    []
                | (Ast.Wildcard,_)::tl when not unused ->
                    analyze true unrecognized useless tl
                | (Ast.Pattern (head,_,_),_)::tl when not unused ->
                    analyze unused (head::unrecognized) useless tl
                | _::tl -> analyze unused unrecognized (useless + 1) tl
              in
               analyze false [] 0 branches
           | (_,name,ty)::cltl ->
              let rec find_and_remove =
               function
                  [] ->
                   raise
                    (DisambiguateTypes.Invalid_choice
                     (lazy (loc, "Missing case: " ^ name)))
                | ((Ast.Wildcard, _) as branch :: _) as branches ->
                    branch, branches
                | (Ast.Pattern (name',_,_),_) as branch :: tl
                   when name = name' ->
                    branch,tl
                | branch::tl ->
                   let found,rest = find_and_remove tl in
                    found, branch::rest
              in
               let branch,tl = find_and_remove branches in
               match branch with
                  Ast.Pattern (name,y,args),term ->
                   if List.length args = count_prod ty - leftsno then
                    ((name,y,args),term)::sort tl cltl
                   else
                    raise
                     (DisambiguateTypes.Invalid_choice
                      (lazy (loc,"Wrong number of arguments for " ^ name)))
                | Ast.Wildcard,term ->
                   let rec mk_lambdas =
                    function
                       0 -> term
                     | n ->
                        CicNotationPt.Binder
                         (`Lambda, (CicNotationPt.Ident ("_", None), None),
                           mk_lambdas (n - 1))
                   in
                    (("wildcard",None,[]),
                     mk_lambdas (count_prod ty - leftsno)) :: sort tl cltl
         in
          let branches = sort branches cl in
           NCic.Match (indtype_ref, cic_outtype, cic_term,
            (List.map do_branch branches))
    | CicNotationPt.Cast (t1, t2) ->
        let cic_t1 = aux ~localize loc context t1 in
        let cic_t2 = aux ~localize loc context t2 in
        NCic.LetIn ("_",cic_t2,cic_t1, NCic.Rel 1)
    | CicNotationPt.LetIn ((name, typ), def, body) ->
        let cic_def = aux ~localize loc context def in
        let cic_name = cic_name_of_name name in
        let cic_typ =
          match typ with
          | None -> NCic.Implicit `Type
          | Some t -> aux ~localize loc context t
        in
        let cic_body = aux ~localize loc (cic_name :: context) body in
        NCic.LetIn (cic_name, cic_typ, cic_def, cic_body)
    | CicNotationPt.LetRec (_kind, _defs, _body) -> assert false 
    | CicNotationPt.Ident _
    | CicNotationPt.Uri _ when is_path -> raise Disambiguate.PathNotWellFormed
    | CicNotationPt.Ident (name, subst) ->
       assert (subst = None);
       (try
         NCic.Rel (find_in_context name context)
       with Not_found -> 
         try NCic.Const (List.assoc name obj_context)
         with Not_found ->
           Disambiguate.resolve ~env ~mk_choice (Id name) (`Args []))
    | CicNotationPt.Uri (name, subst) ->
       assert (subst = None);
       (try
         NCic.Const (NRef.reference_of_string name)
        with NRef.IllFormedReference _ ->
         CicNotationPt.fail loc "Ill formed reference")
    | CicNotationPt.Implicit -> NCic.Implicit `Term
    | CicNotationPt.UserInput -> assert false (*NCic.Implicit (Some `Hole)
patterns not implemented *)
    | CicNotationPt.Num (num, i) -> 
        Disambiguate.resolve ~env ~mk_choice (Num i) (`Num_arg num)
    | CicNotationPt.Meta (index, subst) ->
        let cic_subst =
         List.map
          (function None -> assert false| Some t -> aux ~localize loc context t)
          subst
        in
         NCic.Meta (index, (0, NCic.Ctx cic_subst))
    | CicNotationPt.Sort `Prop -> NCic.Sort NCic.Prop
    | CicNotationPt.Sort `Set -> NCic.Sort (NCic.Type
       [false,NUri.uri_of_string "cic:/matita/pts/Type.univ"])
    | CicNotationPt.Sort (`Type _u) -> NCic.Sort (NCic.Type
       [false,NUri.uri_of_string "cic:/matita/pts/Type.univ"])
    | CicNotationPt.Sort (`NType s) -> NCic.Sort (NCic.Type
       [false,NUri.uri_of_string ("cic:/matita/pts/Type" ^ s ^ ".univ")])
    | CicNotationPt.Sort (`CProp _u) -> NCic.Sort (NCic.Type
       [false,NUri.uri_of_string "cic:/matita/pts/CProp.univ"])
    | CicNotationPt.Symbol (symbol, instance) ->
        Disambiguate.resolve ~env ~mk_choice 
         (Symbol (symbol, instance)) (`Args [])
    | CicNotationPt.Variable _
    | CicNotationPt.Magic _
    | CicNotationPt.Layout _
    | CicNotationPt.Literal _ -> assert false (* god bless Bologna *)
  and aux_option ~localize loc context annotation = function
    | None -> NCic.Implicit annotation
    | Some (CicNotationPt.AttributedTerm (`Loc loc, term)) ->
        let res = aux_option ~localize loc context annotation (Some term) in
        if localize then 
          NCicUntrusted.NCicHash.add localization_tbl res loc;
        res
    | Some (CicNotationPt.AttributedTerm (_, term)) ->
        aux_option ~localize loc context annotation (Some term)
    | Some CicNotationPt.Implicit -> NCic.Implicit annotation
    | Some term -> aux ~localize loc context term
  in
   (fun ~context -> aux ~localize:true HExtlib.dummy_floc context),
   (fun ~context -> aux_option ~localize:true HExtlib.dummy_floc context)
;;

let interpretate_term ?(create_dummy_ids=false) ~context ~env ~uri ~is_path ast
     ~obj_context ~localization_tbl ~mk_choice
=
  let context = List.map fst context in
  fst 
    (interpretate_term_and_interpretate_term_option 
      ~obj_context ~mk_choice ~create_dummy_ids ~env ~uri ~is_path ~localization_tbl)
    ~context ast
;;

let interpretate_term_option 
  ?(create_dummy_ids=false) ~context ~env ~uri ~is_path 
  ~localization_tbl ~mk_choice ~obj_context
=
  let context = List.map fst context in
  snd 
    (interpretate_term_and_interpretate_term_option 
      ~obj_context ~mk_choice ~create_dummy_ids ~env ~uri ~is_path ~localization_tbl)
    ~context 
;;

let new_flavour_of_flavour = function 
  | `Definition -> `Definition
  | `MutualDefinition -> `Definition 
  | `Fact -> `Fact
  | `Lemma -> `Lemma
  | `Remark -> `Corollary
  | `Theorem -> `Theorem
  | `Variant -> `Corollary 
  | `Axiom -> `Fact
;;

(*
let interpretate_obj ?(create_dummy_ids=false) ~context ~env ~uri ~is_path ast
     ~localization_tbl
=
 assert (context = []);
 assert (is_path = false);
 let interpretate_term ?(obj_context=[]) =
  interpretate_term ~mk_choice ~localization_tbl ~obj_context in
 let interpretate_term_option ?(obj_context=[]) =
   interpretate_term_option ~mk_choice ~localization_tbl ~obj_context in
 match obj with
 | CicNotationPt.Theorem (flavour, name, ty, bo) ->
     let attrs = `Provided, new_flavour_of_flavour flavour in
     let ty' = interpretate_term [] env None false ty in
     let height = (* XXX calculate *) 0 in
     uri, height, [], [], 
     (match bo,flavour with
        None,`Axiom ->
         NCic.Constant (name,None,ty',attrs)
      | Some bo,`Axiom -> assert false
      | None,_ ->
         NCic.Constant (name,NCic.Implicit None,ty',attrs)
      | Some bo,_ ->
         match bo with
         | CicNotationPt.LetRec (kind, defs, _) ->
             let inductive = kind = `Inductive in
             let obj_context =
               List.split 
                 (List.fold_left
                   (fun (i,acc) (_,(name,_),_,k) -> 
                    ((name, NReference.reference_of_spec uri 
                       (if inductive then NReference.Fix (i,k,0)
                        else NReference.CoFix i)) :: acc)
                   (0,[]) defs))
             in
             let inductiveFuns =
               List.map
                 (fun (params, (name, typ), body, decr_idx) ->
                   let add_binders kind t =
                    List.fold_right
                     (fun var t -> 
                        CicNotationPt.Binder (kind, var, t)) params t
                   in
                   let cic_body =
                     interpretate_term ~context ~env ~uri:None ~is_path:false
                       (add_binders `Lambda body) 
                   in
                   let cic_type =
                     interpretate_term_option ~context ~env ~uri:None
                       ~is_path:false `Type
                       (HExtlib.map_option (add_binders `Pi) typ)
                   in
                   (name, decr_idx, cic_type, cic_body))
                 defs
             in
             NCic.Fixpoint (inductive,inductiveFuns,attrs)
         | bo -> 
             let bo = 
               interpretate_term ~context:[] ~env ~uri:None ~is_path:false bo
             in
             NCic.Constant (name,Some bo,ty',attrs))
  | _ -> assert false
(*
  | CicNotationPt.Inductive (params,tyl) ->
     let uri = match uri with Some uri -> uri | None -> assert false in
     let context,params =
      let context,res =
       List.fold_left
        (fun (context,res) (name,t) ->
          let t =
           match t with
              None -> CicNotationPt.Implicit
            | Some t -> t in
          let name = CicNotationUtil.cic_name_of_name name in
           name::context,(name, interpretate_term context env None false t)::res
        ) ([],[]) params
      in
       context,List.rev res in
     let add_params =
      List.fold_right (fun (name,ty) t -> Cic.Prod (name,ty,t)) params in
     let obj_context =
      snd (
       List.fold_left
        (*here the explicit_named_substituion is assumed to be of length 0 *)
        (fun (i,res) (name,_,_,_) -> i + 1,(name,Cic.MutInd (uri,i,[]))::res)
        (0,[]) tyl) in
     let tyl =
      List.map
       (fun (name,b,ty,cl) ->
         let ty' = add_params (interpretate_term context env None false ty) in
         let cl' =
          List.map
           (fun (name,ty) ->
             let ty' =
              add_params
               (interpretate_term ~obj_context ~context ~env ~uri:None
                 ~is_path:false ty)
             in
              name,ty'
           ) cl
         in
          name,b,ty',cl'
       ) tyl
     in
      Cic.InductiveDefinition (tyl,[],List.length params,[])
  | CicNotationPt.Record (params,name,ty,fields) ->
     let uri = match uri with Some uri -> uri | None -> assert false in
     let context,params =
      let context,res =
       List.fold_left
        (fun (context,res) (name,t) ->
          let t =
           match t with
              None -> CicNotationPt.Implicit
            | Some t -> t in
          let name = CicNotationUtil.cic_name_of_name name in
           name::context,(name, interpretate_term context env None false t)::res
        ) ([],[]) params
      in
       context,List.rev res in
     let add_params =
      List.fold_right
       (fun (name,ty) t -> Cic.Prod (name,ty,t)) params in
     let ty' = add_params (interpretate_term context env None false ty) in
     let fields' =
      snd (
       List.fold_left
        (fun (context,res) (name,ty,_coercion,arity) ->
          let context' = Cic.Name name :: context in
           context',(name,interpretate_term context env None false ty)::res
        ) (context,[]) fields) in
     let concl =
      (*here the explicit_named_substituion is assumed to be of length 0 *)
      let mutind = Cic.MutInd (uri,0,[]) in
      if params = [] then mutind
      else
       Cic.Appl
        (mutind::CicUtil.mk_rels (List.length params) (List.length fields)) in
     let con =
      List.fold_left
       (fun t (name,ty) -> Cic.Prod (Cic.Name name,ty,t))
       concl fields' in
     let con' = add_params con in
     let tyl = [name,true,ty',["mk_" ^ name,con']] in
     let field_names = List.map (fun (x,_,y,z) -> x,y,z) fields in
      Cic.InductiveDefinition
       (tyl,[],List.length params,[`Class (`Record field_names)])
*)
;;
*)

let disambiguate_term ~context ~metasenv ~subst ?goal
   ~mk_implicit ~description_of_alias ~mk_choice
   ~aliases ~universe ~coercion_db ~lookup_in_library 
   (text,prefix_len,term) 
 =
  let mk_localization_tbl x = NCicUntrusted.NCicHash.create x in
  let hint =
   match goal with
      None -> (fun _ y -> y),(fun x -> x)
    | Some n ->
       (fun metasenv y ->
         let _,_,ty = NCicUtils.lookup_meta n metasenv in
          NCic.LetIn ("_",ty,y,NCic.Rel 1)),
       (function  
        | Disambiguate.Ok (t,m,s,ug) ->
            (match t with
            | NCic.LetIn ("_",_,y,NCic.Rel 1) -> Disambiguate.Ok (y,m,s,ug)
            | _ -> assert false)
        | k -> k)
  in
   let res,b =
    MultiPassDisambiguator.disambiguate_thing
     ~freshen_thing:CicNotationUtil.freshen_term
     ~context ~metasenv ~initial_ugraph:() ~aliases
     ~mk_implicit ~description_of_alias
     ~string_context_of_context:(List.map (fun (x,_) -> Some x))
     ~universe ~uri:None ~pp_thing:CicNotationPp.pp_term
     ~passes:(MultiPassDisambiguator.passes ())
     ~lookup_in_library ~domain_of_thing:Disambiguate.domain_of_term
     ~interpretate_thing:(interpretate_term ~obj_context:[] ~mk_choice (?create_dummy_ids:None))
     ~refine_thing:(refine_term ~coercion_db) (text,prefix_len,term)
     ~mk_localization_tbl ~hint ~subst
   in
    List.map (function (a,b,c,d,_) -> a,b,c,d) res, b
;;

let _ = 
let mk_type n = 
  if n = 0 then
     [false, NUri.uri_of_string ("cic:/matita/pts/Type.univ")]
  else
     [false, NUri.uri_of_string ("cic:/matita/pts/Type"^string_of_int n^".univ")]
in
let mk_cprop n = 
  if n = 0 then 
    [false, NUri.uri_of_string ("cic:/matita/pts/CProp.univ")]
  else
    [false, NUri.uri_of_string ("cic:/matita/pts/CProp"^string_of_int n^".univ")]
in
         NCicEnvironment.add_constraint true (mk_type 0) (mk_type 1);
         NCicEnvironment.add_constraint true (mk_cprop 0) (mk_cprop 1);
         NCicEnvironment.add_constraint true (mk_cprop 0) (mk_type 1);
         NCicEnvironment.add_constraint true (mk_type 0) (mk_cprop 1);
         NCicEnvironment.add_constraint false (mk_cprop 0) (mk_type 0);
         NCicEnvironment.add_constraint false (mk_type 0) (mk_cprop 0);

         NCicEnvironment.add_constraint true (mk_type 1) (mk_type 2);
         NCicEnvironment.add_constraint true (mk_cprop 1) (mk_cprop 2);
         NCicEnvironment.add_constraint true (mk_cprop 1) (mk_type 2);
         NCicEnvironment.add_constraint true (mk_type 1) (mk_cprop 2);
         NCicEnvironment.add_constraint false (mk_cprop 1) (mk_type 1);
         NCicEnvironment.add_constraint false (mk_type 1) (mk_cprop 1);

         NCicEnvironment.add_constraint true (mk_type 2) (mk_type 3);
         NCicEnvironment.add_constraint true (mk_cprop 2) (mk_cprop 3);
         NCicEnvironment.add_constraint true (mk_cprop 2) (mk_type 3);
         NCicEnvironment.add_constraint true (mk_type 2) (mk_cprop 3);
         NCicEnvironment.add_constraint false (mk_cprop 2) (mk_type 2);
         NCicEnvironment.add_constraint false (mk_type 2) (mk_cprop 2);

         NCicEnvironment.add_constraint false (mk_cprop 3) (mk_type 3);
         NCicEnvironment.add_constraint false (mk_type 3) (mk_cprop 3);

;;

