#!/usr/bin/tclsh
# Part of MCU 8051 IDE ( http://mcu8051ide.sf.net )

############################################################################
#    Copyright (C) 2007-2009 by Martin Ošmera                              #
#    martin.osmera@gmail.com                                               #
#                                                                          #
#    This program is free software; you can redistribute it and#or modify  #
#    it under the terms of the GNU General Public License as published by  #
#    the Free Software Foundation; either version 2 of the License, or     #
#    (at your option) any later version.                                   #
#                                                                          #
#    This program is distributed in the hope that it will be useful,       #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#    GNU General Public License for more details.                          #
#                                                                          #
#    You should have received a copy of the GNU General Public License     #
#    along with this program; if not, write to the                         #
#    Free Software Foundation, Inc.,                                       #
#    59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             #
############################################################################


# --------------------------------------------------------------------------
# DESCRIPTION
# Initilizator of the program
# --------------------------------------------------------------------------

# GENERAL CONSTANTS
# -----------------------------
encoding system {utf-8}					;# System encoding
set LIB_DIRNAME	[file normalize [file dirname $argv0]]	;# Path to directory where the *.tcl file are located
set VERSION	"1.3.7"					;# Program version
set SHORTNAME	"MCU8051IDE"				;# Program short name (without white space)
set APPNAME	"MCU 8051 IDE v$VERSION"		;# Full program name
set MIN_TCL_VER	"8.5"					;# Minimum required Tcl version
set APPLICATION_LOADED	0				;# True if program is loaded
set TRANSLATION_LOADED	0				;# Bool: Translation loaded
set MICROSOFT_WINDOWS	{}				;# Bool: Windows OS running on the host computer
set CONFIG_DIR		{}				;# Directory containing configuration files

# Check for correct Tcl version
if {[package vcompare $::tcl_version $::MIN_TCL_VER] < 0} {
	puts stderr "ERROR: This program requires Tcl version $::MIN_TCL_VER or higher but you have Tcl $::tcl_version ."
	puts stderr "       Please install Tcl $::MIN_TCL_VER in order to run this program"

	exit 1
}

## Tcl packages used by this prgram
 # Format: {
 # 	{pkg_name	pkg_verison}
 # 	...
 # }
set LIBRARIES_TO_LOAD {
	{BWidget	1.7}
	{Itcl		3.4}
	{Tcl		8.2}
	{md5		2.0}
	{Tk		8.5}
	{img::png	1.3}
	{tdom		0.8}
	{Tclx		8.0}
}

## Bool:
 #     1 == library TclX is avaliable
 #     0 == library TclX is NOT avaliable
 #
 # TclX is used here only to handle signals (e.g. SIGINT), so the IDE can run
 # without it, that's the reason for this variable. If TCLX_AVAILABLE is 0 then
 # we are not able to handle signals, but everything else works normally.
set ::TCLX_AVAILABLE 1

## Determinate the host OS
set ::MICROSOFT_WINDOWS 0
if {[string first {Windows} ${tcl_platform(os)}] != -1} {
	# Note:
	#   Microsoft Windows is NOT a POSIX system and because of that we need
	#   to do some workarounds here in order to make the IDE functional there.
	set ::MICROSOFT_WINDOWS 1
}

# Set directory containing configuration files according to the host OS
if {!$::MICROSOFT_WINDOWS} {
	set CONFIG_DIR [file join $::env(HOME) .[string tolower ${::SHORTNAME}]]
} {
	set CONFIG_DIR [file join ${::env(USERPROFILE)} ".[string tolower ${::SHORTNAME}]"]
}

# Handle CLI options
# -----------------------------
proc mc args {return [eval "format $args"]}
source "${::LIB_DIRNAME}/cli.tcl"

# SHOW WARNING MESSAGE
# -----------------------------

if {!$::CLI_OPTION(quiet)} {
	if {$::CLI_OPTION(nocolor)} {
		puts "WARNING:"
		puts "\tThis program is distributed with ABSOLUTELY NO WARRANTY !"
		puts "\tPlease report bugs at http://mcu8051ide.sf.net"
		puts "\tAuthor: Martin Osmera <martin.osmera@gmail.com>"
	} {
		puts "WARNING:"
		puts "\tThis program is distributed with ABSOLUTELY NO WARRANTY !"
		puts "\tPlease report bugs at \033\[34;1mhttp://mcu8051ide.sf.net\033\[m"
		puts "\tAuthor: Martin Osmera \033\[33;1m<martin.osmera@gmail.com>\033\[m"
	}
}

## This function should be called when some Tcl library fail to load
 # @parm String library - Name of failed library
 # @return void
proc libraryLoadFailed {library} {

	# Itcl workarond for Debian
	if {$library == {Itcl}} {
		set library_version "3.4"
		set libname "libitcl"

		set ::env(ITCL_LIBRARY) ${::LIB_DIRNAME}

		puts stderr "\nERROR: Unable to load Itcl library compatible with this version of Tcl/Tk !"
		puts stderr "Trying to workaround ..."

		if {[lsearch {Linux} ${::tcl_platform(os)}] == -1} {
			puts stderr "FATAL ERROR: Unsupported operating system. ${::tcl_platform(os)}"
			puts stderr "You can contact author of the project at <martin.osmera@gmail.com> if you want to get you OS supported."
			exit 1
		}

		if {[lsearch {x86_64 i386 i486 i586 i686 x86} ${::tcl_platform(machine)}] == -1} {
			puts stderr "FATAL ERROR: Unsupported system architecture. ${::tcl_platform(machine)}"
			puts stderr "You can contact author of the project at <martin.osmera@gmail.com> if you want to get you OS supported."
			exit 1
		}

		puts stderr "Loading library $library for ${::tcl_platform(os)} on ${::tcl_platform(machine)} ... (filename: ${libname}${library_version}.so.${::tcl_platform(os)}.${::tcl_platform(machine)})"
		if {[catch {load "${::LIB_DIRNAME}/${libname}${library_version}.so.${::tcl_platform(os)}.${::tcl_platform(machine)}" Itcl} error_info]} {
			puts stderr "FAILED !"
			puts stderr "Reason: ${error_info}"
			puts "\nPlease try to run mcu8051ide with --check-libraries to see what's wrong."

			exit 1
		} {
			puts stderr "WORKAROUND SUCCESSFULL ... \n(But don't be much happy about this, it is still serious failure. And please don't forget to comply to developers of your Linux distribution. Missing library is: ${library} version ${library_version})"
			return
		}

	# Tclx workarond for Debian
	} elseif {$library == {Tclx}} {
		set ::TCLX_AVAILABLE 0
		puts stderr "\nERROR: Unable to load Tclx library, functionality will be limited"
		return
	}

	# Print error message
	if {$::CLI_OPTION(nocolor)} {
		puts stderr "\n\nERROR: Unable to load library $library"
	} {
		puts stderr "\n\n\033\[31mERROR:\033\[m Unable to load library \033\[32m$library\033\[m"
	}

	# Print tip
	puts "\nTip: try to run mcu8051ide with --check-libraries to see what's wrong."

	# Terminate the program
	exit 1
}

# PRE-INITIALIZATION
# -----------------------------
# Load Tk ToolKit
set T [lindex [time {
	if {[catch {package require img::png 1.3}]} {
		libraryLoadFailed "img::png"
	}
	if {[catch {package require Tk $::MIN_TCL_VER} errinfo]} {
		puts stderr "Unable to initialize Tk\n$errinfo"
	}
}] 0]
# Hide main window
wm withdraw .
update

# Determinate default Fixed font
set ::DEFAULT_FIXED_FONT {DejaVu Sans Mono}
if {[lsearch -ascii -exact [font families] {DejaVu Sans Mono}] == -1} {
	set ::DEFAULT_FIXED_FONT {Courier}
}

# ------------------------------------------------------------------------------
# Microsoft Windows OS specific code
# ------------------------------------------------------------------------------
if {$::MICROSOFT_WINDOWS} {
	# Print windows related warning
	puts ""
	puts "        THE IDE WAS ORIGINALY DESIGNED FOR POSIX, SO IT IS POSSIBLE THAT SOME"
	puts "        FUNCTIONALITY WILL BE LIMITED ON YOUR OS DUE TO ABSENCE OF CERTAIN"
	puts "        POSIX FUNCTIONALITY !"
	puts ""

	# Redefine the "bind" command, because on Windows binding a callback to
	# event, which does not exist on Windows would cause program crash
	rename bind original_command_bind
	proc microsoft_windows_specific_bind args {
		if {
			[string first {ISO_} [lindex $args 1]] != -1	||
			[string first {XF86_} [lindex $args 1]] != -1
		} {
			return
		}

		eval "original_command_bind $args"
	}
	rename microsoft_windows_specific_bind bind
}
# ------------------------------------------------------------------------------


# Load base config file
# -----------------------------

# Load i18n library
# (It must be loaded here because ::msgcat::mclocale must be available when
# base config file is being loaded)
incr T [lindex [time {
	if {[catch {package require msgcat 1.3.4}]} {
		libraryLoadFailed "msgcat"
	} {
		namespace import -force ::msgcat::*
	}
}] 0]
# Check if the file exits
if {![file exists ${::CONFIG_DIR}]} {
	file mkdir ${::CONFIG_DIR}
	puts "\nCreating program configuration files in directory: \"[file normalize ${::CONFIG_DIR}]\""
	if {!$::MICROSOFT_WINDOWS} {
		puts "Welcome in this IDE, [file tail [file normalize ~]] !"
	} {
		catch {	;# Make the configuration directory in Microsoft Windows hidden
			file attributes $::CONFIG_DIR -hidden 1
		}
		puts "Welcome in this IDE, ${::env(USERNAME)} !"
	}
}
## Open and read the file
if {[catch {
	set conf_file [open "${::CONFIG_DIR}/base.conf" r]
 # File doesn't exits -> create it with default configuration
}]} then {
	# Default settings
	array set GLOBAL_CONFIG [list			\
		splash		1			\
		tips		1			\
		language	[::msgcat::mclocale]	\
	]

	# Create the file
	if {[catch {
		set conf_file [open "${::CONFIG_DIR}/base.conf" w]
		puts -nonewline $conf_file [list	\
			$GLOBAL_CONFIG(splash)		\
			$GLOBAL_CONFIG(splash)		\
			$GLOBAL_CONFIG(language)	\
		]
		close $conf_file
	}]} {
		puts stderr "Unable to create base configuration file"
	}
 # File exits -> read configuration from it
} else {
	# Read file contents
	set data [read $conf_file]
	close $conf_file

	# Set configuration acording to the file contents
	set GLOBAL_CONFIG(splash)	[lindex $data 0]
	set GLOBAL_CONFIG(tips)		[lindex $data 1]
	set GLOBAL_CONFIG(language)	[lindex $data 2]

	## Validate read values
	if {![string is boolean -strict ${GLOBAL_CONFIG(splash)}]} {
		set GLOBAL_CONFIG(splash) 1
	}
	if {![string is boolean -strict ${GLOBAL_CONFIG(tips)}]} {
		set GLOBAL_CONFIG(tips) 1
	}
	 # Check if the cpecified translation is valid
	set tmp [list]
	catch {	;# For Microsoft Windows it has to be enclosed by catch
		set tmp [glob -nocomplain -types f -tails			\
			-directory "${LIB_DIRNAME}/../translations" *.msg	\
		]
	}
	set translations {en}
	foreach translation $tmp {
		lappend translations [file rootname $translation]
	}
	if {[lsearch $translations ${GLOBAL_CONFIG(language)}] == -1} {
		set GLOBAL_CONFIG(language) {en}
	}
}

# Load translation
# -----------------------------

# Load language specific translation file
if {!${::CLI_OPTION(notranslation)} && ${GLOBAL_CONFIG(language)} != {en}} {
	if {[catch {
		mclocale ${GLOBAL_CONFIG(language)}
		mcload "${LIB_DIRNAME}/../translations/"

	} result]} then {
		puts stderr "Unable to load translation"
		puts stderr "\tFile: '${LIB_DIRNAME}/../translations/${GLOBAL_CONFIG(language)}.msg'"
		puts "\n"
		puts $result

	} else {
		set ::TRANSLATION_LOADED 1
	}
}

# CREATE SPLASH SCREEN
# -----------------------------
if {!$::CLI_OPTION(nosplash) && ${::GLOBAL_CONFIG(splash)}} {

	# Crete toplevel  window
	toplevel .splash -class {Splash creen} -bg {#EEEEEE}

	# Show image of splash creen
	place [label .splash.bg	\
		-image [	\
			image create photo -format png	\
				-file "${::LIB_DIRNAME}/../icons/other/splash.png"	\
			]	\
	] -x 0 -y 0 -width 400 -height 199

	# Show status bar
	place [label .splash.status		\
		-bg {#FFFFFF} -fg {#0000FF}	\
		-text [mc "Initializing %s" $APPNAME]	\
	] -x 200 -y 180 -anchor center

	# Set window parameters
	wm geometry .splash "=400x199+[expr {[winfo screenwidth .] / 2 - 200}]+[expr {[winfo screenheight .] / 2 - 100}]"
	wm overrideredirect .splash 1

	# Click on splash creen destroys it
	bind .splash <1> {wm withdraw .splash}

	# Done ..
	update
}


# BASIC FUNCTIONS
# -----------------------------

## Print content of $T in mili seconds ($T must contain value in [us])
 # @return void
proc time_in_msec {} {
	global T	;# Time in microseconds

	# Determinate number of miliseconds
	set msec [lindex $T 0]
	set msec [expr {$msec / 1000}]

	# print the message
	if {!$::CLI_OPTION(quiet)} {
		if {$::CLI_OPTION(nocolor)} {
			puts "... $msec ms"
		} {
			puts "... \033\[33m$msec ms\033\[m"
		}
	}
}

## Print some initialization message (splash screen and CLI)
 # @parm String message - text of the message
 # @return void
proc showInitMessage {message} {

	# Change content of splash screen status bar
	if {!${::CLI_OPTION(nosplash)} && ${::GLOBAL_CONFIG(splash)}} {
		if {[winfo exists .splash.status]} {
			.splash.status configure -text [string trim $message]
			update
		}
	}

	# Print message to console output
	if {!${::CLI_OPTION(quiet)}} {
		if {${::CLI_OPTION(nocolor)}} {
			puts -nonewline $message
		} {
			puts -nonewline "\033\[37m$message\033\[m"
		}

		puts -nonewline [string repeat { } [expr {38 - [string length $message]}]]
		flush stdout
	}
}

## Set status bar tip for some widget
 # Usage:
 #	setStatusTip -widget $some_widget -text "some text"
 #
 # @return void
proc setStatusTip args {

	# Local variables
	set widgetIsSet	0		;# True if widget is set
	set textIsSet	0		;# True if text is set
	set argsLength	[llength $args]	;# Number of arguments
	set widget	{}		;# ID of widget specified by argument '-widget'
	set helpText	{}		;# Help text specified by argument '-text'

	# Iterate over given arguments and evaluate them
	for {set i 0} {$i < $argsLength} {incr i} {
		# Currently parsed argument
		set arg [lindex $args $i]
		# Decide what $arg means
		switch -- $arg {
			-widget {	;# ID of the widget

				# check if that widget wasn't already specified
				if {$widgetIsSet} {
					error "Widget has been already specified"
				}

				# Check if widget's ID follow the arument
				incr i
				if {$i >= $argsLength} {
					error "Expected widget name after -widget option"
				}

				# Set ID of the widget
				set widget [lindex $args $i]
				if {![winfo exists $widget]} {
					error "The specified widget does not exist"
				}

				# Widget is now set
				set widgetIsSet 1
			}

			-text {		;# The help text

				# Check if help text follow the argument
				incr i
				if {$i >= $argsLength} {
					error "Expected text after -text option"
				}

				# Set the help text
				set helpText [lindex $args $i]

				# Help text is now set
				set textIsSet 1
			}

			default {	;# Unrecognized opton -> invoke ERROR
				error "Invalid argument '$arg', possible options are -widget and -text"
			}
		}
	}

	# Ckeck if both aruments are properly specified
	if {!$widgetIsSet || !$textIsSet} {
		error "You must specify text and widget"
	}

	# Create binding
	bind $widget <Enter> "Sbar -freeze {$helpText}"
	bind $widget <Leave> "Sbar {}"
}



# INITIALIZATION
# -----------------------------

# Show "first line message"
if {!$CLI_OPTION(quiet)} {
	if {$CLI_OPTION(nocolor)} {
		puts [mc "\nInitializing MCU 8051 IDE %s" $VERSION]
	} {
		puts [mc "\nInitializing \033\[1mMCU 8051 IDE \033\[32m%s\033\[m" $VERSION]
	}
}

## Load libraries
showInitMessage [mc "\tLoading libraries"]
incr T [lindex [time {
	# Iterate over list of libraries and lod each of them
	foreach library $::LIBRARIES_TO_LOAD {
		# Loading successful
		if {[catch {package require [lindex $library 0] [lindex $library 1]}]} {
			libraryLoadFailed [lindex $library 0]
		# Loading failed
		} {
			if {!$::CLI_OPTION(nosplash)} update
		}
	}

	if {$::MICROSOFT_WINDOWS} { ;# Load dde - Dynamic Data Exchange on Microsoft Windows
		package require dde
	}

	# Import NS for managing OOP in Tcl
	namespace import -force ::itcl::*
}] 0]

# Look for some external programs
foreach program {
		urxvt	vim	emacs	kwrite	gedit	nano	dav
		le	sdcc	indent	doxygen	asl	asem	doxywizard
		as31	sdcc-sdcc	gvim
	} {
		if {[auto_execok $program] == {}} {
			set ::PROGRAM_AVALIABLE($program) 0
		} {
			set ::PROGRAM_AVALIABLE($program) 1
		}
}
time_in_msec	;# Print time info

## Load program sources
showInitMessage [mc "\tLoading program sources"]
set T [time {
	source "${::LIB_DIRNAME}/lib/innerwindow.tcl"		;# Tool for creating inner windows
	source "${::LIB_DIRNAME}/dialogs/errorhandler.tcl"	;# Background error handler
	source "${::LIB_DIRNAME}/dialogs/my_tk_messageBox.tcl"	;# A replacement for tk_messageBox
	source "${::LIB_DIRNAME}/lib/settings.tcl"		;# Settings management
	source "${::LIB_DIRNAME}/project.tcl"			;# Project management
	source "${::LIB_DIRNAME}/dialogs/fsd.tcl"		;# File selection dialog
	source "${::LIB_DIRNAME}/X.tcl"				;# GUI <==> Implementation Interface
	source "${::LIB_DIRNAME}/configdialogs/configdialogs.tcl";# Configuration dialogs
	source "${::LIB_DIRNAME}/editor/editor.tcl"		;# Source code editor
	source "${::LIB_DIRNAME}/lib/Math.tcl"			;# Special mathematical operations
	source "${::LIB_DIRNAME}/compiler/compiler.tcl"		;# 8051 Assemly language compiler
	source "${::LIB_DIRNAME}/dialogs/tips.tcl"		;# Tips on startup
	source "${::LIB_DIRNAME}/lib/hexeditor.tcl"		;# Hexadecimal editor
	source "${::LIB_DIRNAME}/utilities/hexeditdlg.tcl"	;# Hexadecimal editor dialog
	source "${::LIB_DIRNAME}/environment.tcl"		;# Main window "trappings" (menu and such)
	source "${::LIB_DIRNAME}/rightpanel/rightpanel.tcl"	;# Right panel
	source "${::LIB_DIRNAME}/leftpanel/filelist.tcl"	;# Left and middle panel
	source "${::LIB_DIRNAME}/simulator/simulator.tcl"	;# MCU Simulator
	source "${::LIB_DIRNAME}/bottompanel/bottomnotebook.tcl";# Bottom panel
	source "${::LIB_DIRNAME}/maintab.tcl"			;# Central widget
	source "${::LIB_DIRNAME}/lib/ihextools.tcl"		;# Tools for manipulating Intel 8 HEX
	source "${::LIB_DIRNAME}/utilities/symbol_viewer.tcl"	;# Assembly symbols viewer
	source "${::LIB_DIRNAME}/utilities/eightsegment.tcl"	;# 8-segment LED display editor
	source "${::LIB_DIRNAME}/utilities/asciichart.tcl"	;# ASCII chart
	source "${::LIB_DIRNAME}/utilities/notes.tcl"		;# Scribble notepad
	source "${::LIB_DIRNAME}/utilities/baseconvertor.tcl"	;# Base convertor
	source "${::LIB_DIRNAME}/utilities/speccalc.tcl"	;# Special calculator for x51 MCU's
	source "${::LIB_DIRNAME}/utilities/rs232debugger.tcl"	;# UART/RS232 applications debugger
}]
time_in_msec	;# Print time info

# CHECK FOR VALIDITY OF THE MCU DATABASE
if {${::X::avaliable_processors} == {}} {
	destroy .splash
	bell
	tk_messageBox		\
		-icon error	\
		-type ok	\
		-title [mc "FATAL ERROR"]	\
		-message [mc "MCUs database file is currupted,\nthis program cannot run without it.\nPlease reinstall MCU 8051 IDE."]
	exit 1
}

# Load global configuration
loadApplicationConfiguration

# Initialize GUI environment
iconbar_redraw		;# Main toolbar
mainmenu_redraw		;# Main menu
shortcuts_reevaluate	;# Key shortcuts

## Remove splash screen
destroy .splash
if {$CLI_OPTION(minimalized)} {
	wm state . iconic
} {
	wm deiconify .
}

# Configure signal handling
if {$::TCLX_AVAILABLE} {
	proc signal_handler {signal_name} {
		global cntrlc_flag
		puts stdout [mc "\nExiting on signal %s" $signal_name]
		if {[catch {::X::__exit 1}]} {
			exit 1
		}
	}

	signal trap SIGINT	{signal_handler SIGINT}
	signal trap SIGTERM	{signal_handler SIGTERM}
}


# ---------------------------------------------------------
# Ugly job ... dirty workarounds and such a shit ... :(
# ---------------------------------------------------------

catch {
	NoteBook .foo
	proc NoteBook::_getoption {path page option} {
		if {$option == {-background}} {
			return {#eeeeee}
		}

		set value [Widget::cget $path.f$page $option]
		if {![string length $value]} {
			set value [Widget::cget $path $option]
		}

		return $value
	}
	destroy .foo
}

# ---------------------------------------------------------


## Open the last session
# Print message
showInitMessage [mc "\tOpening last session"]
flush stdout
# Evaluate new geometry of the main window
update
evaluate_new_window_geometry
# Open projects of last session
set T [time {
	foreach project_file $::CONFIG(OPENED_PROJECTS) {
		if {![Project::open_project_file $project_file]} {
			tk_messageBox			\
				-title [mc "File not found"]	\
				-icon warning		\
				-type ok		\
				-message [mc "Unable to open project file:\n\"%s\"" $project_file]
		}
	}
}]

# Reopen base convertors
foreach cfg $::CONFIG(BASE_CONVERTORS) {
	if {[catch {
		set obj [::X::__base_convertor]
		::X::$obj set_config $cfg
	}]} {
		puts stderr {}
		puts stderr $::errorInfo
	}
}

time_in_msec	;# Print time info

# Create binding for panes management
bind . <Configure> {X::redraw_panes}
# Print final message
if {!$CLI_OPTION(quiet)} {
	puts [mc "%s is now operational\n" $APPNAME]
}

# Program is now operational
set ::Compiler::in_IDE 1
set APPLICATION_LOADED 1
set X::critical_procedure_in_progress 0
update idle
X::redraw_panes
foreach project ${::X::openedProjects} {
	$project filelist_adjust_size_of_tabbar
	$project ensure_that_both_editors_are_properly_initialized
}
# Focus on the active editor
if {${::X::actualProject} != {}} {
	update
	focus [${::X::actualProject} editor_procedure {} cget -editor]
	focus [${::X::actualProject} editor_procedure {} Configure {}]
	focus [${::X::actualProject} editor_procedure {} highlight_visible_area {}]
}

# Just workaround for disapearing main menu
raise .

# Correct strange behavior concerning restoration of the last window size and position
if {$::MICROSOFT_WINDOWS} {
	update
	wm geometry . $::CONFIG(WINDOW_GEOMETRY)
}
