/*******************************************************************************
 * Copyright (C) 2004-2007 Intel Corp. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *   - Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 * 
 *   - Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 * 
 *   - Neither the name of Intel Corp. nor the names of its
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL Intel Corp. OR THE CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *******************************************************************************/

#ifndef _IPC_BUFFER_H
#define _IPC_BUFFER_H

//*****************************************************************************
//
// Class:			IPCBuffer
// Description:	    IPCBuffer is a class that provides a section of named shared
//                  memory, to be shared between processes. The bIsOwner
//                  parameter of the ctor determines whether or not the object is
//                  the one that creates the shared memory (server), or just uses
//                  it (client) after it was created. Use isOk() and lastError()
//                  after construction to check for errors.
//                  The object may also me secure, in which case only the owner of the
//                  buffer can change it. To make the object secure, set the
//                  bSecure parameter of the ctor to true.
//*****************************************************************************

#ifdef _WIN32
#include <windows.h> //for HANDLE
#else
#include <sys/types.h> //for key_t
#endif

class IPCBuffer
{
public:
	static const unsigned long IPC_BUF_ACCESS_READ;
	static const unsigned long IPC_BUF_ACCESS_WRITE;

	//default ctor: must use init() afterwards
	IPCBuffer();

	//ctor
	IPCBuffer(const char* pName, bool bIsOwner, unsigned int nSize, bool bSecure,
		unsigned long nAccess = IPC_BUF_ACCESS_READ | IPC_BUF_ACCESS_WRITE);

	//initialize this buffer after it was created.
	bool init(const char* pName, bool bIsOwner, unsigned int nSize, bool bSecure,
		unsigned long nAccess = IPC_BUF_ACCESS_READ | IPC_BUF_ACCESS_WRITE);

	//free resources allocated by IPC Buffer
	void close();

	//dtor
	~IPCBuffer();

#ifndef _WIN32
	//connect to shared memory
	bool shmConnect();

	//disconnect from shared memory
	void shmDisconnect();
#endif

	//get name of this IPCBuffer
	const char* name() const {return m_pName;}

	//returns true if this object is the owner of the shared memory
	bool isOwner() const {return m_bIsOwner;}

	//safely read data from the buffer
	bool read(unsigned int nOffset, void *pDest, unsigned int nSize) const;

	//safely write data to the buffer
	bool write(unsigned int nOffset, const void *pSource, unsigned int nSize);

	//returns the size of the buffer
	unsigned int size() const {return m_nSize;}

	//returns the code of the last error that occurred in this object
	unsigned long lastError() const {return m_nLastError;}

	//returns true iff this object had no errors since it was created
	bool isOk() const {return (m_nLastError == SUCCESS);}

private:
	static const unsigned int SUCCESS;
	void reset();

	const char* m_pName;
	bool m_bIsOwner;
	char *m_pData;
	unsigned int m_nSize;
	unsigned long m_nLastError;
	unsigned long m_nAccess;
#ifdef _WIN32
	HANDLE CreateSecurableFileMapping(bool bSecure);
	HANDLE m_hMapFile;
#else
	int m_shmid;
	key_t m_key;
	int m_shmget_flag;
	int m_shmat_flag;
	static const char PROJ;
#define SHM_MARKER_RESERVED_BYTES 8
	char m_marker[SHM_MARKER_RESERVED_BYTES];
#endif
};

#endif //_IPC_BUFFER_H
